"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const test_utils_1 = require("@joplin/lib/testing/test-utils");
const testUtils_1 = require("./utils/testUtils");
const Folder_1 = require("@joplin/lib/models/Folder");
const Note_1 = require("@joplin/lib/models/Note");
const Command = require('./command-rmbook');
const setUpCommand = () => {
    const command = (0, testUtils_1.setupCommandForTesting)(Command);
    const promptMock = jest.fn(() => true);
    command.setPrompt(promptMock);
    return { command, promptMock };
};
describe('command-rmbook', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await (0, testUtils_1.setupApplication)();
    });
    it('should ask before moving to the trash', async () => {
        await Folder_1.default.save({ title: 'folder1' });
        const { command, promptMock } = setUpCommand();
        await command.action({ 'notebook': 'folder1', options: {} });
        expect(promptMock).toHaveBeenCalledTimes(1);
        const folder1 = await Folder_1.default.loadByTitle('folder1');
        expect(folder1.deleted_time).not.toBeFalsy();
        expect((await Note_1.default.allItemsInTrash()).folderIds).toHaveLength(1);
    });
    it('cancelling a prompt should prevent deletion', async () => {
        await Folder_1.default.save({ title: 'folder1' });
        const { command, promptMock } = setUpCommand();
        promptMock.mockImplementation(() => false);
        await command.action({ 'notebook': 'folder1', options: {} });
        expect((await Note_1.default.allItemsInTrash()).folderIds).toHaveLength(0);
    });
    it('should not prompt when the force flag is given', async () => {
        const { id: folder1Id } = await Folder_1.default.save({ title: 'folder1' });
        const { id: folder2Id } = await Folder_1.default.save({ title: 'folder2', parent_id: folder1Id });
        const { command, promptMock } = setUpCommand();
        await command.action({ 'notebook': 'folder1', options: { force: true } });
        expect(promptMock).toHaveBeenCalledTimes(0);
        expect((await Note_1.default.allItemsInTrash()).folderIds.includes(folder1Id)).toBe(true);
        expect((await Note_1.default.allItemsInTrash()).folderIds.includes(folder2Id)).toBe(true);
    });
    it('should support permanent deletion', async () => {
        const { id: folder1Id } = await Folder_1.default.save({ title: 'folder1' });
        const { id: folder2Id } = await Folder_1.default.save({ title: 'folder2' });
        const { command, promptMock } = setUpCommand();
        await command.action({ 'notebook': 'folder1', options: { permanent: true, force: true } });
        expect(promptMock).not.toHaveBeenCalled();
        // Should be permanently deleted.
        expect((await Note_1.default.allItemsInTrash()).folderIds.includes(folder1Id)).toBe(false);
        expect(await Folder_1.default.load(folder1Id, { includeDeleted: true })).toBe(undefined);
        // folder2 should not be deleted
        expect(await Folder_1.default.load(folder2Id, { includeDeleted: false })).toBeTruthy();
        // Should prompt before deleting
        await command.action({ 'notebook': 'folder2', options: { permanent: true } });
        expect(promptMock).toHaveBeenCalled();
        expect(await Folder_1.default.load(folder2Id, { includeDeleted: false })).toBeUndefined();
    });
});
//# sourceMappingURL=command-rmbook.test.js.map