"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const locale_1 = require("@joplin/lib/locale");
const base_command_1 = require("./base-command");
const app_1 = require("./app");
const registry_1 = require("@joplin/lib/registry");
const Logger_1 = require("@joplin/utils/Logger");
const ShareService_1 = require("@joplin/lib/services/share/ShareService");
const BaseModel_1 = require("@joplin/lib/BaseModel");
const reducer_1 = require("@joplin/lib/services/share/reducer");
const Folder_1 = require("@joplin/lib/models/Folder");
const invitationRespond_1 = require("@joplin/lib/services/share/invitationRespond");
const CommandService_1 = require("@joplin/lib/services/CommandService");
const string_utils_1 = require("@joplin/lib/string-utils");
const logger = Logger_1.default.create('command-share');
const folderTitle = (folder) => {
    return folder ? (0, string_utils_1.substrWithEllipsis)(folder.title, 0, 32) : (0, locale_1._)('[None]');
};
const getShareState = () => (0, app_1.default)().store().getState().shareService;
const getShareFromFolderId = (folderId) => {
    const shareState = getShareState();
    const allShares = shareState.shares;
    const share = allShares.find(share => share.folder_id === folderId);
    return share;
};
const getShareUsers = (folderId) => {
    const share = getShareFromFolderId(folderId);
    if (!share) {
        throw new Error(`No share found for folder ${folderId}`);
    }
    return getShareState().shareUsers[share.id];
};
class Command extends base_command_1.default {
    usage() {
        return 'share <command> [notebook] [user]';
    }
    description() {
        return [
            (0, locale_1._)('Shares or unshares the specified [notebook] with [user]. Requires Joplin Cloud or Joplin Server.'),
            (0, locale_1._)('Commands: `add`, `remove`, `list`, `delete`, `accept`, `leave`, and `reject`.'),
        ].join('\n');
    }
    options() {
        return [
            ['--read-only', (0, locale_1._)('Don\'t allow the share recipient to write to the shared notebook. Valid only for the `add` subcommand.')],
            ['-f, --force', (0, locale_1._)('Do not ask for user confirmation.')],
            ['--json', (0, locale_1._)('Prefer JSON output.')],
        ];
    }
    async action(args) {
        const commandShareAdd = async (folder, email) => {
            await registry_1.reg.waitForSyncFinishedThenSync();
            const share = await ShareService_1.default.instance().shareFolder(folder.id);
            const permissions = {
                can_read: 1,
                can_write: args.options['read-only'] ? 0 : 1,
            };
            logger.debug('Sharing folder', folder.id, 'with', email, 'permissions=', permissions);
            await ShareService_1.default.instance().addShareRecipient(share.id, share.master_key_id, email, permissions);
            await ShareService_1.default.instance().refreshShares();
            await ShareService_1.default.instance().refreshShareUsers(share.id);
            await registry_1.reg.waitForSyncFinishedThenSync();
        };
        const commandShareRemove = async (folder, email) => {
            await ShareService_1.default.instance().refreshShares();
            const share = getShareFromFolderId(folder.id);
            if (!share) {
                throw new Error(`No share found for folder ${folder.id}`);
            }
            await ShareService_1.default.instance().refreshShareUsers(share.id);
            const shareUsers = getShareUsers(folder.id);
            if (!shareUsers) {
                throw new Error(`No share found for folder ${folder.id}`);
            }
            const targetUser = shareUsers.find(user => { var _a; return ((_a = user.user) === null || _a === void 0 ? void 0 : _a.email) === email; });
            if (!targetUser) {
                throw new Error(`No recipient found with email ${email}`);
            }
            await ShareService_1.default.instance().deleteShareRecipient(targetUser.id);
            this.stdout((0, locale_1._)('Removed %s from share.', targetUser.user.email));
        };
        const commandShareList = async () => {
            var _a, _b;
            let folder = null;
            if (args.notebook) {
                folder = await (0, app_1.default)().loadItemOrFail(BaseModel_1.ModelType.Folder, args.notebook);
            }
            await ShareService_1.default.instance().maintenance();
            if (folder) {
                const share = getShareFromFolderId(folder.id);
                await ShareService_1.default.instance().refreshShareUsers(share.id);
                const shareUsers = getShareUsers(folder.id);
                const output = {
                    folderTitle: folderTitle(folder),
                    sharedWith: (shareUsers !== null && shareUsers !== void 0 ? shareUsers : []).map(user => ({
                        email: user.user.email,
                        readOnly: user.can_read && !user.can_write,
                    })),
                };
                if (args.options.json) {
                    this.stdout(JSON.stringify(output));
                }
                else {
                    this.stdout((0, locale_1._)('Folder "%s" is shared with:', output.folderTitle));
                    for (const user of output.sharedWith) {
                        this.stdout(`\t${user.email}\t${user.readOnly ? (0, locale_1._)('(Read-only)') : ''}`);
                    }
                }
            }
            else {
                const shareState = getShareState();
                const output = {
                    invitations: shareState.shareInvitations.map(invitation => {
                        var _a;
                        return ({
                            accepted: invitation.status === reducer_1.ShareUserStatus.Accepted,
                            waiting: invitation.status === reducer_1.ShareUserStatus.Waiting,
                            rejected: invitation.status === reducer_1.ShareUserStatus.Rejected,
                            folderId: invitation.share.folder_id,
                            canWrite: !!invitation.can_write,
                            fromUser: {
                                email: (_a = invitation.share.user) === null || _a === void 0 ? void 0 : _a.email,
                            },
                        });
                    }),
                    shares: shareState.shares.map(share => {
                        var _a, _b;
                        return ({
                            isFolder: !!share.folder_id,
                            isNote: !!share.note_id,
                            itemId: (_a = share.folder_id) !== null && _a !== void 0 ? _a : share.note_id,
                            fromUser: {
                                email: (_b = share.user) === null || _b === void 0 ? void 0 : _b.email,
                            },
                        });
                    }),
                };
                if (args.options.json) {
                    this.stdout(JSON.stringify(output));
                }
                else {
                    this.stdout((0, locale_1._)('Incoming shares:'));
                    let loggedInvitation = false;
                    for (const invitation of output.invitations) {
                        let message;
                        if (invitation.waiting) {
                            message = (0, locale_1._)('Waiting: Notebook %s from %s', invitation.folderId, invitation.fromUser.email);
                        }
                        if (invitation.accepted) {
                            const folder = await Folder_1.default.load(invitation.folderId);
                            message = (0, locale_1._)('Accepted: Notebook %s from %s', folderTitle(folder), invitation.fromUser.email);
                        }
                        if (message) {
                            this.stdout(`\t${message}`);
                            loggedInvitation = true;
                        }
                    }
                    if (!loggedInvitation) {
                        this.stdout(`\t${(0, locale_1._)('None')}`);
                    }
                    this.stdout((0, locale_1._)('All shared folders:'));
                    if (output.shares.length) {
                        for (const share of output.shares) {
                            let title;
                            if (share.isFolder) {
                                title = folderTitle(await Folder_1.default.load(share.itemId));
                            }
                            else {
                                title = share.itemId;
                            }
                            if ((_a = share.fromUser) === null || _a === void 0 ? void 0 : _a.email) {
                                this.stdout(`\t${(0, locale_1._)('%s from %s', title, (_b = share.fromUser) === null || _b === void 0 ? void 0 : _b.email)}`);
                            }
                            else {
                                this.stdout(`\t${title} - ${share.itemId}`);
                            }
                        }
                    }
                    else {
                        this.stdout(`\t${(0, locale_1._)('None')}`);
                    }
                }
            }
        };
        const commandShareAcceptOrReject = async (folderId, accept) => {
            await ShareService_1.default.instance().maintenance();
            const shareState = getShareState();
            const invitations = shareState.shareInvitations.filter(invitation => {
                return invitation.share.folder_id === folderId && invitation.status === reducer_1.ShareUserStatus.Waiting;
            });
            if (invitations.length === 0)
                throw new Error('No such invitation found');
            // If there are multiple invitations for the same folder, stop early to avoid
            // accepting the wrong invitation.
            if (invitations.length > 1)
                throw new Error('Multiple invitations found with the same ID');
            const invitation = invitations[0];
            this.stdout(accept ? (0, locale_1._)('Accepting share...') : (0, locale_1._)('Rejecting share...'));
            await (0, invitationRespond_1.default)(invitation.id, invitation.share.folder_id, invitation.master_key, accept);
        };
        const commandShareAccept = (folderId) => (commandShareAcceptOrReject(folderId, true));
        const commandShareReject = (folderId) => (commandShareAcceptOrReject(folderId, false));
        const commandShareDelete = async (folder) => {
            const force = args.options.force;
            const ok = force ? true : await this.prompt((0, locale_1._)('Unshare notebook "%s"? This may cause other users to lose access to the notebook.', folderTitle(folder)), { booleanAnswerDefault: 'n' });
            if (!ok)
                return;
            logger.info('Unsharing folder', folder.id);
            await ShareService_1.default.instance().unshareFolder(folder.id);
            await registry_1.reg.scheduleSync();
        };
        if (args.command === 'add' || args.command === 'remove' || args.command === 'delete') {
            if (!args.notebook)
                throw new Error('[notebook] is required');
            const folder = await (0, app_1.default)().loadItemOrFail(BaseModel_1.ModelType.Folder, args.notebook);
            if (args.command === 'delete') {
                return commandShareDelete(folder);
            }
            else {
                if (!args.user)
                    throw new Error('[user] is required');
                const email = args.user;
                if (args.command === 'add') {
                    return commandShareAdd(folder, email);
                }
                else if (args.command === 'remove') {
                    return commandShareRemove(folder, email);
                }
            }
        }
        if (args.command === 'leave') {
            const folder = args.notebook ? await (0, app_1.default)().loadItemOrFail(BaseModel_1.ModelType.Folder, args.notebook) : null;
            await ShareService_1.default.instance().maintenance();
            return CommandService_1.default.instance().execute('leaveSharedFolder', folder === null || folder === void 0 ? void 0 : folder.id, { force: args.options.force });
        }
        if (args.command === 'list') {
            return commandShareList();
        }
        if (args.command === 'accept') {
            return commandShareAccept(args.notebook);
        }
        if (args.command === 'reject') {
            return commandShareReject(args.notebook);
        }
        throw new Error(`Unknown subcommand: ${args.command}`);
    }
}
module.exports = Command;
//# sourceMappingURL=command-share.js.map