"use strict";
/* eslint-disable import/prefer-default-export */
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadKeychainServiceAndSettings = loadKeychainServiceAndSettings;
const KeychainService_1 = require("./keychain/KeychainService");
const Setting_1 = require("../models/Setting");
const uuid_1 = require("../uuid");
const syncInfoUtils_1 = require("./synchronizer/syncInfoUtils");
const shim_1 = require("../shim");
// This function takes care of initialising both the keychain service and settings.
//
// Loading the settings became more complicated with the keychain integration. This is because
// the settings needs a keychain service, and the keychain service needs a clientId, which
// is set dynamically and saved to the settings.
// In other words, it's not possible to load the settings without the KS service and it's not
// possible to initialise the KS service without the settings.
// The solution is to fetch just the client ID directly from the database.
async function loadKeychainServiceAndSettings(keychainServiceDrivers) {
    const clientIdSetting = await Setting_1.default.loadOne('clientId');
    const clientId = clientIdSetting ? clientIdSetting.value : uuid_1.default.create();
    // Temporary workaround: For a short time, pre-release versions of Joplin Portable encrypted
    // saved keys using the keychain. This can break sync when transferring Joplin between devices.
    // To prevent secure keys from being lost, we enable read-only keychain access in portable mode.
    if (shim_1.default.isPortable()) {
        KeychainService_1.default.instance().readOnly = true;
    }
    await KeychainService_1.default.instance().initialize(keychainServiceDrivers.map(Driver => new Driver(Setting_1.default.value('appId'), clientId)));
    Setting_1.default.setKeychainService(KeychainService_1.default.instance());
    await Setting_1.default.load();
    // Using Linux with the keychain has been observed to cause all secure settings to be lost
    // on Fedora 40 + GNOME. (This may have been related to running multiple Joplin instances).
    // For now, make saving to the keychain opt-in until more feedback is received.
    if (shim_1.default.isLinux() && !Setting_1.default.value('featureFlag.linuxKeychain')) {
        KeychainService_1.default.instance().readOnly = true;
    }
    // This is part of the migration to the new sync target info. It needs to be
    // set as early as possible since it's used to tell if E2EE is enabled, it
    // contains the master keys, etc. Once it has been set, it becomes a noop
    // on future calls.
    await (0, syncInfoUtils_1.migrateLocalSyncInfo)(Setting_1.default.db());
    if (!clientIdSetting)
        Setting_1.default.setValue('clientId', clientId);
    await KeychainService_1.default.instance().detectIfKeychainSupported();
}
//# sourceMappingURL=SettingUtils.js.map