"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const test_utils_1 = require("../../testing/test-utils");
const MasterKey_1 = require("../../models/MasterKey");
const utils_1 = require("./utils");
const syncInfoUtils_1 = require("../synchronizer/syncInfoUtils");
const Setting_1 = require("../../models/Setting");
const ppk_1 = require("./ppk/ppk");
const types_1 = require("./types");
describe('e2ee/utils', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    afterAll(async () => {
        await (0, test_utils_1.afterAllCleanUp)();
    });
    it('should tell if the missing master key message should be shown', async () => {
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey('111111'));
        const mk2 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey('111111'));
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [mk1.id])).toBe(true);
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [mk1.id, mk2.id])).toBe(true);
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [])).toBe(false);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk1.id, false);
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [mk1.id])).toBe(false);
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [mk1.id, mk2.id])).toBe(true);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk2.id, false);
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [mk1.id, mk2.id])).toBe(false);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk1.id, true);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk2.id, true);
        expect((0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), [mk1.id, mk2.id])).toBe(true);
        await (0, test_utils_1.expectNotThrow)(async () => (0, utils_1.showMissingMasterKeyMessage)((0, syncInfoUtils_1.localSyncInfo)(), ['not_downloaded_yet']));
        const syncInfo = (0, syncInfoUtils_1.localSyncInfo)();
        syncInfo.masterKeys = [];
        expect((0, utils_1.showMissingMasterKeyMessage)(syncInfo, [mk1.id, mk2.id])).toBe(false);
    });
    it('should do ppk migration', async () => {
        const { reset } = (0, ppk_1.testing__setPpkMigrations_)([types_1.PublicKeyAlgorithm.RsaV1, types_1.PublicKeyAlgorithm.RsaV2]);
        try {
            const syncInfo = (0, syncInfoUtils_1.localSyncInfo)();
            const testPassword = 'test--TEST';
            Setting_1.default.setValue('encryption.masterPassword', testPassword);
            syncInfo.ppk = await (0, ppk_1.generateKeyPairWithAlgorithm)(types_1.PublicKeyAlgorithm.RsaV1, (0, test_utils_1.encryptionService)(), testPassword);
            (0, syncInfoUtils_1.saveLocalSyncInfo)(syncInfo);
            expect((0, ppk_1.getPpkAlgorithm)((0, syncInfoUtils_1.localSyncInfo)().ppk)).toBe(types_1.PublicKeyAlgorithm.RsaV1);
            await (0, utils_1.migratePpk)();
            expect((0, ppk_1.getPpkAlgorithm)((0, syncInfoUtils_1.localSyncInfo)().ppk)).toBe(types_1.PublicKeyAlgorithm.RsaV2);
        }
        finally {
            reset();
        }
    });
    it('should not migrate ppk if the key is up-to-date', async () => {
        const syncInfo = (0, syncInfoUtils_1.localSyncInfo)();
        const testPassword = 'test 🔑';
        Setting_1.default.setValue('encryption.masterPassword', testPassword);
        const originalPpk = await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), testPassword);
        syncInfo.ppk = originalPpk;
        (0, syncInfoUtils_1.saveLocalSyncInfo)(syncInfo);
        const lastChangeTime = (0, syncInfoUtils_1.localSyncInfo)().keyTimestamp('ppk');
        await (0, utils_1.migratePpk)();
        expect((0, syncInfoUtils_1.localSyncInfo)().keyTimestamp('ppk')).toBe(lastChangeTime);
        expect((0, syncInfoUtils_1.localSyncInfo)().ppk.createdTime).toBe(originalPpk.createdTime);
    });
    it('should do the master password migration', async () => {
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey('111111'));
        const mk2 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey('222222'));
        Setting_1.default.setValue('encryption.passwordCache', {
            [mk1.id]: '111111',
            [mk2.id]: '222222',
        });
        await (0, utils_1.migrateMasterPassword)();
        {
            expect(Setting_1.default.value('encryption.masterPassword')).toBe('');
            const newCache = Setting_1.default.value('encryption.passwordCache');
            expect(newCache[mk1.id]).toBe('111111');
            expect(newCache[mk2.id]).toBe('222222');
        }
        (0, syncInfoUtils_1.setActiveMasterKeyId)(mk1.id);
        await (0, utils_1.migrateMasterPassword)();
        {
            expect(Setting_1.default.value('encryption.masterPassword')).toBe('111111');
            const newCache = Setting_1.default.value('encryption.passwordCache');
            expect(newCache[mk1.id]).toBe(undefined);
            expect(newCache[mk2.id]).toBe('222222');
        }
    });
    it('should update the master password', async () => {
        const masterPassword1 = '111111';
        const masterPassword2 = '222222';
        Setting_1.default.setValue('encryption.masterPassword', masterPassword1);
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        const mk2 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        (0, syncInfoUtils_1.setPpk)(await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), masterPassword1));
        await (0, utils_1.updateMasterPassword)(masterPassword1, masterPassword2);
        expect(Setting_1.default.value('encryption.masterPassword')).toBe(masterPassword2);
        expect(await (0, ppk_1.ppkPasswordIsValid)((0, test_utils_1.encryptionService)(), (0, syncInfoUtils_1.localSyncInfo)().ppk, masterPassword1)).toBe(false);
        expect(await (0, ppk_1.ppkPasswordIsValid)((0, test_utils_1.encryptionService)(), (0, syncInfoUtils_1.localSyncInfo)().ppk, masterPassword2)).toBe(true);
        expect(await (0, test_utils_1.encryptionService)().checkMasterKeyPassword(await MasterKey_1.default.load(mk1.id), masterPassword1)).toBe(false);
        expect(await (0, test_utils_1.encryptionService)().checkMasterKeyPassword(await MasterKey_1.default.load(mk2.id), masterPassword1)).toBe(false);
        expect(await (0, test_utils_1.encryptionService)().checkMasterKeyPassword(await MasterKey_1.default.load(mk1.id), masterPassword2)).toBe(true);
        expect(await (0, test_utils_1.encryptionService)().checkMasterKeyPassword(await MasterKey_1.default.load(mk2.id), masterPassword2)).toBe(true);
        await (0, test_utils_1.expectThrow)(async () => (0, utils_1.updateMasterPassword)('wrong', masterPassword1));
    });
    it('should set and verify master password when a data key exists', async () => {
        const password = '111111';
        await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(password));
        await (0, test_utils_1.expectThrow)(async () => (0, utils_1.updateMasterPassword)('', 'wrong'));
        await (0, test_utils_1.expectNotThrow)(async () => (0, utils_1.updateMasterPassword)('', password));
        expect(Setting_1.default.value('encryption.masterPassword')).toBe(password);
    });
    it('should set and verify master password when a private key exists', async () => {
        const password = '111111';
        (0, syncInfoUtils_1.setPpk)(await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), password));
        await (0, test_utils_1.expectThrow)(async () => (0, utils_1.updateMasterPassword)('', 'wrong'));
        await (0, test_utils_1.expectNotThrow)(async () => (0, utils_1.updateMasterPassword)('', password));
        expect(Setting_1.default.value('encryption.masterPassword')).toBe(password);
    });
    it('should only set the master password if not already set', async () => {
        expect((0, syncInfoUtils_1.localSyncInfo)().ppk).toBeFalsy();
        await (0, utils_1.updateMasterPassword)('', '111111');
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('111111');
        expect((0, syncInfoUtils_1.localSyncInfo)().ppk).toBeFalsy();
        expect((0, syncInfoUtils_1.localSyncInfo)().masterKeys.length).toBe(0);
    });
    it('should change the master password even if no key is present', async () => {
        await (0, utils_1.updateMasterPassword)('', '111111');
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('111111');
        await (0, utils_1.updateMasterPassword)('111111', '222222');
        expect(Setting_1.default.value('encryption.masterPassword')).toBe('222222');
    });
    it('should reset a master password', async () => {
        const masterPassword1 = '111111';
        const masterPassword2 = '222222';
        Setting_1.default.setValue('encryption.masterPassword', masterPassword1);
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        const mk2 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        (0, syncInfoUtils_1.setPpk)(await (0, ppk_1.generateKeyPair)((0, test_utils_1.encryptionService)(), masterPassword1));
        const previousPpk = (0, syncInfoUtils_1.localSyncInfo)().ppk;
        await (0, utils_1.resetMasterPassword)((0, test_utils_1.encryptionService)(), (0, test_utils_1.kvStore)(), null, masterPassword2);
        expect((0, syncInfoUtils_1.masterKeyEnabled)((0, syncInfoUtils_1.masterKeyById)(mk1.id))).toBe(false);
        expect((0, syncInfoUtils_1.masterKeyEnabled)((0, syncInfoUtils_1.masterKeyById)(mk2.id))).toBe(false);
        expect((0, syncInfoUtils_1.localSyncInfo)().ppk.id).not.toBe(previousPpk.id);
        expect((0, syncInfoUtils_1.localSyncInfo)().ppk.privateKey.ciphertext).not.toBe(previousPpk.privateKey.ciphertext);
        expect((0, syncInfoUtils_1.localSyncInfo)().ppk.publicKey).not.toBe(previousPpk.publicKey);
        // Also check that a new master key has been created, that it is active and enabled
        expect((0, syncInfoUtils_1.localSyncInfo)().masterKeys.length).toBe(3);
        expect((0, syncInfoUtils_1.localSyncInfo)().activeMasterKeyId).toBe((0, syncInfoUtils_1.localSyncInfo)().masterKeys[2].id);
        expect((0, syncInfoUtils_1.masterKeyEnabled)((0, syncInfoUtils_1.localSyncInfo)().masterKeys[2])).toBe(true);
    });
    it('should fix active key selection issues - 1', async () => {
        const masterPassword1 = '111111';
        Setting_1.default.setValue('encryption.masterPassword', masterPassword1);
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        await (0, test_utils_1.msleep)(1);
        await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        await (0, test_utils_1.msleep)(1);
        const mk3 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        (0, syncInfoUtils_1.setActiveMasterKeyId)(mk1.id);
        (0, syncInfoUtils_1.setMasterKeyEnabled)(mk1.id, false);
        (0, utils_1.activeMasterKeySanityCheck)();
        const syncInfo = (0, syncInfoUtils_1.localSyncInfo)();
        expect(syncInfo.activeMasterKeyId).toBe(mk3.id);
    });
    it('should fix active key selection issues - 2', async () => {
        // Should not do anything if the active key is already an enabled one.
        const masterPassword1 = '111111';
        Setting_1.default.setValue('encryption.masterPassword', masterPassword1);
        const mk1 = await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        await (0, test_utils_1.msleep)(1);
        await MasterKey_1.default.save(await (0, test_utils_1.encryptionService)().generateMasterKey(masterPassword1));
        (0, syncInfoUtils_1.setActiveMasterKeyId)(mk1.id);
        (0, utils_1.activeMasterKeySanityCheck)();
        const syncInfo = (0, syncInfoUtils_1.localSyncInfo)();
        expect(syncInfo.activeMasterKeyId).toBe(mk1.id);
    });
});
//# sourceMappingURL=utils.test.js.map