"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const InteropService_1 = require("../../services/interop/InteropService");
const test_utils_1 = require("../../testing/test-utils");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const fs = require("fs-extra");
const test_utils_2 = require("../../testing/test-utils");
const types_1 = require("../../services/plugins/api/types");
const types_2 = require("./types");
const promises_1 = require("fs/promises");
async function recreateExportDir() {
    const dir = (0, test_utils_1.exportDir)();
    await fs.remove(dir);
    await fs.mkdirp(dir);
}
describe('interop/InteropService_Exporter_Html', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
        await recreateExportDir();
    });
    test('should export HTML file', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ body: '**ma note**', parent_id: folder1.id });
        const filePath = `${(0, test_utils_1.exportDir)()}/test.html`;
        await service.export({
            path: filePath,
            format: types_2.ExportModuleOutputFormat.Html,
            packIntoSingleFile: false,
        });
        const content = await fs.readFile(filePath, 'utf8');
        expect(content).toContain('<strong>ma note</strong>');
    }));
    test('should export HTML directory', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Folder_1.default.save({ title: 'folder2' });
        await Note_1.default.save({ title: 'note1', parent_id: folder1.id });
        await Note_1.default.save({ title: 'note2', parent_id: folder1.id });
        const dir = (0, test_utils_1.exportDir)();
        await service.export({
            path: dir,
            format: types_2.ExportModuleOutputFormat.Html,
            target: types_2.FileSystemItem.Directory,
        });
        const rootDirs = await fs.readdir(dir);
        rootDirs.sort();
        expect(rootDirs).toEqual(['folder1', 'folder2']);
        const files = await fs.readdir(`${dir}/${rootDirs[0]}`);
        expect(files).toContain('note1.html');
        expect(files).toContain('note2.html');
    }));
    test('should export plugin assets', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ body: '**ma note**', parent_id: folder1.id });
        const filePath = `${(0, test_utils_1.exportDir)()}/test.html`;
        const contentScriptPath = (0, test_utils_2.tempFilePath)('js');
        await fs.writeFile(contentScriptPath, `module.exports = {
			default: function(_context) { 
				return {
					plugin: function (markdownIt, _options) {
						
					}, 
					assets: function() {
						return [
							{ name: 'fence.css' }
						];
					},		
				}
			},
		}`);
        const assetPath = `${require('path').dirname(contentScriptPath)}/fence.css`;
        const fenceContent = 'strong { color: red; }';
        await fs.writeFile(assetPath, fenceContent);
        const plugins = {
            'test': {
                id: 'test',
                contentScripts: {
                    [types_1.ContentScriptType.MarkdownItPlugin]: [
                        {
                            id: 'mdContentScript',
                            path: contentScriptPath,
                        },
                    ],
                },
                views: {},
            },
        };
        await service.export({
            path: filePath,
            format: types_2.ExportModuleOutputFormat.Html,
            packIntoSingleFile: false,
            plugins,
        });
        const fenceRelativePath = 'pluginAssets/mdContentScript/fence.css';
        const content = await fs.readFile(filePath, 'utf8');
        expect(content).toContain(fenceRelativePath);
        const readFenceContent = await fs.readFile(`${(0, test_utils_1.exportDir)()}/${fenceRelativePath}`, 'utf8');
        expect(readFenceContent).toBe(fenceContent);
    }));
    test('should not throw an error on invalid resource paths', (async () => {
        const service = InteropService_1.default.instance();
        const folder1 = await Folder_1.default.save({ title: 'folder1' });
        await Note_1.default.save({ title: 'note1', parent_id: folder1.id, body: '[a link starts with slash](/)' });
        const filePath = `${(0, test_utils_1.exportDir)()}/test.html`;
        await service.export({
            path: filePath,
            format: types_2.ExportModuleOutputFormat.Html,
            packIntoSingleFile: true,
            target: types_2.FileSystemItem.File,
        });
        const content = await (0, promises_1.readFile)(filePath, 'utf-8');
        expect(content).toContain('<a data-from-md="" title="/" href="" download="">a link starts with slash</a>');
    }));
});
//# sourceMappingURL=InteropService_Exporter_Html.test.js.map