"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const BaseItem_1 = require("../../models/BaseItem");
const Note_1 = require("../../models/Note");
const test_utils_1 = require("../../testing/test-utils");
const handleConflictAction_1 = require("./utils/handleConflictAction");
const types_1 = require("./utils/types");
describe('handleConflictAction', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    test('note conflict is created', async () => {
        const local = await Note_1.default.save({ title: 'Test', body: 'body' });
        // Pass the local note with unsaved changes to verify that the note is reloaded before creating the conflict
        const changedLocal = Object.assign(Object.assign({}, local), { title: 'TestChanged' });
        const remoteContent = Object.assign(Object.assign({}, local), { title: 'TestRemote' });
        const initialSyncItem = await BaseItem_1.default.syncItem(1, local.id);
        await (0, handleConflictAction_1.default)(types_1.SyncAction.NoteConflict, Note_1.default, true, remoteContent, changedLocal, 1, false, (action) => (action));
        const createdSyncItem = await BaseItem_1.default.syncItem(1, local.id);
        const updatedLocal = await Note_1.default.load(local.id);
        const notes = await Note_1.default.all();
        const conflictNote = await Note_1.default.loadByTitle('Test');
        expect(initialSyncItem).toBeUndefined();
        expect(createdSyncItem).toBeDefined();
        expect(updatedLocal.title).toBe('TestRemote');
        expect(notes.length).toBe(2);
        expect(conflictNote.id).not.toBe(local.id);
    });
    test('note conflict is not created when remote and local contents match', async () => {
        const local = await Note_1.default.save({ title: 'Test', body: 'body' });
        // Pass the local note with unsaved changes to verify that the note is reloaded before checking if eligible to create a conflict
        const changedLocal = Object.assign(Object.assign({}, local), { title: 'TestChanged' });
        const remoteContent = local;
        const initialSyncItem = await BaseItem_1.default.syncItem(1, local.id);
        await (0, handleConflictAction_1.default)(types_1.SyncAction.NoteConflict, Note_1.default, true, remoteContent, changedLocal, 1, false, (action) => (action));
        const createdSyncItem = await BaseItem_1.default.syncItem(1, local.id);
        const notes = await Note_1.default.all();
        expect(initialSyncItem).toBeUndefined();
        expect(createdSyncItem).toBeDefined();
        expect(notes.length).toBe(1);
    });
});
//# sourceMappingURL=handleConflictAction.test.js.map