"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const time_1 = require("@joplin/utils/time");
const Folder_1 = require("../../models/Folder");
const Note_1 = require("../../models/Note");
const test_utils_1 = require("../../testing/test-utils");
const permanentlyDeleteOldItems_1 = require("./permanentlyDeleteOldItems");
const Setting_1 = require("../../models/Setting");
describe('permanentlyDeleteOldItems', () => {
    beforeEach(async () => {
        await (0, test_utils_1.setupDatabaseAndSynchronizer)(1);
        await (0, test_utils_1.switchClient)(1);
    });
    it('should auto-delete old items', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder1.id });
        const note2 = await Note_1.default.save({});
        await Folder_1.default.delete(folder1.id, { toTrash: true, deleteChildren: true });
        // First check that it doesn't auto-delete if it's not within the right interval
        await (0, permanentlyDeleteOldItems_1.default)(time_1.Day);
        expect((await Folder_1.default.load(folder1.id))).toBeTruthy();
        expect((await Folder_1.default.load(folder2.id))).toBeTruthy();
        expect((await Note_1.default.load(note1.id))).toBeTruthy();
        expect((await Note_1.default.load(note2.id))).toBeTruthy();
        await (0, time_1.msleep)(1);
        await (0, permanentlyDeleteOldItems_1.default)(0);
        expect((await Folder_1.default.load(folder1.id))).toBeFalsy();
        expect((await Folder_1.default.load(folder2.id))).toBeTruthy();
        expect((await Note_1.default.load(note1.id))).toBeFalsy();
        expect((await Note_1.default.load(note2.id))).toBeTruthy();
    });
    it('should not auto-delete non-empty folders', async () => {
        const folder1 = await Folder_1.default.save({});
        const folder2 = await Folder_1.default.save({});
        const note1 = await Note_1.default.save({ parent_id: folder1.id });
        const note2 = await Note_1.default.save({});
        await Folder_1.default.delete(folder1.id, { toTrash: true, deleteChildren: true });
        // Simulates a folder having been deleted a long time ago - so it should be deleted. But
        // since it contains a note it should not.
        await Folder_1.default.save({ id: folder1.id, deleted_time: 1000 });
        await (0, permanentlyDeleteOldItems_1.default)(time_1.Day);
        expect((await Folder_1.default.load(folder1.id))).toBeTruthy();
        expect((await Note_1.default.load(note1.id))).toBeTruthy();
        // Now both folders and items are within the deletion interval, so they should be both be
        // auto-deleted
        await Note_1.default.save({ id: note1.id, deleted_time: 1000 });
        await (0, permanentlyDeleteOldItems_1.default)(1);
        expect((await Folder_1.default.load(folder1.id))).toBeFalsy();
        expect((await Folder_1.default.load(folder2.id))).toBeTruthy();
        expect((await Note_1.default.load(note1.id))).toBeFalsy();
        expect((await Note_1.default.load(note2.id))).toBeTruthy();
    });
    it('should not do anything if auto-deletion is not enabled', async () => {
        Setting_1.default.setValue('trash.autoDeletionEnabled', false);
        const folder1 = await Folder_1.default.save({});
        await Folder_1.default.delete(folder1.id, { toTrash: true });
        await (0, time_1.msleep)(1);
        await (0, permanentlyDeleteOldItems_1.default)(0);
        expect(await Folder_1.default.count()).toBe(1);
    });
    it('should not auto-delete read-only items', async () => {
        const shareId = 'testShare';
        // Simulates a folder having been deleted a long time ago
        const longTimeAgo = 1000;
        const readOnlyFolder = await Folder_1.default.save({
            title: 'Read-only folder',
            share_id: shareId,
            deleted_time: longTimeAgo,
        });
        const readOnlyNote1 = await Note_1.default.save({
            title: 'Read-only note',
            parent_id: readOnlyFolder.id,
            share_id: shareId,
            deleted_time: longTimeAgo,
        });
        const readOnlyNote2 = await Note_1.default.save({
            title: 'Read-only note 2',
            share_id: shareId,
            deleted_time: longTimeAgo,
        });
        const writableNote = await Note_1.default.save({
            title: 'Editable note',
            deleted_time: longTimeAgo,
        });
        const cleanup = (0, test_utils_1.simulateReadOnlyShareEnv)(shareId);
        await (0, permanentlyDeleteOldItems_1.default)(time_1.Day);
        // Should preserve only the read-only items.
        expect(await Folder_1.default.load(readOnlyFolder.id)).toBeTruthy();
        expect(await Note_1.default.load(readOnlyNote1.id)).toBeTruthy();
        expect(await Note_1.default.load(readOnlyNote2.id)).toBeTruthy();
        expect(await Note_1.default.load(writableNote.id)).toBeFalsy();
        cleanup();
    });
});
//# sourceMappingURL=permanentlyDeleteOldItems.test.js.map