/*-*- c++ -*-******************************************************************
**
**  Copyright (c) 1999,2000 Red Hat, Inc. Harald Hoyer (Harald.Hoyer@RedHat.de)
**
**  $Id: cdrom.cc,v 1.5 2005/09/14 13:15:55 saturn_de Exp $
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
******************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>		// mkdir
#include <sys/ioctl.h>		// ioctls
#include <unistd.h>		// lseek, read. etc
#include <fcntl.h>		// O_RDONLY etc.
#include <linux/iso_fs.h>	// isofs constants
#include <linux/cdrom.h>	// old ioctls for cdrom
#include <string>
#include <fstream>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include "cdrom.h"
#include <iostream>
#include <libintl.h>
#include <errno.h>
#define _(String) gettext (String)

#ifndef MNTTYPE_ISO9660
#define MNTTYPE_ISO9660 "iso9660"
#endif

CDRom::CDRom(string _devpath) :
   devpath(_devpath)
{  
   allowEject = true;
   cdromfd = -1;
   cdstatus = (isMounted()) ? mounted : unknown;
}

CDRom::~CDRom()
{
   CDRom::close();
}

bool CDRom::isOpen()
{
   return (cdromfd > 0);
}

int CDRom::open()
{
   if(cdromfd == -1) {
      cdromfd = ::open(devpath.c_str(), O_RDONLY | O_NONBLOCK | O_EXCL);
      if (cdromfd < 0)
	return cdromfd;

      ::ioctl(cdromfd, CDROM_CLEAR_OPTIONS, CDO_AUTO_CLOSE);
      if(allowEject) {
//	 ::ioctl(cdromfd, CDROM_CLEAR_OPTIONS, CDO_LOCK);
#ifdef CDROM_LOCKDOOR
	 ::ioctl(cdromfd, CDROM_LOCKDOOR, 0);
#else
#warning "Need Linux kernel >= 2.2.4 to work with IDE."
#endif
      }
   }
   return cdromfd;
}

int CDRom::test()
{
  bool opened = false;
  if (cdromfd < 0) {      
    if ( open() < 0 ) {
      if (errno == EBUSY)
	return isMounted() ? 1 : 0;
      else {
	return -1;
      }
    }
    opened = true;    
  }
  
  int ret = ioctl(cdromfd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
  
  if (opened) {
    close();
  }
  
  return (ret >= 0 ? 1 : -1);
}

int CDRom::close()
{
   int ret = 0;
   if (cdromfd >= 0)
       ret = ::close(cdromfd);
   cdromfd = -1;
   return ret;
}


bool CDRom::mount() 
{   
   bool ret = false;

   string execstr = string("/bin/mount ") + devpath;

   /* 
    * If the cdrom is mounted with O_NONBLOCK, removing
    * of a mounted cdrom is allowed 
    */
   if(!allowEject) 
      close();

   if (::system(execstr.c_str()) == 0) {
      cdstatus = mounted;
      ret = true;
   }
   else 
      cdstatus = unknown;

   if(!allowEject) 
      open();

   return ret;
}

bool CDRom::mediaChanged()
{  
  int ret;

  ret = ioctl(cdromfd, CDROM_MEDIA_CHANGED, CDSL_CURRENT);

  return ( ret > 0);
}

bool CDRom::isMounted()
{
   ifstream mnts("/proc/mounts");
   char line[PATH_MAX*3];

   char lpath[PATH_MAX];

   while(mnts.getline(line, PATH_MAX*3-1)) {
      char mpoint[PATH_MAX], dname[PATH_MAX];
      sscanf(line, "%s %s", dname, mpoint);
      int len = readlink(dname, lpath, PATH_MAX);
      if( len > 0 && len < PATH_MAX) {
	lpath[len] = 0;
      }
      if(devpath == dname || devpath == lpath)
      {
         // cout << "devpath=" << devpath << ",dname=" << dname << "\n";
         // cout << "mountpath=" << mountpath << ",mpoint=" << mpoint << "\n";
	 mountpath = mpoint;
	 mnts.close();
	 return true;
      }
   }
   mnts.close();
   return false;
}

bool CDRom::checkOK()
{
   return (::ioctl(cdromfd, CDROM_DRIVE_STATUS, CDSL_CURRENT) == CDS_DISC_OK);
}

CDRom::cdstatus_t CDRom::checkMedia()
{
  if (isMounted ()) {
    cdstatus = mounted;
    return cdstatus;
  }
  
  if (cdromfd >= 0) {      
    int ret = ioctl(cdromfd, CDROM_DRIVE_STATUS, CDSL_CURRENT);

    switch(ret) {
    case CDS_DISC_OK:
      break;
    case CDS_TRAY_OPEN:
    case CDS_NO_DISC:
      cdstatus = tray_is_open;
      return cdstatus;
    case CDS_NO_INFO:
    case CDS_DRIVE_NOT_READY:
    default:
      cdstatus = unknown;
      return cdstatus;
    }

    if (mediaChanged()) {
      cdstatus = tray_is_open;
    } else {
      if((cdstatus == tray_is_open) || (cdstatus == unknown)) {
	  int type = ::ioctl(cdromfd, CDROM_DISC_STATUS, CDSL_CURRENT);
	  
	  switch(type) {
	  case CDS_DATA_1:
	  case CDS_DATA_2:
	    cdstatus = unmounted;
	    cdtype = t_data;
	    break;
	  case CDS_AUDIO:
	    cdstatus = audio;
	    cdtype = t_audio;
	    break;
	  case CDS_MIXED:
	    cdstatus = unmounted;
	    cdtype = t_mixed;
	    break;
	  case CDS_NO_INFO:
	  case CDS_NO_DISC:
	    cdstatus = unknown;
	    break;
	  default:
	    fprintf(stderr, _("Unknown data type: 0x%x\n"), type);
	    cdstatus = unknown;
	  }	    
	}
      else if (cdstatus == mounted || cdstatus == unmounted) {
	cdstatus = unmounted;
      }
      if(allowEject) {
	//	 ::ioctl(cdromfd, CDROM_CLEAR_OPTIONS, CDO_LOCK);
#ifdef CDROM_LOCKDOOR
	::ioctl(cdromfd, CDROM_LOCKDOOR, 0);
#else
# warning "Need Linux kernel >= 2.2.4 to work with IDE."
#endif
      }
    }
  }
  else cdstatus = unknown;
  return cdstatus;
}

bool CDRom::eject()
{
   return (::ioctl(cdromfd, CDROMEJECT) == 0);
}     
