/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.aws.kinesis;

import com.amazonaws.regions.Region;
import com.amazonaws.services.kinesis.clientlibrary.exceptions.InvalidStateException;
import com.amazonaws.services.kinesis.clientlibrary.exceptions.ShutdownException;
import com.amazonaws.services.kinesis.clientlibrary.exceptions.ThrottlingException;
import com.amazonaws.services.kinesis.clientlibrary.interfaces.v2.IRecordProcessor;
import com.amazonaws.services.kinesis.clientlibrary.interfaces.v2.IRecordProcessorFactory;
import com.amazonaws.services.kinesis.clientlibrary.lib.worker.KinesisClientLibConfiguration;
import com.amazonaws.services.kinesis.clientlibrary.lib.worker.Worker;
import com.amazonaws.services.kinesis.clientlibrary.types.InitializationInput;
import com.amazonaws.services.kinesis.clientlibrary.types.ProcessRecordsInput;
import com.amazonaws.services.kinesis.clientlibrary.types.ShutdownInput;
import com.amazonaws.services.kinesis.model.Record;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.rholder.retry.Attempt;
import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.RetryListener;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import okhttp3.HttpUrl;
import org.graylog.aws.auth.AWSAuthProvider;
import org.graylog.aws.cloudwatch.CloudWatchLogData;
import org.graylog.aws.cloudwatch.CloudWatchLogEntry;
import org.graylog.aws.config.AWSPluginConfiguration;
import org.graylog.aws.config.Proxy;
import org.graylog.aws.inputs.transports.KinesisTransport;
import org.graylog.aws.inputs.transports.KinesisTransportState;
import org.graylog2.plugin.Tools;
import org.graylog2.plugin.system.NodeId;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KinesisConsumer
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(KinesisConsumer.class);
    private final Region region;
    private final String kinesisStreamName;
    private final AWSAuthProvider authProvider;
    private final NodeId nodeId;
    private final HttpUrl proxyUrl;
    private final AWSPluginConfiguration awsConfig;
    private final Consumer<byte[]> dataHandler;
    private final Integer maxThrottledWaitMillis;
    private final Integer recordBatchSize;
    private Worker worker;
    private KinesisTransport transport;
    private final ObjectMapper objectMapper;
    private String lastSuccessfulRecordSequence = null;

    public KinesisConsumer(String kinesisStreamName, Region region, Consumer<byte[]> dataHandler, AWSPluginConfiguration awsConfig, AWSAuthProvider authProvider, NodeId nodeId, @Nullable HttpUrl proxyUrl, KinesisTransport transport, ObjectMapper objectMapper, Integer maxThrottledWaitMillis, Integer recordBatchSize) {
        this.kinesisStreamName = Objects.requireNonNull(kinesisStreamName, "kinesisStreamName");
        this.region = Objects.requireNonNull(region, "region");
        this.dataHandler = Objects.requireNonNull(dataHandler, "dataHandler");
        this.awsConfig = Objects.requireNonNull(awsConfig, "awsConfig");
        this.authProvider = Objects.requireNonNull(authProvider, "authProvider");
        this.nodeId = Objects.requireNonNull(nodeId, "nodeId");
        this.proxyUrl = proxyUrl;
        this.transport = transport;
        this.objectMapper = objectMapper;
        this.maxThrottledWaitMillis = maxThrottledWaitMillis;
        this.recordBatchSize = recordBatchSize;
    }

    @Override
    public void run() {
        this.transport.consumerState.set(KinesisTransportState.STARTING);
        LOG.debug("Max wait millis [{}]", (Object)this.maxThrottledWaitMillis);
        LOG.debug("Record batch size [{}]", (Object)this.recordBatchSize);
        String workerId = String.format(Locale.ENGLISH, "graylog-node-%s", this.nodeId.anonymize());
        String applicationName = String.format(Locale.ENGLISH, "graylog-aws-plugin-%s", this.kinesisStreamName);
        KinesisClientLibConfiguration config = new KinesisClientLibConfiguration(applicationName, this.kinesisStreamName, this.authProvider, workerId);
        config.withRegionName(this.region.getName());
        if (this.recordBatchSize != null) {
            config.withMaxRecords(this.recordBatchSize);
        }
        if (this.awsConfig.proxyEnabled() && this.proxyUrl != null) {
            config.withCommonClientConfig(Proxy.forAWS(this.proxyUrl));
        }
        IRecordProcessorFactory recordProcessorFactory = () -> new IRecordProcessor(){
            private DateTime lastCheckpoint = DateTime.now();

            @Override
            public void initialize(InitializationInput initializationInput) {
                LOG.debug("Initializing Kinesis worker for stream <{}>", (Object)KinesisConsumer.this.kinesisStreamName);
                ((KinesisConsumer)KinesisConsumer.this).transport.consumerState.set(KinesisTransportState.RUNNING);
            }

            @Override
            public void processRecords(ProcessRecordsInput processRecordsInput) {
                LOG.debug("processRecords called. Received {} Kinesis events", (Object)processRecordsInput.getRecords().size());
                if (KinesisConsumer.this.transport.isThrottled()) {
                    LOG.info("[throttled] Waiting up to [{}ms] for throttling to clear.", (Object)KinesisConsumer.this.maxThrottledWaitMillis);
                    if (!KinesisConsumer.this.transport.blockUntilUnthrottled(KinesisConsumer.this.maxThrottledWaitMillis.intValue(), TimeUnit.MILLISECONDS)) {
                        LOG.info("[throttled] Throttling did not clear in [{}]ms. Stopping the Kinesis worker to let the throttle clear.\ufe0f It will start again automatically once throttling clears.", (Object)KinesisConsumer.this.maxThrottledWaitMillis);
                        if (KinesisConsumer.this.lastSuccessfulRecordSequence != null) {
                            this.checkpoint(processRecordsInput, KinesisConsumer.this.lastSuccessfulRecordSequence);
                        }
                        ((KinesisConsumer)KinesisConsumer.this).transport.consumerState.set(KinesisTransportState.STOPPING);
                        KinesisConsumer.this.worker.shutdown();
                        ((KinesisConsumer)KinesisConsumer.this).transport.stoppedDueToThrottling.set(true);
                        return;
                    }
                    LOG.debug("[unthrottled] Kinesis consumer will now resume processing records.");
                }
                for (Record record : processRecordsInput.getRecords()) {
                    try {
                        ByteBuffer dataBuffer = record.getData().asReadOnlyBuffer();
                        byte[] dataBytes = new byte[dataBuffer.remaining()];
                        dataBuffer.get(dataBytes);
                        byte[] bytes = Tools.decompressGzip((byte[])dataBytes).getBytes();
                        CloudWatchLogData data = (CloudWatchLogData)KinesisConsumer.this.objectMapper.readValue(bytes, CloudWatchLogData.class);
                        Iterator iterator = data.logEvents.stream().map(le -> new CloudWatchLogEntry(data.logGroup, data.logStream, le.timestamp, le.message)).iterator();
                        while (iterator.hasNext()) {
                            CloudWatchLogEntry next = (CloudWatchLogEntry)iterator.next();
                            KinesisConsumer.this.dataHandler.accept(KinesisConsumer.this.objectMapper.writeValueAsBytes((Object)next));
                        }
                        KinesisConsumer.this.lastSuccessfulRecordSequence = record.getSequenceNumber();
                    }
                    catch (Exception e) {
                        LOG.error("Couldn't read Kinesis record from stream <{}>", (Object)KinesisConsumer.this.kinesisStreamName, (Object)e);
                    }
                }
                if (this.lastCheckpoint.plusMinutes(1).isBeforeNow()) {
                    this.lastCheckpoint = DateTime.now();
                    LOG.debug("Checkpointing stream <{}>", (Object)KinesisConsumer.this.kinesisStreamName);
                    this.checkpoint(processRecordsInput, null);
                }
            }

            private void checkpoint(ProcessRecordsInput processRecordsInput, String lastSequence) {
                Retryer retryer = RetryerBuilder.newBuilder().retryIfExceptionOfType(ThrottlingException.class).withWaitStrategy(WaitStrategies.fixedWait((long)1L, (TimeUnit)TimeUnit.SECONDS)).withStopStrategy(StopStrategies.stopAfterDelay((long)10L, (TimeUnit)TimeUnit.MINUTES)).withRetryListener(new RetryListener(){

                    public <V> void onRetry(Attempt<V> attempt) {
                        if (attempt.hasException()) {
                            LOG.warn("Checkpointing stream <{}> failed, retrying. (attempt {})", (Object)KinesisConsumer.this.kinesisStreamName, (Object)attempt.getAttemptNumber());
                        }
                    }
                }).build();
                try {
                    retryer.call(() -> {
                        try {
                            if (lastSequence != null) {
                                processRecordsInput.getCheckpointer().checkpoint(lastSequence);
                            } else {
                                processRecordsInput.getCheckpointer().checkpoint();
                            }
                        }
                        catch (InvalidStateException e) {
                            LOG.error("Couldn't save checkpoint to DynamoDB table used by the Kinesis client library - check database table", (Throwable)e);
                        }
                        catch (ShutdownException e) {
                            LOG.debug("Processor is shutting down, skipping checkpoint");
                        }
                        return null;
                    });
                }
                catch (ExecutionException e) {
                    LOG.error("Couldn't checkpoint stream <{}>", (Object)KinesisConsumer.this.kinesisStreamName, (Object)e);
                }
                catch (RetryException e) {
                    LOG.error("Checkpoint retry for stream <{}> finally failed", (Object)KinesisConsumer.this.kinesisStreamName, (Object)e);
                }
            }

            @Override
            public void shutdown(ShutdownInput shutdownInput) {
                LOG.info("Shutting down Kinesis worker for stream <{}>", (Object)KinesisConsumer.this.kinesisStreamName);
            }
        };
        this.worker = new Worker.Builder().recordProcessorFactory(recordProcessorFactory).config(config).build();
        LOG.debug("Before Kinesis worker runs");
        this.worker.run();
        this.transport.consumerState.set(KinesisTransportState.STOPPED);
        LOG.debug("After Kinesis worker runs");
    }

    public void stop() {
        if (this.worker != null) {
            this.worker.shutdown();
        }
    }
}

