const std = @import("std");
const assert = std.debug.assert;
const codegen = @import("../../codegen.zig");
const link = @import("../../link.zig");
const log = std.log.scoped(.codegen);
const tracking_log = std.log.scoped(.tracking);
const verbose_tracking_log = std.log.scoped(.verbose_tracking);
const wip_mir_log = std.log.scoped(.wip_mir);
const Writer = std.io.Writer;

const Air = @import("../../Air.zig");
const Allocator = std.mem.Allocator;
const Emit = @import("Emit.zig");
const Lower = @import("Lower.zig");
const Mir = @import("Mir.zig");
const Zcu = @import("../../Zcu.zig");
const Module = @import("../../Package/Module.zig");
const InternPool = @import("../../InternPool.zig");
const Type = @import("../../Type.zig");
const Value = @import("../../Value.zig");

const abi = @import("abi.zig");
const bits = @import("bits.zig");
const encoder = @import("encoder.zig");

const Condition = bits.Condition;
const Immediate = bits.Immediate;
const Memory = bits.Memory;
const Register = bits.Register;
const RegisterManager = abi.RegisterManager;
const RegisterLock = RegisterManager.RegisterLock;
const FrameIndex = bits.FrameIndex;

const InnerError = codegen.CodeGenError || error{OutOfRegisters};

pub fn legalizeFeatures(_: *const std.Target) *const Air.Legalize.Features {
    return comptime &.initMany(&.{
        .scalarize_mul_sat,
        .scalarize_div_floor,
        .scalarize_mod,
        .scalarize_add_with_overflow,
        .scalarize_sub_with_overflow,
        .scalarize_mul_with_overflow,
        .scalarize_shl_with_overflow,
        .scalarize_shr,
        .scalarize_shr_exact,
        .scalarize_shl,
        .scalarize_shl_exact,
        .scalarize_shl_sat,
        .scalarize_bitcast,
        .scalarize_ctz,
        .scalarize_popcount,
        .scalarize_byte_swap,
        .scalarize_bit_reverse,
        .scalarize_cmp_vector,
        .scalarize_cmp_vector_optimized,
        .scalarize_shuffle_one,
        .scalarize_shuffle_two,
        .scalarize_select,

        //.unsplat_shift_rhs,
        .reduce_one_elem_to_bitcast,
        .splat_one_elem_to_bitcast,

        .expand_intcast_safe,
        .expand_int_from_float_safe,
        .expand_int_from_float_optimized_safe,
        .expand_add_safe,
        .expand_sub_safe,
        .expand_mul_safe,

        .expand_packed_load,
        .expand_packed_store,
        .expand_packed_struct_field_val,
        .expand_packed_aggregate_init,
    });
}

/// Set this to `false` to uncover Sema OPV bugs.
/// https://github.com/ziglang/zig/issues/22419
const hack_around_sema_opv_bugs = true;

const err_ret_trace_index: Air.Inst.Index = @enumFromInt(std.math.maxInt(u32));

gpa: Allocator,
pt: Zcu.PerThread,
air: Air,
liveness: Air.Liveness,
target: *const std.Target,
owner: union(enum) {
    nav_index: InternPool.Nav.Index,
    lazy_sym: link.File.LazySymbol,
},
inline_func: InternPool.Index,
mod: *Module,
args: []MCValue,
va_info: union {
    sysv: struct {
        gp_count: u32,
        fp_count: u32,
        overflow_arg_area: bits.FrameAddr,
        reg_save_area: bits.FrameAddr,
    },
    win64: struct {},
},
ret_mcv: InstTracking,
err_ret_trace_reg: Register,
fn_type: Type,
src_loc: Zcu.LazySrcLoc,

eflags_inst: ?Air.Inst.Index = null,

/// MIR Instructions
mir_instructions: std.MultiArrayList(Mir.Inst) = .empty,
/// MIR extra data
mir_extra: std.ArrayListUnmanaged(u32) = .empty,
mir_string_bytes: std.ArrayListUnmanaged(u8) = .empty,
mir_strings: std.HashMapUnmanaged(
    u32,
    void,
    std.hash_map.StringIndexContext,
    std.hash_map.default_max_load_percentage,
) = .empty,
mir_locals: std.ArrayListUnmanaged(Mir.Local) = .empty,
mir_table: std.ArrayListUnmanaged(Mir.Inst.Index) = .empty,

/// The value is an offset into the `Function` `code` from the beginning.
/// To perform the reloc, write 32-bit signed little-endian integer
/// which is a relative jump, based on the address following the reloc.
epilogue_relocs: std.ArrayListUnmanaged(Mir.Inst.Index) = .empty,

reused_operands: std.StaticBitSet(Air.Liveness.bpi - 1) = undefined,
inst_tracking: InstTrackingMap = .empty,

// Key is the block instruction
blocks: std.AutoHashMapUnmanaged(Air.Inst.Index, BlockData) = .empty,

register_manager: RegisterManager = .{},

/// Generation of the current scope, increments by 1 for every entered scope.
scope_generation: u32 = 0,

frame_allocs: std.MultiArrayList(FrameAlloc) = .empty,
free_frame_indices: std.AutoArrayHashMapUnmanaged(FrameIndex, void) = .empty,
frame_locs: std.MultiArrayList(Mir.FrameLoc) = .empty,

loops: std.AutoHashMapUnmanaged(Air.Inst.Index, struct {
    /// The state to restore before branching.
    state: State,
    /// The branch target.
    target: Mir.Inst.Index,
}) = .empty,
loop_switches: std.AutoHashMapUnmanaged(Air.Inst.Index, struct {
    start: u31,
    len: u11,
    min: Value,
    else_relocs: union(enum) {
        @"unreachable",
        forward: std.ArrayListUnmanaged(Mir.Inst.Index),
        backward: Mir.Inst.Index,
    },
}) = .empty,

next_temp_index: Temp.Index = @enumFromInt(0),
temp_type: [Temp.Index.max]Type = undefined,

const MaskInfo = packed struct {
    kind: enum(u1) { sign, all },
    inverted: bool = false,
    scalar: Memory.Size,
};

pub const MCValue = union(enum) {
    /// No runtime bits. `void` types, empty structs, u0, enums with 1 tag, etc.
    /// TODO Look into deleting this tag and using `dead` instead, since every use
    /// of MCValue.none should be instead looking at the type and noticing it is 0 bits.
    none,
    /// Control flow will not allow this value to be observed.
    unreach,
    /// No more references to this value remain.
    /// The payload is the value of scope_generation at the point where the death occurred
    dead: u32,
    /// The value is undefined.
    undef,
    /// A pointer-sized integer that fits in a register.
    /// If the type is a pointer, this is the pointer address in virtual address space.
    immediate: u64,
    /// The value resides in the EFLAGS register.
    eflags: Condition,
    /// The value is in a register.
    register: Register,
    /// The value is split across two registers.
    register_pair: [2]Register,
    /// The value is split across three registers.
    register_triple: [3]Register,
    /// The value is split across four registers.
    register_quadruple: [4]Register,
    /// The value is a constant offset from the value in a register.
    register_offset: bits.RegisterOffset,
    /// The value is a tuple { wrapped, overflow } where wrapped value is stored in the GP register.
    register_overflow: struct { reg: Register, eflags: Condition },
    /// The value is a bool vector stored in a vector register with a different scalar type.
    register_mask: struct { reg: Register, info: MaskInfo },
    /// The value is in memory at a hard-coded address.
    /// If the type is a pointer, it means the pointer address is stored at this memory location.
    memory: u64,
    /// The value is in memory at a constant offset from the address in a register.
    indirect: bits.RegisterOffset,
    indirect_load_frame: bits.FrameAddr,
    /// The value stored at an offset from a frame index
    /// Payload is a frame address.
    load_frame: bits.FrameAddr,
    /// The address of an offset from a frame index
    /// Payload is a frame address.
    lea_frame: bits.FrameAddr,
    load_nav: InternPool.Nav.Index,
    lea_nav: InternPool.Nav.Index,
    load_uav: InternPool.Key.Ptr.BaseAddr.Uav,
    lea_uav: InternPool.Key.Ptr.BaseAddr.Uav,
    load_lazy_sym: link.File.LazySymbol,
    lea_lazy_sym: link.File.LazySymbol,
    load_extern_func: Mir.NullTerminatedString,
    lea_extern_func: Mir.NullTerminatedString,
    /// Supports integer_per_element abi
    elementwise_args: packed struct { regs: u3, frame_off: i29, frame_index: FrameIndex },
    /// This indicates that we have already allocated a frame index for this instruction,
    /// but it has not been spilled there yet in the current control flow.
    /// Payload is a frame index.
    reserved_frame: FrameIndex,
    air_ref: Air.Inst.Ref,

    fn isModifiable(mcv: MCValue) bool {
        return switch (mcv) {
            .none,
            .unreach,
            .dead,
            .undef,
            .immediate,
            .register_offset,
            .register_mask,
            .eflags,
            .register_overflow,
            .indirect_load_frame,
            .lea_frame,
            .lea_nav,
            .load_uav,
            .lea_uav,
            .load_lazy_sym,
            .lea_lazy_sym,
            .lea_extern_func,
            .load_extern_func,
            .elementwise_args,
            .reserved_frame,
            .air_ref,
            => false,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .memory,
            .indirect,
            .load_nav,
            => true,
            .load_frame => |frame_addr| !frame_addr.index.isNamed(),
        };
    }

    // hack around linker relocation bugs
    fn isBase(mcv: MCValue) bool {
        return switch (mcv) {
            .memory, .indirect, .load_frame => true,
            else => false,
        };
    }

    fn isMemory(mcv: MCValue) bool {
        return switch (mcv) {
            .memory,
            .indirect,
            .load_frame,
            .load_nav,
            .load_uav,
            .load_lazy_sym,
            .load_extern_func,
            => true,
            else => false,
        };
    }

    fn isImmediate(mcv: MCValue) bool {
        return switch (mcv) {
            .immediate => true,
            else => false,
        };
    }

    fn isRegister(mcv: MCValue) bool {
        return switch (mcv) {
            .register => true,
            .register_offset => |reg_off| return reg_off.off == 0,
            else => false,
        };
    }

    fn isRegisterOffset(mcv: MCValue) bool {
        return switch (mcv) {
            .register, .register_offset => true,
            else => false,
        };
    }

    fn getReg(mcv: MCValue) ?Register {
        return switch (mcv) {
            .register => |reg| reg,
            .register_offset, .indirect => |ro| ro.reg,
            .register_overflow => |ro| ro.reg,
            .register_mask => |rm| rm.reg,
            else => null,
        };
    }

    fn getRegs(mcv: *const MCValue) []const Register {
        return switch (mcv.*) {
            .register => |*reg| reg[0..1],
            inline .register_pair,
            .register_triple,
            .register_quadruple,
            => |*regs| regs,
            inline .register_offset,
            .indirect,
            .register_overflow,
            .register_mask,
            => |*pl| (&pl.reg)[0..1],
            else => &.{},
        };
    }

    fn getCondition(mcv: MCValue) ?Condition {
        return switch (mcv) {
            .eflags => |cc| cc,
            .register_overflow => |reg_ov| reg_ov.eflags,
            else => null,
        };
    }

    fn isAddress(mcv: MCValue) bool {
        return switch (mcv) {
            .immediate, .register, .register_offset, .lea_frame => true,
            else => false,
        };
    }

    fn address(mcv: MCValue) MCValue {
        return switch (mcv) {
            .none => .none,
            .unreach,
            .dead,
            .undef,
            .immediate,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .lea_frame,
            .lea_nav,
            .lea_uav,
            .lea_lazy_sym,
            .lea_extern_func,
            .elementwise_args,
            .reserved_frame,
            .air_ref,
            => unreachable, // not in memory
            .memory => |addr| .{ .immediate = addr },
            .indirect => |reg_off| switch (reg_off.off) {
                0 => .{ .register = reg_off.reg },
                else => .{ .register_offset = reg_off },
            },
            .indirect_load_frame => |frame_addr| .{ .load_frame = frame_addr },
            .load_frame => |frame_addr| .{ .lea_frame = frame_addr },
            .load_nav => |nav| .{ .lea_nav = nav },
            .load_uav => |uav| .{ .lea_uav = uav },
            .load_lazy_sym => |lazy_sym| .{ .lea_lazy_sym = lazy_sym },
            .load_extern_func => |extern_func| .{ .lea_extern_func = extern_func },
        };
    }

    fn deref(mcv: MCValue) MCValue {
        return switch (mcv) {
            .none => .none,
            .unreach,
            .dead,
            .undef,
            .eflags,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_overflow,
            .register_mask,
            .memory,
            .indirect,
            .indirect_load_frame,
            .load_nav,
            .load_uav,
            .load_lazy_sym,
            .load_extern_func,
            .elementwise_args,
            .reserved_frame,
            .air_ref,
            => unreachable, // not dereferenceable
            .immediate => |addr| .{ .memory = addr },
            .register => |reg| .{ .indirect = .{ .reg = reg } },
            .register_offset => |reg_off| .{ .indirect = reg_off },
            .load_frame => |frame_addr| .{ .indirect_load_frame = frame_addr },
            .lea_frame => |frame_addr| .{ .load_frame = frame_addr },
            .lea_nav => |nav| .{ .load_nav = nav },
            .lea_uav => |uav| .{ .load_uav = uav },
            .lea_lazy_sym => |lazy_sym| .{ .load_lazy_sym = lazy_sym },
            .lea_extern_func => |extern_func| .{ .load_extern_func = extern_func },
        };
    }

    fn offset(mcv: MCValue, off: i32) MCValue {
        return switch (mcv) {
            .none => .none,
            .unreach,
            .dead,
            .undef,
            .elementwise_args,
            .reserved_frame,
            .air_ref,
            => unreachable, // not valid
            .eflags,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_overflow,
            .register_mask,
            .memory,
            .indirect,
            .indirect_load_frame,
            .load_frame,
            .load_nav,
            .lea_nav,
            .load_uav,
            .lea_uav,
            .load_lazy_sym,
            .lea_lazy_sym,
            .load_extern_func,
            .lea_extern_func,
            => switch (off) {
                0 => mcv,
                else => unreachable, // not offsettable
            },
            .immediate => |imm| .{ .immediate = @bitCast(@as(i64, @bitCast(imm)) +% off) },
            .register => |reg| .{ .register_offset = .{ .reg = reg, .off = off } },
            .register_offset => |reg_off| .{
                .register_offset = .{ .reg = reg_off.reg, .off = reg_off.off + off },
            },
            .lea_frame => |frame_addr| .{
                .lea_frame = .{ .index = frame_addr.index, .off = frame_addr.off + off },
            },
        };
    }

    fn mem(mcv: MCValue, function: *CodeGen, mod_rm: Memory.Mod.Rm) !Memory {
        return switch (mcv) {
            .none => .{ .mod = .{ .rm = mod_rm } },
            .unreach,
            .dead,
            .undef,
            .immediate,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .indirect_load_frame,
            .lea_frame,
            .elementwise_args,
            .reserved_frame,
            .lea_nav,
            .lea_uav,
            .lea_lazy_sym,
            .lea_extern_func,
            => unreachable,
            .memory => |addr| if (std.math.cast(i32, @as(i64, @bitCast(addr)))) |small_addr| .{
                .base = .{ .reg = .ds },
                .mod = .{ .rm = .{
                    .size = mod_rm.size,
                    .index = mod_rm.index,
                    .scale = mod_rm.scale,
                    .disp = small_addr + mod_rm.disp,
                } },
            } else .{ .base = .{ .reg = .ds }, .mod = .{ .off = addr } },
            .indirect => |reg_off| .{
                .base = .{ .reg = reg_off.reg.toSize(.ptr, function.target) },
                .mod = .{ .rm = .{
                    .size = mod_rm.size,
                    .index = mod_rm.index,
                    .scale = mod_rm.scale,
                    .disp = reg_off.off + mod_rm.disp,
                } },
            },
            .load_frame => |frame_addr| .{
                .base = .{ .frame = frame_addr.index },
                .mod = .{ .rm = .{
                    .size = mod_rm.size,
                    .index = mod_rm.index,
                    .scale = mod_rm.scale,
                    .disp = frame_addr.off + mod_rm.disp,
                } },
            },
            .load_nav => |nav| .{ .base = .{ .nav = nav }, .mod = .{ .rm = mod_rm } },
            .load_uav => |uav| .{ .base = .{ .uav = uav }, .mod = .{ .rm = mod_rm } },
            .load_lazy_sym => |lazy_sym| .{ .base = .{ .lazy_sym = lazy_sym }, .mod = .{ .rm = mod_rm } },
            .load_extern_func => |extern_func| .{ .base = .{ .extern_func = extern_func }, .mod = .{ .rm = mod_rm } },
            .air_ref => |ref| (try function.resolveInst(ref)).mem(function, mod_rm),
        };
    }

    pub fn format(mcv: MCValue, w: *Writer) Writer.Error!void {
        switch (mcv) {
            .none, .unreach, .dead, .undef => try w.print("({s})", .{@tagName(mcv)}),
            .immediate => |pl| try w.print("0x{x}", .{pl}),
            .memory => |pl| try w.print("[ds:0x{x}]", .{pl}),
            inline .eflags, .register => |pl| try w.print("{s}", .{@tagName(pl)}),
            .register_pair => |pl| try w.print("{s}:{s}", .{ @tagName(pl[1]), @tagName(pl[0]) }),
            .register_triple => |pl| try w.print("{s}:{s}:{s}", .{
                @tagName(pl[2]), @tagName(pl[1]), @tagName(pl[0]),
            }),
            .register_quadruple => |pl| try w.print("{s}:{s}:{s}:{s}", .{
                @tagName(pl[3]), @tagName(pl[2]), @tagName(pl[1]), @tagName(pl[0]),
            }),
            .register_offset => |pl| try w.print("{s} + 0x{x}", .{ @tagName(pl.reg), pl.off }),
            .register_overflow => |pl| try w.print("{s}:{s}", .{
                @tagName(pl.eflags),
                @tagName(pl.reg),
            }),
            .register_mask => |pl| try w.print("mask({s},{f}):{c}{s}", .{
                @tagName(pl.info.kind),
                pl.info.scalar,
                @as(u8, if (pl.info.inverted) '!' else ' '),
                @tagName(pl.reg),
            }),
            .indirect => |pl| try w.print("[{s} + 0x{x}]", .{ @tagName(pl.reg), pl.off }),
            .indirect_load_frame => |pl| try w.print("[[{} + 0x{x}]]", .{ pl.index, pl.off }),
            .load_frame => |pl| try w.print("[{} + 0x{x}]", .{ pl.index, pl.off }),
            .lea_frame => |pl| try w.print("{} + 0x{x}", .{ pl.index, pl.off }),
            .load_nav => |pl| try w.print("[nav:{d}]", .{@intFromEnum(pl)}),
            .lea_nav => |pl| try w.print("nav:{d}", .{@intFromEnum(pl)}),
            .load_uav => |pl| try w.print("[uav:{d}]", .{@intFromEnum(pl.val)}),
            .lea_uav => |pl| try w.print("uav:{d}", .{@intFromEnum(pl.val)}),
            .load_lazy_sym => |pl| try w.print("[lazy:{s}:{d}]", .{ @tagName(pl.kind), @intFromEnum(pl.ty) }),
            .lea_lazy_sym => |pl| try w.print("lazy:{s}:{d}", .{ @tagName(pl.kind), @intFromEnum(pl.ty) }),
            .load_extern_func => |pl| try w.print("[extern:{d}]", .{@intFromEnum(pl)}),
            .lea_extern_func => |pl| try w.print("extern:{d}", .{@intFromEnum(pl)}),
            .elementwise_args => |pl| try w.print("elementwise:{d}:[{} + 0x{x}]", .{
                pl.regs, pl.frame_index, pl.frame_off,
            }),
            .reserved_frame => |pl| try w.print("(dead:{})", .{pl}),
            .air_ref => |pl| try w.print("(air:0x{x})", .{@intFromEnum(pl)}),
        }
    }
};

const InstTrackingMap = std.AutoArrayHashMapUnmanaged(Air.Inst.Index, InstTracking);
const ConstTrackingMap = std.AutoArrayHashMapUnmanaged(InternPool.Index, InstTracking);
const InstTracking = struct {
    long: MCValue,
    short: MCValue,

    fn init(result: MCValue) InstTracking {
        return .{ .long = switch (result) {
            .none,
            .unreach,
            .undef,
            .immediate,
            .memory,
            .indirect_load_frame,
            .load_frame,
            .lea_frame,
            .load_nav,
            .lea_nav,
            .load_uav,
            .lea_uav,
            .load_lazy_sym,
            .lea_lazy_sym,
            .load_extern_func,
            .lea_extern_func,
            => result,
            .dead,
            .elementwise_args,
            .reserved_frame,
            .air_ref,
            => unreachable,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .indirect,
            => .none,
        }, .short = result };
    }

    fn getReg(self: InstTracking) ?Register {
        return self.short.getReg();
    }

    fn getRegs(self: *const InstTracking) []const Register {
        return self.short.getRegs();
    }

    fn getCondition(self: InstTracking) ?Condition {
        return self.short.getCondition();
    }

    fn spill(self: *InstTracking, cg: *CodeGen, inst: Air.Inst.Index) !void {
        if (std.meta.eql(self.long, self.short)) return; // Already spilled
        // Allocate or reuse frame index
        switch (self.long) {
            .none => self.long = try cg.allocRegOrMem(inst, false),
            .load_frame => {},
            .lea_frame => return,
            .reserved_frame => |index| self.long = .{ .load_frame = .{ .index = index } },
            else => unreachable,
        }
        tracking_log.debug("spill {f} from {f} to {f}", .{ inst, self.short, self.long });
        try cg.genCopy(cg.typeOfIndex(inst), self.long, self.short, .{});
        for (self.short.getRegs()) |reg| if (reg.isClass(.x87)) try cg.asmRegister(.{ .f_, .free }, reg);
    }

    fn reuseFrame(self: *InstTracking) void {
        self.* = .init(switch (self.long) {
            .none => switch (self.short) {
                .dead => .none,
                else => |short| short,
            },
            .reserved_frame => |index| .{ .load_frame = .{ .index = index } },
            else => |long| long,
        });
    }

    fn trackSpill(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index) !void {
        switch (self.short) {
            .register => |reg| function.register_manager.freeReg(reg),
            inline .register_pair,
            .register_triple,
            .register_quadruple,
            => |regs| for (regs) |reg| function.register_manager.freeReg(reg),
            .register_offset, .indirect => |reg_off| function.register_manager.freeReg(reg_off.reg),
            .register_overflow => |reg_ov| {
                function.register_manager.freeReg(reg_ov.reg);
                function.eflags_inst = null;
            },
            .register_mask => |reg_mask| function.register_manager.freeReg(reg_mask.reg),
            .eflags => function.eflags_inst = null,
            else => {}, // TODO process stack allocation death
        }
        self.reuseFrame();
        tracking_log.debug("{f} => {f} (spilled)", .{ inst, self.* });
    }

    fn verifyMaterialize(self: InstTracking, target: InstTracking) void {
        switch (self.long) {
            .none,
            .load_frame,
            .reserved_frame,
            => switch (target.long) {
                .none,
                .load_frame,
                .reserved_frame,
                => {},
                else => unreachable,
            },
            .unreach,
            .undef,
            .immediate,
            .memory,
            .indirect_load_frame,
            .lea_frame,
            .load_nav,
            .lea_nav,
            .load_uav,
            .lea_uav,
            .load_lazy_sym,
            .lea_lazy_sym,
            .load_extern_func,
            .lea_extern_func,
            => assert(std.meta.eql(self.long, target.long)),
            .dead,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .indirect,
            .elementwise_args,
            .air_ref,
            => unreachable,
        }
    }

    fn materialize(
        self: *InstTracking,
        function: *CodeGen,
        inst: Air.Inst.Index,
        target: InstTracking,
    ) !void {
        self.verifyMaterialize(target);
        try self.materializeUnsafe(function, inst, target);
    }

    fn materializeUnsafe(
        self: InstTracking,
        function: *CodeGen,
        inst: Air.Inst.Index,
        target: InstTracking,
    ) !void {
        const ty = function.typeOfIndex(inst);
        if ((self.long == .none or self.long == .reserved_frame) and target.long == .load_frame)
            try function.genCopy(ty, target.long, self.short, .{});
        try function.genCopy(ty, target.short, self.short, .{});
    }

    fn trackMaterialize(self: *InstTracking, inst: Air.Inst.Index, target: InstTracking) void {
        self.verifyMaterialize(target);
        // Don't clobber reserved frame indices
        self.long = if (target.long == .none) switch (self.long) {
            .load_frame => |addr| .{ .reserved_frame = addr.index },
            .reserved_frame => self.long,
            else => target.long,
        } else target.long;
        self.short = target.short;
        tracking_log.debug("{f} => {f} (materialize)", .{ inst, self.* });
    }

    fn resurrect(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index, scope_generation: u32) !void {
        switch (self.short) {
            .dead => |die_generation| if (die_generation >= scope_generation) {
                self.reuseFrame();
                try function.getValue(self.short, inst);
                tracking_log.debug("{f} => {f} (resurrect)", .{ inst, self.* });
            },
            else => {},
        }
    }

    fn die(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index, comptime opts: FreeOptions) !void {
        if (self.short == .dead) return;
        try function.freeValue(self.short, opts);
        if (self.long == .none) self.long = self.short;
        self.short = .{ .dead = function.scope_generation };
        tracking_log.debug("{f} => {f} (death)", .{ inst, self.* });
    }

    fn reuse(
        self: *InstTracking,
        function: *CodeGen,
        new_inst: ?Air.Inst.Index,
        old_inst: Air.Inst.Index,
    ) void {
        self.short = .{ .dead = function.scope_generation };
        tracking_log.debug("{?f} => {f} (reuse {f})", .{ new_inst, self.*, old_inst });
    }

    fn liveOut(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index) void {
        for (self.getRegs()) |reg| {
            if (function.register_manager.isRegFree(reg)) {
                tracking_log.debug("{f} => {f} (live-out)", .{ inst, self.* });
                continue;
            }

            const index = RegisterManager.indexOfRegIntoTracked(reg).?;
            const tracked_inst = function.register_manager.registers[index];
            const tracking = function.getResolvedInstValue(tracked_inst);

            // Disable death.
            var found_reg = false;
            var remaining_reg: Register = .none;
            for (tracking.getRegs()) |tracked_reg| if (tracked_reg.id() == reg.id()) {
                assert(!found_reg);
                found_reg = true;
            } else {
                assert(remaining_reg == .none);
                remaining_reg = tracked_reg;
            };
            assert(found_reg);
            if (tracking.long == .none) tracking.long = tracking.short;
            tracking.short = switch (remaining_reg) {
                .none => .{ .dead = function.scope_generation },
                else => .{ .register = remaining_reg },
            };

            // Perform side-effects of freeValue manually.
            function.register_manager.freeReg(reg);

            tracking_log.debug("{f} => {f} (live-out {f})", .{ inst, self.*, tracked_inst });
        }
    }

    pub fn format(tracking: InstTracking, bw: *Writer) Writer.Error!void {
        if (!std.meta.eql(tracking.long, tracking.short)) try bw.print("|{f}| ", .{tracking.long});
        try bw.print("{f}", .{tracking.short});
    }
};

const FrameAlloc = struct {
    abi_size: u31,
    spill_pad: u3,
    abi_align: InternPool.Alignment,
    ref_count: u16,

    fn init(alloc_abi: struct { size: u64, pad: u3 = 0, alignment: InternPool.Alignment }) FrameAlloc {
        return .{
            .abi_size = @intCast(alloc_abi.size),
            .spill_pad = alloc_abi.pad,
            .abi_align = alloc_abi.alignment,
            .ref_count = 0,
        };
    }
    fn initType(ty: Type, zcu: *Zcu) FrameAlloc {
        return init(.{
            .size = ty.abiSize(zcu),
            .alignment = ty.abiAlignment(zcu),
        });
    }
    fn initSpill(ty: Type, zcu: *Zcu) FrameAlloc {
        const abi_size = ty.abiSize(zcu);
        const spill_size = if (abi_size < 8)
            std.math.ceilPowerOfTwoAssert(u64, abi_size)
        else
            std.mem.alignForward(u64, abi_size, 8);
        return init(.{
            .size = spill_size,
            .pad = @intCast(spill_size - abi_size),
            .alignment = ty.abiAlignment(zcu).maxStrict(
                .fromNonzeroByteUnits(@min(spill_size, 8)),
            ),
        });
    }
};

const StackAllocation = struct {
    inst: ?Air.Inst.Index,
    /// TODO do we need size? should be determined by inst.ty.abiSize(zcu)
    size: u32,
};

const BlockData = struct {
    relocs: std.ArrayListUnmanaged(Mir.Inst.Index) = .empty,
    state: State,

    fn deinit(self: *BlockData, gpa: Allocator) void {
        self.relocs.deinit(gpa);
        self.* = undefined;
    }
};

const CodeGen = @This();

pub fn generate(
    bin_file: *link.File,
    pt: Zcu.PerThread,
    src_loc: Zcu.LazySrcLoc,
    func_index: InternPool.Index,
    air: *const Air,
    liveness: *const ?Air.Liveness,
) codegen.CodeGenError!Mir {
    _ = bin_file;
    const zcu = pt.zcu;
    const gpa = zcu.gpa;
    const ip = &zcu.intern_pool;
    const func = zcu.funcInfo(func_index);
    const func_zir = func.zir_body_inst.resolveFull(ip).?;
    const file = zcu.fileByIndex(func_zir.file);
    const fn_type: Type = .fromInterned(func.ty);
    const mod = file.mod.?;

    var function: CodeGen = .{
        .gpa = gpa,
        .pt = pt,
        .air = air.*,
        .liveness = liveness.*.?,
        .target = &mod.resolved_target.result,
        .mod = mod,
        .owner = .{ .nav_index = func.owner_nav },
        .inline_func = func_index,
        .args = undefined, // populated after `resolveCallingConventionValues`
        .va_info = undefined, // populated after `resolveCallingConventionValues`
        .ret_mcv = undefined, // populated after `resolveCallingConventionValues`
        .err_ret_trace_reg = undefined, // populated after `resolveCallingConventionValues`
        .fn_type = fn_type,
        .src_loc = src_loc,
    };
    defer {
        function.frame_allocs.deinit(gpa);
        function.free_frame_indices.deinit(gpa);
        function.frame_locs.deinit(gpa);
        function.loops.deinit(gpa);
        function.loop_switches.deinit(gpa);
        var block_it = function.blocks.valueIterator();
        while (block_it.next()) |block| block.deinit(gpa);
        function.blocks.deinit(gpa);
        function.inst_tracking.deinit(gpa);
        function.epilogue_relocs.deinit(gpa);
        function.mir_instructions.deinit(gpa);
        function.mir_string_bytes.deinit(gpa);
        function.mir_strings.deinit(gpa);
        function.mir_locals.deinit(gpa);
        function.mir_extra.deinit(gpa);
        function.mir_table.deinit(gpa);
    }
    try function.inst_tracking.ensureTotalCapacity(gpa, Temp.Index.max);
    for (0..Temp.Index.max) |temp_index| {
        const temp: Temp.Index = @enumFromInt(temp_index);
        function.inst_tracking.putAssumeCapacityNoClobber(temp.toIndex(), .init(.none));
    }

    wip_mir_log.debug("{f}:", .{fmtNav(func.owner_nav, ip)});

    try function.frame_allocs.resize(gpa, FrameIndex.named_count);
    function.frame_allocs.set(
        @intFromEnum(FrameIndex.stack_frame),
        .init(.{ .size = 0, .alignment = .@"1" }),
    );
    function.frame_allocs.set(
        @intFromEnum(FrameIndex.call_frame),
        .init(.{ .size = 0, .alignment = .@"1" }),
    );

    const fn_info = zcu.typeToFunc(fn_type).?;
    var call_info = function.resolveCallingConventionValues(fn_info, &.{}, .args_frame) catch |err| switch (err) {
        error.CodegenFail => return error.CodegenFail,
        else => |e| return e,
    };
    defer call_info.deinit(&function);

    function.args = call_info.args;
    function.ret_mcv = call_info.return_value;
    function.err_ret_trace_reg = call_info.err_ret_trace_reg;
    function.frame_allocs.set(@intFromEnum(FrameIndex.ret_addr), .init(.{
        .size = Type.usize.abiSize(zcu),
        .alignment = Type.usize.abiAlignment(zcu).min(call_info.stack_align),
    }));
    function.frame_allocs.set(@intFromEnum(FrameIndex.base_ptr), .init(.{
        .size = Type.usize.abiSize(zcu),
        .alignment = call_info.stack_align.min(
            .fromNonzeroByteUnits(function.target.stackAlignment()),
        ),
    }));
    function.frame_allocs.set(
        @intFromEnum(FrameIndex.args_frame),
        .init(.{
            .size = call_info.stack_byte_count,
            .alignment = call_info.stack_align,
        }),
    );
    function.va_info = switch (fn_info.cc) {
        else => undefined,
        .x86_64_sysv => .{ .sysv = .{
            .gp_count = call_info.gp_count,
            .fp_count = call_info.fp_count,
            .overflow_arg_area = .{ .index = .args_frame, .off = call_info.stack_byte_count },
            .reg_save_area = undefined,
        } },
        .x86_64_win => .{ .win64 = .{} },
    };
    if (call_info.err_ret_trace_reg != .none) {
        function.register_manager.getRegAssumeFree(call_info.err_ret_trace_reg, err_ret_trace_index);
        try function.inst_tracking.putNoClobber(
            gpa,
            err_ret_trace_index,
            .init(.{ .register = call_info.err_ret_trace_reg }),
        );
    }

    function.gen(&file.zir.?, func_zir.inst, func.comptime_args, call_info.air_arg_count) catch |err| switch (err) {
        error.CodegenFail => return error.CodegenFail,
        error.OutOfRegisters => return function.fail("ran out of registers (Zig compiler bug)", .{}),
        else => |e| return e,
    };

    // Drop them off at the rbrace.
    if (!mod.strip) _ = try function.addInst(.{
        .tag = .pseudo,
        .ops = .pseudo_dbg_line_line_column,
        .data = .{ .line_column = .{
            .line = func.rbrace_line,
            .column = func.rbrace_column,
        } },
    });

    var mir: Mir = .{
        .instructions = .empty,
        .extra = &.{},
        .string_bytes = &.{},
        .locals = &.{},
        .table = &.{},
        .frame_locs = .empty,
    };
    errdefer mir.deinit(gpa);
    mir.instructions = function.mir_instructions.toOwnedSlice();
    mir.extra = try function.mir_extra.toOwnedSlice(gpa);
    mir.string_bytes = try function.mir_string_bytes.toOwnedSlice(gpa);
    mir.locals = try function.mir_locals.toOwnedSlice(gpa);
    mir.table = try function.mir_table.toOwnedSlice(gpa);
    mir.frame_locs = function.frame_locs.toOwnedSlice();
    return mir;
}

pub fn getTmpMir(cg: *CodeGen) Mir {
    return .{
        .instructions = cg.mir_instructions.slice(),
        .extra = cg.mir_extra.items,
        .string_bytes = cg.mir_string_bytes.items,
        .locals = cg.mir_locals.items,
        .table = cg.mir_table.items,
        .frame_locs = cg.frame_locs.slice(),
    };
}

pub fn generateLazy(
    bin_file: *link.File,
    pt: Zcu.PerThread,
    src_loc: Zcu.LazySrcLoc,
    lazy_sym: link.File.LazySymbol,
    code: *std.ArrayListUnmanaged(u8),
    debug_output: link.File.DebugInfoOutput,
) codegen.CodeGenError!void {
    const gpa = pt.zcu.gpa;
    // This function is for generating global code, so we use the root module.
    const mod = pt.zcu.comp.root_mod;
    var function: CodeGen = .{
        .gpa = gpa,
        .pt = pt,
        .air = undefined,
        .liveness = undefined,
        .target = &mod.resolved_target.result,
        .mod = mod,
        .owner = .{ .lazy_sym = lazy_sym },
        .inline_func = undefined,
        .args = undefined,
        .va_info = undefined,
        .ret_mcv = undefined,
        .err_ret_trace_reg = undefined,
        .fn_type = undefined,
        .src_loc = src_loc,
    };
    defer {
        function.inst_tracking.deinit(gpa);
        function.mir_instructions.deinit(gpa);
        function.mir_string_bytes.deinit(gpa);
        function.mir_strings.deinit(gpa);
        function.mir_locals.deinit(gpa);
        function.mir_extra.deinit(gpa);
        function.mir_table.deinit(gpa);
    }
    try function.inst_tracking.ensureTotalCapacity(gpa, Temp.Index.max);
    for (0..Temp.Index.max) |temp_index| {
        const temp: Temp.Index = @enumFromInt(temp_index);
        function.inst_tracking.putAssumeCapacityNoClobber(temp.toIndex(), .init(.none));
    }

    function.genLazy(lazy_sym) catch |err| switch (err) {
        error.CodegenFail => return error.CodegenFail,
        error.OutOfRegisters => return function.fail("ran out of registers (Zig compiler bug)", .{}),
        else => |e| return e,
    };

    try function.getTmpMir().emitLazy(bin_file, pt, src_loc, lazy_sym, code, debug_output);
}

const FormatNavData = struct {
    ip: *const InternPool,
    nav_index: InternPool.Nav.Index,
};
fn formatNav(data: FormatNavData, w: *Writer) Writer.Error!void {
    try w.print("{f}", .{data.ip.getNav(data.nav_index).fqn.fmt(data.ip)});
}
fn fmtNav(nav_index: InternPool.Nav.Index, ip: *const InternPool) std.fmt.Formatter(FormatNavData, formatNav) {
    return .{ .data = .{
        .ip = ip,
        .nav_index = nav_index,
    } };
}

const FormatAirData = struct {
    self: *CodeGen,
    inst: Air.Inst.Index,
};
fn formatAir(data: FormatAirData, w: *std.io.Writer) Writer.Error!void {
    data.self.air.writeInst(w, data.inst, data.self.pt, data.self.liveness);
}
fn fmtAir(self: *CodeGen, inst: Air.Inst.Index) std.fmt.Formatter(FormatAirData, formatAir) {
    return .{ .data = .{ .self = self, .inst = inst } };
}

const FormatWipMirData = struct {
    self: *CodeGen,
    inst: Mir.Inst.Index,
};
fn formatWipMir(data: FormatWipMirData, w: *Writer) Writer.Error!void {
    var lower: Lower = .{
        .target = data.self.target,
        .allocator = data.self.gpa,
        .mir = data.self.getTmpMir(),
        .cc = .auto,
        .src_loc = data.self.src_loc,
    };
    var first = true;
    for ((lower.lowerMir(data.inst) catch |err| switch (err) {
        error.LowerFail => {
            defer {
                lower.err_msg.?.deinit(data.self.gpa);
                lower.err_msg = null;
            }
            try w.writeAll(lower.err_msg.?.msg);
            return;
        },
        else => |e| {
            try w.writeAll(@errorName(e));
            return;
        },
    }).insts) |lowered_inst| {
        if (!first) try w.writeAll("\ndebug(wip_mir): ");
        try w.print("  | {f}", .{lowered_inst});
        first = false;
    }
    if (first) {
        const ip = &data.self.pt.zcu.intern_pool;
        const mir_inst = lower.mir.instructions.get(data.inst);
        try w.print("  | .{s}", .{@tagName(mir_inst.ops)});
        switch (mir_inst.ops) {
            else => unreachable,
            .pseudo_dbg_prologue_end_none,
            .pseudo_dbg_epilogue_begin_none,
            .pseudo_dbg_enter_block_none,
            .pseudo_dbg_leave_block_none,
            .pseudo_dbg_arg_none,
            .pseudo_dbg_var_args_none,
            .pseudo_dbg_var_none,
            .pseudo_dead_none,
            => {},
            .pseudo_dbg_line_stmt_line_column, .pseudo_dbg_line_line_column => try w.print(
                " {[line]d}, {[column]d}",
                mir_inst.data.line_column,
            ),
            .pseudo_dbg_enter_inline_func, .pseudo_dbg_leave_inline_func => try w.print(" {f}", .{
                ip.getNav(ip.indexToKey(mir_inst.data.ip_index).func.owner_nav).name.fmt(ip),
            }),
            .pseudo_dbg_arg_i_s, .pseudo_dbg_var_i_s => try w.print(" {d}", .{
                @as(i32, @bitCast(mir_inst.data.i.i)),
            }),
            .pseudo_dbg_arg_i_u, .pseudo_dbg_var_i_u => try w.print(" {d}", .{
                mir_inst.data.i.i,
            }),
            .pseudo_dbg_arg_i_64, .pseudo_dbg_var_i_64 => try w.print(" {d}", .{
                mir_inst.data.i64,
            }),
            .pseudo_dbg_arg_ro, .pseudo_dbg_var_ro => {
                const mem_op: encoder.Instruction.Operand = .{ .mem = .initSib(.qword, .{
                    .base = .{ .reg = mir_inst.data.ro.reg },
                    .disp = mir_inst.data.ro.off,
                }) };
                try w.print(" {f}", .{mem_op.fmt(.m)});
            },
            .pseudo_dbg_arg_fa, .pseudo_dbg_var_fa => {
                const mem_op: encoder.Instruction.Operand = .{ .mem = .initSib(.qword, .{
                    .base = .{ .frame = mir_inst.data.fa.index },
                    .disp = mir_inst.data.fa.off,
                }) };
                try w.print(" {f}", .{mem_op.fmt(.m)});
            },
            .pseudo_dbg_arg_m, .pseudo_dbg_var_m => {
                const mem_op: encoder.Instruction.Operand = .{
                    .mem = lower.mir.extraData(Mir.Memory, mir_inst.data.x.payload).data.decode(),
                };
                try w.print(" {f}", .{mem_op.fmt(.m)});
            },
            .pseudo_dbg_arg_val, .pseudo_dbg_var_val => try w.print(" {f}", .{
                Value.fromInterned(mir_inst.data.ip_index).fmtValue(data.self.pt),
            }),
        }
    }
}
fn fmtWipMir(self: *CodeGen, inst: Mir.Inst.Index) std.fmt.Formatter(FormatWipMirData, formatWipMir) {
    return .{ .data = .{ .self = self, .inst = inst } };
}

const FormatTrackingData = struct {
    self: *CodeGen,
};
fn formatTracking(data: FormatTrackingData, w: *Writer) Writer.Error!void {
    var it = data.self.inst_tracking.iterator();
    while (it.next()) |entry| try w.print("\n{f} = {f}", .{ entry.key_ptr.*, entry.value_ptr.* });
}
fn fmtTracking(self: *CodeGen) std.fmt.Formatter(FormatTrackingData, formatTracking) {
    return .{ .data = .{ .self = self } };
}

fn addInst(self: *CodeGen, inst: Mir.Inst) error{OutOfMemory}!Mir.Inst.Index {
    const gpa = self.gpa;
    try self.mir_instructions.ensureUnusedCapacity(gpa, 1);
    const result_index: Mir.Inst.Index = @intCast(self.mir_instructions.len);
    self.mir_instructions.appendAssumeCapacity(inst);
    if (inst.ops != .pseudo_dead_none) wip_mir_log.debug("{f}", .{self.fmtWipMir(result_index)});
    return result_index;
}

fn addExtra(self: *CodeGen, extra: anytype) Allocator.Error!u32 {
    const fields = std.meta.fields(@TypeOf(extra));
    try self.mir_extra.ensureUnusedCapacity(self.gpa, fields.len);
    return self.addExtraAssumeCapacity(extra);
}

fn addExtraAssumeCapacity(self: *CodeGen, extra: anytype) u32 {
    const fields = std.meta.fields(@TypeOf(extra));
    const result: u32 = @intCast(self.mir_extra.items.len);
    inline for (fields) |field| {
        self.mir_extra.appendAssumeCapacity(switch (field.type) {
            u32 => @field(extra, field.name),
            i32, Mir.Memory.Info => @bitCast(@field(extra, field.name)),
            FrameIndex => @intFromEnum(@field(extra, field.name)),
            else => @compileError("bad field type: " ++ field.name ++ ": " ++ @typeName(field.type)),
        });
    }
    return result;
}

fn addString(cg: *CodeGen, string: []const u8) Allocator.Error!Mir.NullTerminatedString {
    try cg.mir_string_bytes.ensureUnusedCapacity(cg.gpa, string.len + 1);
    try cg.mir_strings.ensureUnusedCapacityContext(cg.gpa, 1, .{ .bytes = &cg.mir_string_bytes });

    const mir_string_gop = cg.mir_strings.getOrPutAssumeCapacityAdapted(
        string,
        std.hash_map.StringIndexAdapter{ .bytes = &cg.mir_string_bytes },
    );
    if (!mir_string_gop.found_existing) {
        mir_string_gop.key_ptr.* = @intCast(cg.mir_string_bytes.items.len);
        cg.mir_string_bytes.appendSliceAssumeCapacity(string);
        cg.mir_string_bytes.appendAssumeCapacity(0);
    }
    return @enumFromInt(mir_string_gop.key_ptr.*);
}

fn asmOps(self: *CodeGen, tag: Mir.Inst.FixedTag, ops: [4]Operand) !void {
    return switch (ops[0]) {
        .none => self.asmOpOnly(tag),
        .reg => |reg0| switch (ops[1]) {
            .none => self.asmRegister(tag, reg0),
            .reg => |reg1| switch (ops[2]) {
                .none => self.asmRegisterRegister(tag, reg0, reg1),
                .reg => |reg2| switch (ops[3]) {
                    .none => self.asmRegisterRegisterRegister(tag, reg0, reg1, reg2),
                    .reg => |reg3| self.asmRegisterRegisterRegisterRegister(tag, reg0, reg1, reg2, reg3),
                    .imm => |imm3| self.asmRegisterRegisterRegisterImmediate(tag, reg0, reg1, reg2, imm3),
                    else => error.InvalidInstruction,
                },
                .mem => |mem2| switch (ops[3]) {
                    .none => self.asmRegisterRegisterMemory(tag, reg0, reg1, mem2),
                    .reg => |reg3| self.asmRegisterRegisterMemoryRegister(tag, reg0, reg1, mem2, reg3),
                    .imm => |imm3| self.asmRegisterRegisterMemoryImmediate(tag, reg0, reg1, mem2, imm3),
                    else => error.InvalidInstruction,
                },
                .imm => |imm2| switch (ops[3]) {
                    .none => self.asmRegisterRegisterImmediate(tag, reg0, reg1, imm2),
                    else => error.InvalidInstruction,
                },
                else => error.InvalidInstruction,
            },
            .mem => |mem1| switch (ops[2]) {
                .none => self.asmRegisterMemory(tag, reg0, mem1),
                .reg => |reg2| switch (ops[3]) {
                    .none => self.asmRegisterMemoryRegister(tag, reg0, mem1, reg2),
                    else => error.InvalidInstruction,
                },
                .imm => |imm2| switch (ops[3]) {
                    .none => self.asmRegisterMemoryImmediate(tag, reg0, mem1, imm2),
                    else => error.InvalidInstruction,
                },
                else => error.InvalidInstruction,
            },
            .imm => |imm1| switch (ops[2]) {
                .none => self.asmRegisterImmediate(tag, reg0, imm1),
                else => error.InvalidInstruction,
            },
            else => error.InvalidInstruction,
        },
        .mem => |mem0| switch (ops[1]) {
            .none => self.asmMemory(tag, mem0),
            .reg => |reg1| switch (ops[2]) {
                .none => self.asmMemoryRegister(tag, mem0, reg1),
                .reg => |reg2| switch (ops[3]) {
                    .none => self.asmMemoryRegisterRegister(tag, mem0, reg1, reg2),
                    else => error.InvalidInstruction,
                },
                .imm => |imm2| switch (ops[3]) {
                    .none => self.asmMemoryRegisterImmediate(tag, mem0, reg1, imm2),
                    else => error.InvalidInstruction,
                },
                else => error.InvalidInstruction,
            },
            .imm => |imm1| switch (ops[2]) {
                .none => self.asmMemoryImmediate(tag, mem0, imm1),
                else => error.InvalidInstruction,
            },
            else => error.InvalidInstruction,
        },
        .imm => |imm0| switch (ops[1]) {
            .none => self.asmImmediate(tag, imm0),
            .reg => |reg1| switch (ops[2]) {
                .none => self.asmImmediateRegister(tag, imm0, reg1),
                else => error.InvalidInstruction,
            },
            .imm => |imm1| switch (ops[2]) {
                .none => self.asmImmediateImmediate(tag, imm0, imm1),
                else => error.InvalidInstruction,
            },
            else => error.InvalidInstruction,
        },
        .inst => |inst0| switch (ops[1]) {
            .none => self.asmReloc(tag, inst0),
            else => error.InvalidInstruction,
        },
    };
}

/// A `cc` of `.z_and_np` clobbers `reg2`!
fn asmCmovccRegisterRegister(self: *CodeGen, cc: Condition, reg1: Register, reg2: Register) !void {
    if (self.hasFeature(.cmov)) _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .cmov,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .rr,
            .z_and_np => .pseudo_cmov_z_and_np_rr,
            .nz_or_p => .pseudo_cmov_nz_or_p_rr,
        },
        .data = .{ .rr = .{
            .fixes = switch (cc) {
                else => .fromCond(cc),
                .z_and_np, .nz_or_p => ._,
            },
            .r1 = reg1,
            .r2 = reg2,
        } },
    }) else {
        const reloc = try self.asmJccReloc(cc.negate(), undefined);
        try self.asmRegisterRegister(.{ ._, .mov }, reg1, reg2);
        self.performReloc(reloc);
    }
}

/// A `cc` of `.z_and_np` is not supported by this encoding!
fn asmCmovccRegisterMemory(self: *CodeGen, cc: Condition, reg: Register, m: Memory) !void {
    if (self.hasFeature(.cmov)) _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .cmov,
            .z_and_np => unreachable,
            .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .rm,
            .z_and_np => unreachable,
            .nz_or_p => .pseudo_cmov_nz_or_p_rm,
        },
        .data = .{ .rx = .{
            .fixes = switch (cc) {
                else => .fromCond(cc),
                .z_and_np => unreachable,
                .nz_or_p => ._,
            },
            .r1 = reg,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    }) else {
        const reloc = try self.asmJccReloc(cc.negate(), undefined);
        try self.asmRegisterMemory(.{ ._, .mov }, reg, m);
        self.performReloc(reloc);
    }
}

fn asmSetccRegister(self: *CodeGen, cc: Condition, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .set,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .r,
            .z_and_np => .pseudo_set_z_and_np_r,
            .nz_or_p => .pseudo_set_nz_or_p_r,
        },
        .data = switch (cc) {
            else => .{ .r = .{
                .fixes = .fromCond(cc),
                .r1 = reg,
            } },
            .z_and_np, .nz_or_p => .{ .rr = .{
                .r1 = reg,
                .r2 = (try self.register_manager.allocReg(null, abi.RegisterClass.gp)).to8(),
            } },
        },
    });
}

fn asmSetccMemory(self: *CodeGen, cc: Condition, m: Memory) !void {
    const payload = try self.addExtra(Mir.Memory.encode(m));
    _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .set,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .m,
            .z_and_np => .pseudo_set_z_and_np_m,
            .nz_or_p => .pseudo_set_nz_or_p_m,
        },
        .data = switch (cc) {
            else => .{ .x = .{
                .fixes = .fromCond(cc),
                .payload = payload,
            } },
            .z_and_np, .nz_or_p => .{ .rx = .{
                .r1 = (try self.register_manager.allocReg(null, abi.RegisterClass.gp)).to8(),
                .payload = payload,
            } },
        },
    });
}

fn asmJmpReloc(self: *CodeGen, target: Mir.Inst.Index) !Mir.Inst.Index {
    return self.addInst(.{
        .tag = .j,
        .ops = .inst,
        .data = .{ .inst = .{
            .fixes = ._mp,
            .inst = target,
        } },
    });
}

fn asmJccReloc(self: *CodeGen, cc: Condition, target: Mir.Inst.Index) !Mir.Inst.Index {
    return self.addInst(.{
        .tag = switch (cc) {
            else => .j,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .inst,
            .z_and_np => .pseudo_j_z_and_np_inst,
            .nz_or_p => .pseudo_j_nz_or_p_inst,
        },
        .data = .{ .inst = .{
            .fixes = switch (cc) {
                else => .fromCond(cc),
                .z_and_np, .nz_or_p => ._,
            },
            .inst = target,
        } },
    });
}

fn asmReloc(self: *CodeGen, tag: Mir.Inst.FixedTag, target: Mir.Inst.Index) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .inst,
        .data = .{ .inst = .{
            .fixes = tag[0],
            .inst = target,
        } },
    });
}

fn asmPlaceholder(self: *CodeGen) !Mir.Inst.Index {
    return self.addInst(.{
        .tag = .pseudo,
        .ops = .pseudo_dead_none,
        .data = undefined,
    });
}

fn asmOpOnly(self: *CodeGen, tag: Mir.Inst.FixedTag) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .none,
        .data = .{ .none = .{
            .fixes = tag[0],
        } },
    });
}

fn asmPseudo(self: *CodeGen, ops: Mir.Inst.Ops) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_none"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = undefined,
    });
}

fn asmPseudoRegister(self: *CodeGen, ops: Mir.Inst.Ops, reg: Register) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_r"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .r = .{ .r1 = reg } },
    });
}

fn asmPseudoImmediate(self: *CodeGen, ops: Mir.Inst.Ops, imm: Immediate) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_i_s"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .i = .{ .i = @bitCast(imm.signed) } },
    });
}

fn asmPseudoRegisterRegister(self: *CodeGen, ops: Mir.Inst.Ops, reg1: Register, reg2: Register) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_rr"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .rr = .{ .r1 = reg1, .r2 = reg2 } },
    });
}

fn asmPseudoRegisterImmediate(self: *CodeGen, ops: Mir.Inst.Ops, reg: Register, imm: Immediate) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_ri_s"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .ri = .{ .r1 = reg, .i = @bitCast(imm.signed) } },
    });
}

fn asmRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .r,
        .data = .{ .r = .{
            .fixes = tag[0],
            .r1 = reg,
        } },
    });
}

fn asmImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, imm: Immediate) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = switch (imm) {
            .signed => .i_s,
            .unsigned => .i_u,
            .nav => .nav,
            .uav => .uav,
            .lazy_sym => .lazy_sym,
            .extern_func => .extern_func,
        },
        .data = switch (imm) {
            .signed, .unsigned => .{ .i = .{
                .fixes = tag[0],
                .i = switch (imm) {
                    .signed => |s| @bitCast(s),
                    .unsigned => |u| @intCast(u),
                    .nav, .uav, .lazy_sym, .extern_func => unreachable,
                },
            } },
            .nav => |nav| switch (tag[0]) {
                ._ => .{ .nav = nav },
                else => unreachable,
            },
            .uav => |uav| switch (tag[0]) {
                ._ => .{ .uav = uav },
                else => unreachable,
            },
            .lazy_sym => |lazy_sym| switch (tag[0]) {
                ._ => .{ .lazy_sym = lazy_sym },
                else => unreachable,
            },
            .extern_func => |extern_func| switch (tag[0]) {
                ._ => .{ .extern_func = extern_func },
                else => unreachable,
            },
        },
    });
}

fn asmImmediateRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, imm: Immediate, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .ir,
        .data = .{ .ri = .{
            .fixes = tag[0],
            .r1 = reg,
            .i = @as(u8, switch (imm) {
                .signed => |s| @bitCast(@as(i8, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .nav, .uav, .lazy_sym, .extern_func => unreachable,
            }),
        } },
    });
}

fn asmImmediateImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, imm1: Immediate, imm2: Immediate) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .ii,
        .data = .{ .ii = .{
            .fixes = tag[0],
            .i1 = switch (imm1) {
                .signed => |s| @bitCast(@as(i16, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .nav, .uav, .lazy_sym, .extern_func => unreachable,
            },
            .i2 = switch (imm2) {
                .signed => |s| @bitCast(@as(i8, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .nav, .uav, .lazy_sym, .extern_func => unreachable,
            },
        } },
    });
}

fn asmRegisterRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, reg1: Register, reg2: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rr,
        .data = .{ .rr = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
        } },
    });
}

fn asmRegisterImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, reg: Register, imm: Immediate) !void {
    const ops: Mir.Inst.Ops, const i: u32 = switch (imm) {
        .signed => |s| .{ .ri_s, @bitCast(s) },
        .unsigned => |u| if (std.math.cast(u32, u)) |small|
            .{ .ri_u, small }
        else
            .{ .ri_64, try self.addExtra(Mir.Imm64.encode(imm.unsigned)) },
        .nav, .uav, .lazy_sym, .extern_func => unreachable,
    };
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = ops,
        .data = .{ .ri = .{
            .fixes = tag[0],
            .r1 = reg,
            .i = i,
        } },
    });
}

fn asmRegisterRegisterRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    reg3: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrr,
        .data = .{ .rrr = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
        } },
    });
}

fn asmRegisterRegisterRegisterRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    reg3: Register,
    reg4: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrrr,
        .data = .{ .rrrr = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
            .r4 = reg4,
        } },
    });
}

fn asmRegisterRegisterRegisterImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    reg3: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrri,
        .data = .{ .rrri = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
            .i = switch (imm) {
                .signed => |s| @bitCast(@as(i8, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .nav, .uav, .lazy_sym, .extern_func => unreachable,
            },
        } },
    });
}

fn asmRegisterRegisterImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = switch (imm) {
            .signed => .rri_s,
            .unsigned => .rri_u,
            .nav, .uav, .lazy_sym, .extern_func => unreachable,
        },
        .data = .{ .rri = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .i = switch (imm) {
                .signed => |s| @bitCast(s),
                .unsigned => |u| @intCast(u),
                .nav, .uav, .lazy_sym, .extern_func => unreachable,
            },
        } },
    });
}

fn asmRegisterRegisterMemory(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    m: Memory,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrm,
        .data = .{ .rrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterRegisterMemoryRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    m: Memory,
    reg3: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrmr,
        .data = .{ .rrrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemory(self: *CodeGen, tag: Mir.Inst.FixedTag, m: Memory) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .m,
        .data = .{ .x = .{
            .fixes = tag[0],
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterMemory(self: *CodeGen, tag: Mir.Inst.FixedTag, reg: Register, m: Memory) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rm,
        .data = .{ .rx = .{
            .fixes = tag[0],
            .r1 = reg,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterMemoryRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    m: Memory,
    reg2: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rmr,
        .data = .{ .rrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterMemoryImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg: Register,
    m: Memory,
    imm: Immediate,
) !void {
    if (switch (imm) {
        .signed => |s| if (std.math.cast(i16, s)) |x| @as(u16, @bitCast(x)) else null,
        .unsigned => |u| std.math.cast(u16, u),
        .nav, .uav, .lazy_sym, .extern_func => unreachable,
    }) |small_imm| {
        _ = try self.addInst(.{
            .tag = tag[1],
            .ops = .rmi,
            .data = .{ .rix = .{
                .fixes = tag[0],
                .r1 = reg,
                .i = small_imm,
                .payload = try self.addExtra(Mir.Memory.encode(m)),
            } },
        });
    } else {
        const payload = try self.addExtra(Mir.Imm32{ .imm = switch (imm) {
            .signed => |s| @bitCast(s),
            .unsigned => |u| @as(u32, @intCast(u)),
            .nav, .uav, .lazy_sym, .extern_func => unreachable,
        } });
        assert(payload + 1 == try self.addExtra(Mir.Memory.encode(m)));
        _ = try self.addInst(.{
            .tag = tag[1],
            .ops = switch (imm) {
                .signed => .rmi_s,
                .unsigned => .rmi_u,
                .nav, .uav, .lazy_sym, .extern_func => unreachable,
            },
            .data = .{ .rx = .{
                .fixes = tag[0],
                .r1 = reg,
                .payload = payload,
            } },
        });
    }
}

fn asmRegisterRegisterMemoryImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    m: Memory,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrmi,
        .data = .{ .rrix = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .i = @intCast(imm.unsigned),
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemoryRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, m: Memory, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .mr,
        .data = .{ .rx = .{
            .fixes = tag[0],
            .r1 = reg,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemoryImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, m: Memory, imm: Immediate) !void {
    const payload = try self.addExtra(Mir.Imm32{ .imm = switch (imm) {
        .signed => |s| @bitCast(s),
        .unsigned => |u| @intCast(u),
        .nav, .uav, .lazy_sym, .extern_func => unreachable,
    } });
    assert(payload + 1 == try self.addExtra(Mir.Memory.encode(m)));
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = switch (imm) {
            .signed => .mi_s,
            .unsigned => .mi_u,
            .nav, .uav, .lazy_sym, .extern_func => unreachable,
        },
        .data = .{ .x = .{
            .fixes = tag[0],
            .payload = payload,
        } },
    });
}

fn asmMemoryRegisterRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    m: Memory,
    reg1: Register,
    reg2: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .mrr,
        .data = .{ .rrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemoryRegisterImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    m: Memory,
    reg: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .mri,
        .data = .{ .rix = .{
            .fixes = tag[0],
            .r1 = reg,
            .i = @intCast(imm.unsigned),
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn gen(
    self: *CodeGen,
    zir: *const std.zig.Zir,
    func_zir_inst: std.zig.Zir.Inst.Index,
    comptime_args: InternPool.Index.Slice,
    air_arg_count: u32,
) InnerError!void {
    const pt = self.pt;
    const zcu = pt.zcu;
    const fn_info = zcu.typeToFunc(self.fn_type).?;
    if (fn_info.cc != .naked) {
        try self.asmRegister(.{ ._, .push }, .rbp);
        try self.asmPseudoImmediate(.pseudo_cfi_adjust_cfa_offset_i_s, .s(8));
        try self.asmPseudoRegisterImmediate(.pseudo_cfi_rel_offset_ri_s, .rbp, .s(0));
        try self.asmRegisterRegister(.{ ._, .mov }, .rbp, .rsp);
        try self.asmPseudoRegister(.pseudo_cfi_def_cfa_register_r, .rbp);
        const backpatch_push_callee_preserved_regs = try self.asmPlaceholder();
        const backpatch_frame_align = try self.asmPlaceholder();
        const backpatch_frame_align_extra = try self.asmPlaceholder();
        const backpatch_stack_alloc = try self.asmPlaceholder();
        const backpatch_stack_alloc_extra = try self.asmPlaceholder();

        switch (self.ret_mcv.long) {
            .none, .unreach => {},
            .indirect => {
                // The address where to store the return value for the caller is in a
                // register which the callee is free to clobber. Therefore, we purposely
                // spill it to stack immediately.
                const frame_index = try self.allocFrameIndex(.initSpill(.usize, zcu));
                try self.genSetMem(
                    .{ .frame = frame_index },
                    0,
                    .usize,
                    self.ret_mcv.long.address().offset(-self.ret_mcv.short.indirect.off),
                    .{},
                );
                self.ret_mcv.long = .{ .load_frame = .{ .index = frame_index } };
                tracking_log.debug("spill {f} to {}", .{ self.ret_mcv.long, frame_index });
            },
            else => unreachable,
        }

        if (fn_info.is_var_args) switch (fn_info.cc) {
            .x86_64_sysv => {
                const info = &self.va_info.sysv;
                const reg_save_area_fi = try self.allocFrameIndex(.init(.{
                    .size = abi.SysV.c_abi_int_param_regs.len * 8 +
                        abi.SysV.c_abi_sse_param_regs.len * 16,
                    .alignment = .@"16",
                }));
                info.reg_save_area = .{ .index = reg_save_area_fi };

                for (abi.SysV.c_abi_int_param_regs[info.gp_count..], info.gp_count..) |reg, reg_i|
                    try self.genSetMem(.{ .frame = reg_save_area_fi }, @intCast(reg_i * 8), .usize, .{ .register = reg }, .{});

                try self.asmRegisterImmediate(.{ ._, .cmp }, .al, .u(info.fp_count));
                const skip_sse_reloc = try self.asmJccReloc(.na, undefined);

                const vec_2_f64 = try pt.vectorType(.{ .len = 2, .child = .f64_type });
                for (abi.SysV.c_abi_sse_param_regs[info.fp_count..], info.fp_count..) |reg, reg_i|
                    try self.genSetMem(
                        .{ .frame = reg_save_area_fi },
                        @intCast(abi.SysV.c_abi_int_param_regs.len * 8 + reg_i * 16),
                        vec_2_f64,
                        .{ .register = reg },
                        .{},
                    );

                self.performReloc(skip_sse_reloc);
            },
            .x86_64_win => return self.fail("TODO implement gen var arg function for Win64", .{}),
            else => |cc| return self.fail("{s} does not support var args", .{@tagName(cc)}),
        };

        if (!self.mod.strip) try self.asmPseudo(.pseudo_dbg_prologue_end_none);

        try self.genMainBody(zir, func_zir_inst, comptime_args, air_arg_count);

        const epilogue = if (self.epilogue_relocs.items.len > 0) epilogue: {
            var last_inst: Mir.Inst.Index = @intCast(self.mir_instructions.len - 1);
            while (self.epilogue_relocs.getLastOrNull() == last_inst) {
                self.epilogue_relocs.items.len -= 1;
                self.mir_instructions.set(last_inst, .{
                    .tag = .pseudo,
                    .ops = .pseudo_dead_none,
                    .data = undefined,
                });
                last_inst -= 1;
            }
            for (self.epilogue_relocs.items) |epilogue_reloc| self.performReloc(epilogue_reloc);

            if (!self.mod.strip) try self.asmPseudo(.pseudo_dbg_epilogue_begin_none);
            const backpatch_stack_dealloc = try self.asmPlaceholder();
            const backpatch_pop_callee_preserved_regs = try self.asmPlaceholder();
            try self.asmRegister(.{ ._, .pop }, .rbp);
            try self.asmPseudoRegisterImmediate(.pseudo_cfi_def_cfa_ri_s, .rsp, .s(8));
            try self.asmOpOnly(.{ ._, .ret });
            break :epilogue .{
                .backpatch_stack_dealloc = backpatch_stack_dealloc,
                .backpatch_pop_callee_preserved_regs = backpatch_pop_callee_preserved_regs,
            };
        } else null;

        const frame_layout = try self.computeFrameLayout(fn_info.cc);
        const need_frame_align = frame_layout.stack_mask != std.math.maxInt(u32);
        const need_stack_adjust = frame_layout.stack_adjust > 0;
        const need_save_reg = frame_layout.save_reg_list.count() > 0;
        if (need_frame_align) {
            const page_align = @as(u32, std.math.maxInt(u32)) << 12;
            self.mir_instructions.set(backpatch_frame_align, .{
                .tag = .@"and",
                .ops = .ri_s,
                .data = .{ .ri = .{
                    .r1 = .rsp,
                    .i = @max(frame_layout.stack_mask, page_align),
                } },
            });
            if (frame_layout.stack_mask < page_align) {
                self.mir_instructions.set(backpatch_frame_align_extra, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_align_ri_s,
                    .data = .{ .ri = .{
                        .r1 = .rsp,
                        .i = ~frame_layout.stack_mask & page_align,
                    } },
                });
            }
        }
        if (need_stack_adjust) {
            const page_size: u32 = 1 << 12;
            if (frame_layout.stack_adjust <= page_size) {
                self.mir_instructions.set(backpatch_stack_alloc, .{
                    .tag = .sub,
                    .ops = .ri_s,
                    .data = .{ .ri = .{
                        .r1 = .rsp,
                        .i = frame_layout.stack_adjust,
                    } },
                });
            } else if (frame_layout.stack_adjust <
                page_size * Lower.pseudo_probe_adjust_unrolled_max_insts)
            {
                self.mir_instructions.set(backpatch_stack_alloc, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_adjust_unrolled_ri_s,
                    .data = .{ .ri = .{
                        .r1 = .rsp,
                        .i = frame_layout.stack_adjust,
                    } },
                });
            } else {
                const scratch_reg = abi.getCAbiLinkerScratchReg(fn_info.cc);
                self.mir_instructions.set(backpatch_stack_alloc, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_adjust_setup_rri_s,
                    .data = .{ .rri = .{
                        .r1 = .rsp,
                        .r2 = scratch_reg,
                        .i = frame_layout.stack_adjust,
                    } },
                });
                self.mir_instructions.set(backpatch_stack_alloc_extra, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_adjust_loop_rr,
                    .data = .{ .rr = .{
                        .r1 = .rsp,
                        .r2 = scratch_reg,
                    } },
                });
            }
        }
        if (epilogue) |e| if (need_frame_align or need_stack_adjust) {
            self.mir_instructions.set(e.backpatch_stack_dealloc, switch (-frame_layout.save_reg_list.size(self.target)) {
                0 => .{
                    .tag = .mov,
                    .ops = .rr,
                    .data = .{ .rr = .{
                        .r1 = .rsp,
                        .r2 = .rbp,
                    } },
                },
                else => |disp| .{
                    .tag = .lea,
                    .ops = .rm,
                    .data = .{ .rx = .{
                        .r1 = .rsp,
                        .payload = try self.addExtra(Mir.Memory.encode(.{
                            .base = .{ .reg = .rbp },
                            .mod = .{ .rm = .{
                                .size = .qword,
                                .disp = disp,
                            } },
                        })),
                    } },
                },
            });
        };
        if (need_save_reg) {
            self.mir_instructions.set(backpatch_push_callee_preserved_regs, .{
                .tag = .pseudo,
                .ops = .pseudo_push_reg_list,
                .data = .{ .reg_list = frame_layout.save_reg_list },
            });
            if (epilogue) |e| self.mir_instructions.set(e.backpatch_pop_callee_preserved_regs, .{
                .tag = .pseudo,
                .ops = .pseudo_pop_reg_list,
                .data = .{ .reg_list = frame_layout.save_reg_list },
            });
        }
    } else {
        if (!self.mod.strip) try self.asmPseudo(.pseudo_dbg_prologue_end_none);
        try self.genMainBody(zir, func_zir_inst, comptime_args, air_arg_count);
        if (!self.mod.strip) try self.asmPseudo(.pseudo_dbg_epilogue_begin_none);
    }
}

fn genMainBody(
    cg: *CodeGen,
    zir: *const std.zig.Zir,
    func_zir_inst: std.zig.Zir.Inst.Index,
    comptime_args: InternPool.Index.Slice,
    air_arg_count: u32,
) InnerError!void {
    const pt = cg.pt;
    const zcu = pt.zcu;
    const ip = &zcu.intern_pool;

    const main_body = cg.air.getMainBody();
    const air_args_body = main_body[0..air_arg_count];
    try cg.genBody(air_args_body);

    if (!cg.mod.strip) {
        var air_arg_index: usize = 0;
        const fn_info = zcu.typeToFunc(cg.fn_type).?;
        var fn_param_index: usize = 0;
        var zir_param_index: usize = 0;
        for (zir.getParamBody(func_zir_inst)) |zir_param_inst| {
            const name = switch (zir.getParamName(zir_param_inst) orelse break) {
                .empty => .none,
                else => |zir_name| try cg.addString(zir.nullTerminatedString(zir_name)),
            };
            defer zir_param_index += 1;

            if (comptime_args.len > 0) switch (comptime_args.get(ip)[zir_param_index]) {
                .none => {},
                else => |comptime_arg| {
                    try cg.mir_locals.append(cg.gpa, .{ .name = name, .type = ip.typeOf(comptime_arg) });
                    _ = try cg.addInst(.{
                        .tag = .pseudo,
                        .ops = .pseudo_dbg_arg_val,
                        .data = .{ .ip_index = comptime_arg },
                    });
                    continue;
                },
            };

            const arg_ty = fn_info.param_types.get(ip)[fn_param_index];
            try cg.mir_locals.append(cg.gpa, .{ .name = name, .type = arg_ty });
            fn_param_index += 1;

            if (air_arg_index == air_args_body.len) {
                try cg.asmPseudo(.pseudo_dbg_arg_none);
                continue;
            }
            const air_arg_inst = air_args_body[air_arg_index];
            const air_arg_data = cg.air.instructions.items(.data)[air_arg_index].arg;
            if (air_arg_data.zir_param_index != zir_param_index) {
                try cg.asmPseudo(.pseudo_dbg_arg_none);
                continue;
            }
            air_arg_index += 1;
            try cg.genLocalDebugInfo(
                .arg,
                .fromInterned(arg_ty),
                cg.getResolvedInstValue(air_arg_inst).short,
            );
        }
        if (fn_info.is_var_args) try cg.asmPseudo(.pseudo_dbg_var_args_none);
    }

    try cg.genBody(main_body[air_arg_count..]);
}

fn checkInvariantsAfterAirInst(cg: *CodeGen) void {
    assert(!cg.register_manager.lockedRegsExist());

    if (std.debug.runtime_safety) {
        // check consistency of tracked registers
        var it = cg.register_manager.free_registers.iterator(.{ .kind = .unset });
        while (it.next()) |index| {
            const tracked_inst = cg.register_manager.registers[index];
            const tracking = cg.getResolvedInstValue(tracked_inst);
            for (tracking.getRegs()) |reg| {
                if (RegisterManager.indexOfRegIntoTracked(reg).? == index) break;
            } else unreachable; // tracked register not in use
        }
    }
}

fn genBodyBlock(cg: *CodeGen, body: []const Air.Inst.Index) InnerError!void {
    if (!cg.mod.strip) try cg.asmPseudo(.pseudo_dbg_enter_block_none);
    try cg.genBody(body);
    if (!cg.mod.strip) try cg.asmPseudo(.pseudo_dbg_leave_block_none);
}

fn genBody(cg: *CodeGen, body: []const Air.Inst.Index) InnerError!void {
    @setEvalBranchQuota(29_400);
    const pt = cg.pt;
    const zcu = pt.zcu;
    const ip = &zcu.intern_pool;
    const air_tags = cg.air.instructions.items(.tag);
    const air_datas = cg.air.instructions.items(.data);

    for (body) |inst| {
        if (cg.liveness.isUnused(inst) and !cg.air.mustLower(inst, ip)) continue;
        wip_mir_log.debug("{f}", .{cg.fmtAir(inst)});
        verbose_tracking_log.debug("{f}", .{cg.fmtTracking()});

        cg.reused_operands = .initEmpty();
        try cg.inst_tracking.ensureUnusedCapacity(cg.gpa, 1);
        switch (air_tags[@intFromEnum(inst)]) {
            .select => try cg.airSelect(inst),
            .shuffle_one, .shuffle_two => @panic("x86_64 TODO: shuffle_one/shuffle_two"),

            .arg => try cg.airArg(inst),
            .add, .add_optimized, .add_wrap => |air_tag| {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .int = .byte }, .{ .int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm8, .none } },
                        .{ .src = .{ .imm8, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .word }, .{ .int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm16, .none } },
                        .{ .src = .{ .imm16, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .dword }, .{ .int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm32, .none } },
                        .{ .src = .{ .imm32, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .qword }, .{ .int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .simm32, .none } },
                        .{ .src = .{ .simm32, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memsia(.src0d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp1d, .memsia(.src1d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0d, .@"4", .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5d, .leasi(.tmp1d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5d, .leasi(.tmp2d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3d, .@"4", .tmp4), .tmp5d, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .add, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } },
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } },
                            .none,
                        } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .extern_func = "__addhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .add, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .add, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .add, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addhf3" } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .add, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .add, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .add, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .add, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0q, ._, ._, ._ },
                        .{ ._, .f_, .add, .src1q, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .add, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .fn_cw, .st, .tmp2w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp3d, .memd(.tmp2b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp2b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp2w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp2b, 1), .tmp3b, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp2w, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .add, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .fn_cw, .st, .tmp1w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memd(.tmp1b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp1b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp1w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp1b, 1), .tmp2b, ._, ._ },
                        .{ ._, .f_, .add, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp1w, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .fn_cw, .st, .tmp0w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memd(.tmp0b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp0b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp0w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp0b, 1), .tmp1b, ._, ._ },
                        .{ ._, ._, .mov, .tmp1p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp1, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp1, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp1, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp1p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp0w, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } },
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } },
                            .none,
                        } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .extern_func = "__addtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__addtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {f} {f} {f}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                switch (air_tag) {
                    else => unreachable,
                    .add, .add_optimized => {},
                    .add_wrap => res[0].wrapInt(cg) catch |err| switch (err) {
                        error.SelectFailed => return cg.fail("failed to select {s} wrap {f} {f}", .{
                            @tagName(air_tag),
                            cg.typeOf(bin_op.lhs).fmt(pt),
                            res[0].tracking(cg),
                        }),
                        else => |e| return e,
                    },
                }
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .add_safe => unreachable,
            .add_sat => |air_tag| {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .exact_int = 1 }, .{ .exact_int = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .@"or", .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 8 }, .{ .exact_signed_int = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp0b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 8 }, .{ .exact_signed_int = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .dst0b, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0b, .ui(7), ._, ._ },
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .xor, .tmp0b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .@"test", .dst0b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._po, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .@"test", .dst0b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._pe, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .dst0b, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_unsigned_int = 8 }, .{ .exact_unsigned_int = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u8, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"or", .dst0b, .tmp0b, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u8, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .cmp, .dst0b, .tmp0b, ._, ._ },
                        .{ ._, ._a, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .cmp, .dst0b, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0b, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, .fast_imm16, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, .fast_imm16, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_unsigned_int = 16 }, .{ .exact_unsigned_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"or", .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .cmp, .dst0w, .tmp0w, ._, ._ },
                        .{ ._, ._a, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .cmp, .dst0w, .tmp0w, ._, ._ },
                        .{ ._, ._a, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .cmp, .dst0w, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0w, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .cmp, .dst0w, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0w, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .cmp, .dst0w, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0d, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .cmp, .dst0w, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0d, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 32 }, .{ .exact_signed_int = 32 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 32 }, .{ .exact_signed_int = 32 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 31 }, .{ .exact_signed_int = 31 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 31 }, .{ .exact_signed_int = 31 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_unsigned_int = 32 }, .{ .exact_unsigned_int = 32 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"or", .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .cmp, .dst0d, .tmp0d, ._, ._ },
                        .{ ._, ._a, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .cmp, .dst0d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0d, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 64 }, .{ .exact_signed_int = 64 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 64 }, .{ .exact_signed_int = 64 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 63 }, .{ .exact_signed_int = 63 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .dst0q, ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp1q, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 63 }, .{ .exact_signed_int = 63 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .dst0q, ._, ._ },
                        .{ ._, ._, .add, .tmp0q, .tmp0q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .cmov, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1q, .dst0q, .sia(-31, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .dst0q, ._, ._ },
                        .{ ._, ._r, .sh, .tmp1q, .sia(-31, .src0, .add_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .dst0q, ._, ._ },
                        .{ ._, ._r, .sh, .tmp0q, .sia(-31, .src0, .add_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .exact_unsigned_int = 64 }, .{ .exact_unsigned_int = 64 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0q, .tmp0q, ._, ._ },
                        .{ ._, ._, .@"or", .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .cmp, .dst0q, .tmp0q, ._, ._ },
                        .{ ._, ._a, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .cmp, .dst0q, .tmp0q, ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .cmp, .dst0q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0q, .ua(.src0, .add_umax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 128 } },
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 128 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -8), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .memad(.src0q, .add_size, -8), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memad(.src1q, .add_size, -8), ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ .@"0:", ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .exact_remainder_unsigned_int = .{ .of = .xword, .is = 128 } },
                        .{ .exact_remainder_unsigned_int = .{ .of = .xword, .is = 128 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._nc, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .si(-1), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sa(.src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(2, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -16), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -16), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -16), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .memad(.src0q, .add_size, -16), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memad(.src1q, .add_size, -16), ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-2, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ .@"0:", ._, .mov, .memad(.dst0q, .add_size, -16), .tmp1q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .exact_remainder_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .exact_remainder_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -8), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._nc, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .si(-1), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .memad(.dst0q, .add_size, -8), .si(0), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -8), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1q, .uia(63, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp1q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-2, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -16), .tmp1q, ._, ._ },
                        .{ .@"0:", ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .remainder_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .adc, .tmp2q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -8), .tmp2q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .cmp, .tmp2q, .tmp1q, ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -16), .tmp1q, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .si(-1), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-2, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .memad(.dst0q, .add_size, -8), .si(0), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1q, .uia(63, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp1q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .{ .remainder_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp2q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp2q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .cmp, .tmp2q, .tmp1q, ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .si(-1), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .@"or", .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_, .@"or", .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .@"or", .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .@"or", .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_, .@"or", .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .adds, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .adds, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .adds, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .adds, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_b, .maxs, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_b, .mins, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_b, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .maxs, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_b, .mins, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .slimit_delta_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_b, .sub, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .adds, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .add, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .adds, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .sub, .dst0x, .tmp2x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_32_i8, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_b, .maxs, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_b, .mins, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .qword, .is = 8 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .qword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .addus, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .addus, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .addus, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 8 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .addus, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .mmx, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_b, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .minu, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_b, .minu, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_b, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .minu, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_b, .minu, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .adds, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .adds, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .adds, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .adds, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .mmx, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .qword } } },
                        .{ .type = .vector_4_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .qword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_w, .add, .dst0q, .src1q, ._, ._ },
                        .{ ._, .p_w, .maxs, .dst0q, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_w, .mins, .dst0q, .lea(.tmp0q), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_w, .maxs, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_w, .mins, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_w, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_w, .maxs, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_w, .mins, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_16_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_w, .maxs, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_w, .mins, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .qword, .is = 16 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .qword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .addus, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .addus, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .addus, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 16 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .addus, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_w, .minu, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_w, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_w, .minu, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .umax_delta_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_w, .add, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .addus, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_w, .sub, .dst0x, .tmp2x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_w, .minu, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_d, .sra, .tmp2x, .src0x, .ui(31), ._ },
                        .{ ._, .vp_d, .cmpgt, .tmp3x, .dst0x, .src0x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .lea(.tmp0x), ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp3x, .src1x, ._ },
                        .{ ._, .v_ps, .blendv, .dst0x, .tmp2x, .dst0x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_d, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .src0x, ._, ._ },
                        .{ ._, .p_d, .sra, .tmp2x, .ui(31), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .src0x, ._, ._ },
                        .{ ._, .p_d, .cmpgt, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .src1x, ._, ._ },
                        .{ ._, ._ps, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 32 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_d, .sra, .tmp2y, .src0y, .ui(31), ._ },
                        .{ ._, .vp_d, .cmpgt, .tmp3y, .dst0y, .src0y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .lea(.tmp0y), ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp3y, .src1y, ._ },
                        .{ ._, .v_ps, .blendv, .dst0y, .tmp2y, .dst0y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_d, .maxs, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_d, .mins, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_d, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_d, .maxs, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_d, .mins, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_8_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_d, .maxs, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_d, .mins, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .cmpeq, .dst0x, .dst0x, .dst0x, ._ },
                        .{ ._, .vp_, .xor, .dst0x, .src0x, .dst0x, ._ },
                        .{ ._, .vp_d, .minu, .dst0x, .dst0x, .src1x, ._ },
                        .{ ._, .vp_d, .add, .dst0x, .dst0x, .src0x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .cmpeq, .dst0x, .dst0x, ._, ._ },
                        .{ ._, .p_, .xor, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_d, .minu, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_d, .add, .dst0x, .src0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 32 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .cmpeq, .dst0y, .dst0y, .dst0y, ._ },
                        .{ ._, .vp_, .xor, .dst0y, .src0y, .dst0y, ._ },
                        .{ ._, .vp_d, .minu, .dst0y, .dst0y, .src1y, ._ },
                        .{ ._, .vp_d, .add, .dst0y, .dst0y, .src0y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_d, .minu, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_d, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_d, .minu, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_d, .minu, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2x, .tmp2x, .src0x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .src0x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .lea(.tmp0x), ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp3x, .src1x, ._ },
                        .{ ._, .v_pd, .blendv, .dst0x, .tmp2x, .dst0x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp2x, .src0x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .src0x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .src1x, ._, ._ },
                        .{ ._, ._pd, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 64 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2y, .tmp2y, .src0y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .dst0y, .src0y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .lea(.tmp0y), ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp3y, .src1y, ._ },
                        .{ ._, .v_pd, .blendv, .dst0y, .tmp2y, .dst0y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp2x, .dst0x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                        .{ ._, .vp_q, .cmpeq, .tmp3x, .tmp3x, .tmp3x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp3x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp2x, .dst0x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                        .{ ._, .vp_q, .cmpeq, .tmp3x, .tmp3x, .tmp3x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp3x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_b, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                        .{ ._, .p_q, .cmpeq, .tmp3x, .tmp3x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .tmp2y, .dst0y, ._ },
                        .{ ._, .vp_b, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                        .{ ._, .vp_q, .cmpeq, .tmp3y, .tmp3y, .tmp3y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .tmp3y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .dst0y, .tmp2y, ._ },
                        .{ ._, .vp_b, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .src0x, .tmp2x, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .vp_, .@"or", .dst0x, .dst0x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .src0x, .tmp2x, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .vp_, .@"or", .dst0x, .dst0x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .src0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .dst0x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .@"or", .dst0x, .tmp3x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .src0y, .tmp2y, ._ },
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .dst0y, .tmp2y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .vp_, .@"or", .dst0y, .dst0y, .tmp3y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2x, .tmp2x, .dst0x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .lea(.tmp0x), .tmp2x },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2x, .tmp2x, .dst0x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .lea(.tmp0x), .tmp2x },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_u64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp2x, .dst0x, ._, ._ },
                        .{ ._, .p_b, .blendv, .dst0x, .lea(.tmp0x), .tmp2x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 63 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2y, .tmp2y, .dst0y, ._ },
                        .{ ._, .vp_b, .blendv, .dst0y, .dst0y, .lea(.tmp0y), .tmp2y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .dst0y, .tmp2y, ._ },
                        .{ ._, .vp_b, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .@"or", .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .@"or", .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .@"or", .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .@"or", .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u8, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .@"or", .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .adds, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .adds, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .adds, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i8, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .adds, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smin), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smin), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .cmpeq, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_b, .broadcast, .tmp3y, .lea(.tmp0b), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp3y, .tmp2y, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp4y, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_b, .maxs, .tmp4y, .tmp4y, .tmp3y, ._ },
                        .{ ._, .vp_b, .mins, .tmp4y, .tmp4y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp4y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .cmpeq, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp3x, .tmp2x, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp4x, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_b, .maxs, .tmp4x, .tmp4x, .tmp3x, ._ },
                        .{ ._, .vp_b, .mins, .tmp4x, .tmp4x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_b, .cmpeq, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .maxs, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_b, .mins, .tmp4x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .slimit_delta_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .sub, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .adds, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .adds, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .sub, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._po, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._po, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._pe, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._pe, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 8 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .addus, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .addus, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .addus, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 8 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u8, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .addus, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .broadcast, .tmp2y, .lea(.tmp0b), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_b, .minu, .tmp3y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_b, .minu, .tmp3x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .minu, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .mmx, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u8, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .qword } } },
                        .{ .type = .vector_8_u8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._q, .mov, .tmp2q, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp3q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp3q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .minu, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp3q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp2d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp2b, .tmp1b, ._, ._ },
                        .{ ._, ._a, .cmov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp2b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp2d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp2b, .tmp1b, ._, ._ },
                        .{ ._, ._a, .cmov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp2b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1b, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1b, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .adds, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .adds, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .adds, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp2w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1w, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .cmpeq, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_w, .broadcast, .tmp3y, .lea(.tmp0w), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp3y, .tmp2y, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp4y, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_w, .maxs, .tmp4y, .tmp4y, .tmp3y, ._ },
                        .{ ._, .vp_w, .mins, .tmp4y, .tmp4y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp4y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .cmpeq, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp3x, .tmp2x, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp4x, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_w, .maxs, .tmp4x, .tmp4x, .tmp3x, ._ },
                        .{ ._, .vp_w, .mins, .tmp4x, .tmp4x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_w, .cmpeq, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .add, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .maxs, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_w, .mins, .tmp4x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp3w, .tmp3w, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp3w, .tmp3w, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp2w, .tmp2w, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1w, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 15 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp2w, .tmp2w, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, .fast_imm16, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp3d, .tmp1d, .uia(1, .src0, .add_bit_size_rem_64), ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp3d, .tmp1d, .uia(1, .src0, .add_bit_size_rem_64), ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp1d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp3d, .uia(31, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp1d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp3d, .uia(31, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp2d, .uia(31, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1w, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp2d, .uia(31, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 16 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i16, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .addus, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 16 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .addus, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .addus, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .addus, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 16 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u16, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .addus, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .broadcast, .tmp2y, .lea(.tmp0w), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_w, .minu, .tmp3y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_w, .minu, .tmp3x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .add, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .minu, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .umax_delta_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .add, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .addus, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .sub, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp2d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._a, .cmov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp2w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 32 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .broadcast, .tmp2y, .lea(.tmp0d), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp5y, .tmp3y, .tmp4y, ._ },
                        .{ ._, .vp_d, .sra, .tmp6y, .tmp3y, .ui(31), ._ },
                        .{ ._, .vp_d, .cmpgt, .tmp3y, .tmp5y, .tmp3y, ._ },
                        .{ ._, .vp_, .xor, .tmp6y, .tmp6y, .tmp2y, ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp3y, .tmp4y, ._ },
                        .{ ._, .v_ps, .blendv, .tmp3y, .tmp6y, .tmp5y, .tmp3y },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_ss, .broadcast, .tmp2x, .lea(.tmp0d), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp5x, .tmp3x, .tmp4x, ._ },
                        .{ ._, .vp_d, .sra, .tmp6x, .tmp3x, .ui(31), ._ },
                        .{ ._, .vp_d, .cmpgt, .tmp3x, .tmp5x, .tmp3x, ._ },
                        .{ ._, .vp_, .xor, .tmp6x, .tmp6x, .tmp2x, ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp3x, .tmp4x, ._ },
                        .{ ._, .v_ps, .blendv, .tmp3x, .tmp6x, .tmp5x, .tmp3x },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp5x, .tmp3x, ._, ._ },
                        .{ ._, .p_d, .add, .tmp5x, .tmp4x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp6x, .tmp3x, ._, ._ },
                        .{ ._, .p_d, .sra, .tmp6x, .ui(31), ._, ._ },
                        .{ ._, .p_d, .cmpgt, .tmp3x, .tmp5x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp6x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .tmp4x, ._, ._ },
                        .{ ._, ._ps, .blendv, .tmp5x, .tmp6x, .tmp3x, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp5x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 32 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 32 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .sa(.src0, .add_smin), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .cmpeq, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_d, .broadcast, .tmp3y, .lea(.tmp0d), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp3y, .tmp2y, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp4y, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_d, .maxs, .tmp4y, .tmp4y, .tmp3y, ._ },
                        .{ ._, .vp_d, .mins, .tmp4y, .tmp4y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp4y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .cmpeq, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .v_ss, .broadcast, .tmp3x, .lea(.tmp0d), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp3x, .tmp2x, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp4x, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_d, .maxs, .tmp4x, .tmp4x, .tmp3x, ._ },
                        .{ ._, .vp_d, .mins, .tmp4x, .tmp4x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_d, .cmpeq, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .add, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .maxs, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_d, .mins, .tmp4x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 31 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 31 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 31 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .dword, .is = 31 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp3d, .tmp1d, .uia(1, .src0, .add_bit_size_rem_64), ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp1d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp3d, .uia(31, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp2d, .uia(31, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 32 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .cmpeq, .tmp1y, .tmp1y, .tmp1y, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp2y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp2y, .tmp1y, ._ },
                        .{ ._, .vp_d, .minu, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_d, .add, .tmp2y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp2y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .cmpeq, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp2x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp2x, .tmp1x, ._ },
                        .{ ._, .vp_d, .minu, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_d, .add, .tmp2x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp2x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_d, .cmpeq, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .xor, .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, .p_d, .minu, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .add, .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .dword, .is = 32 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .dword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .broadcast, .tmp2y, .lea(.tmp0d), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_d, .minu, .tmp3y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_ss, .broadcast, .tmp2x, .lea(.tmp0d), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_d, .minu, .tmp3x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .add, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .minu, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._a, .cmov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp2d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 64 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp5y, .tmp5y, .tmp5y, ._ },
                        .{ ._, .vp_q, .add, .tmp6y, .tmp3y, .tmp4y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp5y, .tmp5y, .tmp3y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .tmp6y, .tmp3y, ._ },
                        .{ ._, .vp_, .xor, .tmp5y, .tmp5y, .tmp2y, ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp3y, .tmp4y, ._ },
                        .{ ._, .v_pd, .blendv, .tmp5y, .tmp5y, .tmp6y, .tmp3y },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp5y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp5x, .tmp5x, .tmp5x, ._ },
                        .{ ._, .vp_q, .add, .tmp6x, .tmp3x, .tmp4x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp5x, .tmp5x, .tmp3x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp6x, .tmp3x, ._ },
                        .{ ._, .vp_, .xor, .tmp5x, .tmp5x, .tmp2x, ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp3x, .tmp4x, ._ },
                        .{ ._, .v_pd, .blendv, .tmp5x, .tmp5x, .tmp6x, .tmp3x },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp5x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .xor, .tmp5x, .tmp5x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp6x, .tmp3x, ._, ._ },
                        .{ ._, .p_q, .add, .tmp6x, .tmp4x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp5x, .tmp3x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp7x, .tmp6x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp7x, .tmp3x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp5x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp7x, .tmp4x, ._, ._ },
                        .{ ._, ._pd, .blendv, .tmp5x, .tmp6x, .tmp7x, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp5x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 64 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp3q, .ui(63), ._, ._ },
                        .{ ._, ._, .xor, .tmp3q, .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp2q, .tmp3q, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 64 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smin), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2q, .ui(63), ._, ._ },
                        .{ ._, ._, .xor, .tmp2q, .tmp1q, ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .cmpeq, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp3y, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp3y, .tmp2y, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp4y, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp5y, .tmp3y, .tmp4y, ._ },
                        .{ ._, .vp_b, .blendv, .tmp4y, .tmp4y, .tmp3y, .tmp5y },
                        .{ ._, .vp_q, .cmpgt, .tmp5y, .tmp4y, .tmp2y, ._ },
                        .{ ._, .vp_b, .blendv, .tmp4y, .tmp4y, .tmp2y, .tmp5y },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp4y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .cmpeq, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .v_, .movddup, .tmp3x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp3x, .tmp2x, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp4x, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp5x, .tmp3x, .tmp4x, ._ },
                        .{ ._, .vp_b, .blendv, .tmp4x, .tmp4x, .tmp3x, .tmp5x },
                        .{ ._, .vp_q, .cmpgt, .tmp5x, .tmp4x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .tmp4x, .tmp4x, .tmp2x, .tmp5x },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_q, .cmpeq, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._, .movddup, .tmp3x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .add, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp5x, .tmp3x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp5x, .tmp4x, ._, ._ },
                        .{ ._, .p_b, .blendv, .tmp4x, .tmp3x, .tmp5x, ._ },
                        .{ ._, ._dqa, .mov, .tmp5x, .tmp4x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp5x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .blendv, .tmp4x, .tmp2x, .tmp5x, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 63 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp3q, .ui(63), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp3q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp4q, .tmp4q, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp2q, .tmp3q, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 63 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp3q, .tmp3q, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2q, .ui(63), ._, ._ },
                        .{ ._, ._, .xor, .tmp2q, .tmp1q, ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .cmov, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp3q, .ui(63), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp3q, .tmp1q, ._, ._ },
                        .{ ._, ._rx, .ro, .tmp4q, .tmp2q, .sia(-31, .src0, .add_bit_size_rem_64), ._ },
                        .{ ._, ._, .add, .tmp4d, .tmp4d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp2q, .tmp3q, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp3q, .ui(63), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp3q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp4q, .sia(-31, .src0, .add_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp4d, .tmp4d, ._, ._ },
                        .{ ._, ._o, .cmov, .tmp2q, .tmp3q, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .tmp2q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp3q, .sia(-31, .src0, .add_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2q, .ui(63), ._, ._ },
                        .{ ._, ._, .xor, .tmp2q, .tmp1q, ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 64 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp4y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .vp_q, .add, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_, .xor, .tmp5y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp4y, .tmp4y, .tmp5y, ._ },
                        .{ ._, .vp_, .@"or", .tmp3y, .tmp3y, .tmp4y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp4x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .vp_q, .add, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_, .xor, .tmp5x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp4x, .tmp4x, .tmp5x, ._ },
                        .{ ._, .vp_, .@"or", .tmp3x, .tmp3x, .tmp4x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp4x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_q, .add, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp5x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp5x, .tmp3x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp4x, .tmp5x, ._, ._ },
                        .{ ._, .p_, .@"or", .tmp3x, .tmp4x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 64 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .qword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp2q, .tmp2q, ._, ._ },
                        .{ ._, ._, .@"or", .tmp1q, .tmp2q, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 63 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .yword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp4y, .tmp4y, .tmp4y, ._ },
                        .{ ._, .vp_q, .add, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp4y, .tmp4y, .tmp3y, ._ },
                        .{ ._, .vp_b, .blendv, .tmp3y, .tmp3y, .tmp2y, .tmp4y },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp4x, .tmp4x, .tmp4x, ._ },
                        .{ ._, .vp_q, .add, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp4x, .tmp4x, .tmp3x, ._ },
                        .{ ._, .vp_b, .blendv, .tmp3x, .tmp3x, .tmp2x, .tmp4x },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 63 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .xor, .tmp4x, .tmp4x, ._, ._ },
                        .{ ._, .p_q, .add, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_b, .blendv, .tmp3x, .tmp2x, .tmp4x, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp3y, .tmp3y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp4y, .tmp3y, .tmp2y, ._ },
                        .{ ._, .vp_b, .blendv, .tmp3y, .tmp3y, .tmp2y, .tmp4y },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp3y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp3x, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp4x, .tmp3x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .tmp3x, .tmp3x, .tmp2x, .tmp4x },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .umax_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_u64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .add, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp4x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .blendv, .tmp3x, .tmp2x, .tmp4x, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp2q, .tmp1q, ._, ._ },
                        .{ ._, ._a, .cmov, .tmp2q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_umax), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp2q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp2q, .tmp1q, ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .tmp1q, ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_exact_remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .reg = .rdi } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rcx } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memiad(.src0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memiad(.dst0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sia(1, .src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .lea(.tmp1q), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .lea(.tmp2q), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .leaad(.tmp3, .sub_src0_elem_size, 8), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, ._r, .sa, .tmp5q, .ui(63), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5q, .tmp4q, ._, ._ },
                        .{ .@"1:", ._, .mov, .lea(.tmp3q), .tmp5q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_remainder_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_exact_remainder_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .si(-1), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp2p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp4p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp5p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp6q, .leasi(.tmp2q, .@"8", .tmp5), ._, ._ },
                        .{ ._, ._, .adc, .tmp6q, .leasi(.tmp3q, .@"8", .tmp5), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp4q, .@"8", .tmp5), .tmp6q, ._, ._ },
                        .{ ._, ._c, .in, .tmp5p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._nc, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp5p, .leaa(.tmp4, .sub_src0_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .sa(.src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_exact_remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .reg = .rdi } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rcx } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memiad(.src0, .tmp0, .add_unaligned_size_add_elem_size, -16), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_unaligned_size_add_elem_size, -16), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memiad(.dst0, .tmp0, .add_unaligned_size_add_elem_size, -16), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sia(2, .src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .lea(.tmp1q), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .lea(.tmp2q), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .leaad(.tmp3, .sub_src0_elem_size, 16), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-2, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, ._r, .sa, .tmp5q, .ui(63), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5q, .tmp4q, ._, ._ },
                        .{ .@"1:", ._, .mov, .lea(.tmp3q), .tmp5q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp5q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .lead(.tmp3q, 8), .tmp5q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_remainder_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_exact_remainder_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .si(-1), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp2p, .memiad(.src0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memiad(.src1, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp4p, .memiad(.dst0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .tmp5p, .sia(1, .src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp6q, .leasi(.tmp2q, .@"8", .tmp5), ._, ._ },
                        .{ ._, ._, .adc, .tmp6q, .leasi(.tmp3q, .@"8", .tmp5), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp4q, .@"8", .tmp5), .tmp6q, ._, ._ },
                        .{ ._, ._c, .in, .tmp5p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._nc, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp5p, .leaad(.tmp4, .sub_src0_elem_size, 8), ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .lea(.tmp4q), .si(0), ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .reg = .rdi } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rcx } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memiad(.src0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memiad(.dst0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sia(1, .src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._l, .sa, .tmp5q, .uia(63, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp5q, .tmp5q, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .leaad(.tmp3, .sub_src0_elem_size, 8), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-2, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, ._r, .sa, .tmp5q, .ui(63), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5q, .tmp4q, ._, ._ },
                        .{ ._, ._, .mov, .lead(.tmp3q, -8), .tmp5q, ._, ._ },
                        .{ .@"1:", ._r, .sa, .tmp5q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .lea(.tmp3q), .tmp5q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_remainder_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .si(-1), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp3p, .memiad(.src0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp4p, .memiad(.src1, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp5p, .memiad(.dst0, .tmp0, .add_unaligned_size_add_elem_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .tmp6p, .sia(1, .src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp7q, .leasi(.tmp3q, .@"8", .tmp6), ._, ._ },
                        .{ ._, ._, .adc, .tmp7q, .leasi(.tmp4q, .@"8", .tmp6), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp5q, .@"8", .tmp6), .tmp7q, ._, ._ },
                        .{ ._, ._c, .in, .tmp6p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .cmp, .tmp7q, .tmp1q, ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .lead(.tmp5q, -8), .tmp1q, ._, ._ },
                        .{ ._, ._, .lea, .tmp6p, .leaad(.tmp5, .sub_src0_elem_size, 8), ._, ._ },
                        .{ ._, ._, .mov, .tmp7d, .sia(-2, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .lea(.tmp5q), .si(0), ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .reg = .rdi } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rcx } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._l, .sa, .tmp5q, .uia(63, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp5q, .tmp5q, ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .leaa(.tmp3, .sub_src0_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, ._r, .sa, .tmp5q, .ui(63), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5q, .tmp4q, ._, ._ },
                        .{ ._, ._, .mov, .lead(.tmp3q, -8), .tmp5q, ._, ._ },
                        .{ .@"1:", ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_remainder_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_umax), ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .si(-1), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp3p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp4p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp5p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp6p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp7q, .leasi(.tmp3q, .@"8", .tmp6), ._, ._ },
                        .{ ._, ._, .adc, .tmp7q, .leasi(.tmp4q, .@"8", .tmp6), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp5q, .@"8", .tmp6), .tmp7q, ._, ._ },
                        .{ ._, ._c, .in, .tmp6p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .cmp, .tmp7q, .tmp1q, ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp6p, .leaa(.tmp5, .sub_src0_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp7d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .lead(.tmp5q, -8), .tmp1q, ._, ._ },
                        .{ .@"1:", ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {f} {f} {f}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .sub, .sub_optimized, .sub_wrap => |air_tag| {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .int = .byte }, .{ .int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .word }, .{ .int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .dword }, .{ .int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .qword }, .{ .int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memsia(.src0d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1d, .memsia(.src1d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0d, .@"4", .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_q, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5d, .leasi(.tmp1d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5d, .leasi(.tmp2d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3d, .@"4", .tmp4), .tmp5d, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .sub, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } },
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } },
                            .none,
                        } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .extern_func = "__subhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .sub, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .sub, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .sub, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subhf3" } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f16, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subhf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .sub, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .sub, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .sub, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .sub, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0q, ._, ._, ._ },
                        .{ ._, .f_, .sub, .src1q, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .sub, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .sub, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .fn_cw, .st, .tmp2w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp3d, .memd(.tmp2b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp2b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp2w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp2b, 1), .tmp3b, ._, ._ },
                        .{ ._, .f_p, .sub, ._, ._, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp2w, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .subr, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .fn_cw, .st, .tmp1w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memd(.tmp1b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp1b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp1w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp1b, 1), .tmp2b, ._, ._ },
                        .{ ._, .f_, .subr, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp1w, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .sub, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .fn_cw, .st, .tmp1w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memd(.tmp1b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp1b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp1w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp1b, 1), .tmp2b, ._, ._ },
                        .{ ._, .f_, .sub, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp1w, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .gnu,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .sub, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_abi = .msvc,
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .mem },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .fn_cw, .st, .tmp0w, ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memd(.tmp0b, 1), ._, ._ },
                        .{ ._, ._, .@"or", .memd(.tmp0b, 1), .ui(0b000_0_00_11), ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp0w, ._, ._, ._ },
                        .{ ._, ._, .mov, .memd(.tmp0b, 1), .tmp1b, ._, ._ },
                        .{ ._, ._, .mov, .tmp1p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp1, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp1, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .sub, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp1, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp1p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        .{ ._, .f_cw, .ld, .tmp0w, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } },
                            .{ .to_param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } },
                            .none,
                        } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .extern_func = "__subtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 0, .at = 0 } } },
                        .{ .type = .f128, .kind = .{ .param_sse = .{ .cc = .ccc, .after = 1, .at = 1 } } },
                        .{ .type = .usize, .kind = .{ .extern_func = "__subtf3" } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {f} {f} {f}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                switch (air_tag) {
                    else => unreachable,
                    .sub, .sub_optimized => {},
                    .sub_wrap => res[0].wrapInt(cg) catch |err| switch (err) {
                        error.SelectFailed => return cg.fail("failed to select {s} wrap {f} {f}", .{
                            @tagName(air_tag),
                            cg.typeOf(bin_op.lhs).fmt(pt),
                            res[0].tracking(cg),
                        }),
                        else => |e| return e,
                    },
                }
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .sub_safe => unreachable,
            .sub_sat => |air_tag| {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .exact_int = 1 }, .{ .exact_int = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .{ .imm = 0 }, .none } },
                        .{ .src = .{ .to_mut_gpr, .{ .imm = 0 }, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{} },
                }, .{
                    .src_constraints = .{ .{ .exact_int = 1 }, .{ .exact_int = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .imm = 0 }, .to_mut_gpr, .none } },
                        .{ .src = .{ .any, .imm8, .none } },
                    },
                    .dst_temps = .{ .{ .imm = 0 }, .unused },
                    .each = .{ .once = &.{} },
                }, .{
                    .required_features = .{ .bmi, null, null, null },
                    .src_constraints = .{ .{ .exact_int = 1 }, .{ .exact_int = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .general_purpose } }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .andn, .dst0d, .src1d, .src0d, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_int = 1 }, .{ .exact_int = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_mut_gpr, .none } },
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .not, .src1b, ._, ._, ._ },
                        .{ ._, ._, .@"and", .dst0b, .src0b, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 8 }, .{ .exact_signed_int = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp0b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 8 }, .{ .exact_signed_int = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .dst0b, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0b, .ui(7), ._, ._ },
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .xor, .tmp0b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .@"test", .dst0b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._po, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .@"test", .dst0b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._pe, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .dst0b, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._c, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u8, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                        .{ ._, ._, .cmc, ._, ._, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .dst0b, .tmp0b, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 16 }, .{ .exact_signed_int = 16 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .tmp1w, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 15 }, .{ .exact_signed_int = 15 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0w, .tmp0w, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, .fast_imm16, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, .fast_imm16, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0w, .ui(15), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0w, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0w, .ui(15), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._c, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                        .{ ._, ._, .cmc, ._, ._, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 32 }, .{ .exact_signed_int = 32 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 32 }, .{ .exact_signed_int = 32 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smin), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 31 }, .{ .exact_signed_int = 31 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .exact_signed_int = 31 }, .{ .exact_signed_int = 31 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .bmi2, .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1d, .dst0d, .uia(1, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .src0d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0d, .ui(31), ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .xor, .tmp0d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .mov, .tmp0d, .dst0d, ._, ._ },
                        .{ ._, ._l, .sa, .tmp0d, .uia(31, .src0, .sub_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .dst0d, .sa(.src0, .add_smax), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._c, .cmov, .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                        .{ ._, ._, .cmc, ._, ._, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .dst0d, .tmp0d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 64 }, .{ .exact_signed_int = 64 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 64 }, .{ .exact_signed_int = 64 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 63 }, .{ .exact_signed_int = 63 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .dst0q, ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp1q, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .exact_signed_int = 63 }, .{ .exact_signed_int = 63 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .dst0q, ._, ._ },
                        .{ ._, ._, .add, .tmp0q, .tmp0q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .cmov, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._rx, .ro, .tmp1q, .dst0q, .sia(-31, .src0, .add_bit_size), ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp0q, .ui(63), ._, ._ },
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .tmp0q, .tmp1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .dst0q, ._, ._ },
                        .{ ._, ._r, .sh, .tmp1q, .sia(-31, .src0, .add_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .tmp1d, ._, ._ },
                        .{ ._, ._o, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .i64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .dst0q, ._, ._ },
                        .{ ._, ._r, .sh, .tmp0q, .sia(-31, .src0, .add_bit_size), ._, ._ },
                        .{ ._, ._, .add, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .dst0q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .ua(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .xor, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .tmp0d, .tmp0d, ._, ._ },
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._c, .cmov, .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u64, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .general_purpose } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, ._, .cmc, ._, ._, ._, ._ },
                        .{ ._, ._, .sbb, .tmp0q, .tmp0q, ._, ._ },
                        .{ ._, ._, .@"and", .dst0q, .tmp0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 128 } },
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 128 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -8), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .memad(.src0q, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memad(.src1q, .add_size, -8), ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ .@"0:", ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .exact_remainder_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(2, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -16), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -16), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -16), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .memad(.src0q, .add_size, -16), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memad(.src1q, .add_size, -16), ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-2, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ .@"0:", ._, .mov, .memad(.dst0q, .add_size, -16), .tmp1q, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .mov, .memsiad(.dst0q, .@"8", .tmp0, .add_size, -8), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1q, .uia(63, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp1q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-2, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -16), .tmp1q, ._, ._ },
                        .{ .@"0:", ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .{ .remainder_signed_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._l, .sa, .tmp1q, .uia(63, .src0, .sub_bit_size_rem_64), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp1q, ._, ._ },
                        .{ ._, ._no, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._r, .sa, .tmp1q, .ui(63), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1q, .tmp2q, ._, ._ },
                        .{ ._, ._, .mov, .memad(.dst0q, .add_size, -8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .tmp1d, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp2q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp2q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp2q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._nc, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sa(.src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .sto, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_mut_mmx, .none } },
                        .{ .src = .{ .to_mmx, .to_mut_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .andn, .dst0q, .src0q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_, .andn, .dst0x, .src1x, .src0x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_mut_sse, .none } },
                        .{ .src = .{ .to_sse, .to_mut_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .andn, .dst0x, .src0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_mut_sse, .none } },
                        .{ .src = .{ .to_sse, .to_mut_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .andn, .dst0x, .src0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .{ .scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_, .andn, .dst0y, .src1y, .src0y, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .subs, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .subs, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .subs, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .subs, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_b, .maxs, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_b, .mins, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_b, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .maxs, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_b, .mins, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .slimit_delta_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_b, .sub, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .subs, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .add, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .adds, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .sub, .dst0x, .tmp2x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_32_i8, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_b, .maxs, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_b, .mins, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .subus, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .subus, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .subus, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .subus, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .qword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .subs, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .subs, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .subs, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .subs, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .mmx, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .qword } } },
                        .{ .type = .vector_4_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .qword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_w, .sub, .dst0q, .src1q, ._, ._ },
                        .{ ._, .p_w, .maxs, .dst0q, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_w, .mins, .dst0q, .lea(.tmp0q), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_w, .maxs, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_w, .mins, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_8_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_w, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_w, .maxs, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_w, .mins, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_16_i16, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_w, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_w, .maxs, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_w, .mins, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .subus, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .subus, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .subus, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .subus, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_d, .sra, .tmp2x, .src0x, .ui(31), ._ },
                        .{ ._, .vp_d, .cmpgt, .tmp3x, .dst0x, .src0x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .lea(.tmp0x), ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp3x, .src1x, ._ },
                        .{ ._, .v_ps, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_d, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .src0x, ._, ._ },
                        .{ ._, .p_d, .sra, .tmp2x, .ui(31), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_d, .cmpgt, .tmp3x, .src0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .src1x, ._, ._ },
                        .{ ._, ._ps, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 32 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_d, .sra, .tmp2y, .src0y, .ui(31), ._ },
                        .{ ._, .vp_d, .cmpgt, .tmp3y, .dst0y, .src0y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .lea(.tmp0y), ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp3y, .src1y, ._ },
                        .{ ._, .v_ps, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_d, .maxs, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_d, .mins, .dst0x, .dst0x, .lea(.tmp0x), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_d, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_d, .maxs, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .p_d, .mins, .dst0x, .lea(.tmp0x), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_8_i32, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_d, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_d, .maxs, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp2), ._, ._ },
                        .{ ._, .vp_d, .mins, .dst0y, .dst0y, .lea(.tmp0y), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .maxu, .dst0x, .src1x, .src0x, ._ },
                        .{ ._, .vp_d, .sub, .dst0x, .dst0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .maxu, .src0x, .src1x, ._, ._ },
                        .{ ._, .p_d, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .maxu, .dst0y, .src1y, .src0y, ._ },
                        .{ ._, .vp_d, .sub, .dst0y, .dst0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2x, .tmp2x, .src0x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .src0x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .lea(.tmp0x), ._ },
                        .{ ._, .vp_, .xor, .tmp3x, .tmp3x, .src1x, ._ },
                        .{ ._, .v_pd, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_signed_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_q, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp2x, .src0x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .src0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .src1x, ._, ._ },
                        .{ ._, ._pd, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 64 } },
                        .{ .scalar_exact_signed_int = .{ .of = .yword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i64, .kind = .{ .smax_mem = .{ .ref = .src0, .vectorize_to = .yword } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_q, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2y, .tmp2y, .src0y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .dst0y, .src0y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .lea(.tmp0y), ._ },
                        .{ ._, .vp_, .xor, .tmp3y, .tmp3y, .src1y, ._ },
                        .{ ._, .v_pd, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp2x, .dst0x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                        .{ ._, .vp_q, .cmpeq, .tmp3x, .tmp3x, .tmp3x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp3x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .tmp2x, .dst0x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                        .{ ._, .vp_q, .cmpeq, .tmp3x, .tmp3x, .tmp3x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp2x, .tmp3x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_b, .blendv, .dst0x, .dst0x, .tmp2x, .tmp3x },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .reg = .xmm0 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .p_q, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_b, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                        .{ ._, .p_q, .cmpeq, .tmp3x, .tmp3x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .dst0x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .blendv, .dst0x, .tmp2x, .tmp3x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_q, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .tmp2y, .dst0y, ._ },
                        .{ ._, .vp_b, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                        .{ ._, .vp_q, .cmpeq, .tmp3y, .tmp3y, .tmp3y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp2y, .tmp3y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp3y, .dst0y, .tmp2y, ._ },
                        .{ ._, .vp_b, .blendv, .dst0y, .dst0y, .tmp2y, .tmp3y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .dst0x, .src0x, .tmp2x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .src1x, .tmp2x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_, .@"and", .dst0x, .dst0x, .tmp2x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .v_, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .dst0x, .src0x, .tmp2x, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .src1x, .tmp2x, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2x, .dst0x, .tmp2x, ._ },
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .vp_, .@"and", .dst0x, .dst0x, .tmp2x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_2_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._, .movddup, .tmp2x, .lea(.tmp0q), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .src0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .src1x, ._, ._ },
                        .{ ._, .p_q, .cmpgt, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_q, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .@"and", .dst0x, .tmp3x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 64 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .yword, .is = 64 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_4_i64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_q, .broadcast, .tmp2y, .lea(.tmp0q), ._, ._ },
                        .{ ._, .vp_, .xor, .dst0y, .src0y, .tmp2y, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .src1y, .tmp2y, ._ },
                        .{ ._, .vp_q, .cmpgt, .tmp2y, .dst0y, .tmp2y, ._ },
                        .{ ._, .vp_q, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .vp_, .@"and", .dst0y, .dst0y, .tmp2y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_, .xor, .tmp0x, .tmp0x, .tmp0x, ._ },
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .v_pd, .blendv, .dst0x, .dst0x, .tmp0x, .dst0x },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .xor, .tmp0x, .tmp0x, ._, ._ },
                        .{ ._, .p_q, .sub, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._pd, .blendv, .dst0x, .tmp0x, .dst0x, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_, .xor, .tmp0y, .tmp0y, .tmp0y, ._ },
                        .{ ._, .vp_q, .sub, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .v_pd, .blendv, .dst0y, .dst0y, .tmp0y, .dst0y },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .yword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .andn, .tmp1y, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_, .andn, .tmp1x, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .andn, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .xword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .andn, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .{ .multiple_scalar_exact_int = .{ .of = .qword, .is = 1 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u8, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .andn, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .subs, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .subs, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .subs, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .qword, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i8, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .subs, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smin), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .byte, .is = 8 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._no, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smin), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .none } } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .cmpeq, .tmp2y, .tmp2y, .tmp2y, ._ },
                        .{ ._, .vp_b, .broadcast, .tmp3y, .lea(.tmp0b), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2y, .tmp3y, .tmp2y, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .sub, .tmp4y, .tmp4y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_b, .maxs, .tmp4y, .tmp4y, .tmp3y, ._ },
                        .{ ._, .vp_b, .mins, .tmp4y, .tmp4y, .tmp2y, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp4y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .vp_b, .cmpeq, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .vp_, .xor, .tmp2x, .tmp3x, .tmp2x, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .sub, .tmp4x, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .vp_b, .maxs, .tmp4x, .tmp4x, .tmp3x, ._ },
                        .{ ._, .vp_b, .mins, .tmp4x, .tmp4x, .tmp2x, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, .p_b, .cmpeq, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp3x, .lea(.tmp0x), ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp4x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .sub, .tmp4x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .maxs, .tmp4x, .tmp3x, ._, ._ },
                        .{ ._, .p_b, .mins, .tmp4x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp4x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .slimit_delta_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .sub, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .subs, .tmp3x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .adds, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .sub, .tmp3x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._po, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2b, .ui(7), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2b, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._po, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._pe, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .@"test", .tmp1b, .sia(-1 << 7, .src0, .sub_smin), ._, ._ },
                        .{ ._, ._pe, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._r, .sa, .tmp1b, .ui(7), ._, ._ },
                        .{ ._, ._, .xor, .tmp1b, .sa(.src0, .add_smax), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .subus, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .subus, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .subus, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .mmx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u8, .kind = .{ .rc = .mmx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._q, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .subus, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .tmp1d, ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp2d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp2b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._c, .cmov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp2b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .tmp1d, ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp2d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp2b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._c, .cmov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp2b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmc, ._, ._, ._, ._ },
                        .{ ._, ._, .sbb, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cmc, ._, ._, ._, ._ },
                        .{ ._, ._, .sbb, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .yword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .subs, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .subs, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .xword, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .subs, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp2w, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .tmp1d, ._, ._ },
                        .{ ._, ._r, .sa, .tmp2d, .ui(31), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .sa(.src0, .add_smax), ._, ._ },
                        .{ ._, ._, .sub, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._o, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .fast_imm16, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .{ .multiple_scalar_exact_signed_int = .{ .of = .word, .is = 16 } },
                        .any,
                    },
             