module Octet (
  Octet(..),
  octet_properties,
  octet_tests )
where

import Test.Tasty ( TestTree, testGroup )
import Test.Tasty.HUnit ( (@?=), testCase )
import Test.Tasty.QuickCheck (
  Arbitrary( arbitrary ),
  Gen,
  Property,
  (==>),
  testProperty )

import Bit as B( Bit( Zero, One) )
import Maskable( Maskable( apply_mask) )
import Maskbits(
  Maskbits( Zero, One, Two, Three, Four, Five, Six, Seven, Eight ) )

-- | An Octet consists of eight bits. For our purposes, the most
--   significant bit will come "first." That is, b1 is in the 2^7
--   place while b8 is in the 2^0 place.
data Octet =
  Octet { b1 :: Bit,
          b2 :: Bit,
          b3 :: Bit,
          b4 :: Bit,
          b5 :: Bit,
          b6 :: Bit,
          b7 :: Bit,
          b8 :: Bit }
    deriving (Eq, Ord)


instance Show Octet where
  show oct = show (fromEnum oct)


instance Arbitrary Octet where
  arbitrary = do
    a1 <- arbitrary :: Gen Bit
    a2 <- arbitrary :: Gen Bit
    a3 <- arbitrary :: Gen Bit
    a4 <- arbitrary :: Gen Bit
    a5 <- arbitrary :: Gen Bit
    a6 <- arbitrary :: Gen Bit
    a7 <- arbitrary :: Gen Bit
    a8 <- arbitrary :: Gen Bit
    return (Octet a1 a2 a3 a4 a5 a6 a7 a8)


instance Maskable Octet where
  apply_mask oct Eight _ =  oct

  apply_mask oct Seven bit =
    oct { b8 = bit }

  apply_mask oct Six bit =
    oct { b8 = bit, b7 = bit }

  apply_mask oct Five bit =
    oct { b8 = bit, b7 = bit, b6 = bit }

  apply_mask oct Four bit =
    oct { b8 = bit, b7 = bit, b6 = bit, b5 = bit }

  apply_mask oct Three bit =
    oct { b8 = bit, b7 = bit, b6 = bit, b5 = bit, b4 = bit }

  apply_mask oct Two bit =
    oct { b8 = bit, b7 = bit, b6 = bit, b5 = bit, b4 = bit, b3 = bit }

  apply_mask oct Maskbits.One bit =
    oct { b8 = bit, b7 = bit, b6 = bit, b5 = bit,
          b4 = bit, b3 = bit, b2 = bit }

  apply_mask oct Maskbits.Zero bit =
    oct { b8 = bit, b7 = bit, b6 = bit, b5 = bit,
          b4 = bit, b3 = bit, b2 = bit, b1 = bit }

  -- The Maskbits must be in [Eight..ThirtyTwo].
  apply_mask oct _ _ = oct


instance Bounded Octet where
  -- | The octet with the least possible value.
  minBound =
    Octet B.Zero B.Zero B.Zero B.Zero B.Zero B.Zero B.Zero B.Zero

  -- | The octet with the greatest possible value.
  maxBound =
    Octet B.One B.One B.One B.One B.One B.One B.One B.One


instance Enum Octet where

  -- | Create an 'Octet' from an 'Int'. The docs for Enum say we
  --   should throw a runtime error on out-of-bounds, so we do.
  toEnum x
    | x < minBound || x > maxBound = error "octet out of bounds"
    | otherwise = Octet a1 a2 a3 a4 a5 a6 a7 a8
        where
          a1 = if (x >= 128) then B.One else B.Zero
          a2 = if ((x `mod` 128) >= 64) then B.One else B.Zero
          a3 = if ((x `mod` 64)  >= 32) then B.One else B.Zero
          a4 = if ((x `mod` 32)  >= 16) then B.One else B.Zero
          a5 = if ((x `mod` 16)  >= 8)  then B.One else B.Zero
          a6 = if ((x `mod` 8)   >= 4)  then B.One else B.Zero
          a7 = if ((x `mod` 4)   >= 2)  then B.One else B.Zero
          a8 = if ((x `mod` 2)   == 1)  then B.One else B.Zero

  -- | Convert each bit to its integer value, and multiply by the
  --   appropriate power of two. Sum them up, and we should get an integer
  --   between 0 and 255.
  fromEnum x =
    128 * (fromEnum (b1 x)) +
    64  * (fromEnum (b2 x)) +
    32  * (fromEnum (b3 x)) +
    16  * (fromEnum (b4 x)) +
    8   * (fromEnum (b5 x)) +
    4   * (fromEnum (b6 x)) +
    2   * (fromEnum (b7 x)) +
    1   * (fromEnum (b8 x))



instance Read Octet where
  readsPrec _ s =
    case (reads s :: [(Int, String)]) of
      []              -> []
      (x,leftover):_  -> go x leftover
    where
      go :: Int -> String -> [(Octet, String)]
      go x' leftover'
        | x' < minBound || x' > maxBound = []
        | otherwise = [(toEnum x', leftover')]


-- Test lists.
octet_tests :: TestTree
octet_tests =
  testGroup "Octet Tests" [
    test_octet_to_enum1,
    test_octet_mask1,
    test_octet_mask2 ]

octet_properties :: TestTree
octet_properties =
  testGroup
    "Octet Properties "
    [ prop_from_enum_to_enum_inverses,
      prop_read_show_inverses,
      prop_ord_instances_agree ]

-- QuickCheck properties
prop_from_enum_to_enum_inverses :: TestTree
prop_from_enum_to_enum_inverses =
  testProperty "fromEnum and toEnum are inverses" prop
  where
    prop :: Int -> Property
    prop x =
      (0 <= x) && (x <= 255) ==>
        fromEnum (toEnum x :: Octet) == x

prop_read_show_inverses :: TestTree
prop_read_show_inverses =
  testProperty "read and show are inverses" prop
  where
    prop :: Int -> Property
    prop x = (0 <= x) && (x <= 255) ==> x' == x
      where
        oct :: Octet
        oct = read $ show x

        x' :: Int
        x' = read $ show oct

-- | Ensure that the Ord instance for Octets agrees with the Ord
--   instance for Int (i.e. that numerical comparisons work).
prop_ord_instances_agree :: TestTree
prop_ord_instances_agree =
  testProperty "the Octet and Int Ord instances agree" prop
  where
    prop :: Int -> Int -> Property
    prop x y = (0 <= x) && (x <= 255) && (0 <= y) && (y <= 255) ==> ord == ord'
      where
        ord = (x <= y)

        oct1 = toEnum x :: Octet
        oct2 = toEnum y :: Octet
        ord' = (oct1 <= oct2)

-- HUnit Tests
test_octet_to_enum1 :: TestTree
test_octet_to_enum1 =
  testCase desc $ actual @?= expected
  where
    desc = "toEnum 128 should parse as 10000000"
    expected = Octet B.One B.Zero B.Zero B.Zero B.Zero B.Zero B.Zero B.Zero
    actual = toEnum 128 :: Octet


test_octet_mask1 :: TestTree
test_octet_mask1 =
  testCase desc $ actual @?= expected
  where
    desc = "The network bits of 255/4 should equal 240"
    expected = toEnum 240 :: Octet
    actual = apply_mask (toEnum 255) Four B.Zero :: Octet


test_octet_mask2 :: TestTree
test_octet_mask2 =
  testCase desc $ actual @?= expected
  where
    desc = "The network bits of 255/1 should equal 128"
    expected = toEnum 128 :: Octet
    actual = apply_mask (toEnum 255) Maskbits.One B.Zero :: Octet
