import { StatsigClient } from '@statsig/js-client';
import { DynamicConfig } from './compat/DynamicConfig';
import { Layer } from './compat/Layer';
import { type FetcherOptions } from './fetcher';
import { type LocalOverrides, PersistentOverrideAdapter } from './PersistentOverrideAdapter';
import { type BaseClientOptions, type CheckGateOptions, type ClientOptions, type CustomAttributes, type FromValuesClientOptions, type GetExperimentOptions, type GetExperimentValueOptions, type GetLayerOptions, type GetLayerValueOptions, type Identifiers, type Provider } from './types';
export declare class Client {
    private statsigClient?;
    private user?;
    private initOptions?;
    private initPromise;
    /** True if an initialize method was called and completed successfully. */
    private initCompleted;
    /**
     * True if an initialize method was called and completed, meaning the client is now usable.
     * However if there was an error during initialization it may have initialized with default
     * values. Use {@link initCompleted} to check for this.
     */
    private initWithDefaults;
    private currentIdentifiers?;
    private currentAttributes?;
    private hasCheckGateErrorOccurred;
    private hasGetExperimentErrorOccurred;
    private hasGetExperimentValueErrorOccurred;
    private hasGetLayerErrorOccurred;
    private hasGetLayerValueErrorOccurred;
    private provider?;
    private subscriptions;
    private dataAdapter;
    private overrideAdapter;
    constructor({ localStorageKey, overrideAdapter, }?: {
        localStorageKey?: string;
        overrideAdapter?: PersistentOverrideAdapter;
    });
    /**
     * @description
     * This method initializes the client using a network call to fetch the bootstrap values.
     * If the client is inialized with an `analyticsWebClient`, it will send an operational event
     * to GASv3 with the following attributes:
     * - targetApp: the target app of the client
     * - clientVersion: the version of the client
     * - success: whether the initialization was successful
     * - startTime: the time when the initialization started
     * - totalTime: the total time it took to initialize the client
     * - apiKey: the api key used to initialize the client
     * @param clientOptions {ClientOptions}
     * @param identifiers {Identifiers}
     * @param customAttributes {CustomAttributes}
     * @returns {Promise<void>}
     */
    initialize(clientOptions: ClientOptions, identifiers: Identifiers, customAttributes?: CustomAttributes): Promise<void>;
    /**
     * @description
     * This method initializes the client using the provider given to call to fetch the bootstrap values.
     * If the client is initialized with an `analyticsWebClient`, it will send an operational event
     * to GASv3 with the following attributes:
     * - targetApp: the target app of the client
     * - clientVersion: the version of the client
     * - success: whether the initialization was successful
     * - startTime: the time when the initialization started
     * - totalTime: the total time it took to initialize the client
     * - apiKey: the api key used to initialize the client
     * @param clientOptions {ClientOptions}
     * @param provider {Provider}
     * @param identifiers {Identifiers}
     * @param customAttributes {CustomAttributes}
     * @returns {Promise<void>}
     */
    initializeWithProvider(clientOptions: BaseClientOptions, provider: Provider, identifiers: Identifiers, customAttributes?: CustomAttributes): Promise<void>;
    private applyUpdateCallback;
    private fireClientEvent;
    initializeFromValues(clientOptions: FromValuesClientOptions, identifiers: Identifiers, customAttributes?: CustomAttributes, initializeValues?: Record<string, unknown>): Promise<void>;
    protected assertInitialized(statsigClient: StatsigClient | undefined): asserts statsigClient;
    /**
     * This method updates the user using a network call to fetch the new set of values.
     * @param fetchOptions {FetcherOptions}
     * @param identifiers {Identifiers}
     * @param customAttributes {CustomAttributes}
     */
    updateUser(fetchOptions: FetcherOptions, identifiers: Identifiers, customAttributes?: CustomAttributes): Promise<void>;
    /**
     * This method updates the user using the provider given on initialisation to get the new set of
     * values.
     * @param identifiers {Identifiers}
     * @param customAttributes {CustomAttributes}
     */
    updateUserWithProvider(identifiers: Identifiers, customAttributes?: CustomAttributes): Promise<void>;
    /**
     * This method updates the user given a new set of bootstrap values obtained from one of the
     * server-side SDKs.
     *
     * @param identifiers {Identifiers}
     * @param customAttributes {CustomAttributes}
     * @param initializeValues {Record<string,unknown>}
     */
    updateUserWithValues(identifiers: Identifiers, customAttributes?: CustomAttributes, initializeValues?: Record<string, unknown>): Promise<void>;
    initializeCalled(): boolean;
    initializeCompleted(): boolean;
    /**
     * Returns the value for a feature gate. Returns false if there are errors.
     * @param {string} gateName - The name of the feature gate.
     * @param {Object} options
     * @param {boolean} options.fireGateExposure
     *        Whether or not to fire the exposure event for the gate. Defaults to true.
     *        To log an exposure event manually at a later time, use {@link Client.manuallyLogGateExposure}
     *        (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-)).
     */
    checkGate(gateName: string, options?: CheckGateOptions): boolean;
    isGateExist(gateName: string): boolean;
    isExperimentExist(experimentName: string): boolean;
    /**
     * Manually log a gate exposure (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-)).
     * This is useful if you have evaluated a gate earlier via {@link Client.checkGate} where
     * <code>options.fireGateExposure</code> is false.
     * @param gateName
     */
    manuallyLogGateExposure(gateName: string): void;
    /**
     * Returns the entire config for a given experiment.
     *
     * @param {string} experimentName - The name of the experiment
     * @param {Object} options
     * @param {boolean} options.fireExperimentExposure - Whether or not to fire the exposure event
     * for the experiment. Defaults to true. To log an exposure event manually at a later time, use
     * {@link Client.manuallyLogExperimentExposure} (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-)).
     * @returns The config for an experiment
     * @example
     * ```ts
     * const experimentConfig = client.getExperiment('example-experiment-name');
     * const backgroundColor: string = experimentConfig.get('backgroundColor', 'yellow');
     * ```
     */
    getExperiment(experimentName: string, options?: GetExperimentOptions): DynamicConfig;
    /**
     * Returns the value of a given parameter in an experiment config.
     *
     * @template T
     * @param {string} experimentName - The name of the experiment
     * @param {string} parameterName - The name of the parameter to fetch from the experiment config
     * @param {T} defaultValue - The value to serve if the experiment or parameter do not exist, or
     * if the returned value does not match the expected type.
     * @param {Object} options
     * @param {boolean} options.fireExperimentExposure - Whether or not to fire the exposure event
     * for the experiment. Defaults to true. To log an exposure event manually at a later time, use
     * {@link Client.manuallyLogExperimentExposure} (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-))
     * @param {function} options.typeGuard - A function that asserts that the return value has the
     * expected type. If this function returns false, then the default value will be returned
     * instead. This can be set to protect your code from unexpected values being set remotely. By
     * default, this will be done by asserting that the default value and value are the same primitive
     * type.
     * @returns The value of the parameter if the experiment and parameter both exist, otherwise the
     * default value.
     * @example
     ``` ts
     type ValidColor = 'blue' | 'red' | 'yellow';
     type ValidColorTypeCheck = (value: unknown) => value is ValidColor;

     const isValidColor: ValidColorTypeCheck =
            (value: unknown) => typeof value === 'string' && ['blue', 'red', 'yellow'].includes(value);

     const buttonColor: ValidColor = client.getExperimentValue(
            'example-experiment-name',
            'backgroundColor',
            'yellow',
            {
                    typeGuard: isValidColor
            }
     );
     ```
    */
    getExperimentValue<T>(experimentName: string, parameterName: string, defaultValue: T, options?: GetExperimentValueOptions<T>): T;
    /**
     * Manually log an experiment exposure (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-)).
     * This is useful if you have evaluated an experiment earlier via {@link Client.getExperimentValue} or
     * {@link Client.getExperiment} where <code>options.fireExperimentExposure</code> is false.
     * @param experimentName
     */
    manuallyLogExperimentExposure(experimentName: string): void;
    /**
     * Manually log a layer exposure (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-)).
     * This is useful if you have evaluated a layer earlier via {@link Client.getLayerValue} where <code>options.fireExperimentExposure</code> is false.
     * @param layerName
     * @param parameterName
     */
    manuallyLogLayerExposure(layerName: string, parameterName: string): void;
    shutdownStatsig(): void;
    /**
     * Adds a new override for the given gate.
     *
     * This method is additive, meaning you can call it multiple times with different gate names to
     * build your full set of overrides.
     *
     * Overrides are persisted to the `STATSIG_OVERRIDES` key in localStorage, so they
     * will continue to affect every client that is initialized on the same domain after this method
     * is called. If you are using this API for testing purposes, you should call
     * {@link Client.clearGateOverride} after your tests are completed to remove this
     * localStorage entry.
     *
     * @param {string} gateName
     * @param {boolean} value
     */
    overrideGate(gateName: string, value: boolean): void;
    /**
     * Removes any overrides that have been set for the given gate.
     */
    clearGateOverride(gateName: string): void;
    /**
     * Adds a new override for the given config (or experiment).
     *
     * This method is additive, meaning you can call it multiple times with different experiment
     * names to build your full set of overrides.
     *
     * Overrides are persisted to the `STATSIG_OVERRIDES` key in localStorage, so they
     * will continue to affect every client that is initialized on the same domain after this method
     * is called. If you are using this API for testing purposes, you should call
     * {@link Client.clearConfigOverride} after your tests are completed to remove this
     * localStorage entry.
     *
     * @param {string} experimentName
     * @param {object} values
     */
    overrideConfig(experimentName: string, values: Record<string, unknown>): void;
    /**
     * Removes any overrides that have been set for the given experiment.
     * @param {string} experimentName
     */
    clearConfigOverride(experimentName: string): void;
    /**
     * Set overrides for gates, experiments and layers in batch.
     *
     * Note that these overrides are **not** additive and will completely replace any that have been
     * added via prior calls to {@link Client.overrideConfig} or
     * {@link Client.overrideGate}.
     *
     * Overrides are persisted to the `STATSIG_OVERRIDES` key in localStorage, so they
     * will continue to affect every client that is initialized on the same domain after this method
     * is called. If you are using this API for testing purposes, you should call
     * {@link Client.clearAllOverrides} after your tests are completed to remove this
     * localStorage entry.
     */
    setOverrides(overrides: Partial<LocalOverrides>): void;
    /**
     * @returns The current overrides for gates, configs (including experiments) and layers.
     */
    getOverrides(): LocalOverrides;
    /**
     * Clears overrides for all gates, configs (including experiments) and layers.
     */
    clearAllOverrides(): void;
    /**
     * Returns whether the given identifiers and customAttributes align with the current
     * set that is being used by the client.
     *
     * If this method returns false, then the {@link Client.updateUser},
     * {@link Client.updateUserWithValues} or {@link Client.updateUserWithProvider}
     * methods can be used to re-align these values.
     *
     * @param identifiers
     * @param customAttributes
     * @returns a flag indicating whether the clients current configuration aligns with the given values
     */
    isCurrentUser(identifiers: Identifiers, customAttributes?: CustomAttributes): boolean;
    /**
     * Subscribe to updates where the given callback will be called with the current checkGate value
     * @param gateName
     * @param callback
     * @param options
     * @returns off function to unsubscribe from updates
     */
    onGateUpdated(gateName: string, callback: (value: boolean) => void, options?: CheckGateOptions): () => void;
    /**
     * Subscribe to updates where the given callback will be called with the current experiment value
     * @param experimentName
     * @param parameterName
     * @param defaultValue
     * @param callback
     * @param options
     * @returns off function to unsubscribe from updates
     */
    onExperimentValueUpdated<T>(experimentName: string, parameterName: string, defaultValue: T, callback: (value: T) => void, options?: GetExperimentValueOptions<T>): () => void;
    /**
     * Subscribe so on any update the callback will be called.
     * NOTE: The callback will be called whenever the values are updated even if the values have not
     * changed.
     * @param callback
     * @returns off function to unsubscribe from updates
     */
    onAnyUpdated(callback: () => void): () => void;
    /**
     * This method initializes the client using a network call to fetch the bootstrap values for the
     * given user.
     *
     * @param clientOptions
     * @param identifiers
     * @param customAttributes
     * @private
     */
    private init;
    private initWithProvider;
    /**
     * This method initializes the client using a set of boostrap values obtained from one of the
     * server-side SDKs.
     *
     * @param clientOptions
     * @param identifiers
     * @param customAttributes
     * @param initializeValues
     * @private
     */
    private initFromValues;
    /**
     * This method updates the user for this client with the bootstrap values returned from a given
     * Promise.
     * It uses the customAttributes from fetching experiment values to update the Statsig user but
     * uses the customAttributes from given input to check if the user has changed.
     *
     * @param {Identifiers} identifiers
     * @param {CustomAttributes} customAttributes
     * @param {Promise<InitializeValues>} getInitializeValues
     * @private
     */
    private updateUserUsingInitializeValuesProducer;
    /**
     * This method updates the user on the nested Statsig client
     *
     * @param identifiers
     * @param customAttributes
     * @param initializeValuesPromise
     * @private
     */
    private updateStatsigClientUser;
    /**
     * Call this if modifying the values being served by the Statsig library since it has its own
     * memoization cache which will not be updated if the values are changed outside of the library.
     */
    protected statsigValuesUpdated: () => void;
    /**
     * @returns string version of the current package in semver style.
     */
    getPackageVersion(): string;
    /**
     * Returns a specified layer otherwise returns an empty layer as a default value if the layer doesn't exist.
     *
     * @param {string} layerName - The name of the layer
     * @param {Object} options
     * @param {boolean} options.fireLayerExposure - Whether or not to fire the exposure event for the
     * layer. Defaults to true. To log an exposure event manually at a later time, use
     * {@link Client.manuallyLogLayerExposure} (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-)).
     * @returns A layer
     * @example
     * ```ts
     * const layer = client.getLayer('example-layer-name');
     * const exampletitle: string = layer.get("title", "Welcome to Statsig!");
     * ```
     */
    getLayer(
    /** The name of the layer */
    layerName: string, options?: GetLayerOptions): Layer;
    /**
     * Returns the value of a given parameter in a layer config.
     *
     * @template T
     * @param {string} layerName - The name of the layer
     * @param {string} parameterName - The name of the parameter to fetch from the layer config
     * @param {T} defaultValue - The value to serve if the layer or parameter do not exist, or if the
     * returned value does not match the expected type.
     * @param {Object} options
     * @param {boolean} options.fireLayerExposure - Whether or not to fire the exposure event for the
     * layer. Defaults to true. To log an exposure event manually at a later time, use
     * {@link Client.manuallyLogLayerExposure} (see [Statsig docs about manually logging exposures](https://docs.statsig.com/client/jsClientSDK#manual-exposures-))
     * @param {function} options.typeGuard - A function that asserts that the return value has the expected type. If this function returns false, then the default value will be returned instead. This can be set to protect your code from unexpected values being set remotely. By default, this will be done by asserting that the default value and value are the same primitive type.
     * @returns The value of the parameter if the layer and parameter both exist, otherwise the default value.
     * @example
     * ``` ts
     * type ValidColor = 'blue' | 'red' | 'yellow';
     * type ValidColorTypeCheck = (value: unknown) => value is ValidColor;
     *
     * const isValidColor: ValidColorTypeCheck =
     *    (value: unknown) => typeof value === 'string' && ['blue', 'red', 'yellow'].includes(value);
     *
     * const buttonColor: ValidColor = client.getLayerValue(
     *    'example-layer-name',
     *    'backgroundColor',
     *    'yellow',
     *    {
     *        typeGuard: isValidColor
     *    }
     * );
     * ```
     */
    getLayerValue<T>(layerName: string, parameterName: string, defaultValue: T, options?: GetLayerValueOptions<T>): T;
}
