#ifndef MICROTEX_FONT_STYLE_H
#define MICROTEX_FONT_STYLE_H

#include "microtexexport.h"
#include "utils/types.h"

namespace microtex {

/** Enum represents all supported font style */
enum class FontStyle : u16 {
  none = 0,         // unspecified, choose an appropriate style depends on context
  rm = 0b1,         // roman
  bf = 0b10,        // bold
  it = 0b100,       // italic
  cal = 0b1000,     // calligraphic
  frak = 0b10000,   // fraktur
  bb = 0b100000,    // double-struck
  sf = 0b1000000,   // sans-serif
  tt = 0b10000000,  // type-writer (monospaced)
  // composed styles
  bfit = bf | it,
  bfcal = bf | cal,
  bffrak = bf | frak,
  sfbf = bf | sf,
  sfit = it | sf,
  sfbfit = bf | it | sf,
  // invalid
  invalid = 0xffff
};

/**
 * Get the most similar font style from the given source font style.
 *
 * <p>
 * The similarity is measured by the number of set bits of the result
 * that generated by the bitwise operation '&' between source font style
 * and the composed font styles.
 */
FontStyle MICROTEX_EXPORT findClosestStyle(FontStyle src);

/** Test if the given font style is unspecified*/
bool MICROTEX_EXPORT isUnspecified(FontStyle style);

/** Test if the given font style is roman */
bool MICROTEX_EXPORT isRoman(FontStyle style);

/** Test if the given font style is bold */
bool MICROTEX_EXPORT isBold(FontStyle style);

/** Test if the given font style is italic */
bool MICROTEX_EXPORT isItalic(FontStyle style);

/** Test if the given font style is sans-serif */
bool MICROTEX_EXPORT isSansSerif(FontStyle style);

/** Test if the given font style is mono-spaced */
bool MICROTEX_EXPORT isMono(FontStyle style);

}  // namespace microtex

#endif  // MICROTEX_FONT_STYLE_H
