/*******************************************************************************
* File:   		localkeygen.c
* Date:   		2004-05-21
* Author: 		Fernando Tarín
* Last Modified:	2004-5-23
*
* Description: Generates keywords for 64 an 128 bit mode
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU Library General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*******************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <openssl/rand.h>
#include "localkeygen.h"

unsigned int  generator_mode, start_value, end_value, key_length;
unsigned int  current_key[13], end_key[13];
unsigned char start_key[13];

// old vars
unsigned int mode;
unsigned int temp_key[13], bitkeys[13];


int hextoi(const char * num){
	int number;
	switch (num[0]){
		case '0':
			number = 0;
			break;
		case '1':
			number = 16;
			break;
		case '2':
			number = 32;
			break;
		case '3':
			number = 48;
			break;
		case '4':
			number = 64;
			break;
		case '5':
			number = 80;
			break;
		case '6':
			number = 96;
			break;
		case '7':
			number = 112;
			break;
		case '8':
			number = 128;
			break;
		case '9':
			number = 144;
			break;
		case 'A':
			number = 160;
			break;
		case 'B':
			number = 176;
			break;
		case 'C':
			number = 192;
			break;
		case 'D':
			number = 208;
			break;
		case 'E':
			number = 224;
			break;
		case 'F':
			number = 240;
			break;
		default:
			number = 0;
			break;
	}
	switch (num[1]){
		case '0':
			break;
		case '1':
			number += 1;
			break;
		case '2':
			number += 2;
			break;
		case '3':
			number += 3;
			break;
		case '4':
			number += 4;
			break;
		case '5':
			number += 5;
			break;
		case '6':
			number += 6;
			break;
		case '7':
			number += 7;
			break;
		case '8':
			number += 8;
			break;
		case '9':
			number += 9;
			break;
		case 'A':
			number += 10;
			break;
		case 'B':
			number += 11;
			break;
		case 'C':
			number += 12;
			break;
		case 'D':
			number += 13;
			break;
		case 'E':
			number += 14;
			break;
		case 'F':
			number += 15;
			break;
		default:		
			break;
	}
	return number;
}

int key_cmp(const unsigned char * start_key, const unsigned char * end_key, int number_bits){
	int i, total_b, greater=0;
	unsigned char key1[13],key2[13];

	if (!parse_key(start_key, key1, number_bits)) return 0;
	if (!parse_key(end_key, key2, number_bits)) return 0;

	if (number_bits == 64) total_b = 5;
	else total_b = 13;

	for (i=0;i<total_b;i++){	
		if (key1[i] > key2[i])
			if (!greater) return 0;
		if (key2[i] > key1[i])
			greater = 1;
		}

	return 1;
}


int parse_key(const unsigned char * key_to_parse, unsigned char * key, int number_bits){
	int i=0,j=0,k=0;
	int total_b, total_c;
	char numero[5];
	if (number_bits == 64){
		total_b = 5; total_c = 15;
	}
	else{
		total_b = 13; total_c = 39;
	}
	while (j < total_b && i < total_c ){
		if (key_to_parse[i] == ':' || key_to_parse[i] == '\0'){
			numero[k] = '\0';
			key[j] = hextoi(numero);
			i++; j++; k=0;
		}
		if (isdigit(key_to_parse[i])) numero[k] = key_to_parse[i];
		else numero[k] = toupper(key_to_parse[i]);
		if (key_to_parse[i] == '\0') break;
		i++; k++;
	}
	if (number_bits == 64 && j == 5) return 1;
	if (number_bits == 128 && j == 13) return 1;
	return 0;
}


int initialize_generator(int klength, unsigned char * skey, unsigned char * ekey, int gen_mode){
	int i, bkeys;
	
	key_length = klength;
	
	switch (key_length){
		case 64:
			bkeys = 5;
			break;
		case 128:
			bkeys = 13;
			break;
		default:
			return -1;
	}
	
	generator_mode = gen_mode;
	
	switch (generator_mode){
		case ALL:
			start_value=0;
			end_value=255;
			break;
		case ALPHA:
			start_value=65;
			end_value=122;
			break;
		case ALPHANUM:
			start_value=48;
			end_value=122;
			break;
		case NUMERIC:
			start_value=48;
			end_value=57;
			break;
		case PRINTABLE:
			start_value=33;
			end_value=126;
			break;
		case RANDOM:
			RAND_bytes(start_key, bkeys);
			start_key[bkeys-1] = 255;
			start_value=0;
			end_value=255;
			break;
		case UDEF:
			start_value=0;
			end_value=255;
			break;
		default:
			return -1;
	}
	
	if (generator_mode == UDEF){
		if (skey == NULL || ekey == NULL) return -1;
		for (i=0; i<bkeys; i++){
			start_key[i] = skey[i];
			end_key[i] = ekey[i];
		}
		end_key[bkeys-1] += 1;
	}
	
	else if (generator_mode == RANDOM){
		for (i=0; i<bkeys; i++){
			end_key[i] = start_key[i];
		}
		end_key[bkeys-1] -= 1;
	}
	
	else{
		for (i=0; i<bkeys; i++){
			start_key[i] = start_value;
			end_key[i] = end_value;
		}
		end_key[bkeys-1] += 1;
	}
	
	for (i=0; i<bkeys ;i++){
		current_key[i] = start_key[i];
		if (skey != NULL) skey[i] = start_key[i];
		if (ekey != NULL) ekey[i] = end_key[i];
	}
	
	return 0;
}
	

int generate_key(unsigned char * key){
	int i;
	
	switch (key_length){
		case 64:
			if (!((current_key[0] == end_key[0]) && (current_key[1] == end_key[1]) && (current_key[2] == end_key[2]) && (current_key[3] == end_key[3]) && (current_key[4] == end_key[4]))){
				i=4;
				while (current_key[i] > end_value){
					if (current_key[0] > 255) current_key[0] = current_key[1] = current_key[2] = current_key[3] = current_key[4] = 0;
					current_key[i] = start_value;					
					current_key[i-1] = current_key[i-1] + 1;
					
					if (generator_mode == ALPHANUM){
						if (current_key[i-1] == 58) current_key[i-1] = 65;
						if (current_key[i-1] == 91) current_key[i-1] = 97;
					}
					else if (generator_mode == ALPHA){
						if (current_key[i-1] == 91) current_key[i-1] = 97;
					}
					else if (generator_mode == PRINTABLE){
						if (current_key[i-1] == 127) current_key[i-1] = 33;
					}
					
					i = i - 1;
				}
				
				sprintf(key,"%c%c%c%c%c",current_key[0],current_key[1],current_key[2],current_key[3],current_key[4]);
				current_key[4] += 1;
				
				if (generator_mode == ALPHANUM) {
					if (current_key[4] == 58) current_key[4] = 65;
					if (current_key[4] == 91) current_key[4] = 97;
				}
				
				else if (generator_mode == ALPHA){
					if (current_key[4] == 91) current_key[4] = 97;
				}
				
				return 1;
			}
			if (generator_mode == RANDOM){
				sprintf(key,"%c%c%c%c%c",current_key[0],current_key[1],current_key[2],current_key[3],current_key[4]);
				generator_mode = -1;
				return 1;
			}				
			break;			
		case 128:
			if (!((current_key[0] == end_key[0]) && (current_key[1] == end_key[1]) && (current_key[2] == end_key[2]) && (current_key[3] == end_key[3]) && (current_key[4] == end_key[4]) &&\
				(current_key[5] == end_key[5]) && (current_key[6] == end_key[6]) && (current_key[7] == end_key[7]) && (current_key[8] == end_key[8]) && (current_key[9] == end_key[9]) &&\
				(current_key[10] == end_key[10]) && (current_key[11] == end_key[11]) && (current_key[12] == end_key[12]))){
				i=12;
				while (current_key[i] > end_value){
					if (current_key[0] > 255){
						current_key[0] = current_key[1] = current_key[2] = current_key[3] = current_key[4] = current_key[5] = current_key[6] = current_key[7] =\
					    current_key[8] = current_key[9] = current_key[10] = current_key[11] = current_key[12] = 0;
					}
					current_key[i] = start_value;					
					current_key[i-1] = current_key[i-1] + 1;
					
					if (generator_mode == ALPHANUM){
						if (current_key[i-1] == 58) current_key[i-1] = 65;
						if (current_key[i-1] == 91) current_key[i-1] = 97;
					}
					else if (generator_mode == ALPHA){
						if (current_key[i-1] == 91) current_key[i-1] = 97;
					}
					else if (generator_mode == PRINTABLE){
						if (current_key[i-1] == 127) current_key[i-1] = 33;
					}
					
					i = i - 1;	
				}
				
				sprintf(key,"%c%c%c%c%c%c%c%c%c%c%c%c%c",current_key[0],current_key[1],current_key[2],current_key[3],current_key[4]\
					,current_key[5],current_key[6],current_key[7],current_key[8],current_key[9],current_key[10],current_key[11],current_key[12]);
				current_key[12] += 1;
				
				if (generator_mode == ALPHANUM) {
					if (current_key[12] == 58) current_key[12] = 65;
					if (current_key[12] == 91) current_key[12] = 97;
				}
				
				else if (generator_mode == ALPHA){
					if (current_key[12] == 91) current_key[12] = 97;
				}
				
				return 1;
			}
			if (generator_mode == RANDOM){
				sprintf(key,"%c%c%c%c%c%c%c%c%c%c%c%c%c",current_key[0],current_key[1],current_key[2],current_key[3],current_key[4]\
					,current_key[5],current_key[6],current_key[7],current_key[8],current_key[9],current_key[10],current_key[11],current_key[12]);
				generator_mode = -1;
				return 1;
			}
			break;
		default:
			return 0;
			break;
	}
	return 0;
}

void end_generator(void){
	int i;
	
	generator_mode = start_value = end_value = key_length = -1;
	
	for (i=0; i<13; i++){
		current_key[i] = end_key[i] = start_key[i] = 0;
	}
}


// --------------- Old functions --------------------
int initialize_generator_begin(int mode_keygen, ...){
	int i;
	va_list args;
	va_start(args, mode_keygen);
	mode = mode_keygen;
	for (i=0;i<13;i++) temp_key[i] = 0;
	switch (mode){
		case 64:
			for (i=0;i<5;i++){
				temp_key[i] = va_arg(args, int);
			}
			break;
		case 128:
			for (i=0;i<13;i++){
				temp_key[i] = va_arg(args, int);
			}
			break;
		default:
			mode = 0;
			va_end(args);
			return -1;
		}
	va_end(args);
	return 0;
}


int initialize_generator_end(int mode_keygen, ...){
	int i;
	va_list args;
	va_start(args, mode_keygen);
	mode = mode_keygen;
	switch (mode){
		case 64:
			for (i=0;i<5;i++){
				bitkeys[i] = va_arg(args, int);
			}
			break;
		case 128:
			for (i=0;i<13;i++){
				bitkeys[i] = va_arg(args, int);
			}
			break;
		default:
			mode = 0;
			va_end(args);
			return -1;
		}	
	va_end(args);
	return 0;
}


int generate_key_old(unsigned char * key){
	int i;
	
	switch (mode){
		case 64:
			if (!((temp_key[0] == bitkeys[0]) && (temp_key[1] == bitkeys[1]) && (temp_key[2] == bitkeys[2]) && (temp_key[3] == bitkeys[3]) && (temp_key[4] == bitkeys[4]+1))){
				i=4;
				while (temp_key[i] > 255){
					temp_key[i]=0;
					temp_key[i-1] = temp_key[i-1] + 1;
					i = i - 1;
				}
				sprintf(key,"%c%c%c%c%c",temp_key[0],temp_key[1],temp_key[2],temp_key[3],temp_key[4]);
				temp_key[4] = temp_key[4] + 1;
				return 1;
			}
			break;
		case 128:
			if (!((temp_key[0] == bitkeys[0]) && (temp_key[1] == bitkeys[1]) && (temp_key[2] == bitkeys[2]) && (temp_key[3] == bitkeys[3]) && (temp_key[4] == bitkeys[4]) &&\
				(temp_key[5] == bitkeys[5]) && (temp_key[6] == bitkeys[6]) && (temp_key[7] == bitkeys[7]) && (temp_key[8] == bitkeys[8]) && (temp_key[9] == bitkeys[9]) &&\
				(temp_key[10] == bitkeys[10]) && (temp_key[11] == bitkeys[11]) && (temp_key[12] == bitkeys[12]+1))){
				i=12;
				while (temp_key[i] > 255){
					temp_key[i]=0;
					temp_key[i-1] = temp_key[i-1] + 1;
					i = i - 1;
				}
				sprintf(key,"%c%c%c%c%c%c%c%c%c%c%c%c%c",temp_key[0],temp_key[1],temp_key[2],temp_key[3],temp_key[4]\
					,temp_key[5],temp_key[6],temp_key[7],temp_key[8],temp_key[9],temp_key[10],temp_key[11],temp_key[12]);
				temp_key[12] = temp_key[12] + 1;
				return 1;
			}
			break;
		default:
			return 0;
			break;
	}
	return 0;
}
