/*
 * Copyright (c) 2002, 2017 Jens Keiner, Stefan Kunis, Daniel Potts
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

#define NFFT_PRECISION_DOUBLE

#include "nfft3mp.h"

static void simple_test_nfft_1d(void)
{
  NFFT(plan) p;

  int N = 14;
  int M = 19;

  const char *error_str;

  /** init an one dimensional plan */
  NFFT(init_1d)(&p, N, M);

  /** init pseudo random nodes */
  NFFT(vrand_shifted_unit_double)(p.x, p.M_total);
 
  /** precompute psi, the entries of the matrix B */
  if (p.flags & PRE_ONE_PSI)
      NFFT(precompute_one_psi)(&p);

  /** init pseudo random Fourier coefficients and show them */
  NFFT(vrand_unit_complex)(p.f_hat,p.N_total);
  NFFT(vpr_complex)(p.f_hat, p.N_total, "given Fourier coefficients, vector f_hat");

  /** check for valid parameters before calling any trafo/adjoint method */
  error_str = NFFT(check)(&p);
  if (error_str != 0)
  {
    printf("Error in nfft module: %s\n", error_str);
    return;
  }

  /** direct trafo and show the result */
  NFFT(trafo_direct)(&p);
  NFFT(vpr_complex)(p.f,p.M_total,"ndft, vector f");

  /** approx. trafo and show the result */
  NFFT(trafo)(&p);
  NFFT(vpr_complex)(p.f,p.M_total,"nfft, vector f");

  /** approx. adjoint and show the result */
  NFFT(adjoint_direct)(&p);
  NFFT(vpr_complex)(p.f_hat,p.N_total,"adjoint ndft, vector f_hat");

  /** approx. adjoint and show the result */
  NFFT(adjoint)(&p);
  NFFT(vpr_complex)(p.f_hat,p.N_total,"adjoint nfft, vector f_hat");

  /** finalise the one dimensional plan */
  NFFT(finalize)(&p);
}

static void simple_test_nfft_2d(void)
{
  int K, N[2], n[2], M;
  NFFT_R t0, t1;

  NFFT(plan) p;

  const char *error_str;

  N[0] = 32; n[0] = 64;
  N[1] = 14; n[1] = 32;
  M = N[0] * N[1];
  K = 16;

  t0 = NFFT(clock_gettime_seconds)();
  /** init a two dimensional plan */
  NFFT(init_guru)(&p, 2, N, M, n, 7,
     PRE_PHI_HUT| PRE_FULL_PSI| MALLOC_F_HAT| MALLOC_X| MALLOC_F |
     FFTW_INIT| FFT_OUT_OF_PLACE,
     FFTW_ESTIMATE| FFTW_DESTROY_INPUT);

  /** init pseudo random nodes */
  NFFT(vrand_shifted_unit_double)(p.x, p.d * p.M_total);

  /** precompute psi, the entries of the matrix B */
  if(p.flags & PRE_ONE_PSI)
    NFFT(precompute_one_psi)(&p);

  /** init pseudo random Fourier coefficients and show them */
  NFFT(vrand_unit_complex)(p.f_hat, p.N_total);

  t1 = NFFT(clock_gettime_seconds)();
  NFFT(vpr_complex)(p.f_hat,K, "given Fourier coefficients, vector f_hat (first few entries)");
  printf(" ... initialisation took %.2" NFFT__FES__ " seconds.\n",t1-t0);

  /** check for valid parameters before calling any trafo/adjoint method */
  error_str = NFFT(check)(&p);
  if (error_str != 0)
  {
    printf("Error in nfft module: %s\n", error_str);
    return;
  }

  /** direct trafo and show the result */
  t0 = NFFT(clock_gettime_seconds)();
  NFFT(trafo_direct)(&p);
  t1 = NFFT(clock_gettime_seconds)();
  NFFT(vpr_complex)(p.f, K, "ndft, vector f (first few entries)");
  printf(" took %.2" NFFT__FES__ " seconds.\n",t1-t0);

  /** approx. trafo and show the result */
  t0 = NFFT(clock_gettime_seconds)();
  NFFT(trafo)(&p);
  t1 = NFFT(clock_gettime_seconds)();
  NFFT(vpr_complex)(p.f, K, "nfft, vector f (first few entries)");
  printf(" took %.2" NFFT__FES__ " seconds.\n",t1-t0);

  /** direct adjoint and show the result */
  t0 = NFFT(clock_gettime_seconds)();
  NFFT(adjoint_direct)(&p);
  t1 = NFFT(clock_gettime_seconds)();
  NFFT(vpr_complex)(p.f_hat, K, "adjoint ndft, vector f_hat (first few entries)");
  printf(" took %.2" NFFT__FES__ " seconds.\n",t1-t0);

  /** approx. adjoint and show the result */
  t0 = NFFT(clock_gettime_seconds)();
  NFFT(adjoint)(&p);
  t1 = NFFT(clock_gettime_seconds)();
  NFFT(vpr_complex)(p.f_hat, K, "adjoint nfft, vector f_hat (first few entries)");
  printf(" took %.2" NFFT__FES__ " seconds.\n",t1-t0);

  /** finalise the two dimensional plan */
  NFFT(finalize)(&p);
}

int main(void)
{
  printf("1) computing a one dimensional ndft, nfft and an adjoint nfft\n\n");
  simple_test_nfft_1d();

  getc(stdin);

  printf("2) computing a two dimensional ndft, nfft and an adjoint nfft\n\n");
  simple_test_nfft_2d();

  return EXIT_SUCCESS;
}
