package j

import (
	. "github.com/alecthomas/chroma" // nolint
	"github.com/alecthomas/chroma/lexers/internal"
)

// Julia lexer.
var Julia = internal.Register(MustNewLazyLexer(
	&Config{
		Name:      "Julia",
		Aliases:   []string{"julia", "jl"},
		Filenames: []string{"*.jl"},
		MimeTypes: []string{"text/x-julia", "application/x-julia"},
	},
	juliaRules,
))

func juliaRules() Rules {
	return Rules{
		"root": {
			{`\n`, Text, nil},
			{`[^\S\n]+`, Text, nil},
			{`#=`, CommentMultiline, Push("blockcomment")},
			{`#.*$`, Comment, nil},
			{`[\[\](),;]`, Punctuation, nil},
			{`((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))(\s*)(:)((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))`, ByGroups(Name, Text, Operator, Name), nil},
			{`(?<![\]):<>\d.])(:(?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))`, LiteralStringSymbol, nil},
			{`(?<=::)(\s*)((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))\b(?![(\[])`, ByGroups(Text, KeywordType), nil},
			{`((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))(\s*)([<>]:)(\s*)((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))\b(?![(\[])`, ByGroups(KeywordType, Text, Operator, Text, KeywordType), nil},
			{`([<>]:)(\s*)((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))\b(?![(\[])`, ByGroups(Operator, Text, KeywordType), nil},
			{`\b((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))(\s*)([<>]:)`, ByGroups(KeywordType, Text, Operator), nil},
			{Words(``, `[²³¹ʰʲʳʷʸˡˢˣᴬᴮᴰᴱᴳᴴᴵᴶᴷᴸᴹᴺᴼᴾᴿᵀᵁᵂᵃᵇᵈᵉᵍᵏᵐᵒᵖᵗᵘᵛᵝᵞᵟᵠᵡᵢᵣᵤᵥᵦᵧᵨᵩᵪᶜᶠᶥᶦᶫᶰᶸᶻᶿ′″‴‵‶‷⁗⁰ⁱ⁴⁵⁶⁷⁸⁹⁺⁻⁼⁽⁾ⁿ₀₁₂₃₄₅₆₇₈₉₊₋₌₍₎ₐₑₒₓₕₖₗₘₙₚₛₜⱼⱽ]*`, `->`, `:=`, `$=`, `?`, `||`, `&&`, `:`, `$`, `::`, `=`, `+=`, `-=`, `*=`, `/=`, `//=`, `\=`, `^=`, `÷=`, `%=`, `<<=`, `>>=`, `>>>=`, `|=`, `&=`, `⊻=`, `≔`, `⩴`, `≕'`, `~`, `=>`, `→`, `↔`, `↚`, `↛`, `↞`, `↠`, `↢`, `↣`, `↦`, `↤`, `↮`, `⇎`, `⇍`, `⇏`, `⇐`, `⇒`, `⇔`, `⇴`, `⇶`, `⇷`, `⇸`, `⇹`, `⇺`, `⇻`, `⇼`, `⇽`, `⇾`, `⇿`, `⟵`, `⟶`, `⟷`, `⟹`, `⟺`, `⟻`, `⟼`, `⟽`, `⟾`, `⟿`, `⤀`, `⤁`, `⤂`, `⤃`, `⤄`, `⤅`, `⤆`, `⤇`, `⤌`, `⤍`, `⤎`, `⤏`, `⤐`, `⤑`, `⤔`, `⤕`, `⤖`, `⤗`, `⤘`, `⤝`, `⤞`, `⤟`, `⤠`, `⥄`, `⥅`, `⥆`, `⥇`, `⥈`, `⥊`, `⥋`, `⥎`, `⥐`, `⥒`, `⥓`, `⥖`, `⥗`, `⥚`, `⥛`, `⥞`, `⥟`, `⥢`, `⥤`, `⥦`, `⥧`, `⥨`, `⥩`, `⥪`, `⥫`, `⥬`, `⥭`, `⥰`, `⧴`, `⬱`, `⬰`, `⬲`, `⬳`, `⬴`, `⬵`, `⬶`, `⬷`, `⬸`, `⬹`, `⬺`, `⬻`, `⬼`, `⬽`, `⬾`, `⬿`, `⭀`, `⭁`, `⭂`, `⭃`, `⭄`, `⭇`, `⭈`, `⭉`, `⭊`, `⭋`, `⭌`, `￩`, `￫`, `⇜`, `⇝`, `↜`, `↝`, `↩`, `↪`, `↫`, `↬`, `↼`, `↽`, `⇀`, `⇁`, `⇄`, `⇆`, `⇇`, `⇉`, `⇋`, `⇌`, `⇚`, `⇛`, `⇠`, `⇢`, `↷`, `↶`, `↺`, `↻`, `-->`, `<--`, `<-->`, `>`, `<`, `>=`, `≥`, `<=`, `≤`, `==`, `===`, `≡`, `!=`, `≠`, `!==`, `≢`, `∈`, `∉`, `∋`, `∌`, `⊆`, `⊈`, `⊂`, `⊄`, `⊊`, `∝`, `∊`, `∍`, `∥`, `∦`, `∷`, `∺`, `∻`, `∽`, `∾`, `≁`, `≃`, `≂`, `≄`, `≅`, `≆`, `≇`, `≈`, `≉`, `≊`, `≋`, `≌`, `≍`, `≎`, `≐`, `≑`, `≒`, `≓`, `≖`, `≗`, `≘`, `≙`, `≚`, `≛`, `≜`, `≝`, `≞`, `≟`, `≣`, `≦`, `≧`, `≨`, `≩`, `≪`, `≫`, `≬`, `≭`, `≮`, `≯`, `≰`, `≱`, `≲`, `≳`, `≴`, `≵`, `≶`, `≷`, `≸`, `≹`, `≺`, `≻`, `≼`, `≽`, `≾`, `≿`, `⊀`, `⊁`, `⊃`, `⊅`, `⊇`, `⊉`, `⊋`, `⊏`, `⊐`, `⊑`, `⊒`, `⊜`, `⊩`, `⊬`, `⊮`, `⊰`, `⊱`, `⊲`, `⊳`, `⊴`, `⊵`, `⊶`, `⊷`, `⋍`, `⋐`, `⋑`, `⋕`, `⋖`, `⋗`, `⋘`, `⋙`, `⋚`, `⋛`, `⋜`, `⋝`, `⋞`, `⋟`, `⋠`, `⋡`, `⋢`, `⋣`, `⋤`, `⋥`, `⋦`, `⋧`, `⋨`, `⋩`, `⋪`, `⋫`, `⋬`, `⋭`, `⋲`, `⋳`, `⋴`, `⋵`, `⋶`, `⋷`, `⋸`, `⋹`, `⋺`, `⋻`, `⋼`, `⋽`, `⋾`, `⋿`, `⟈`, `⟉`, `⟒`, `⦷`, `⧀`, `⧁`, `⧡`, `⧣`, `⧤`, `⧥`, `⩦`, `⩧`, `⩪`, `⩫`, `⩬`, `⩭`, `⩮`, `⩯`, `⩰`, `⩱`, `⩲`, `⩳`, `⩵`, `⩶`, `⩷`, `⩸`, `⩹`, `⩺`, `⩻`, `⩼`, `⩽`, `⩾`, `⩿`, `⪀`, `⪁`, `⪂`, `⪃`, `⪄`, `⪅`, `⪆`, `⪇`, `⪈`, `⪉`, `⪊`, `⪋`, `⪌`, `⪍`, `⪎`, `⪏`, `⪐`, `⪑`, `⪒`, `⪓`, `⪔`, `⪕`, `⪖`, `⪗`, `⪘`, `⪙`, `⪚`, `⪛`, `⪜`, `⪝`, `⪞`, `⪟`, `⪠`, `⪡`, `⪢`, `⪣`, `⪤`, `⪥`, `⪦`, `⪧`, `⪨`, `⪩`, `⪪`, `⪫`, `⪬`, `⪭`, `⪮`, `⪯`, `⪰`, `⪱`, `⪲`, `⪳`, `⪴`, `⪵`, `⪶`, `⪷`, `⪸`, `⪹`, `⪺`, `⪻`, `⪼`, `⪽`, `⪾`, `⪿`, `⫀`, `⫁`, `⫂`, `⫃`, `⫄`, `⫅`, `⫆`, `⫇`, `⫈`, `⫉`, `⫊`, `⫋`, `⫌`, `⫍`, `⫎`, `⫏`, `⫐`, `⫑`, `⫒`, `⫓`, `⫔`, `⫕`, `⫖`, `⫗`, `⫘`, `⫙`, `⫷`, `⫸`, `⫹`, `⫺`, `⊢`, `⊣`, `⟂`, `<:`, `>:`, `<|`, `|>`, `…`, `⁝`, `⋮`, `⋱`, `⋰`, `⋯`, `+`, `-`, `¦`, `|`, `⊕`, `⊖`, `⊞`, `⊟`, `++`, `∪`, `∨`, `⊔`, `±`, `∓`, `∔`, `∸`, `≏`, `⊎`, `⊻`, `⊽`, `⋎`, `⋓`, `⧺`, `⧻`, `⨈`, `⨢`, `⨣`, `⨤`, `⨥`, `⨦`, `⨧`, `⨨`, `⨩`, `⨪`, `⨫`, `⨬`, `⨭`, `⨮`, `⨹`, `⨺`, `⩁`, `⩂`, `⩅`, `⩊`, `⩌`, `⩏`, `⩐`, `⩒`, `⩔`, `⩖`, `⩗`, `⩛`, `⩝`, `⩡`, `⩢`, `⩣`, `*`, `/`, `⌿`, `÷`, `%`, `&`, `⋅`, `∘`, `×`, `\`, `∩`, `∧`, `⊗`, `⊘`, `⊙`, `⊚`, `⊛`, `⊠`, `⊡`, `⊓`, `∗`, `∙`, `∤`, `⅋`, `≀`, `⊼`, `⋄`, `⋆`, `⋇`, `⋉`, `⋊`, `⋋`, `⋌`, `⋏`, `⋒`, `⟑`, `⦸`, `⦼`, `⦾`, `⦿`, `⧶`, `⧷`, `⨇`, `⨰`, `⨱`, `⨲`, `⨳`, `⨴`, `⨵`, `⨶`, `⨷`, `⨸`, `⨻`, `⨼`, `⨽`, `⩀`, `⩃`, `⩄`, `⩋`, `⩍`, `⩎`, `⩑`, `⩓`, `⩕`, `⩘`, `⩚`, `⩜`, `⩞`, `⩟`, `⩠`, `⫛`, `⊍`, `▷`, `⨝`, `⟕`, `⟖`, `⟗`, `⨟`, `//`, `>>`, `<<`, `>>>`, `^`, `↑`, `↓`, `⇵`, `⟰`, `⟱`, `⤈`, `⤉`, `⤊`, `⤋`, `⤒`, `⤓`, `⥉`, `⥌`, `⥍`, `⥏`, `⥑`, `⥔`, `⥕`, `⥘`, `⥙`, `⥜`, `⥝`, `⥠`, `⥡`, `⥣`, `⥥`, `⥮`, `⥯`, `￪`, `￬`, `!`, `¬`, `√`, `∛`, `∜`), Operator, nil},
			{Words(``, `[²³¹ʰʲʳʷʸˡˢˣᴬᴮᴰᴱᴳᴴᴵᴶᴷᴸᴹᴺᴼᴾᴿᵀᵁᵂᵃᵇᵈᵉᵍᵏᵐᵒᵖᵗᵘᵛᵝᵞᵟᵠᵡᵢᵣᵤᵥᵦᵧᵨᵩᵪᶜᶠᶥᶦᶫᶰᶸᶻᶿ′″‴‵‶‷⁗⁰ⁱ⁴⁵⁶⁷⁸⁹⁺⁻⁼⁽⁾ⁿ₀₁₂₃₄₅₆₇₈₉₊₋₌₍₎ₐₑₒₓₕₖₗₘₙₚₛₜⱼⱽ]*`, `.=`, `.+=`, `.-=`, `.*=`, `./=`, `.//=`, `.\=`, `.^=`, `.÷=`, `.%=`, `.<<=`, `.>>=`, `.>>>=`, `.|=`, `.&=`, `.⊻=`, `.≔`, `.⩴`, `.≕'`, `.~`, `.=>`, `.→`, `.↔`, `.↚`, `.↛`, `.↞`, `.↠`, `.↢`, `.↣`, `.↦`, `.↤`, `.↮`, `.⇎`, `.⇍`, `.⇏`, `.⇐`, `.⇒`, `.⇔`, `.⇴`, `.⇶`, `.⇷`, `.⇸`, `.⇹`, `.⇺`, `.⇻`, `.⇼`, `.⇽`, `.⇾`, `.⇿`, `.⟵`, `.⟶`, `.⟷`, `.⟹`, `.⟺`, `.⟻`, `.⟼`, `.⟽`, `.⟾`, `.⟿`, `.⤀`, `.⤁`, `.⤂`, `.⤃`, `.⤄`, `.⤅`, `.⤆`, `.⤇`, `.⤌`, `.⤍`, `.⤎`, `.⤏`, `.⤐`, `.⤑`, `.⤔`, `.⤕`, `.⤖`, `.⤗`, `.⤘`, `.⤝`, `.⤞`, `.⤟`, `.⤠`, `.⥄`, `.⥅`, `.⥆`, `.⥇`, `.⥈`, `.⥊`, `.⥋`, `.⥎`, `.⥐`, `.⥒`, `.⥓`, `.⥖`, `.⥗`, `.⥚`, `.⥛`, `.⥞`, `.⥟`, `.⥢`, `.⥤`, `.⥦`, `.⥧`, `.⥨`, `.⥩`, `.⥪`, `.⥫`, `.⥬`, `.⥭`, `.⥰`, `.⧴`, `.⬱`, `.⬰`, `.⬲`, `.⬳`, `.⬴`, `.⬵`, `.⬶`, `.⬷`, `.⬸`, `.⬹`, `.⬺`, `.⬻`, `.⬼`, `.⬽`, `.⬾`, `.⬿`, `.⭀`, `.⭁`, `.⭂`, `.⭃`, `.⭄`, `.⭇`, `.⭈`, `.⭉`, `.⭊`, `.⭋`, `.⭌`, `.￩`, `.￫`, `.⇜`, `.⇝`, `.↜`, `.↝`, `.↩`, `.↪`, `.↫`, `.↬`, `.↼`, `.↽`, `.⇀`, `.⇁`, `.⇄`, `.⇆`, `.⇇`, `.⇉`, `.⇋`, `.⇌`, `.⇚`, `.⇛`, `.⇠`, `.⇢`, `.↷`, `.↶`, `.↺`, `.↻`, `.-->`, `.<--`, `.<-->`, `.>`, `.<`, `.>=`, `.≥`, `.<=`, `.≤`, `.==`, `.===`, `.≡`, `.!=`, `.≠`, `.!==`, `.≢`, `.∈`, `.∉`, `.∋`, `.∌`, `.⊆`, `.⊈`, `.⊂`, `.⊄`, `.⊊`, `.∝`, `.∊`, `.∍`, `.∥`, `.∦`, `.∷`, `.∺`, `.∻`, `.∽`, `.∾`, `.≁`, `.≃`, `.≂`, `.≄`, `.≅`, `.≆`, `.≇`, `.≈`, `.≉`, `.≊`, `.≋`, `.≌`, `.≍`, `.≎`, `.≐`, `.≑`, `.≒`, `.≓`, `.≖`, `.≗`, `.≘`, `.≙`, `.≚`, `.≛`, `.≜`, `.≝`, `.≞`, `.≟`, `.≣`, `.≦`, `.≧`, `.≨`, `.≩`, `.≪`, `.≫`, `.≬`, `.≭`, `.≮`, `.≯`, `.≰`, `.≱`, `.≲`, `.≳`, `.≴`, `.≵`, `.≶`, `.≷`, `.≸`, `.≹`, `.≺`, `.≻`, `.≼`, `.≽`, `.≾`, `.≿`, `.⊀`, `.⊁`, `.⊃`, `.⊅`, `.⊇`, `.⊉`, `.⊋`, `.⊏`, `.⊐`, `.⊑`, `.⊒`, `.⊜`, `.⊩`, `.⊬`, `.⊮`, `.⊰`, `.⊱`, `.⊲`, `.⊳`, `.⊴`, `.⊵`, `.⊶`, `.⊷`, `.⋍`, `.⋐`, `.⋑`, `.⋕`, `.⋖`, `.⋗`, `.⋘`, `.⋙`, `.⋚`, `.⋛`, `.⋜`, `.⋝`, `.⋞`, `.⋟`, `.⋠`, `.⋡`, `.⋢`, `.⋣`, `.⋤`, `.⋥`, `.⋦`, `.⋧`, `.⋨`, `.⋩`, `.⋪`, `.⋫`, `.⋬`, `.⋭`, `.⋲`, `.⋳`, `.⋴`, `.⋵`, `.⋶`, `.⋷`, `.⋸`, `.⋹`, `.⋺`, `.⋻`, `.⋼`, `.⋽`, `.⋾`, `.⋿`, `.⟈`, `.⟉`, `.⟒`, `.⦷`, `.⧀`, `.⧁`, `.⧡`, `.⧣`, `.⧤`, `.⧥`, `.⩦`, `.⩧`, `.⩪`, `.⩫`, `.⩬`, `.⩭`, `.⩮`, `.⩯`, `.⩰`, `.⩱`, `.⩲`, `.⩳`, `.⩵`, `.⩶`, `.⩷`, `.⩸`, `.⩹`, `.⩺`, `.⩻`, `.⩼`, `.⩽`, `.⩾`, `.⩿`, `.⪀`, `.⪁`, `.⪂`, `.⪃`, `.⪄`, `.⪅`, `.⪆`, `.⪇`, `.⪈`, `.⪉`, `.⪊`, `.⪋`, `.⪌`, `.⪍`, `.⪎`, `.⪏`, `.⪐`, `.⪑`, `.⪒`, `.⪓`, `.⪔`, `.⪕`, `.⪖`, `.⪗`, `.⪘`, `.⪙`, `.⪚`, `.⪛`, `.⪜`, `.⪝`, `.⪞`, `.⪟`, `.⪠`, `.⪡`, `.⪢`, `.⪣`, `.⪤`, `.⪥`, `.⪦`, `.⪧`, `.⪨`, `.⪩`, `.⪪`, `.⪫`, `.⪬`, `.⪭`, `.⪮`, `.⪯`, `.⪰`, `.⪱`, `.⪲`, `.⪳`, `.⪴`, `.⪵`, `.⪶`, `.⪷`, `.⪸`, `.⪹`, `.⪺`, `.⪻`, `.⪼`, `.⪽`, `.⪾`, `.⪿`, `.⫀`, `.⫁`, `.⫂`, `.⫃`, `.⫄`, `.⫅`, `.⫆`, `.⫇`, `.⫈`, `.⫉`, `.⫊`, `.⫋`, `.⫌`, `.⫍`, `.⫎`, `.⫏`, `.⫐`, `.⫑`, `.⫒`, `.⫓`, `.⫔`, `.⫕`, `.⫖`, `.⫗`, `.⫘`, `.⫙`, `.⫷`, `.⫸`, `.⫹`, `.⫺`, `.⊢`, `.⊣`, `.⟂`, `.<:`, `.>:`, `.<|`, `.|>`, `.…`, `.⁝`, `.⋮`, `.⋱`, `.⋰`, `.⋯`, `.+`, `.-`, `.¦`, `.|`, `.⊕`, `.⊖`, `.⊞`, `.⊟`, `.++`, `.∪`, `.∨`, `.⊔`, `.±`, `.∓`, `.∔`, `.∸`, `.≏`, `.⊎`, `.⊻`, `.⊽`, `.⋎`, `.⋓`, `.⧺`, `.⧻`, `.⨈`, `.⨢`, `.⨣`, `.⨤`, `.⨥`, `.⨦`, `.⨧`, `.⨨`, `.⨩`, `.⨪`, `.⨫`, `.⨬`, `.⨭`, `.⨮`, `.⨹`, `.⨺`, `.⩁`, `.⩂`, `.⩅`, `.⩊`, `.⩌`, `.⩏`, `.⩐`, `.⩒`, `.⩔`, `.⩖`, `.⩗`, `.⩛`, `.⩝`, `.⩡`, `.⩢`, `.⩣`, `.*`, `./`, `.⌿`, `.÷`, `.%`, `.&`, `.⋅`, `.∘`, `.×`, `.\`, `.∩`, `.∧`, `.⊗`, `.⊘`, `.⊙`, `.⊚`, `.⊛`, `.⊠`, `.⊡`, `.⊓`, `.∗`, `.∙`, `.∤`, `.⅋`, `.≀`, `.⊼`, `.⋄`, `.⋆`, `.⋇`, `.⋉`, `.⋊`, `.⋋`, `.⋌`, `.⋏`, `.⋒`, `.⟑`, `.⦸`, `.⦼`, `.⦾`, `.⦿`, `.⧶`, `.⧷`, `.⨇`, `.⨰`, `.⨱`, `.⨲`, `.⨳`, `.⨴`, `.⨵`, `.⨶`, `.⨷`, `.⨸`, `.⨻`, `.⨼`, `.⨽`, `.⩀`, `.⩃`, `.⩄`, `.⩋`, `.⩍`, `.⩎`, `.⩑`, `.⩓`, `.⩕`, `.⩘`, `.⩚`, `.⩜`, `.⩞`, `.⩟`, `.⩠`, `.⫛`, `.⊍`, `.▷`, `.⨝`, `.⟕`, `.⟖`, `.⟗`, `.⨟`, `.//`, `.>>`, `.<<`, `.>>>`, `.^`, `.↑`, `.↓`, `.⇵`, `.⟰`, `.⟱`, `.⤈`, `.⤉`, `.⤊`, `.⤋`, `.⤒`, `.⤓`, `.⥉`, `.⥌`, `.⥍`, `.⥏`, `.⥑`, `.⥔`, `.⥕`, `.⥘`, `.⥙`, `.⥜`, `.⥝`, `.⥠`, `.⥡`, `.⥣`, `.⥥`, `.⥮`, `.⥯`, `.￪`, `.￬`, `.!`, `.¬`, `.√`, `.∛`, `.∜`), Operator, nil},
			{Words(``, ``, `...`, `..`), Operator, nil},
			{`'(\\.|\\[0-7]{1,3}|\\x[a-fA-F0-9]{1,3}|\\u[a-fA-F0-9]{1,4}|\\U[a-fA-F0-9]{1,6}|[^\\\'\n])'`, LiteralStringChar, nil},
			{`(?<=[.\w)\]])(\'[²³¹ʰʲʳʷʸˡˢˣᴬᴮᴰᴱᴳᴴᴵᴶᴷᴸᴹᴺᴼᴾᴿᵀᵁᵂᵃᵇᵈᵉᵍᵏᵐᵒᵖᵗᵘᵛᵝᵞᵟᵠᵡᵢᵣᵤᵥᵦᵧᵨᵩᵪᶜᶠᶥᶦᶫᶰᶸᶻᶿ′″‴‵‶‷⁗⁰ⁱ⁴⁵⁶⁷⁸⁹⁺⁻⁼⁽⁾ⁿ₀₁₂₃₄₅₆₇₈₉₊₋₌₍₎ₐₑₒₓₕₖₗₘₙₚₛₜⱼⱽ]*)+`, Operator, nil},
			{`(raw)(""")`, ByGroups(LiteralStringAffix, LiteralString), Push("tqrawstring")},
			{`(raw)(")`, ByGroups(LiteralStringAffix, LiteralString), Push("rawstring")},
			{`(r)(""")`, ByGroups(LiteralStringAffix, LiteralStringRegex), Push("tqregex")},
			{`(r)(")`, ByGroups(LiteralStringAffix, LiteralStringRegex), Push("regex")},
			{`((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))?(""")`, ByGroups(LiteralStringAffix, LiteralString), Push("tqstring")},
			{`((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))?(")`, ByGroups(LiteralStringAffix, LiteralString), Push("string")},
			{"((?:[a-zA-Z_\u00a1-\U0010ffff][a-zA-Z_0-9!\u00a1-\U0010ffff]*))?(```)", ByGroups(LiteralStringAffix, LiteralStringBacktick), Push("tqcommand")},
			{"((?:[a-zA-Z_\u00a1-\U0010ffff][a-zA-Z_0-9!\u00a1-\U0010ffff]*))?(`)", ByGroups(LiteralStringAffix, LiteralStringBacktick), Push("command")},
			{`((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))(\{)`, ByGroups(KeywordType, Punctuation), Push("curly")},
			{`(where)(\s+)((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))`, ByGroups(Keyword, Text, KeywordType), nil},
			{`(\{)`, Punctuation, Push("curly")},
			{`(abstract[ \t]+type|primitive[ \t]+type|mutable[ \t]+struct|struct)([\s()]+)((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*))`, ByGroups(Keyword, Text, KeywordType), nil},
			{`@(?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*)`, NameDecorator, nil},
			{Words(`@`, `[²³¹ʰʲʳʷʸˡˢˣᴬᴮᴰᴱᴳᴴᴵᴶᴷᴸᴹᴺᴼᴾᴿᵀᵁᵂᵃᵇᵈᵉᵍᵏᵐᵒᵖᵗᵘᵛᵝᵞᵟᵠᵡᵢᵣᵤᵥᵦᵧᵨᵩᵪᶜᶠᶥᶦᶫᶰᶸᶻᶿ′″‴‵‶‷⁗⁰ⁱ⁴⁵⁶⁷⁸⁹⁺⁻⁼⁽⁾ⁿ₀₁₂₃₄₅₆₇₈₉₊₋₌₍₎ₐₑₒₓₕₖₗₘₙₚₛₜⱼⱽ]*`, `->`, `:=`, `$=`, `?`, `||`, `&&`, `:`, `$`, `::`, `..`, `.`, `=`, `+=`, `-=`, `*=`, `/=`, `//=`, `\=`, `^=`, `÷=`, `%=`, `<<=`, `>>=`, `>>>=`, `|=`, `&=`, `⊻=`, `≔`, `⩴`, `≕'`, `~`, `=>`, `→`, `↔`, `↚`, `↛`, `↞`, `↠`, `↢`, `↣`, `↦`, `↤`, `↮`, `⇎`, `⇍`, `⇏`, `⇐`, `⇒`, `⇔`, `⇴`, `⇶`, `⇷`, `⇸`, `⇹`, `⇺`, `⇻`, `⇼`, `⇽`, `⇾`, `⇿`, `⟵`, `⟶`, `⟷`, `⟹`, `⟺`, `⟻`, `⟼`, `⟽`, `⟾`, `⟿`, `⤀`, `⤁`, `⤂`, `⤃`, `⤄`, `⤅`, `⤆`, `⤇`, `⤌`, `⤍`, `⤎`, `⤏`, `⤐`, `⤑`, `⤔`, `⤕`, `⤖`, `⤗`, `⤘`, `⤝`, `⤞`, `⤟`, `⤠`, `⥄`, `⥅`, `⥆`, `⥇`, `⥈`, `⥊`, `⥋`, `⥎`, `⥐`, `⥒`, `⥓`, `⥖`, `⥗`, `⥚`, `⥛`, `⥞`, `⥟`, `⥢`, `⥤`, `⥦`, `⥧`, `⥨`, `⥩`, `⥪`, `⥫`, `⥬`, `⥭`, `⥰`, `⧴`, `⬱`, `⬰`, `⬲`, `⬳`, `⬴`, `⬵`, `⬶`, `⬷`, `⬸`, `⬹`, `⬺`, `⬻`, `⬼`, `⬽`, `⬾`, `⬿`, `⭀`, `⭁`, `⭂`, `⭃`, `⭄`, `⭇`, `⭈`, `⭉`, `⭊`, `⭋`, `⭌`, `￩`, `￫`, `⇜`, `⇝`, `↜`, `↝`, `↩`, `↪`, `↫`, `↬`, `↼`, `↽`, `⇀`, `⇁`, `⇄`, `⇆`, `⇇`, `⇉`, `⇋`, `⇌`, `⇚`, `⇛`, `⇠`, `⇢`, `↷`, `↶`, `↺`, `↻`, `-->`, `<--`, `<-->`, `>`, `<`, `>=`, `≥`, `<=`, `≤`, `==`, `===`, `≡`, `!=`, `≠`, `!==`, `≢`, `∈`, `∉`, `∋`, `∌`, `⊆`, `⊈`, `⊂`, `⊄`, `⊊`, `∝`, `∊`, `∍`, `∥`, `∦`, `∷`, `∺`, `∻`, `∽`, `∾`, `≁`, `≃`, `≂`, `≄`, `≅`, `≆`, `≇`, `≈`, `≉`, `≊`, `≋`, `≌`, `≍`, `≎`, `≐`, `≑`, `≒`, `≓`, `≖`, `≗`, `≘`, `≙`, `≚`, `≛`, `≜`, `≝`, `≞`, `≟`, `≣`, `≦`, `≧`, `≨`, `≩`, `≪`, `≫`, `≬`, `≭`, `≮`, `≯`, `≰`, `≱`, `≲`, `≳`, `≴`, `≵`, `≶`, `≷`, `≸`, `≹`, `≺`, `≻`, `≼`, `≽`, `≾`, `≿`, `⊀`, `⊁`, `⊃`, `⊅`, `⊇`, `⊉`, `⊋`, `⊏`, `⊐`, `⊑`, `⊒`, `⊜`, `⊩`, `⊬`, `⊮`, `⊰`, `⊱`, `⊲`, `⊳`, `⊴`, `⊵`, `⊶`, `⊷`, `⋍`, `⋐`, `⋑`, `⋕`, `⋖`, `⋗`, `⋘`, `⋙`, `⋚`, `⋛`, `⋜`, `⋝`, `⋞`, `⋟`, `⋠`, `⋡`, `⋢`, `⋣`, `⋤`, `⋥`, `⋦`, `⋧`, `⋨`, `⋩`, `⋪`, `⋫`, `⋬`, `⋭`, `⋲`, `⋳`, `⋴`, `⋵`, `⋶`, `⋷`, `⋸`, `⋹`, `⋺`, `⋻`, `⋼`, `⋽`, `⋾`, `⋿`, `⟈`, `⟉`, `⟒`, `⦷`, `⧀`, `⧁`, `⧡`, `⧣`, `⧤`, `⧥`, `⩦`, `⩧`, `⩪`, `⩫`, `⩬`, `⩭`, `⩮`, `⩯`, `⩰`, `⩱`, `⩲`, `⩳`, `⩵`, `⩶`, `⩷`, `⩸`, `⩹`, `⩺`, `⩻`, `⩼`, `⩽`, `⩾`, `⩿`, `⪀`, `⪁`, `⪂`, `⪃`, `⪄`, `⪅`, `⪆`, `⪇`, `⪈`, `⪉`, `⪊`, `⪋`, `⪌`, `⪍`, `⪎`, `⪏`, `⪐`, `⪑`, `⪒`, `⪓`, `⪔`, `⪕`, `⪖`, `⪗`, `⪘`, `⪙`, `⪚`, `⪛`, `⪜`, `⪝`, `⪞`, `⪟`, `⪠`, `⪡`, `⪢`, `⪣`, `⪤`, `⪥`, `⪦`, `⪧`, `⪨`, `⪩`, `⪪`, `⪫`, `⪬`, `⪭`, `⪮`, `⪯`, `⪰`, `⪱`, `⪲`, `⪳`, `⪴`, `⪵`, `⪶`, `⪷`, `⪸`, `⪹`, `⪺`, `⪻`, `⪼`, `⪽`, `⪾`, `⪿`, `⫀`, `⫁`, `⫂`, `⫃`, `⫄`, `⫅`, `⫆`, `⫇`, `⫈`, `⫉`, `⫊`, `⫋`, `⫌`, `⫍`, `⫎`, `⫏`, `⫐`, `⫑`, `⫒`, `⫓`, `⫔`, `⫕`, `⫖`, `⫗`, `⫘`, `⫙`, `⫷`, `⫸`, `⫹`, `⫺`, `⊢`, `⊣`, `⟂`, `<:`, `>:`, `<|`, `|>`, `…`, `⁝`, `⋮`, `⋱`, `⋰`, `⋯`, `+`, `-`, `¦`, `|`, `⊕`, `⊖`, `⊞`, `⊟`, `++`, `∪`, `∨`, `⊔`, `±`, `∓`, `∔`, `∸`, `≏`, `⊎`, `⊻`, `⊽`, `⋎`, `⋓`, `⧺`, `⧻`, `⨈`, `⨢`, `⨣`, `⨤`, `⨥`, `⨦`, `⨧`, `⨨`, `⨩`, `⨪`, `⨫`, `⨬`, `⨭`, `⨮`, `⨹`, `⨺`, `⩁`, `⩂`, `⩅`, `⩊`, `⩌`, `⩏`, `⩐`, `⩒`, `⩔`, `⩖`, `⩗`, `⩛`, `⩝`, `⩡`, `⩢`, `⩣`, `*`, `/`, `⌿`, `÷`, `%`, `&`, `⋅`, `∘`, `×`, `\`, `∩`, `∧`, `⊗`, `⊘`, `⊙`, `⊚`, `⊛`, `⊠`, `⊡`, `⊓`, `∗`, `∙`, `∤`, `⅋`, `≀`, `⊼`, `⋄`, `⋆`, `⋇`, `⋉`, `⋊`, `⋋`, `⋌`, `⋏`, `⋒`, `⟑`, `⦸`, `⦼`, `⦾`, `⦿`, `⧶`, `⧷`, `⨇`, `⨰`, `⨱`, `⨲`, `⨳`, `⨴`, `⨵`, `⨶`, `⨷`, `⨸`, `⨻`, `⨼`, `⨽`, `⩀`, `⩃`, `⩄`, `⩋`, `⩍`, `⩎`, `⩑`, `⩓`, `⩕`, `⩘`, `⩚`, `⩜`, `⩞`, `⩟`, `⩠`, `⫛`, `⊍`, `▷`, `⨝`, `⟕`, `⟖`, `⟗`, `⨟`, `//`, `>>`, `<<`, `>>>`, `^`, `↑`, `↓`, `⇵`, `⟰`, `⟱`, `⤈`, `⤉`, `⤊`, `⤋`, `⤒`, `⤓`, `⥉`, `⥌`, `⥍`, `⥏`, `⥑`, `⥔`, `⥕`, `⥘`, `⥙`, `⥜`, `⥝`, `⥠`, `⥡`, `⥣`, `⥥`, `⥮`, `⥯`, `￪`, `￬`, `!`, `¬`, `√`, `∛`, `∜`), NameDecorator, nil},
			{Words(``, `\b`, `baremodule`, `begin`, `break`, `catch`, `ccall`, `const`, `continue`, `do`, `else`, `elseif`, `end`, `export`, `finally`, `for`, `function`, `global`, `if`, `import`, `in`, `isa`, `let`, `local`, `macro`, `module`, `quote`, `return`, `try`, `using`, `where`, `while`), Keyword, nil},
			{Words(``, `\b`, `AbstractArray`, `AbstractChannel`, `AbstractChar`, `AbstractDict`, `AbstractDisplay`, `AbstractFloat`, `AbstractIrrational`, `AbstractMatch`, `AbstractMatrix`, `AbstractPattern`, `AbstractRange`, `AbstractSet`, `AbstractString`, `AbstractUnitRange`, `AbstractVecOrMat`, `AbstractVector`, `Any`, `ArgumentError`, `Array`, `AssertionError`, `BigFloat`, `BigInt`, `BitArray`, `BitMatrix`, `BitSet`, `BitVector`, `Bool`, `BoundsError`, `CapturedException`, `CartesianIndex`, `CartesianIndices`, `Cchar`, `Cdouble`, `Cfloat`, `Channel`, `Char`, `Cint`, `Cintmax_t`, `Clong`, `Clonglong`, `Cmd`, `Colon`, `Complex`, `ComplexF16`, `ComplexF32`, `ComplexF64`, `ComposedFunction`, `CompositeException`, `Condition`, `Cptrdiff_t`, `Cshort`, `Csize_t`, `Cssize_t`, `Cstring`, `Cuchar`, `Cuint`, `Cuintmax_t`, `Culong`, `Culonglong`, `Cushort`, `Cvoid`, `Cwchar_t`, `Cwstring`, `DataType`, `DenseArray`, `DenseMatrix`, `DenseVecOrMat`, `DenseVector`, `Dict`, `DimensionMismatch`, `Dims`, `DivideError`, `DomainError`, `EOFError`, `Enum`, `ErrorException`, `Exception`, `ExponentialBackOff`, `Expr`, `Float16`, `Float32`, `Float64`, `Function`, `GlobalRef`, `HTML`, `IO`, `IOBuffer`, `IOContext`, `IOStream`, `IdDict`, `IndexCartesian`, `IndexLinear`, `IndexStyle`, `InexactError`, `InitError`, `Int`, `Int128`, `Int16`, `Int32`, `Int64`, `Int8`, `Integer`, `InterruptException`, `InvalidStateException`, `Irrational`, `KeyError`, `LinRange`, `LineNumberNode`, `LinearIndices`, `LoadError`, `MIME`, `Matrix`, `Method`, `MethodError`, `Missing`, `MissingException`, `Module`, `NTuple`, `NamedTuple`, `Nothing`, `Number`, `OrdinalRange`, `OutOfMemoryError`, `OverflowError`, `Pair`, `PartialQuickSort`, `PermutedDimsArray`, `Pipe`, `ProcessFailedException`, `Ptr`, `QuoteNode`, `Rational`, `RawFD`, `ReadOnlyMemoryError`, `Real`, `ReentrantLock`, `Ref`, `Regex`, `RegexMatch`, `RoundingMode`, `SegmentationFault`, `Set`, `Signed`, `Some`, `StackOverflowError`, `StepRange`, `StepRangeLen`, `StridedArray`, `StridedMatrix`, `StridedVecOrMat`, `StridedVector`, `String`, `StringIndexError`, `SubArray`, `SubString`, `SubstitutionString`, `Symbol`, `SystemError`, `Task`, `TaskFailedException`, `Text`, `TextDisplay`, `Timer`, `Tuple`, `Type`, `TypeError`, `TypeVar`, `UInt`, `UInt128`, `UInt16`, `UInt32`, `UInt64`, `UInt8`, `UndefInitializer`, `UndefKeywordError`, `UndefRefError`, `UndefVarError`, `Union`, `UnionAll`, `UnitRange`, `Unsigned`, `Val`, `Vararg`, `VecElement`, `VecOrMat`, `Vector`, `VersionNumber`, `WeakKeyDict`, `WeakRef`), KeywordType, nil},
			{Words(``, `\b`, `ARGS`, `C_NULL`, `DEPOT_PATH`, `ENDIAN_BOM`, `ENV`, `Inf`, `Inf16`, `Inf32`, `Inf64`, `InsertionSort`, `LOAD_PATH`, `MergeSort`, `NaN`, `NaN16`, `NaN32`, `NaN64`, `PROGRAM_FILE`, `QuickSort`, `RoundDown`, `RoundFromZero`, `RoundNearest`, `RoundNearestTiesAway`, `RoundNearestTiesUp`, `RoundToZero`, `RoundUp`, `VERSION`, `devnull`, `false`, `im`, `missing`, `nothing`, `pi`, `stderr`, `stdin`, `stdout`, `true`, `undef`, `π`, `ℯ`), NameBuiltin, nil},
			{`(?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*)`, Name, nil},
			{`(\d+((_\d+)+)?\.(?!\.)(\d+((_\d+)+)?)?|\.\d+((_\d+)+)?)([eEf][+-]?[0-9]+)?`, LiteralNumberFloat, nil},
			{`\d+((_\d+)+)?[eEf][+-]?[0-9]+`, LiteralNumberFloat, nil},
			{`0x[a-fA-F0-9]+((_[a-fA-F0-9]+)+)?(\.([a-fA-F0-9]+((_[a-fA-F0-9]+)+)?)?)?p[+-]?\d+`, LiteralNumberFloat, nil},
			{`0b[01]+((_[01]+)+)?`, LiteralNumberBin, nil},
			{`0o[0-7]+((_[0-7]+)+)?`, LiteralNumberOct, nil},
			{`0x[a-fA-F0-9]+((_[a-fA-F0-9]+)+)?`, LiteralNumberHex, nil},
			{`\d+((_\d+)+)?`, LiteralNumberInteger, nil},
			{Words(``, ``, `.`), Operator, nil},
		},
		"blockcomment": {
			{`[^=#]`, CommentMultiline, nil},
			{`#=`, CommentMultiline, Push()},
			{`=#`, CommentMultiline, Pop(1)},
			{`[=#]`, CommentMultiline, nil},
		},
		"curly": {
			{`\{`, Punctuation, Push()},
			{`\}`, Punctuation, Pop(1)},
			{`(?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*)`, KeywordType, nil},
			Include("root"),
		},
		"tqrawstring": {
			{`"""`, LiteralString, Pop(1)},
			{`([^"]|"[^"][^"])+`, LiteralString, nil},
		},
		"rawstring": {
			{`"`, LiteralString, Pop(1)},
			{`\\"`, LiteralStringEscape, nil},
			{`([^"\\]|\\[^"])+`, LiteralString, nil},
		},
		"interp": {
			{`\$(?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*)`, LiteralStringInterpol, nil},
			{`(\$)(\()`, ByGroups(LiteralStringInterpol, Punctuation), Push("in-intp")},
		},
		"in-intp": {
			{`\(`, Punctuation, Push()},
			{`\)`, Punctuation, Pop(1)},
			Include("root"),
		},
		"string": {
			{`(")((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*)|\d+)?`, ByGroups(LiteralString, LiteralStringAffix), Pop(1)},
			{`\\([\\"\'$nrbtfav]|(x|u|U)[a-fA-F0-9]+|\d+)`, LiteralStringEscape, nil},
			Include("interp"),
			{`%[-#0 +]*([0-9]+|[*])?(\.([0-9]+|[*]))?[hlL]?[E-GXc-giorsux%]`, LiteralStringInterpol, nil},
			{`[^"$%\\]+`, LiteralString, nil},
			{`.`, LiteralString, nil},
		},
		"tqstring": {
			{`(""")((?:[a-zA-Z_¡-􏿿][a-zA-Z_0-9!¡-􏿿]*)|\d+)?`, ByGroups(LiteralString, LiteralStringAffix), Pop(1)},
			{`\\([\\"\'$nrbtfav]|(x|u|U)[a-fA-F0-9]+|\d+)`, LiteralStringEscape, nil},
			Include("interp"),
			{`[^"$%\\]+`, LiteralString, nil},
			{`.`, LiteralString, nil},
		},
		"regex": {
			{`(")([imsxa]*)?`, ByGroups(LiteralStringRegex, LiteralStringAffix), Pop(1)},
			{`\\"`, LiteralStringRegex, nil},
			{`[^\\"]+`, LiteralStringRegex, nil},
		},
		"tqregex": {
			{`(""")([imsxa]*)?`, ByGroups(LiteralStringRegex, LiteralStringAffix), Pop(1)},
			{`[^"]+`, LiteralStringRegex, nil},
		},
		"command": {
			{"(`)((?:[a-zA-Z_\u00a1-\U0010ffff][a-zA-Z_0-9!\u00a1-\U0010ffff]*)|\\d+)?", ByGroups(LiteralStringBacktick, LiteralStringAffix), Pop(1)},
			{"\\\\[`$]", LiteralStringEscape, nil},
			Include("interp"),
			{"[^\\\\`$]+", LiteralStringBacktick, nil},
			{`.`, LiteralStringBacktick, nil},
		},
		"tqcommand": {
			{"(```)((?:[a-zA-Z_\u00a1-\U0010ffff][a-zA-Z_0-9!\u00a1-\U0010ffff]*)|\\d+)?", ByGroups(LiteralStringBacktick, LiteralStringAffix), Pop(1)},
			{`\\\$`, LiteralStringEscape, nil},
			Include("interp"),
			{"[^\\\\`$]+", LiteralStringBacktick, nil},
			{`.`, LiteralStringBacktick, nil},
		},
	}
}
