# This file is part of Cantera. See License.txt in the top-level directory or
# at https://cantera.org/license.txt for license and copyright information.

from collections.abc import Sequence
from pathlib import Path
from typing import (
    Any,
    Generic,
    Literal,
    TypedDict,
    TypeVar,
    overload,
)

from pandas import DataFrame
from typing_extensions import Never, Unpack, override

from ._types import (
    Array,
    ArrayLike,
    ArrayPureFluidStateSetter,
    ArrayState2Setter,
    ArrayStateSetter,
    Basis,
    CompositionLike,
    CompressionLevel,
    EquilibriumSolver,
    Index,
    LogLevel,
    PropertyPair,
    State2Setter,
    StateDefinition,
    StateSetter,
)
from .kinetics import InterfaceKinetics, Kinetics, _DerivativeSettings, _KineticsType
from .reaction import Reaction
from .solutionbase import SolutionArrayBase, _SolutionBase, _SortingType, _YamlHeader
from .thermo import (
    InterfacePhase,
    PureFluid,
    Species,
    ThermoPhase,
    _PhaseOfMatter,
    _QuadratureMethod,
    _ThermoType,
)
from .transport import (
    DustyGasTransport,
    Transport,
    _TransportFittingErrors,
    _TransportModel,
)
from .units import Units, UnitSystem, _UnitDict

_T = TypeVar("_T")

class _EquivRatioFraction(TypedDict, Generic[_T], total=False):
    fuel: _T
    oxidizer: _T
    diluent: _T

# Generic representing a valid "phase" input
_P = TypeVar("_P", bound=_SolutionBase)

class Solution(Transport, Kinetics, ThermoPhase): ...
class Interface(InterfaceKinetics, InterfacePhase): ...
class DustyGas(DustyGasTransport, Kinetics, ThermoPhase): ...

class Quantity:
    @property
    def state(self) -> Array: ...
    @state.setter
    def state(self, state: ArrayLike) -> None: ...
    mass: float
    constant: PropertyPair
    def __init__(
        self,
        phase: Solution,
        mass: float | None = None,
        moles: float | None = None,
        constant: PropertyPair = "UV",
    ) -> None: ...
    @property
    def phase(self) -> Solution: ...
    @property
    def moles(self) -> float: ...
    @moles.setter
    def moles(self, n: float) -> None: ...
    @property
    def volume(self) -> float: ...
    @property
    def int_energy(self) -> float: ...
    @property
    def enthalpy(self) -> float: ...
    @property
    def entropy(self) -> float: ...
    @property
    def gibbs(self) -> float: ...
    def equilibrate(
        self,
        XY: PropertyPair | None = None,
        solver: EquilibriumSolver = "auto",
        rtol: float = 1e-9,
        max_steps: int = 1000,
        max_iter: int = 100,
        estimate_equil: int = 0,
        log_level: LogLevel = 0,
    ) -> None: ...
    def set_equivalence_ratio(
        self,
        phi: float,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
        *,
        diluent: CompositionLike | None = None,
        fraction: str | _EquivRatioFraction[float] | None = None,
    ) -> None: ...
    def set_mixture_fraction(
        self,
        mixture_fraction: float,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
    ) -> None: ...
    def __imul__(self, other: float) -> Quantity: ...
    def __mul__(self, other: float) -> Quantity: ...
    def __rmul__(self, other: float) -> Quantity: ...
    def __iadd__(self, other: Quantity) -> Quantity: ...
    def __add__(self, other: Quantity) -> Quantity: ...
    # Synonyms for total properties
    @property
    def V(self) -> float: ...
    @property
    def U(self) -> float: ...
    @property
    def H(self) -> float: ...
    @property
    def S(self) -> float: ...
    @property
    def G(self) -> float: ...
    # Dynamically-added properties acting as pass-throughs to Solution class
    # From _SolutionBase:
    @property
    def name(self) -> str: ...
    @name.setter
    def name(self, name: str) -> None: ...
    @property
    def source(self) -> str: ...
    @property
    def composite(
        self,
    ) -> tuple[_ThermoType | None, _KineticsType | None, _TransportModel | None]: ...
    @property
    def input_data(
        self,
    ) -> dict[str, str | list[str] | dict[str, float | dict[str, float]]]: ...
    @property
    def input_header(self) -> _YamlHeader: ...
    def update_user_data(self, data: dict[str, Any]) -> None: ...
    def clear_user_data(self) -> None: ...
    def update_user_header(self, data: dict[str, str | list[str]]) -> None: ...
    def clear_user_header(self) -> None: ...
    @overload
    def write_yaml(
        self,
        filename: None,
        phases: Sequence[ThermoPhase] | None = None,
        units: UnitSystem | _UnitDict | None = None,
        precision: int | None = None,
        skip_user_defined: bool | None = None,
        header: bool = True,
    ) -> str: ...
    @overload
    def write_yaml(
        self,
        filename: str | Path,
        phases: Sequence[ThermoPhase] | None = None,
        units: UnitSystem | _UnitDict | None = None,
        precision: int | None = None,
        skip_user_defined: bool | None = None,
        header: bool = True,
    ) -> None: ...
    @overload
    def write_yaml(
        self,
        filename: str | Path | None = None,
        phases: Sequence[ThermoPhase] | None = None,
        units: UnitSystem | _UnitDict | None = None,
        precision: int | None = None,
        skip_user_defined: bool | None = None,
        header: bool = True,
    ) -> str | None: ...
    def write_chemkin(
        self,
        mechanism_path: str | Path | None = None,
        thermo_path: str | Path | None = None,
        transport_path: str | Path | None = None,
        sort_species: _SortingType = None,
        sort_elements: _SortingType = None,
        overwrite: bool = False,
        quiet: bool = False,
    ) -> None: ...
    @property
    def selected_species(self) -> list[int]: ...
    @selected_species.setter
    def selected_species(
        self, species: str | int | Sequence[str] | Sequence[int]
    ) -> None: ...

    # From Transport:
    @property
    def transport_model(self) -> _TransportModel: ...
    @transport_model.setter
    def transport_model(self, model: _TransportModel) -> None: ...
    @property
    def CK_mode(self) -> bool: ...
    @property
    def viscosity(self) -> float: ...
    @property
    def species_viscosities(self) -> Array: ...
    @property
    def electrical_conductivity(self) -> float: ...
    @property
    def thermal_conductivity(self) -> float: ...
    @property
    def mix_diff_coeffs(self) -> Array: ...
    @property
    def mix_diff_coeffs_mass(self) -> Array: ...
    @property
    def mix_diff_coeffs_mole(self) -> Array: ...
    @property
    def thermal_diff_coeffs(self) -> Array: ...
    @property
    def multi_diff_coeffs(self) -> Array: ...
    @property
    def binary_diff_coeffs(self) -> Array: ...
    @property
    def mobilities(self) -> Array: ...
    def get_viscosity_polynomial(self, i: int) -> Array: ...
    def get_thermal_conductivity_polynomial(self, i: int) -> Array: ...
    def get_binary_diff_coeffs_polynomial(self, i: int, j: int) -> Array: ...
    def get_collision_integral_polynomials(
        self, i: int, j: int
    ) -> tuple[Array, Array, Array]: ...
    def set_viscosity_polynomial(self, i: int, values: ArrayLike) -> None: ...
    def set_thermal_conductivity_polynomial(
        self, i: int, values: ArrayLike
    ) -> None: ...
    def set_binary_diff_coeffs_polynomial(
        self, i: int, j: int, values: ArrayLike
    ) -> None: ...
    def set_collision_integral_polynomial(
        self,
        i: int,
        j: int,
        avalues: ArrayLike,
        bvalues: ArrayLike,
        cvalues: ArrayLike,
        actualT: bool = False,
    ) -> None: ...
    @property
    def transport_fitting_errors(self) -> _TransportFittingErrors: ...

    # From Kinetics:
    @property
    def kinetics_model(self) -> str: ...
    @property
    def n_total_species(self) -> int: ...
    @property
    def n_reactions(self) -> int: ...
    @property
    def n_phases(self) -> int: ...
    def kinetics_species_index(self, species: str | int, phase: int = 0) -> int: ...
    def kinetics_species_name(self, k: int) -> str: ...
    @property
    def kinetics_species_names(self) -> list[str]: ...
    def reaction(self, i_reaction: int) -> Reaction: ...
    def reactions(self) -> list[Reaction]: ...
    def modify_reaction(self, irxn: int, rxn: Reaction) -> None: ...
    def add_reaction(self, rxn: Reaction) -> None: ...
    def multiplier(self, i_reaction: int) -> float: ...
    def set_multiplier(self, value: float, i_reaction: int = -1) -> None: ...
    def reaction_equations(self, indices: Sequence[int] | None = None) -> list[str]: ...
    def reactant_stoich_coeff(self, k_spec: str | int, i_reaction: int) -> float: ...
    def product_stoich_coeff(self, k_spec: str | int, i_reaction: int) -> float: ...
    @property
    def reactant_stoich_coeffs(self) -> Array: ...
    @property
    def product_stoich_coeffs(self) -> Array: ...
    @property
    def product_stoich_coeffs_reversible(self) -> Array: ...
    @property
    def forward_rates_of_progress(self) -> Array: ...
    @property
    def reverse_rates_of_progress(self) -> Array: ...
    @property
    def net_rates_of_progress(self) -> Array: ...
    @property
    def equilibrium_constants(self) -> Array: ...
    @property
    def forward_rate_constants(self) -> Array: ...
    @property
    def reverse_rate_constants(self) -> Array: ...
    @property
    def creation_rates(self) -> Array: ...
    @property
    def destruction_rates(self) -> Array: ...
    @property
    def net_production_rates(self) -> Array: ...
    @property
    def derivative_settings(self) -> _DerivativeSettings: ...
    @derivative_settings.setter
    def derivative_settings(self, settings: _DerivativeSettings) -> None: ...
    @property
    def forward_rate_constants_ddT(self) -> Array: ...
    @property
    def forward_rate_constants_ddP(self) -> Array: ...
    @property
    def forward_rate_constants_ddC(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddT(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddP(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddC(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddX(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddCi(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddT(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddP(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddC(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddX(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddCi(self) -> Array: ...
    @property
    def net_rates_of_progress_ddT(self) -> Array: ...
    @property
    def net_rates_of_progress_ddP(self) -> Array: ...
    @property
    def net_rates_of_progress_ddC(self) -> Array: ...
    @property
    def net_rates_of_progress_ddX(self) -> Array: ...
    @property
    def net_rates_of_progress_ddCi(self) -> Array: ...
    @property
    def creation_rates_ddT(self) -> Array: ...
    @property
    def creation_rates_ddP(self) -> Array: ...
    @property
    def creation_rates_ddC(self) -> Array: ...
    @property
    def creation_rates_ddX(self) -> Array: ...
    @property
    def creation_rates_ddCi(self) -> Array: ...
    @property
    def destruction_rates_ddT(self) -> Array: ...
    @property
    def destruction_rates_ddP(self) -> Array: ...
    @property
    def destruction_rates_ddC(self) -> Array: ...
    @property
    def destruction_rates_ddX(self) -> Array: ...
    @property
    def destruction_rates_ddCi(self) -> Array: ...
    @property
    def net_production_rates_ddT(self) -> Array: ...
    @property
    def net_production_rates_ddP(self) -> Array: ...
    @property
    def net_production_rates_ddC(self) -> Array: ...
    @property
    def net_production_rates_ddX(self) -> Array: ...
    @property
    def net_production_rates_ddCi(self) -> Array: ...
    @property
    def delta_enthalpy(self) -> Array: ...
    @property
    def delta_gibbs(self) -> Array: ...
    @property
    def delta_entropy(self) -> Array: ...
    @property
    def delta_standard_enthalpy(self) -> Array: ...
    @property
    def third_body_concentrations(self) -> Array: ...
    @property
    def delta_standard_gibbs(self) -> Array: ...
    @property
    def delta_standard_entropy(self) -> Array: ...
    @property
    def heat_release_rate(self) -> float: ...
    @property
    def heat_production_rates(self) -> Array: ...

    # From ThermoPhase:
    @property
    def thermo_model(self) -> _ThermoType: ...
    @property
    def phase_of_matter(self) -> _PhaseOfMatter: ...
    def report(self, show_thermo: bool = True, threshold: float = 1e-14) -> str: ...
    @property
    def is_pure(self) -> bool: ...
    @property
    def has_phase_transition(self) -> bool: ...
    @property
    def is_compressible(self) -> bool: ...
    @property
    def basis(self) -> Basis: ...
    @basis.setter
    def basis(self, value: Basis) -> None: ...
    ####### Composition, species, and elements ########
    @property
    def n_elements(self) -> int: ...
    def element_index(self, element: str | int) -> int: ...
    def element_name(self, m: int) -> str: ...
    @property
    def element_names(self) -> list[str]: ...
    def atomic_weight(self, m: int) -> float: ...
    @property
    def atomic_weights(self) -> Array: ...
    @property
    def n_species(self) -> int: ...
    @property
    def n_selected_species(self) -> int: ...
    def species_name(self, k: int) -> str: ...
    @property
    def species_names(self) -> list[str]: ...
    def species_index(self, species: str | int) -> int: ...
    @property
    def case_sensitive_species_names(self) -> bool: ...
    @case_sensitive_species_names.setter
    def case_sensitive_species_names(self, val: bool) -> None: ...
    @overload
    def species(self, k: None = None) -> list[Species]: ...
    @overload
    def species(self, k: str | int) -> Species: ...
    def modify_species(self, k: int, species: Species) -> None: ...
    def add_species(self, species: Species) -> None: ...
    def add_species_alias(self, name: str, alias: str) -> None: ...
    def find_isomers(self, comp: CompositionLike) -> list[str]: ...
    def n_atoms(self, species: str | int, element: str | int) -> int: ...
    @property
    def molecular_weights(self) -> Array: ...
    @property
    def charges(self) -> Array: ...
    @property
    def mean_molecular_weight(self) -> float: ...
    @property
    def Y(self) -> Array: ...
    @Y.setter
    def Y(self, Y: CompositionLike) -> None: ...
    @property
    def X(self) -> Array: ...
    @X.setter
    def X(self, X: CompositionLike) -> None: ...
    @property
    def concentrations(self) -> Array: ...
    @concentrations.setter
    def concentrations(self, C: ArrayLike) -> None: ...
    def equivalence_ratio(
        self,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
        include_species: list[str | int] | None = None,
    ) -> float: ...
    def mixture_fraction(
        self,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
        element: str | int = "Bilger",
    ) -> float: ...
    def stoich_air_fuel_ratio(
        self,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
    ) -> float: ...
    def elemental_mass_fraction(self, m: str | int) -> float: ...
    def elemental_mole_fraction(self, m: str | int) -> float: ...
    def mass_fraction_dict(self, threshold: float = 0.0) -> dict[str, float]: ...
    def mole_fraction_dict(self, threshold: float = 0.0) -> dict[str, float]: ...
    ######## Read-only thermodynamic properties ########
    @property
    def P(self) -> float: ...
    @property
    def T(self) -> float: ...
    @property
    def density(self) -> float: ...
    @property
    def density_mass(self) -> float: ...
    @property
    def density_mole(self) -> float: ...
    @property
    def v(self) -> float: ...
    @property
    def volume_mass(self) -> float: ...
    @property
    def volume_mole(self) -> float: ...
    @property
    def u(self) -> float: ...
    @property
    def int_energy_mole(self) -> float: ...
    @property
    def int_energy_mass(self) -> float: ...
    @property
    def h(self) -> float: ...
    @property
    def enthalpy_mole(self) -> float: ...
    @property
    def enthalpy_mass(self) -> float: ...
    @property
    def s(self) -> float: ...
    @property
    def entropy_mole(self) -> float: ...
    @property
    def entropy_mass(self) -> float: ...
    @property
    def g(self) -> float: ...
    @property
    def gibbs_mole(self) -> float: ...
    @property
    def gibbs_mass(self) -> float: ...
    @property
    def cv(self) -> float: ...
    @property
    def cv_mole(self) -> float: ...
    @property
    def cv_mass(self) -> float: ...
    @property
    def cp(self) -> float: ...
    @property
    def cp_mole(self) -> float: ...
    @property
    def cp_mass(self) -> float: ...
    @property
    def critical_temperature(self) -> float: ...
    @property
    def critical_pressure(self) -> float: ...
    @property
    def critical_density(self) -> float: ...
    @property
    def P_sat(self) -> float: ...
    @property
    def T_sat(self) -> float: ...
    @property
    def auxiliary_data(self) -> dict[str, Any]: ...
    ######## Methods to get/set the complete thermodynamic state ########
    @property
    def state_size(self) -> int: ...
    @property
    def TD(self) -> tuple[float, float]: ...
    @TD.setter
    def TD(self, values: State2Setter) -> None: ...
    @property
    def TDX(self) -> tuple[float, float, Array]: ...
    @TDX.setter
    def TDX(self, values: StateSetter) -> None: ...
    @property
    def TDY(self) -> tuple[float, float, Array]: ...
    @TDY.setter
    def TDY(self, values: StateSetter) -> None: ...
    @property
    def TP(self) -> tuple[float, float]: ...
    @TP.setter
    def TP(self, values: State2Setter) -> None: ...
    @property
    def TPX(self) -> tuple[float, float, Array]: ...
    @TPX.setter
    def TPX(self, values: StateSetter) -> None: ...
    @property
    def TPY(self) -> tuple[float, float, Array]: ...
    @TPY.setter
    def TPY(self, values: StateSetter) -> None: ...
    @property
    def UV(self) -> tuple[float, float]: ...
    @UV.setter
    def UV(self, values: State2Setter) -> None: ...
    @property
    def UVX(self) -> tuple[float, float, Array]: ...
    @UVX.setter
    def UVX(self, values: StateSetter) -> None: ...
    @property
    def UVY(self) -> tuple[float, float, Array]: ...
    @UVY.setter
    def UVY(self, values: StateSetter) -> None: ...
    @property
    def DP(self) -> tuple[float, float]: ...
    @DP.setter
    def DP(self, values: State2Setter) -> None: ...
    @property
    def DPX(self) -> tuple[float, float, Array]: ...
    @DPX.setter
    def DPX(self, values: StateSetter) -> None: ...
    @property
    def DPY(self) -> tuple[float, float, Array]: ...
    @DPY.setter
    def DPY(self, values: StateSetter) -> None: ...
    @property
    def HP(self) -> tuple[float, float]: ...
    @HP.setter
    def HP(self, values: State2Setter) -> None: ...
    @property
    def HPX(self) -> tuple[float, float, Array]: ...
    @HPX.setter
    def HPX(self, values: StateSetter) -> None: ...
    @property
    def HPY(self) -> tuple[float, float, Array]: ...
    @HPY.setter
    def HPY(self, values: StateSetter) -> None: ...
    @property
    def SP(self) -> tuple[float, float]: ...
    @SP.setter
    def SP(self, values: State2Setter) -> None: ...
    @property
    def SPX(self) -> tuple[float, float, Array]: ...
    @SPX.setter
    def SPX(self, values: StateSetter) -> None: ...
    @property
    def SPY(self) -> tuple[float, float, Array]: ...
    @SPY.setter
    def SPY(self, values: StateSetter) -> None: ...
    @property
    def SV(self) -> tuple[float, float]: ...
    @SV.setter
    def SV(self, values: State2Setter) -> None: ...
    @property
    def SVX(self) -> tuple[float, float, Array]: ...
    @SVX.setter
    def SVX(self, values: StateSetter) -> None: ...
    @property
    def SVY(self) -> tuple[float, float, Array]: ...
    @SVY.setter
    def SVY(self, values: StateSetter) -> None: ...
    # partial molar / non-dimensional properties
    @property
    def partial_molar_enthalpies(self) -> Array: ...
    @property
    def partial_molar_entropies(self) -> Array: ...
    @property
    def partial_molar_int_energies(self) -> Array: ...
    @property
    def chemical_potentials(self) -> Array: ...
    @property
    def electrochemical_potentials(self) -> Array: ...
    @property
    def partial_molar_cp(self) -> Array: ...
    @property
    def partial_molar_volumes(self) -> Array: ...
    @property
    def standard_enthalpies_RT(self) -> Array: ...
    @property
    def standard_entropies_R(self) -> Array: ...
    @property
    def standard_int_energies_RT(self) -> Array: ...
    @property
    def standard_gibbs_RT(self) -> Array: ...
    @property
    def standard_cp_R(self) -> Array: ...
    @property
    def activities(self) -> Array: ...
    @property
    def activity_coefficients(self) -> Array: ...
    ######## Miscellaneous properties ########
    @property
    def isothermal_compressibility(self) -> float: ...
    @property
    def thermal_expansion_coeff(self) -> float: ...
    @property
    def sound_speed(self) -> float: ...
    @property
    def min_temp(self) -> float: ...
    @property
    def max_temp(self) -> float: ...
    @property
    def reference_pressure(self) -> float: ...
    @property
    def electric_potential(self) -> float: ...
    @electric_potential.setter
    def electric_potential(self, value: float) -> None: ...
    @property
    def standard_concentration_units(self) -> Units: ...
    # methods for plasma
    @property
    def Te(self) -> float: ...
    @Te.setter
    def Te(self, value: float) -> None: ...
    @property
    def Pe(self) -> float: ...
    @property
    def reduced_electric_field(self) -> float: ...
    @reduced_electric_field.setter
    def reduced_electric_field(self, value: float) -> None: ...
    @property
    def electric_field(self) -> float: ...
    @electric_field.setter
    def electric_field(self, value: float) -> None: ...
    def set_discretized_electron_energy_distribution(
        self, levels: ArrayLike, distribution: ArrayLike
    ) -> None: ...
    def update_electron_energy_distribution(self) -> None: ...
    @property
    def n_electron_energy_levels(self) -> int: ...
    @property
    def electron_energy_levels(self) -> Array: ...
    @electron_energy_levels.setter
    def electron_energy_levels(self, levels: Array) -> None: ...
    @property
    def electron_energy_distribution(self) -> Array: ...
    @property
    def isotropic_shape_factor(self) -> float: ...
    @isotropic_shape_factor.setter
    def isotropic_shape_factor(self, x: float) -> None: ...
    @property
    def electron_energy_distribution_type(self) -> str: ...
    @electron_energy_distribution_type.setter
    def electron_energy_distribution_type(self, distribution_type: str) -> None: ...
    @property
    def mean_electron_energy(self) -> float: ...
    @mean_electron_energy.setter
    def mean_electron_energy(self, energy: float) -> None: ...
    @property
    def quadrature_method(self) -> _QuadratureMethod: ...
    @quadrature_method.setter
    def quadrature_method(self, method: _QuadratureMethod) -> None: ...
    @property
    def normalize_electron_energy_distribution_enabled(self) -> bool: ...
    @normalize_electron_energy_distribution_enabled.setter
    def normalize_electron_energy_distribution_enabled(self, enable: bool) -> None: ...
    @property
    def electron_species_name(self) -> str: ...
    @property
    def elastic_power_loss(self) -> float: ...

class SolutionArray(SolutionArrayBase, Generic[_P]):
    _phase: _P | None
    def __init__(
        self,
        phase: _P,
        shape: int | tuple[int, ...] = (0,),
        states: ArrayLike | None = None,
        extra: str | Sequence[str] | dict[str, ArrayLike] | None = None,
        meta: dict[str, Any] = {},
        init: bool = True,
    ) -> None: ...
    def __getitem__(self, index: Index) -> SolutionArray[_P]: ...
    def __call__(self, *species: str) -> SolutionArray[_P]: ...
    def __len__(self) -> int: ...
    @property
    def ndim(self) -> int: ...
    @property
    def shape(self) -> tuple[int, ...]: ...
    @shape.setter
    def shape(self, shp: tuple[int, ...]) -> None: ...
    def append(
        self,
        state: ArrayLike | None = None,
        normalize: bool = True,
        **kwargs: Unpack[StateDefinition],
    ) -> None: ...
    def sort(self, col: str, reverse: bool = False) -> None: ...
    def equilibrate(
        self,
        XY: PropertyPair | None = None,
        solver: EquilibriumSolver = "auto",
        rtol: float = 1e-9,
        max_steps: int = 1000,
        max_iter: int = 100,
        estimate_equil: int = 0,
        log_level: LogLevel = 0,
    ) -> None: ...
    def restore_data(self, data: dict[str, Array], normalize: bool = True) -> None: ...
    def set_equivalence_ratio(
        self,
        phi: float | Array,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
        *,
        diluent: CompositionLike | None = None,
        fraction: str | _EquivRatioFraction[float | Array] | None = None,
    ) -> None: ...
    def set_mixture_fraction(
        self,
        mixture_fraction: float | Array,
        fuel: CompositionLike,
        oxidizer: CompositionLike,
        basis: Basis = "mole",
    ) -> None: ...
    def collect_data(
        self,
        cols: Sequence[str] | None = None,
        tabular: bool = False,
        threshold: int = 0,
        species: Literal["X", "Y"] | None = None,
    ) -> dict[str, Array]: ...
    def read_csv(self, filename: str, normalize: bool = True) -> None: ...
    def to_pandas(
        self,
        cols: Sequence[str] | None = None,
        tabular: bool = False,
        threshold: int = 0,
        species: Literal["X", "Y"] | None = None,
    ) -> DataFrame: ...
    def from_pandas(self, df: DataFrame, normalize: bool = True) -> None: ...
    def save(
        self,
        fname: str | Path,
        name: str | None = None,
        sub: str | None = None,
        description: str | None = None,
        *,
        overwrite: bool = False,
        compression: CompressionLevel = 0,
        basis: Basis | None = None,
    ) -> None: ...
    def restore(
        self, fname: str | Path, name: str | None = None, sub: str | None = None
    ) -> None: ...
    @override
    def __reduce__(self) -> Never: ...
    def __copy__(self) -> Never: ...

    # Dynamically-added properties acting as pass-throughs to underlying Solution class
    # State setters/getters
    @property
    def TD(self) -> tuple[Array, Array]: ...
    @TD.setter
    def TD(self, values: ArrayState2Setter) -> None: ...
    @property
    def TDX(self) -> tuple[Array, Array, Array]: ...
    @TDX.setter
    def TDX(self, values: ArrayStateSetter) -> None: ...
    @property
    def TDY(self) -> tuple[Array, Array, Array]: ...
    @TDY.setter
    def TDY(self, values: ArrayStateSetter) -> None: ...
    @property
    def TP(self) -> tuple[Array, Array]: ...
    @TP.setter
    def TP(self, values: ArrayState2Setter) -> None: ...
    @property
    def TPX(self) -> tuple[Array, Array, Array]: ...
    @TPX.setter
    def TPX(self, values: ArrayStateSetter) -> None: ...
    @property
    def TPY(self) -> tuple[Array, Array, Array]: ...
    @TPY.setter
    def TPY(self, values: ArrayStateSetter) -> None: ...
    @property
    def UV(self) -> tuple[Array, Array]: ...
    @UV.setter
    def UV(self, values: ArrayState2Setter) -> None: ...
    @property
    def UVX(self) -> tuple[Array, Array, Array]: ...
    @UVX.setter
    def UVX(self, values: ArrayStateSetter) -> None: ...
    @property
    def UVY(self) -> tuple[Array, Array, Array]: ...
    @UVY.setter
    def UVY(self, values: ArrayStateSetter) -> None: ...
    @property
    def DP(self) -> tuple[Array, Array]: ...
    @DP.setter
    def DP(self, values: ArrayState2Setter) -> None: ...
    @property
    def DPX(self) -> tuple[Array, Array, Array]: ...
    @DPX.setter
    def DPX(self, values: ArrayStateSetter) -> None: ...
    @property
    def DPY(self) -> tuple[Array, Array, Array]: ...
    @DPY.setter
    def DPY(self, values: ArrayStateSetter) -> None: ...
    @property
    def HP(self) -> tuple[Array, Array]: ...
    @HP.setter
    def HP(self, values: ArrayState2Setter) -> None: ...
    @property
    def HPX(self) -> tuple[Array, Array, Array]: ...
    @HPX.setter
    def HPX(self, values: ArrayStateSetter) -> None: ...
    @property
    def HPY(self) -> tuple[Array, Array, Array]: ...
    @HPY.setter
    def HPY(self, values: ArrayStateSetter) -> None: ...
    @property
    def SP(self) -> tuple[Array, Array]: ...
    @SP.setter
    def SP(self, values: ArrayState2Setter) -> None: ...
    @property
    def SPX(self) -> tuple[Array, Array, Array]: ...
    @SPX.setter
    def SPX(self, values: ArrayStateSetter) -> None: ...
    @property
    def SPY(self) -> tuple[Array, Array, Array]: ...
    @SPY.setter
    def SPY(self, values: ArrayStateSetter) -> None: ...
    @property
    def SV(self) -> tuple[Array, Array]: ...
    @SV.setter
    def SV(self, values: ArrayState2Setter) -> None: ...
    @property
    def SVX(self) -> tuple[Array, Array, Array]: ...
    @SVX.setter
    def SVX(self, values: ArrayStateSetter) -> None: ...
    @property
    def SVY(self) -> tuple[Array, Array, Array]: ...
    @SVY.setter
    def SVY(self, values: ArrayStateSetter) -> None: ...
    @property
    def TQ(self) -> tuple[Array, Array]: ...
    @TQ.setter
    def TQ(self, values: ArrayState2Setter) -> None: ...
    @property
    def PQ(self) -> tuple[Array, Array]: ...
    @PQ.setter
    def PQ(self, values: ArrayState2Setter) -> None: ...
    @property
    def ST(self) -> tuple[Array, Array]: ...
    @ST.setter
    def ST(self, values: ArrayState2Setter) -> None: ...
    @property
    def TV(self) -> tuple[Array, Array]: ...
    @TV.setter
    def TV(self, values: ArrayState2Setter) -> None: ...
    @property
    def PV(self) -> tuple[Array, Array]: ...
    @PV.setter
    def PV(self, values: ArrayState2Setter) -> None: ...
    @property
    def UP(self) -> tuple[Array, Array]: ...
    @UP.setter
    def UP(self, values: ArrayState2Setter) -> None: ...
    @property
    def VH(self) -> tuple[Array, Array]: ...
    @VH.setter
    def VH(self, values: ArrayState2Setter) -> None: ...
    @property
    def TH(self) -> tuple[Array, Array]: ...
    @TH.setter
    def TH(self, values: ArrayState2Setter) -> None: ...
    @property
    def SH(self) -> tuple[Array, Array]: ...
    @SH.setter
    def SH(self, values: ArrayState2Setter) -> None: ...
    @property
    def TDQ(self) -> tuple[Array, Array, Array]: ...
    @TDQ.setter
    def TDQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...
    @property
    def TPQ(self) -> tuple[Array, Array, Array]: ...
    @TPQ.setter
    def TPQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...
    @property
    def UVQ(self) -> tuple[Array, Array, Array]: ...
    @UVQ.setter
    def UVQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...
    @property
    def DPQ(self) -> tuple[Array, Array, Array]: ...
    @DPQ.setter
    def DPQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...
    @property
    def HPQ(self) -> tuple[Array, Array, Array]: ...
    @HPQ.setter
    def HPQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...
    @property
    def SPQ(self) -> tuple[Array, Array, Array]: ...
    @SPQ.setter
    def SPQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...
    @property
    def SVQ(self) -> tuple[Array, Array, Array]: ...
    @SVQ.setter
    def SVQ(
        self: SolutionArray[PureFluid], values: ArrayPureFluidStateSetter
    ) -> None: ...

    # scalar
    # From ThermoPhase
    @property
    def mean_molecular_weight(self) -> Array: ...
    @property
    def P(self) -> Array: ...
    @property
    def T(self) -> Array: ...
    @property
    def Te(self) -> Array: ...
    @property
    def density(self) -> Array: ...
    @property
    def density_mass(self) -> Array: ...
    @property
    def density_mole(self) -> Array: ...
    @property
    def v(self) -> Array: ...
    @property
    def volume_mass(self) -> Array: ...
    @property
    def volume_mole(self) -> Array: ...
    @property
    def u(self) -> Array: ...
    @property
    def int_energy_mole(self) -> Array: ...
    @property
    def int_energy_mass(self) -> Array: ...
    @property
    def h(self) -> Array: ...
    @property
    def enthalpy_mole(self) -> Array: ...
    @property
    def enthalpy_mass(self) -> Array: ...
    @property
    def s(self) -> Array: ...
    @property
    def entropy_mole(self) -> Array: ...
    @property
    def entropy_mass(self) -> Array: ...
    @property
    def g(self) -> Array: ...
    @property
    def gibbs_mole(self) -> Array: ...
    @property
    def gibbs_mass(self) -> Array: ...
    @property
    def cv(self) -> Array: ...
    @property
    def cv_mole(self) -> Array: ...
    @property
    def cv_mass(self) -> Array: ...
    @property
    def cp(self) -> Array: ...
    @property
    def cp_mole(self) -> Array: ...
    @property
    def cp_mass(self) -> Array: ...
    @property
    def critical_temperature(self) -> Array: ...
    @property
    def critical_pressure(self) -> Array: ...
    @property
    def critical_density(self) -> Array: ...
    @property
    def P_sat(self) -> Array: ...
    @property
    def T_sat(self) -> Array: ...
    @property
    def isothermal_compressibility(self) -> Array: ...
    @property
    def thermal_expansion_coeff(self) -> Array: ...
    @property
    def sound_speed(self) -> Array: ...
    @property
    def electric_potential(self) -> Array: ...
    # From Kinetics
    @property
    def heat_release_rate(self: SolutionArray[Kinetics]) -> Array: ...
    # From Transport
    @property
    def viscosity(self) -> Array: ...
    @property
    def electrical_conductivity(self) -> Array: ...
    @property
    def thermal_conductivity(self) -> Array: ...
    # strings
    @property
    def phase_of_matter(self) -> _PhaseOfMatter: ...

    # n_species
    # from ThermoPhase
    @property
    def Y(self) -> Array: ...
    @property
    def X(self) -> Array: ...
    @property
    def concentrations(self) -> Array: ...
    @property
    def partial_molar_enthalpies(self) -> Array: ...
    @property
    def partial_molar_entropies(self) -> Array: ...
    @property
    def partial_molar_int_energies(self) -> Array: ...
    @property
    def chemical_potentials(self) -> Array: ...
    @property
    def electrochemical_potentials(self) -> Array: ...
    @property
    def partial_molar_cp(self) -> Array: ...
    @property
    def partial_molar_volumes(self) -> Array: ...
    @property
    def standard_enthalpies_RT(self) -> Array: ...
    @property
    def standard_entropies_R(self) -> Array: ...
    @property
    def standard_int_energies_RT(self) -> Array: ...
    @property
    def standard_gibbs_RT(self) -> Array: ...
    @property
    def standard_cp_R(self) -> Array: ...
    @property
    def activities(self) -> Array: ...
    @property
    def activity_coefficients(self) -> Array: ...
    # From Transport
    @property
    def mix_diff_coeffs(self) -> Array: ...
    @property
    def mix_diff_coeffs_mass(self) -> Array: ...
    @property
    def mix_diff_coeffs_mole(self) -> Array: ...
    @property
    def thermal_diff_coeffs(self) -> Array: ...
    @property
    def mobilities(self) -> Array: ...
    @property
    def species_viscosities(self) -> Array: ...

    # n_total_species
    @property
    def creation_rates(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def destruction_rates(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_production_rates(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def creation_rates_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def creation_rates_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def creation_rates_ddT(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def destruction_rates_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def destruction_rates_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def destruction_rates_ddT(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_production_rates_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_production_rates_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_production_rates_ddT(self: SolutionArray[Kinetics]) -> Array: ...

    # n_species2
    @property
    def multi_diff_coeffs(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def binary_diff_coeffs(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def creation_rates_ddX(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def destruction_rates_ddX(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_production_rates_ddX(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def creation_rates_ddCi(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def destruction_rates_ddCi(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_production_rates_ddCi(self: SolutionArray[Kinetics]) -> Array: ...

    # n_reactions
    @property
    def forward_rates_of_progress(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def reverse_rates_of_progress(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_rates_of_progress(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def equilibrium_constants(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rate_constants(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def reverse_rate_constants(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def delta_enthalpy(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def delta_gibbs(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def delta_entropy(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def delta_standard_enthalpy(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def delta_standard_gibbs(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def delta_standard_entropy(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def heat_production_rates(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rate_constants_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rate_constants_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rate_constants_ddT(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rates_of_progress_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rates_of_progress_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def forward_rates_of_progress_ddT(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_rates_of_progress_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_rates_of_progress_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def net_rates_of_progress_ddT(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def reverse_rates_of_progress_ddC(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def reverse_rates_of_progress_ddP(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def reverse_rates_of_progress_ddT(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def third_body_concentrations(self: SolutionArray[Kinetics]) -> Array: ...
    # call_scalar
    def elemental_mass_fraction(self, m: str | int) -> Array: ...
    def elemental_mole_fraction(self, m: str | int) -> Array: ...

    # passthrough
    # from ThermoPhase
    @property
    def name(self) -> str: ...
    @property
    def source(self) -> str: ...
    @property
    def basis(self) -> Basis: ...
    @property
    def n_elements(self) -> int: ...
    def element_index(self, element: str | int) -> int: ...
    def element_name(self, m: int) -> str: ...
    @property
    def element_names(self) -> list[str]: ...
    def atomic_weight(self, m: int) -> float: ...
    @property
    def atomic_weights(self) -> Array: ...
    @property
    def n_species(self) -> int: ...
    def species_name(self, k: int) -> str: ...
    @property
    def species_names(self) -> list[str]: ...
    def species_index(self, species: str | int) -> int: ...
    @overload
    def species(self, k: None = None) -> list[Species]: ...
    @overload
    def species(self, k: str | int) -> Species: ...
    @overload
    def species(self, k: str | int | None = None) -> Species | list[Species]: ...
    def n_atoms(self, species: str | int, element: str | int) -> int: ...
    @property
    def molecular_weights(self) -> Array: ...
    @property
    def min_temp(self) -> float: ...
    @property
    def max_temp(self) -> float: ...
    @property
    def reference_pressure(self) -> float: ...
    @property
    def charges(self) -> Array: ...
    # From Kinetics
    @property
    def n_total_species(self: SolutionArray[Kinetics]) -> int: ...
    @property
    def n_reactions(self: SolutionArray[Kinetics]) -> int: ...
    @property
    def n_phases(self: SolutionArray[Kinetics]) -> int: ...
    def kinetics_species_index(
        self: SolutionArray[Kinetics], species: str | int, phase: int = 0
    ) -> int: ...
    def reaction(self: SolutionArray[Kinetics], i_reaction: int) -> Reaction: ...
    def reactions(self: SolutionArray[Kinetics]) -> list[Reaction]: ...
    def modify_reaction(
        self: SolutionArray[Kinetics], irxn: int, rxn: Reaction
    ) -> None: ...
    def multiplier(self: SolutionArray[Kinetics], i_reaction: int) -> float: ...
    def set_multiplier(
        self: SolutionArray[Kinetics], value: float, i_reaction: int = -1
    ) -> None: ...
    def reaction_equations(
        self: SolutionArray[Kinetics], indices: Sequence[int] | None = None
    ) -> list[str]: ...
    def reactant_stoich_coeff(
        self: SolutionArray[Kinetics], k_spec: str | int, i_reaction: int
    ) -> float: ...
    def product_stoich_coeff(
        self: SolutionArray[Kinetics], k_spec: str | int, i_reaction: int
    ) -> float: ...
    @property
    def reactant_stoich_coeffs(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def product_stoich_coeffs(self: SolutionArray[Kinetics]) -> Array: ...
    @property
    def product_stoich_coeffs_reversible(self: SolutionArray[Kinetics]) -> Array: ...
    # from Transport
    @property
    def transport_model(self) -> str: ...

    # interface_passthrough
    @property
    def site_density(self: SolutionArray[Interface]) -> Array: ...
    @site_density.setter
    def site_density(self: SolutionArray[Interface], value: Array) -> None: ...

    # interface_n_species
    @property
    def coverages(self: SolutionArray[Interface]) -> Array: ...

    # purefluid_scalar
    @property
    def Q(self: SolutionArray[PureFluid]) -> Array: ...
