// bbppp.cc for bbppp - a ppp connection manager for Blackbox
//
//  Copyright (c) 1998-2000 by John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "bbppp.hh"
#include "BaseDisplay.hh"
#include "version.h"
#include <errno.h>
#include <sys/wait.h>


ToolWindow::ToolWindow(int argc,char **argv, CMDOPTIONS *options) :
		Basewindow(argc,argv,options)
{
	XrmInitialize();
	hour=minute=sec=0;
	tpRate = 0.0;
	resource = new Resource(this);
	ppp_up = False;

	if (options->thruput)
		resource->show.thruputLabel = true;
	if (!options->showtime)
		resource->show.upTimeLabel = false;
	if (options->showsecs)
		resource->show.showsecs = true;
	if (options->compact)
		resource->compact = true;
	if (options->orient == vertical)
		resource->vertOrient = true;
	numButtons = (resource->compact) ? 1 : 3;

	MakeWindow(False);
	pppstatus = new PPPStat(this);
	pppstatus->checkPPP();

	Redraw();
	eventLoop();
}

ToolWindow::~ToolWindow()
{
	XUnmapWindow(getXDisplay(),framewin);

	/* destroy pixmaps */
	if (pixmap.frame) getImageControl()->removeImage(pixmap.frame);
	if (pixmap.pppUpLabel) getImageControl()->removeImage(pixmap.pppUpLabel);
	if (pixmap.thruputLabel) getImageControl()->removeImage(pixmap.thruputLabel);

	/* destroy windows */
	if (pppUpLabelwin)
		XDestroyWindow(getXDisplay(),pppUpLabelwin);
	if (thruputLabelwin)
		XDestroyWindow(getXDisplay(),thruputLabelwin);
	XDestroyWindow(getXDisplay(),framewin);
}

void ToolWindow::reconfigure(void)
{
	/* destroy pixmaps */
	getImageControl()->removeImage(pixmap.frame);
	getImageControl()->removeImage(pixmap.pppUpLabel);
	getImageControl()->removeImage(pixmap.thruputLabel);

	resource->Reload();

	if (options->thruput)
		resource->show.thruputLabel = true;
	if (!options->showtime)
		resource->show.upTimeLabel = false;
	if (options->showsecs)
		resource->show.showsecs = true;
	if (options->compact)
		resource->compact = true;
	if (options->orient == vertical)
		resource->vertOrient = true;
	numButtons = (resource->compact) ? 1 : 3;
	
	MakeWindow(True);

	if (resource->show.upTimeLabel)
		XClearWindow(getXDisplay(), pppUpLabelwin);
	if (resource->show.thruputLabel)
		XClearWindow(getXDisplay(), thruputLabelwin);
	XClearWindow(getXDisplay(), framewin);
	Redraw();
}


void ToolWindow::MakeWindow(bool reconfigure)
{
	XSetWindowAttributes attrib;
	XWMHints wmhints;

	unsigned long create_mask = CWBackPixmap|CWCursor|CWEventMask;

	frame.height=resource->frame.font->ascent+
	             resource->frame.font->descent+4*resource->frame.bevelWidth;

	// height and width of buttons and labels won't change based on user-specified parameters
	button.height = pppUpLabel.height = thruputLabel.height = frame.height-2*resource->frame.bevelWidth;
	if (resource->compact)
		button.width = 15;
	else
		button.width = (int) (3.0/4.0*button.height);
	if (resource->show.showsecs)
		pppUpLabel.width = XTextWidth(resource->label.font," hh:mm:ss ",strlen(" hh:mm:ss ")) +
		                   resource->frame.bevelWidth;
	else
		pppUpLabel.width = XTextWidth(resource->label.font," hh:mm ",strlen(" hh:mm ")) +
		                   resource->frame.bevelWidth;
	thruputLabel.width = XTextWidth(resource->label.font, "  x.yk ", strlen("  x.yk ")) +
	                     resource->frame.bevelWidth;

	// vertical orientation stacks components vertically
	// ie, 1st row contains the buttons,
	// next row is ppp uptime label, if present
	// next row is thruput rate label if present
	if (resource->vertOrient) {
		// the clock time will be widest component
		frame.width = pppUpLabel.width + 2 * resource->frame.bevelWidth;
		if (frame.width < 64)
			frame.width = 64;

		// center the components along x-axis
		button.x = (int)((frame.width - numButtons*button.width)/2.0);
		button.y = resource->frame.bevelWidth;
		frame.height = button.height + 2 * resource->frame.bevelWidth;

		int nextY = 2*resource->frame.bevelWidth + button.height;
		if (resource->show.upTimeLabel) {
			pppUpLabel.x = (int)((frame.width - pppUpLabel.width)/2.0);
			pppUpLabel.y = nextY;
			nextY += pppUpLabel.height + resource->frame.bevelWidth;
			frame.height += pppUpLabel.height + resource->frame.bevelWidth;
		}
		if (resource->show.thruputLabel) {
			thruputLabel.x = (int)((frame.width - thruputLabel.width)/2.0);
			thruputLabel.y = nextY;
			frame.height += thruputLabel.height + resource->frame.bevelWidth;
		}
	} else  // components aligned horizontally
	{
		button.y = thruputLabel.y = pppUpLabel.y = resource->frame.bevelWidth;

		button.x = resource->frame.bevelWidth;
		frame.width = numButtons * button.width + 2 * resource->frame.bevelWidth;

		int nextX = 2 * resource->frame.bevelWidth + numButtons * button.width;
		if (resource->show.upTimeLabel) {
			pppUpLabel.x = nextX;
			nextX = pppUpLabel.x + pppUpLabel.width + resource->frame.bevelWidth;
			frame.width += pppUpLabel.width + resource->frame.bevelWidth;
		}
		if (resource->show.thruputLabel) {
			thruputLabel.x = nextX;
			frame.width += thruputLabel.width + resource->frame.bevelWidth;
		}
	}

	if (resource->compact) {
		switchlight.x=(int)(button.width/6.0);
		switchlight.y=4;
		switchlight.width=switchlight.x;
		switchlight.height=button.height-2*switchlight.y;
		txlight.x=2*switchlight.x+switchlight.width;
		txlight.y=switchlight.y;
		txlight.width=switchlight.width;
		txlight.height=switchlight.height;
		rxlight.x=3*switchlight.x+switchlight.width+txlight.width;
		rxlight.y=switchlight.y;
		rxlight.width=switchlight.width;
		rxlight.height=switchlight.height;
	}

	if (resource->position.mask & XNegative) {
		resource->position.x = getCurrentScreenInfo()->getWidth() +
		                       resource->position.x - frame.width;
	}
	if (resource->position.mask & YNegative) {
		resource->position.y = getCurrentScreenInfo()->getHeight() +
		                       resource->position.y - frame.height;
	}

	if (!options->withdrawn) {
		frame.x=resource->position.x;
		frame.y=resource->position.y;
		wmhints.initial_state = NormalState;
	} else
		wmhints.initial_state = WithdrawnState;


	attrib.background_pixmap = ParentRelative;

	pixmap.frame = getImageControl()->renderImage(frame.width, frame.height,
	               &resource->frame.texture);

	pixmap.pppUpLabel = getImageControl()->renderImage(pppUpLabel.width, pppUpLabel.height,
	                    &resource->label.texture);

	pixmap.thruputLabel = getImageControl()->renderImage(thruputLabel.width, thruputLabel.height,
	                      &resource->label.texture);

	pixmap.button = getImageControl()->renderImage(button.width, button.height,
	                &resource->button.texture);

	pixmap.button_pressed = getImageControl()->renderImage(button.width,button.height,
	                        &resource->button.texture_pressed);

	attrib.cursor = getSessionCursor();
	attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
	                    FocusChangeMask | KeyPressMask | StructureNotifyMask;

	if (!reconfigure) {
		framewin = XCreateWindow(getXDisplay(),
		                         getCurrentScreenInfo()->getRootWindow(),
		                         frame.x,frame.y, frame.width,frame.height,
		                         0, getCurrentScreenInfo()->getDepth(),
		                         InputOutput,getCurrentScreenInfo()->getVisual(),
		                         create_mask, &attrib);
	} else if (!options->withdrawn)
		XMoveResizeWindow(getXDisplay(), framewin, frame.x,
		                  frame.y, frame.width, frame.height);
	else
		XResizeWindow(getXDisplay(),framewin,frame.width,frame.height);

	wmhints.flags = IconWindowHint | StateHint;
	wmhints.icon_window = framewin;

	XTextProperty windowname;
	XClassHint classhints;

	char *name = BBTOOL;
	XSizeHints sizehints;

	classhints.res_name = BBTOOL;
	classhints.res_class = "bbtools";

	sizehints.x = resource->position.x;
	sizehints.y = resource->position.y;
	sizehints.max_width = sizehints.min_width = frame.width;
	sizehints.max_height = sizehints.min_height = frame.height;
	sizehints.flags = USPosition | PMinSize | PMaxSize;

	XStringListToTextProperty(&name,1,&windowname);
	XSetWMProperties(getXDisplay(),framewin,&windowname,NULL,getArgv(),getArgc(),
	                 &sizehints,&wmhints,&classhints);
	Atom wmproto[2];
	wmproto[0] = wm_delete_window;
	wmproto[1] = getBlackboxStructureMessagesAtom();
	XSetWMProtocols(getXDisplay(), framewin,wmproto, 2);

	if (!options->decorated&&!options->withdrawn) {
		BlackboxHints bb_hints;
		bb_hints.decoration = DecorNone;
		bb_hints.attrib = AttribOmnipresent;
		bb_hints.flags = AttribDecoration | AttribOmnipresent;
		XChangeProperty(getXDisplay(), framewin, getBlackboxHintsAtom(),
		                getBlackboxHintsAtom(), 32, PropModeReplace,
		                (unsigned char *) &bb_hints,  PropBlackboxHintsElements);
	}

	if (!options->shape)
		XSetWindowBackgroundPixmap(getXDisplay(), framewin, pixmap.frame);

	if (resource->show.upTimeLabel) {
		if (!reconfigure)
			pppUpLabelwin = XCreateWindow(getXDisplay(), framewin,pppUpLabel.x,pppUpLabel.y,
			                              pppUpLabel.width, pppUpLabel.height,
			                              0,getCurrentScreenInfo()->getDepth(),
			                              InputOutput,getCurrentScreenInfo()->getVisual(),
			                              create_mask, &attrib);
		else
			XMoveResizeWindow(getXDisplay(), pppUpLabelwin, pppUpLabel.x,pppUpLabel.y,
			                  pppUpLabel.width, pppUpLabel.height);

		if (!resource->label.transparent)
			XSetWindowBackgroundPixmap(getXDisplay(), pppUpLabelwin, pixmap.pppUpLabel);
	}

	if (resource->show.thruputLabel) {
		if (!reconfigure)
			thruputLabelwin = XCreateWindow(getXDisplay(), framewin,thruputLabel.x,thruputLabel.y,
			                                thruputLabel.width, thruputLabel.height,
			                                0,getCurrentScreenInfo()->getDepth(),
			                                InputOutput,getCurrentScreenInfo()->getVisual(),
			                                create_mask, &attrib);
		else
			XMoveResizeWindow(getXDisplay(), thruputLabelwin, thruputLabel.x,thruputLabel.y,
			                  thruputLabel.width, thruputLabel.height);

		if (!resource->label.transparent)
			XSetWindowBackgroundPixmap(getXDisplay(), thruputLabelwin, pixmap.thruputLabel);
	}

	if (!reconfigure) {

		switchwin = XCreateWindow(getXDisplay(), framewin, button.x,button.y,
		                          button.width, button.height, 0,
		                          getCurrentScreenInfo()->getDepth(), InputOutput,
		                          getCurrentScreenInfo()->getVisual(), create_mask,
		                          &attrib);
		if (! resource->compact) {
			txwin = XCreateWindow(getXDisplay(), framewin,
			                      button.x+button.width,
			                      button.y,
			                      button.width, button.height, 0,
			                      getCurrentScreenInfo()->getDepth(), InputOutput,
			                      getCurrentScreenInfo()->getVisual(),create_mask,
			                      &attrib);

			rxwin = XCreateWindow(getXDisplay(), framewin,
			                      button.x+2*button.width,
			                      button.y,
			                      button.width, button.height, 0,
			                      getCurrentScreenInfo()->getDepth(), InputOutput,
			                      getCurrentScreenInfo()->getVisual(), create_mask,
			                      &attrib);
		}

	} else {
		XMoveResizeWindow(getXDisplay(), switchwin, button.x,button.y,
		                  button.width, button.height);

		if (! resource->compact) {
			XMoveResizeWindow(getXDisplay(), txwin, button.x+button.width,
			                  button.y,
			                  button.width, button.height);

			XMoveResizeWindow(getXDisplay(), rxwin, button.x+2*button.width,
			                  button.y,
			                  button.width, button.height);
		}
	}

	XSetWindowBackgroundPixmap(getXDisplay(), switchwin, pixmap.button);
	if (! resource->compact) {
		XSetWindowBackgroundPixmap(getXDisplay(), txwin, pixmap.button);
		XSetWindowBackgroundPixmap(getXDisplay(), rxwin, pixmap.button);
	}

	if (!reconfigure) {
		gcv.font = resource->label.font->fid;
		gcv.foreground = resource->label.textColor.getPixel();
		frameGC = XCreateGC(getXDisplay(), framewin,GCFont|GCForeground, &gcv);

		gcv.foreground = resource->button.switch_off_color.getPixel();
		switchGC = XCreateGC(getXDisplay(), switchwin,GCForeground, &gcv);

		gcv.foreground = resource->button.tx_off_color.getPixel();
		if (! resource->compact) {
			txGC = XCreateGC(getXDisplay(), txwin,GCForeground, &gcv);

			gcv.foreground = resource->button.rx_off_color.getPixel();
			rxGC = XCreateGC(getXDisplay(), rxwin,GCForeground, &gcv);
		} else {
			txGC = XCreateGC(getXDisplay(), switchwin,GCForeground, &gcv);

			gcv.foreground = resource->button.rx_off_color.getPixel();
			rxGC = XCreateGC(getXDisplay(), switchwin,GCForeground, &gcv);
		}
	} else {
		gcv.font = resource->label.font->fid;
		gcv.foreground = resource->label.textColor.getPixel();
		XChangeGC(getXDisplay(), frameGC,GCFont|GCForeground, &gcv);

		gcv.foreground = resource->button.switch_off_color.getPixel();
		XChangeGC(getXDisplay(), switchGC,GCForeground, &gcv);

		gcv.foreground = resource->button.tx_off_color.getPixel();
		XChangeGC(getXDisplay(), txGC,GCForeground, &gcv);

		gcv.foreground = resource->button.rx_off_color.getPixel();
		XChangeGC(getXDisplay(), rxGC,GCForeground, &gcv);
	}

	if (!reconfigure) {
		XClearWindow(getXDisplay(), framewin);
		XMapWindow(getXDisplay(), framewin);
		XMapSubwindows(getXDisplay(), framewin);
		if (!options->decorated&&!options->withdrawn)
			XMoveResizeWindow(getXDisplay(),framewin,frame.x,frame.y, frame.width,
			                  frame.height);

	}

} // end MakeWindow

void ToolWindow::Redraw()
{
	char time[12];

	XClearWindow(getXDisplay(),switchwin);
	if (!resource->compact) {
		
		XClearWindow(getXDisplay(),txwin);
		XClearWindow(getXDisplay(),rxwin);


		XFillRectangle(getXDisplay(),switchwin,switchGC,resource->button.bevelWidth,
		               resource->button.bevelWidth,
		               button.width-2*resource->button.bevelWidth,
		               button.height-2*resource->button.bevelWidth);
		XFillRectangle(getXDisplay(),txwin,txGC,resource->button.bevelWidth,
		               resource->button.bevelWidth,
		               button.width-2*resource->button.bevelWidth,
		               button.height-2*resource->button.bevelWidth);
		XFillRectangle(getXDisplay(),rxwin,rxGC,resource->button.bevelWidth,
		               resource->button.bevelWidth,
		               button.width-2*resource->button.bevelWidth,
		               button.height-2*resource->button.bevelWidth);
	} else {
		XFillRectangle(getXDisplay(),switchwin,switchGC,switchlight.x,
		               switchlight.y,
		               switchlight.width,switchlight.height);
		XFillRectangle(getXDisplay(),switchwin,txGC,txlight.x,txlight.y,
		               txlight.width,txlight.height);

		XFillRectangle(getXDisplay(),switchwin,rxGC,rxlight.x,rxlight.y,
		               rxlight.width,rxlight.height);
	}


	if (resource->show.upTimeLabel) {
		if (resource->show.showsecs)
			sprintf(time, " %02d:%02d:%02d ", hour, minute, sec);
		else
			sprintf(time, " %02d:%02d ", hour, minute);
		XClearWindow(getXDisplay(), pppUpLabelwin);
		XDrawString(getXDisplay(), pppUpLabelwin, frameGC, resource->frame.bevelWidth,
		            (pppUpLabel.height+resource->label.font->ascent-
		             resource->label.font->descent) / 2,
		            time, strlen(time));
	}
	if (resource->show.thruputLabel) {
		char tpRatestr[7];
		sprintf(tpRatestr, " %2.1fk ", tpRate);

		XClearWindow(getXDisplay(), thruputLabelwin);
		XDrawString(getXDisplay(), thruputLabelwin, frameGC, resource->frame.bevelWidth,
		            (thruputLabel.height+resource->label.font->ascent-
		             resource->label.font->descent) / 2,
		            tpRatestr, strlen(tpRatestr));
	}
}



void ToolWindow::process_event(XEvent *Event)
{
	/* process events */
	switch (Event->type) {
	case ClientMessage:
		{
			if ((unsigned)Event->xclient.data.l[0]==wm_delete_window) shutdown();
		}
		break;

	case Expose:
		{
			//      CheckPPP(True);
			Redraw();
		}
		break;

	case ButtonPress:
		{
			if (Event->xbutton.button == LEFT_BUTTON) {
				if (Event->xbutton.window == pppUpLabelwin) {
					if (!(raised)) {
						XRaiseWindow(getXDisplay(),framewin);
						raised=True;
					}
				} else if (Event->xbutton.window == switchwin) {
					XSetWindowBackgroundPixmap(getXDisplay(), switchwin,
					                           pixmap.button_pressed);

					XClearWindow(getXDisplay(),switchwin);
					Redraw();
				}
			} else if (Event->xbutton.button == MIDDLE_BUTTON) {
				if (raised) {
					XLowerWindow(getXDisplay(),framewin);
					raised=False;
				}
			}
		}
		break;

	case ButtonRelease:
		{
			if (Event->xbutton.button == LEFT_BUTTON) {
				if (Event->xbutton.window==switchwin) {
					XSetWindowBackgroundPixmap(getXDisplay(), switchwin,
					                           pixmap.button);
					XClearWindow(getXDisplay(),switchwin);
					Redraw();
					if (Event->xbutton.x > 0 && Event->xbutton.x < button.width &&
					        Event->xbutton.y >0 && Event->xbutton.y < button.height ) {
						if (pppstatus->getConnected()) {
							/* quit connection */
							pppstatus->stopPPP();
							ppp_up=False;
						} else {
							pppstatus->startPPP();
							ppp_up=True;
						}
					}
				}
			}
		}
		break;

	case ConfigureNotify:
		{
			if (Event->xconfigure.window==framewin &&
			        Event->xconfigure.send_event) {
				if (options->withdrawn) reconfigure();
				int parent_x,parent_y;
				Window parent_root;
				unsigned int parent_width;
				unsigned int parent_height;
				unsigned int parent_border_width;
				unsigned int parent_depth;
				if (options->withdrawn) {
					XGetGeometry(getXDisplay(),Event->xconfigure.above,&parent_root,
					             &parent_x,&parent_y,&parent_width,&parent_height,
					             &parent_border_width,&parent_depth);
					frame.x=Event->xconfigure.x+parent_x;
					frame.y=Event->xconfigure.y+parent_y;
				} else {
					frame.x=Event->xconfigure.x;
					frame.y=Event->xconfigure.y;
					if (options->decorated) {
						if (options->geometry==NULL) options->geometry = new char [13];
						sprintf(options->geometry,"+%i+%i",frame.x,frame.y);
					}
				}
			}
		}
	}
}
