import { IChangedArgs } from '@jupyterlab/coreutils';
import { ISettingRegistry } from '@jupyterlab/settingregistry';
import { ITranslator } from '@jupyterlab/translation';
import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
import { Widget } from '@lumino/widgets';
import { ISplashScreen } from './splash';
import { IThemeManager } from './tokens';
/**
 * A class that provides theme management.
 */
export declare class ThemeManager implements IThemeManager {
    /**
     * Construct a new theme manager.
     */
    constructor(options: ThemeManager.IOptions);
    /**
     * Get the name of the current theme.
     */
    get theme(): string | null;
    /**
     * The names of the registered themes.
     */
    get themes(): ReadonlyArray<string>;
    /**
     * A signal fired when the application theme changes.
     */
    get themeChanged(): ISignal<this, IChangedArgs<string, string | null>>;
    /**
     * Get the value of a CSS variable from its key.
     *
     * @param key - A Jupyterlab CSS variable, without the leading '--jp-'.
     *
     * @return value - The current value of the Jupyterlab CSS variable
     */
    getCSS(key: string): string;
    /**
     * Load a theme CSS file by path.
     *
     * @param path - The path of the file to load.
     */
    loadCSS(path: string): Promise<void>;
    /**
     * Loads all current CSS overrides from settings. If an override has been
     * removed or is invalid, this function unloads it instead.
     */
    loadCSSOverrides(): void;
    /**
     * Validate a CSS value w.r.t. a key
     *
     * @param key - A Jupyterlab CSS variable, without the leading '--jp-'.
     *
     * @param val - A candidate CSS value
     */
    validateCSS(key: string, val: string): boolean;
    /**
     * Register a theme with the theme manager.
     *
     * @param theme - The theme to register.
     *
     * @returns A disposable that can be used to unregister the theme.
     */
    register(theme: IThemeManager.ITheme): IDisposable;
    /**
     * Add a CSS override to the settings.
     */
    setCSSOverride(key: string, value: string): Promise<void>;
    /**
     * Set the current theme.
     */
    setTheme(name: string): Promise<void>;
    /**
     * Test whether a given theme is light.
     */
    isLight(name: string): boolean;
    /**
     * Increase a font size w.r.t. its current setting or its value in the
     * current theme.
     *
     * @param key - A Jupyterlab font size CSS variable, without the leading '--jp-'.
     */
    incrFontSize(key: string): Promise<void>;
    /**
     * Decrease a font size w.r.t. its current setting or its value in the
     * current theme.
     *
     * @param key - A Jupyterlab font size CSS variable, without the leading '--jp-'.
     */
    decrFontSize(key: string): Promise<void>;
    /**
     * Test whether a given theme styles scrollbars,
     * and if the user has scrollbar styling enabled.
     */
    themeScrollbars(name: string): boolean;
    /**
     * Test if the user has scrollbar styling enabled.
     */
    isToggledThemeScrollbars(): boolean;
    /**
     * Toggle the `theme-scrollbars` setting.
     */
    toggleThemeScrollbars(): Promise<void>;
    /**
     * Get the display name of the theme.
     */
    getDisplayName(name: string): string;
    /**
     * Change a font size by a positive or negative increment.
     */
    private _incrFontSize;
    /**
     * Initialize the key -> property dict for the overrides
     */
    private _initOverrideProps;
    /**
     * Handle the current settings.
     */
    private _loadSettings;
    /**
     * Load the theme.
     *
     * #### Notes
     * This method assumes that the `theme` exists.
     */
    private _loadTheme;
    /**
     * Handle a theme error.
     */
    private _onError;
    protected translator: ITranslator;
    private _trans;
    private _base;
    private _current;
    private _host;
    private _links;
    private _overrides;
    private _overrideProps;
    private _outstanding;
    private _pending;
    private _requests;
    private _settings;
    private _splash;
    private _themes;
    private _themeChanged;
}
export declare namespace ThemeManager {
    /**
     * The options used to create a theme manager.
     */
    interface IOptions {
        /**
         * The host widget for the theme manager.
         */
        host: Widget;
        /**
         * The setting registry key that holds theme setting data.
         */
        key: string;
        /**
         * The settings registry.
         */
        settings: ISettingRegistry;
        /**
         * The splash screen to show when loading themes.
         */
        splash?: ISplashScreen;
        /**
         * The url for local theme loading.
         */
        url: string;
        /**
         * The application language translator.
         */
        translator?: ITranslator;
    }
}
