// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
/**
 * @packageDocumentation
 * @module nbformat
 */
// Notebook format interfaces
// https://nbformat.readthedocs.io/en/latest/format_description.html
// https://github.com/jupyter/nbformat/blob/master/nbformat/v4/nbformat.v4.schema.json
import { JSONExt } from '@lumino/coreutils';
/**
 * The earliest major version of the notebook format we support.
 */
export const MAJOR_VERSION = 4;
/**
 * The earliest minor version of the notebook format we support.
 */
export const MINOR_VERSION = 4;
/**
 * Validate a mime type/value pair.
 *
 * @param type - The mimetype name.
 *
 * @param value - The value associated with the type.
 *
 * @returns Whether the type/value pair are valid.
 */
export function validateMimeValue(type, value) {
    // Check if "application/json" or "application/foo+json"
    const jsonTest = /^application\/(.*?)+\+json$/;
    const isJSONType = type === 'application/json' || jsonTest.test(type);
    const isString = (x) => {
        return Object.prototype.toString.call(x) === '[object String]';
    };
    // If it is an array, make sure if is not a JSON type and it is an
    // array of strings.
    if (Array.isArray(value)) {
        if (isJSONType) {
            return false;
        }
        let valid = true;
        value.forEach(v => {
            if (!isString(v)) {
                valid = false;
            }
        });
        return valid;
    }
    // If it is a string, make sure we are not a JSON type.
    if (isString(value)) {
        return !isJSONType;
    }
    // It is not a string, make sure it is a JSON type.
    if (!isJSONType) {
        return false;
    }
    // It is a JSON type, make sure it is a valid JSON object.
    return JSONExt.isObject(value);
}
/**
 * Test whether a cell is a raw cell.
 */
export function isRaw(cell) {
    return cell.cell_type === 'raw';
}
/**
 * Test whether a cell is a markdown cell.
 */
export function isMarkdown(cell) {
    return cell.cell_type === 'markdown';
}
/**
 * Test whether a cell is a code cell.
 */
export function isCode(cell) {
    return cell.cell_type === 'code';
}
/**
 * Test whether an output is an execute result.
 */
export function isExecuteResult(output) {
    return output.output_type === 'execute_result';
}
/**
 * Test whether an output is from display data.
 */
export function isDisplayData(output) {
    return output.output_type === 'display_data';
}
/**
 * Test whether an output is from updated display data.
 */
export function isDisplayUpdate(output) {
    return output.output_type === 'update_display_data';
}
/**
 * Test whether an output is from a stream.
 */
export function isStream(output) {
    return output.output_type === 'stream';
}
/**
 * Test whether an output is an error.
 */
export function isError(output) {
    return output.output_type === 'error';
}
//# sourceMappingURL=index.js.map