import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
import { IObservable } from './modeldb';
/**
 * A map which can be observed for changes.
 */
export interface IObservableMap<T> extends IDisposable, IObservable {
    /**
     * The type of the Observable.
     */
    type: 'Map';
    /**
     * A signal emitted when the map has changed.
     */
    readonly changed: ISignal<this, IObservableMap.IChangedArgs<T>>;
    /**
     * The number of key-value pairs in the map.
     */
    readonly size: number;
    /**
     * Set a key-value pair in the map
     *
     * @param key - The key to set.
     *
     * @param value - The value for the key.
     *
     * @returns the old value for the key, or undefined
     *   if that did not exist.
     */
    set(key: string, value: T): T | undefined;
    /**
     * Get a value for a given key.
     *
     * @param key - the key.
     *
     * @returns the value for that key.
     */
    get(key: string): T | undefined;
    /**
     * Check whether the map has a key.
     *
     * @param key - the key to check.
     *
     * @returns `true` if the map has the key, `false` otherwise.
     */
    has(key: string): boolean;
    /**
     * Get a list of the keys in the map.
     *
     * @returns - a list of keys.
     */
    keys(): string[];
    /**
     * Get a list of the values in the map.
     *
     * @returns - a list of values.
     */
    values(): T[];
    /**
     * Remove a key from the map
     *
     * @param key - the key to remove.
     *
     * @returns the value of the given key,
     *   or undefined if that does not exist.
     */
    delete(key: string): T | undefined;
    /**
     * Set the ObservableMap to an empty map.
     */
    clear(): void;
    /**
     * Dispose of the resources held by the map.
     */
    dispose(): void;
}
/**
 * The interfaces associated with an IObservableMap.
 */
export declare namespace IObservableMap {
    /**
     * The change types which occur on an observable map.
     */
    type ChangeType = 
    /**
     * An entry was added.
     */
    'add'
    /**
     * An entry was removed.
     */
     | 'remove'
    /**
     * An entry was changed.
     */
     | 'change';
    /**
     * The changed args object which is emitted by an observable map.
     */
    interface IChangedArgs<T> {
        /**
         * The type of change undergone by the map.
         */
        type: ChangeType;
        /**
         * The key of the change.
         */
        key: string;
        /**
         * The old value of the change.
         */
        oldValue: T | undefined;
        /**
         * The new value of the change.
         */
        newValue: T | undefined;
    }
}
/**
 * A concrete implementation of IObservableMap<T>.
 */
export declare class ObservableMap<T> implements IObservableMap<T> {
    /**
     * Construct a new observable map.
     */
    constructor(options?: ObservableMap.IOptions<T>);
    /**
     * The type of the Observable.
     */
    get type(): 'Map';
    /**
     * A signal emitted when the map has changed.
     */
    get changed(): ISignal<this, IObservableMap.IChangedArgs<T>>;
    /**
     * Whether this map has been disposed.
     */
    get isDisposed(): boolean;
    /**
     * The number of key-value pairs in the map.
     */
    get size(): number;
    /**
     * Set a key-value pair in the map
     *
     * @param key - The key to set.
     *
     * @param value - The value for the key.
     *
     * @returns the old value for the key, or undefined
     *   if that did not exist.
     *
     * @throws if the new value is undefined.
     *
     * #### Notes
     * This is a no-op if the value does not change.
     */
    set(key: string, value: T): T | undefined;
    /**
     * Get a value for a given key.
     *
     * @param key - the key.
     *
     * @returns the value for that key.
     */
    get(key: string): T | undefined;
    /**
     * Check whether the map has a key.
     *
     * @param key - the key to check.
     *
     * @returns `true` if the map has the key, `false` otherwise.
     */
    has(key: string): boolean;
    /**
     * Get a list of the keys in the map.
     *
     * @returns - a list of keys.
     */
    keys(): string[];
    /**
     * Get a list of the values in the map.
     *
     * @returns - a list of values.
     */
    values(): T[];
    /**
     * Remove a key from the map
     *
     * @param key - the key to remove.
     *
     * @returns the value of the given key,
     *   or undefined if that does not exist.
     *
     * #### Notes
     * This is a no-op if the value does not change.
     */
    delete(key: string): T | undefined;
    /**
     * Set the ObservableMap to an empty map.
     */
    clear(): void;
    /**
     * Dispose of the resources held by the map.
     */
    dispose(): void;
    private _map;
    private _itemCmp;
    private _changed;
    private _isDisposed;
}
/**
 * The namespace for `ObservableMap` class statics.
 */
export declare namespace ObservableMap {
    /**
     * The options used to initialize an observable map.
     */
    interface IOptions<T> {
        /**
         * An optional initial set of values.
         */
        values?: {
            [key: string]: T;
        };
        /**
         * The item comparison function for change detection on `set`.
         *
         * If not given, strict `===` equality will be used.
         */
        itemCmp?: (first: T, second: T) => boolean;
    }
}
