/*	Copyright  2001 The Regents of the University of California.
*	All Rights Reserved.
*	Permission to use, copy, modify, and distribute this software and its
*	documentation for educational, research and non-profit purposes, without
*	fee, and without a written agreement is hereby granted, provided that the
*	above copyright notice, this paragraph and the following three paragraphs
*	appear in all copies.
*	Permission to incorporate this software into commercial products may
*	be obtained by contacting the University of California: 
*	Bill Hoskins
*	Office of Technology Licensing
*	2150 Shattuck Avenue #150
* 	Berkeley, CA 94720-1620
*	(510) 643-7201
*	bhoskins@uclink2.berkeley.edu
*	
*
*	This software program and documentation are copyrighted by The Regents
*	of the University of California. The software program and documentation
*	are supplied "as is", without any accompanying services from
*	The Regents. The Regents does not warrant that the operation of the program
*	will be uninterrupted or error-free. The end-user understands that the
*	program was developed for research purposes and is advised not to rely
*	exclusively on the program for any reason.
*	IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
*	FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
*	LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION,
*	EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
*	OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY
*	WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
*	AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
*	ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO
*	OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
*	This software is based on Mark Fullmer's flow-tools 
*	http://www.splintered.net/sw/flow-tools/
*	
*      $Id: acl-filter.c,v 1.16 2001/10/03 19:12:14 robert Exp $
*/

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include "acl-filter.h"

int debug;
int ip_net_only;



struct acl_list acl_list;

const int FTIO_OUT_MAX = 128;

int yyparse (void);
	

void init_flow_input(struct ftio* in, struct ftver* version)
{
    
 
  /* input from stdin */
  if (ftio_init(in, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");
  

  if (ftio_check_generic(in) < 0)
    fterr_errx(1, "flow-filter does not yet support PDU format");
  
  ftio_get_ver(in, version);
  version->s_version = FT_IO_SVERSION;
  
}




void init_acls(char** acl_names, struct acl_info** acls, int* list_sizes, struct ftio* ftstdout) 
{

  extern struct acl_list acl_list;
  char* token = NULL;
  char* name = NULL;
  char* delim = ", ";
  struct acl_info* info = NULL;
  int i, size;
  
  
  for (i = 0; i < LAST_ACL; i++) {
    name = acl_names[i];

    if (name != NULL ) {
      token = strtok(name, delim);

      if (!token) token = name;
      
      size = list_sizes[i];
      
      while(token != NULL) {
	if (check_dup(size,info,token))
	  continue;
	info = realloc(info, sizeof(struct acl_info) * (size + 1));
	bzero(&info[size], sizeof(struct acl_info));
	info[size].main_index = acl_find(acl_list, token);
	if (info[size].main_index == -1)
	  fterr_errx(1, "Couldn't find list %s\n", token);
	info[size].sub_index = acl_list.names[info[size].main_index].num;
	info[size].name = strdup(token);
	info[size].bucket = ftstdout;
	info[size].num_written = 0;
	++size;
	token = NULL;
	token = strtok(NULL,delim);
      }
      list_sizes[i] = size;
      acls[i] = info;
      info = NULL;
    }
  }
    
}


struct ftio* init_output(struct ftset* ftset, struct ftver* ftversion, int fd)
{
  
  struct ftio* out = malloc(sizeof(struct ftio));

    if (ftio_init(out, fd, FT_IO_FLAG_WRITE |
		  ((ftset->z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
      fterr_errx(1, "ftio_init(): failed");

    ftio_set_comment(out, ftset->comments);
    ftio_set_byte_order(out, ftset->byte_order);
    ftio_set_z_level(out, ftset->z_level);
    ftio_set_streaming(out, 1);
    ftio_set_debug(out, debug);
  
    if (ftio_set_ver(out, ftversion) < 0)
      fterr_errx(1, "ftio_set_ver(): failed");

    return out;
  

}






void init_split_output(struct ftset* ftset, struct ftver* ftver,
		       struct acl_info** acls, int* sizes)
{
  
  //struct ftio* template;
  struct ftio* out;
  char name[FILENAME_MAX];
  struct acl_info* current;
  int FILE_FLAG = (O_RDWR | O_CREAT);
  int USER_PERM = S_IWUSR | S_IRUSR ;
  int fd[FTIO_OUT_MAX];
  int num_fd, dup_fd;
  int i,j;
  char* bucket_suffix[]= {"_src","_dst"};
  num_fd = 0;
  dup_fd = -1;
  for(i =0; i < LAST_ACL; i++) {
    current = acls[i];
    if (current) {
      for(j = 0; j < sizes[i]; j++ ) {
	strcpy(name, current[j].name);
	strcat(name, bucket_suffix[i%2]);
	fd[num_fd] = open(name, FILE_FLAG, USER_PERM);
	out = init_output(ftset, ftver,fd[num_fd]);
	write_header(out);
	num_fd++;
	current[j].bucket = out;
      }
    }
  }
	
}


FILE*  open_log_file()
{

  char* mode = "w";
  char template[] = "acl-filter.log";
  FILE* log;

  if((log = fopen(template, mode)) > 0){
    fterr_setfile(1,log);
    return log;
  }

  return NULL;
}





/* check for duplicates in acl names: 1 if so, 0 if not */
int  check_dup(int len, struct acl_info* acls, char* name ) 
{

   
  int i; 

  for (i = 0; i < len; i++) {
    if (!strcmp(name, acls[i].name)){
      return 1;
    }
  }
  return 0;
}
      



void close_buckets(struct acl_info **acls, int* sizes) 
{

  struct acl_info* current;
  int i,j;

  for(i = 0; i < LAST_ACL; i++) {
    if (acls[i]) {
      current = acls[i];
      for(j=0;j < sizes[i];j++){
	if (ftio_close(current[j].bucket) < 0){
	  fterr_errx(1, "ftio_close(): failed");
	}
      }
    }
  }
}	


void run_filters(struct ftio* ftin, struct ftio* ftstdout, 
		 struct acl_info** acls, int* sizes, int debug_filters)
{

  
  struct fts3rec_v5 *rec_v5;
  struct fts3rec_gen *rec_gen;
  void* rec;
  
  struct acl_info* acl; 
  int ret;
  int i,j;
  u_int64 num_flows = 0;
  u_int64 num_discarded = 0;
  char* bucket_names[] = {"src", "dst"};
  
  /* grab 1 flow */
  while ((rec = ftio_read(ftin))) {
    
        
    rec_v5 = rec;
    rec_gen = rec;
    num_flows++;

    //filter src and dst acls
    if (acls[SRC_STD] || acls[DST_STD]) { //standard acls

      /*iterate through acl source names */
      if (acls[SRC_STD]) {
	
	acl = acls[SRC_STD];
	ret = filter_std(acl, sizes[SRC_STD], rec_gen->srcaddr);
	
	if (ret >= 0) {
	  pass_flow(&acl[ret], rec);
	  continue;
	}
      }

	if(acls[DST_STD]) {
	
	  acl =  acls[DST_STD];
	  ret = filter_std(acl, sizes[DST_STD], rec_gen->dstaddr);
	
	  if (ret >= 0) {
	    pass_flow(&acl[ret], rec);
	    continue;
	  }
	}
	
	num_discarded++;
	continue; 
    }

    
    if(acls[SRC_AS] || acls[DST_AS]) { //as-path acls
      
      if(acls[SRC_AS]) {
	if (debug_filters > 1) 
	  fterr_info("Examining src as \n");
	acl = acls[SRC_AS];
	ret = filter_as(acl,sizes[SRC_AS],rec_v5->srcaddr, rec_v5->src_mask,debug_filters);
	if (ret >= 0) {
	  pass_flow(&acl[ret], rec);
	 continue;
	}
      }

      if(acls[DST_AS]) {
	if (debug_filters > 1)
	  fterr_info("Examining dst as \n");
	acl = acls[DST_AS];
	ret = filter_as(acl,sizes[DST_AS],rec_v5->dstaddr, rec_v5->dst_mask, debug_filters);
	if (ret >= 0) {
	  pass_flow(&acl[ret], rec);
	 continue;
	}
      }
      num_discarded++;
      continue;
    }


    if(acls[SRC_COMM] || acls[DST_COMM]) { //community acls
       
      if(acls[SRC_COMM]) {
	if (debug_filters > 1) 
	  fterr_info("Examining src community string \n");
	
	acl = acls[SRC_COMM];
	ret = filter_community(acl,sizes[SRC_COMM],rec_v5->srcaddr, rec_v5->src_mask, debug_filters);
	if (ret >= 0) {
	 pass_flow(&acl[ret], rec);
	 continue;
	}
      }

      if(acls[DST_COMM]) {
	if (debug_filters > 1) 
	  fterr_info("Examining dest  community string \n");
	acl = acls[DST_COMM];
	ret = filter_community(acl,sizes[DST_COMM],rec_v5->dstaddr, rec_v5->dst_mask, debug_filters);
	if (ret >= 0) {
	pass_flow(&acl[ret], rec); 
	 continue;
	}
      }
      num_discarded++;
      continue;
    }
	

  }

  
  for(i =0; i < LAST_ACL; i++) {
    acl = acls[i];
    for (j=0; j < sizes[i]; j++) {
      fterr_info("Matched %qu flows to %s acl %s ", acl[j].num_written, bucket_names[i%2],
		 acl[j].name);
    }
  }
  fterr_info("processed %qu flows discarded %qu flows\n", num_flows, num_discarded);
}



void pass_flow(struct acl_info* acl, void* rec) {
  
  write_flow(acl->bucket,rec);
  ++ acl->num_written ;
}


int filter_std(struct acl_info* acl, int len, u_int32 ip) 
{

  extern struct acl_list acl_list;
  int k, ret;
  char addr[32];
  fmt_ipv4(addr, ip, FMT_JUST_LEFT);

  for(k = 0; k  < len; k++){
    
    ret = acl_eval_std(acl_list, acl[k].sub_index,ip);
    if (ret == 0) {
      return k;
    }
  }
  return -1;
}


int filter_as(struct acl_info* acl, int len, u_int32 prefix,
	      u_int8 mask_bitlen, int debug)
{


  extern struct acl_list acl_list;
  char* aspath = NULL;
  char addr[32];
  char mask[8];
  int fmt = FMT_JUST_LEFT;
  int i, ret;
  ret = -1;

  if (debug > 1) {
      fmt_ipv4(addr, prefix,fmt);
      fmt_uint8(mask, mask_bitlen,fmt);
  }

  if ((aspath = lookup_aspath(prefix, mask_bitlen))) {
    if (debug > 1) {
        fterr_info("Prefix %s/%s as-path %s \n", addr,mask,aspath);
    }
    for(i = 0; i < len; i++){
      if (!(acl_eval_as_path(acl_list, acl[i].sub_index,aspath, debug))) {
	ret = i;
	break;
      }
    }
  } else {
    if (debug > 1) {
      fterr_info("Prefix %s/%s as-path FAILED \n", addr,mask);
    }
  }
  free(aspath);
  return ret;
  
}


int filter_community(struct acl_info* acl, int len, u_int32 prefix,u_int8 mask_bitlen, int debug)
{


  extern struct acl_list acl_list;
  char* community;
  
  char addr[32];
  char mask[8];
  int fmt = FMT_JUST_LEFT;
  int i, ret;
  
  ret = -1;
  
  
  if (debug > 1) {
      fmt_ipv4(addr, prefix,fmt);
      fmt_uint8(mask, mask_bitlen,fmt);
  }

  if ((community = lookup_community(prefix, mask_bitlen))) {
    if (debug > 1) {
      fterr_info("Prefix %s/%s community-string %s \n", addr,mask,community);
    }

    for(i = 0; i < len; i++){
      if (!(acl_eval_community(acl_list, acl[i].sub_index,community, debug))){
	ret = i;
	break;
      }
    }
  } else {
    if (debug > 1) {
      fterr_info("Prefix %s/%s community-string FAILED \n", addr,mask);
    }
  }
  
  free (community);
  return ret;
  
}


void write_header(struct ftio* out)
{
  if (ftio_write_header(out) < 0)
    fterr_errx(1, "ftio_write_header(): failed");
}

/* write a single flow to an output stream */
/* exits on error */
void write_flow(struct ftio* out, void* flow) 
{

  if (ftio_write(out,flow) < 0)
    fterr_errx(1, "ftio_write(): failed");
  
}


void usage () {
  
  fprintf(stdout, "acl-filter -f acl-file [-r bgpdump] [-s acl -S acl] |[ -a acl -A acl] |[ -c acl -C acl] [-O] [-d[1|2]]
-s -S : use standard acl(s) to filter on source (-s) destination (-S)

-a -A : use as-path acl(s) to filter on source (-a) destination (-A)

-c -C : use community acl(s) to filter on source (-c) destination (-C)

-O      dump matching flows in files named after matching acl
        (acl buckets) flows matching on source are written to acl_src,
        and flows matching on destination are written to acl_dst.

-d      produces debugging output to acl-filter.log 1 or 2 specifies
        the level of verbosity. 

Lists of acls enclosed in ' '
A bgpdump file must be specified for as-path and community filtering\n");

  exit(0);

}


void yyerror(const char *msg)
{
  extern char *yytext;
  fterr_warnx("%s at '%s'\n", msg, yytext);
}



void check_acls(struct acl_info **acls, int* sizes) 
{
  
  int i,j;
  struct acl_info *current = NULL;
  for(i=0; i < LAST_ACL; i++) {
    if (acls[i]) {
      current = acls[i];
      for (j =0; j < sizes[i]; j++){
	fterr_info("ACL [%i][%i] : index %i sub index %i, name %s, bucket %p \n",
		i,j,current[j].main_index, current[j].sub_index, current[j].name, 
		current[j].bucket);
      }
    }
  }
}




int main(int argc, char **argv)
{
  
 extern char *optarg; //command line arguments
 extern FILE *yyin; 
 extern struct acl_list acl_list;
 
 /*acl options */
 char* acl_fname;
 char* acl_names[LAST_ACL];
 struct acl_info* acls[LAST_ACL];
 int acl_list_sizes[LAST_ACL];
 
 /*bgpdump file options */
 char* rib_file = NULL;

 /* output options */
 struct ftio in;
 struct ftio* out; 
 struct ftset ftset;
 struct ftver ftversion;
 struct ftprof ftp;
 int split_files = 0;
 int debug_filters = 0;
 FILE* log_file;

 /* misc */
 int i;
 
 
 bzero(acl_names, LAST_ACL* (sizeof (char*)));
 bzero(&acl_list, sizeof(struct acl_list));
 bzero(acl_list_sizes,LAST_ACL *(sizeof(int)));
 bzero(acls, LAST_ACL * (sizeof (struct acl_info*)));
 
 fterr_setid(argv[0]);
 
 /* profile */
  ftprof_start (&ftp);

 ftset_init(&ftset, 0);
 
 while ((i = getopt(argc, argv, "f:r:s:S:a:A:c:C:b:m:z:d:O")) != -1){
   switch (i) {
   case 'f': /* acl file name */
     acl_fname = optarg;
     break;
   case 'r' :
     rib_file = optarg;
     break;
   case 's':
     acl_names[SRC_STD] = optarg;
     break;
   case 'S':
     acl_names[DST_STD] = optarg;
     break;
   case 'a':
     acl_names[SRC_AS] = optarg;
     break;
   case 'A':
     acl_names[DST_AS]  = optarg;
     break;
   case 'c':
     acl_names[SRC_COMM] = optarg;
     break;
   case 'C':
     acl_names[DST_COMM] = optarg;
     break;
   case 'd':
     debug_filters = atoi(optarg);
     if (debug_filters < 1 || debug_filters > 3){
       usage();
     }
     log_file = open_log_file();
     if (!log_file) fterr_errx(1, "Failed to open log file \n");
     break;
   case 'b': /* output byte order */
     if (!strcasecmp(optarg, "little"))
       ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
     else if (!strcasecmp(optarg, "big"))
       ftset.byte_order = FT_HEADER_BIG_ENDIAN;
     else
       fterr_errx(1, "expecting \"big\" or \"little\"");
     break;
   case 'm': /* comment field */
     ftset.comments = optarg;
     break;
   case 'z': /* compress level */
     ftset.z_level = atoi(optarg);
     if ((ftset.z_level < 0) || (ftset.z_level > 9))
       fterr_errx(1, "Compression level must be between 0 and 9");
     break;
   case 'O': /* split output by acl match */
     split_files = 1;
     break; 
   case 'h': /* help */
   case '?':
     usage();
     exit (0);
     break;
   default:
     usage();
     exit (1);
     break;

   } /* switch */

  
 
 }
 

/*
   * load acl's
   * XXX add fname check and close
   */
  bzero(&acl_list, sizeof(acl_list));
  if ((yyin = fopen(acl_fname ? acl_fname : "flow.acl", "r"))){
    while (!feof(yyin))
      yyparse();
  }


 init_flow_input(&in, &ftversion);
  out = init_output(&ftset,&ftversion,1); // initiate std output 
 init_acls(acl_names, acls, acl_list_sizes, out);
 
 if (rib_file) {
   load_file(rib_file);
 }

 if (debug_filters)
   check_acls(acls, acl_list_sizes);

 if (split_files) {
   if (debug_filters) fterr_info("Initing split output \n");
   init_split_output(&ftset, &ftversion, acls, acl_list_sizes);
 }else {
   if (debug_filters) fterr_info("Initing output to stdout \n");
   write_header(out);
 }

 if (debug_filters) fterr_info("Running filters \n");
 run_filters(&in, out, acls, acl_list_sizes, debug_filters);

 if(split_files) {
   close_buckets(acls, acl_list_sizes);
   
 }else {
   ftio_close(out);
 }

 if(debug_filters) {
   if (log_file) fclose(log_file);
 }

 ftio_close(&in);

 return 0;

}/*main */
