/**
* Fresh Editor TypeScript Plugin API
*
* This file provides type definitions for the Fresh editor's TypeScript plugin system.
* Plugins have access to the global `editor` object which provides methods to:
* - Query editor state (buffers, cursors, viewports)
* - Modify buffer content (insert, delete text)
* - Add visual decorations (overlays, highlighting)
* - Interact with the editor UI (status messages, prompts)
*
* AUTO-GENERATED FILE - DO NOT EDIT MANUALLY
* Generated by fresh-plugin-api-macros + ts-rs from JsEditorApi impl
*/
/**
* Get the editor API instance.
* Plugins must call this at the top of their file to get a scoped editor object.
*/
declare function getEditor(): EditorAPI;
/** Handle for a cancellable async operation */
interface ProcessHandle<T> extends PromiseLike<T> {
	/** Promise that resolves to the result when complete */
	readonly result: Promise<T>;
	/** Cancel/kill the operation. Returns true if cancelled, false if already completed */
	kill(): Promise<boolean>;
}
/** Buffer identifier */
type BufferId = number;
/** Split identifier */
type SplitId = number;
type TextPropertyEntry = {
	/**
	* Text content for this entry
	*/
	text: string;
	/**
	* Optional properties attached to this text (e.g., file path, line number)
	*/
	properties?: Record<string, unknown>;
};
type TsCompositeLayoutConfig = {
	/**
	* Layout type: "side-by-side", "stacked", or "unified"
	*/
	type: string;
	/**
	* Width ratios for side-by-side (e.g., [0.5, 0.5])
	*/
	ratios: Array<number> | null;
	/**
	* Show separator between panes
	*/
	showSeparator: boolean;
	/**
	* Spacing for stacked layout
	*/
	spacing: number | null;
};
type TsCompositeSourceConfig = {
	/**
	* Buffer ID of the source buffer (required)
	*/
	bufferId: number;
	/**
	* Label for this pane (e.g., "OLD", "NEW")
	*/
	label: string;
	/**
	* Whether this pane is editable
	*/
	editable: boolean;
	/**
	* Style configuration
	*/
	style: TsCompositePaneStyle | null;
};
type TsCompositePaneStyle = {
	/**
	* Background color for added lines (RGB)
	* Using [u8; 3] instead of (u8, u8, u8) for better rquickjs_serde compatibility
	*/
	addBg: [number, number, number] | null;
	/**
	* Background color for removed lines (RGB)
	*/
	removeBg: [number, number, number] | null;
	/**
	* Background color for modified lines (RGB)
	*/
	modifyBg: [number, number, number] | null;
	/**
	* Gutter style: "line-numbers", "diff-markers", "both", or "none"
	*/
	gutterStyle: string | null;
};
type TsCompositeHunk = {
	/**
	* Starting line in old buffer (0-indexed)
	*/
	oldStart: number;
	/**
	* Number of lines in old buffer
	*/
	oldCount: number;
	/**
	* Starting line in new buffer (0-indexed)
	*/
	newStart: number;
	/**
	* Number of lines in new buffer
	*/
	newCount: number;
};
type TsCreateCompositeBufferOptions = {
	/**
	* Buffer name (displayed in tabs/title)
	*/
	name: string;
	/**
	* Mode for keybindings
	*/
	mode: string;
	/**
	* Layout configuration
	*/
	layout: TsCompositeLayoutConfig;
	/**
	* Source pane configurations
	*/
	sources: Array<TsCompositeSourceConfig>;
	/**
	* Diff hunks for alignment (optional)
	*/
	hunks: Array<TsCompositeHunk> | null;
};
type ViewportInfo = {
	/**
	* Byte position of the first visible line
	*/
	topByte: number;
	/**
	* Line number of the first visible line (None when line index unavailable, e.g. large file before scan)
	*/
	topLine: number | null;
	/**
	* Left column offset (horizontal scroll)
	*/
	leftColumn: number;
	/**
	* Viewport width
	*/
	width: number;
	/**
	* Viewport height
	*/
	height: number;
};
type LayoutHints = {
	/**
	* Optional compose width for centering/wrapping
	*/
	composeWidth: number | null;
	/**
	* Optional column guides for aligned tables
	*/
	columnGuides: Array<number> | null;
};
type ViewTokenWire = {
	/**
	* Source byte offset in the buffer. None for injected content (annotations).
	*/
	source_offset: number | null;
	/**
	* The token content
	*/
	kind: ViewTokenWireKind;
	/**
	* Optional styling for injected content (only used when source_offset is None)
	*/
	style?: ViewTokenStyle;
};
type ViewTokenWireKind = {
	"Text": string;
} | "Newline" | "Space" | "Break" | {
	"BinaryByte": number;
};
type ViewTokenStyle = {
	/**
	* Foreground color as RGB tuple
	*/
	fg: [number, number, number] | null;
	/**
	* Background color as RGB tuple
	*/
	bg: [number, number, number] | null;
	/**
	* Whether to render in bold
	*/
	bold: boolean;
	/**
	* Whether to render in italic
	*/
	italic: boolean;
};
type PromptSuggestion = {
	/**
	* The text to display
	*/
	text: string;
	/**
	* Optional description
	*/
	description?: string;
	/**
	* The value to use when selected (defaults to text if None)
	*/
	value?: string;
	/**
	* Whether this suggestion is disabled (greyed out, defaults to false)
	*/
	disabled?: boolean;
	/**
	* Optional keyboard shortcut
	*/
	keybinding?: string;
};
type DirEntry = {
	/**
	* File/directory name
	*/
	name: string;
	/**
	* True if this is a file
	*/
	is_file: boolean;
	/**
	* True if this is a directory
	*/
	is_dir: boolean;
};
type BufferInfo = {
	/**
	* Buffer ID
	*/
	id: number;
	/**
	* File path (if any)
	*/
	path: string;
	/**
	* Whether the buffer has been modified
	*/
	modified: boolean;
	/**
	* Length of buffer in bytes
	*/
	length: number;
	/**
	* Whether this is a virtual buffer (not backed by a file)
	*/
	is_virtual: boolean;
	/**
	* Current view mode of the active split: "source" or "compose"
	*/
	view_mode: string;
	/**
	* True if any split showing this buffer has compose mode enabled.
	* Plugins should use this (not `view_mode`) to decide whether to maintain
	* decorations, since decorations live on the buffer and are filtered
	* per-split at render time.
	*/
	is_composing_in_any_split: boolean;
	/**
	* Compose width (if set), from the active split's view state
	*/
	compose_width: number | null;
	/**
	* The detected language for this buffer (e.g., "rust", "markdown", "text")
	*/
	language: string;
};
type JsDiagnostic = {
	/**
	* Document URI
	*/
	uri: string;
	/**
	* Diagnostic message
	*/
	message: string;
	/**
	* Severity: 1=Error, 2=Warning, 3=Info, 4=Hint, null=unknown
	*/
	severity: number | null;
	/**
	* Range in the document
	*/
	range: JsRange;
	/**
	* Source of the diagnostic (e.g., "typescript", "eslint")
	*/
	source?: string;
};
type JsRange = {
	/**
	* Start position
	*/
	start: JsPosition;
	/**
	* End position
	*/
	end: JsPosition;
};
type JsPosition = {
	/**
	* Zero-indexed line number
	*/
	line: number;
	/**
	* Zero-indexed character offset
	*/
	character: number;
};
type ActionSpec = {
	/**
	* Action name (e.g., "move_word_right", "delete_line")
	*/
	action: string;
	/**
	* Number of times to repeat the action (default 1)
	*/
	count: number;
};
type TsActionPopupAction = {
	/**
	* Unique action identifier (returned in ActionPopupResult)
	*/
	id: string;
	/**
	* Display text for the button (can include command hints)
	*/
	label: string;
};
type ActionPopupOptions = {
	/**
	* Unique identifier for the popup (used in ActionPopupResult)
	*/
	id: string;
	/**
	* Title text for the popup
	*/
	title: string;
	/**
	* Body message (supports basic formatting)
	*/
	message: string;
	/**
	* Action buttons to display
	*/
	actions: Array<TsActionPopupAction>;
};
type FileExplorerDecoration = {
	/**
	* File path to decorate
	*/
	path: string;
	/**
	* Symbol to display (e.g., "●", "M", "A")
	*/
	symbol: string;
	/**
	* Color as RGB array (rquickjs_serde requires array, not tuple)
	*/
	color: [number, number, number];
	/**
	* Priority for display when multiple decorations exist (higher wins)
	*/
	priority: number;
};
type FormatterPackConfig = {
	/**
	* Command to run (e.g., "prettier", "rustfmt")
	*/
	command: string;
	/**
	* Arguments to pass to the formatter
	*/
	args: Array<string>;
};
type TerminalResult = {
	/**
	* The created buffer ID (for use with setSplitBuffer, etc.)
	*/
	bufferId: number;
	/**
	* The terminal ID (for use with sendTerminalInput, closeTerminal)
	*/
	terminalId: number;
	/**
	* The split ID (if created in a new split)
	*/
	splitId: number | null;
};
type CreateTerminalOptions = {
	/**
	* Working directory for the terminal (defaults to editor cwd)
	*/
	cwd?: string;
	/**
	* Split direction: "horizontal" or "vertical" (default: "vertical")
	*/
	direction?: string;
	/**
	* Split ratio 0.0-1.0 (default: 0.5)
	*/
	ratio?: number;
	/**
	* Whether to focus the new terminal split (default: true)
	*/
	focus?: boolean;
};
type CursorInfo = {
	/**
	* Byte position of the cursor
	*/
	position: number;
	/**
	* Selection range (if any)
	*/
	selection: {
		start: number;
		end: number;
	} | null;
};
type BackgroundProcessResult = {
	/**
	* Unique process ID for later reference
	*/
	process_id: number;
	/**
	* Process exit code (0 usually means success, -1 if killed)
	* Only present when the process has exited
	*/
	exit_code: number;
};
type BufferSavedDiff = {
	equal: boolean;
	byte_ranges: Array<[number, number]>;
	line_ranges: Array<[number, number]> | null;
};
type CreateVirtualBufferInExistingSplitOptions = {
	/**
	* Buffer name (displayed in tabs/title)
	*/
	name: string;
	/**
	* Target split ID (required)
	*/
	splitId: number;
	/**
	* Mode for keybindings (e.g., "git-log", "search-results")
	*/
	mode?: string;
	/**
	* Whether buffer is read-only (default: false)
	*/
	readOnly?: boolean;
	/**
	* Show line numbers in gutter (default: true)
	*/
	showLineNumbers?: boolean;
	/**
	* Show cursor (default: true)
	*/
	showCursors?: boolean;
	/**
	* Disable text editing (default: false)
	*/
	editingDisabled?: boolean;
	/**
	* Enable line wrapping
	*/
	lineWrap?: boolean;
	/**
	* Initial content entries with optional properties
	*/
	entries?: Array<TextPropertyEntry>;
};
type CreateVirtualBufferInSplitOptions = {
	/**
	* Buffer name (displayed in tabs/title)
	*/
	name: string;
	/**
	* Mode for keybindings (e.g., "git-log", "search-results")
	*/
	mode?: string;
	/**
	* Whether buffer is read-only (default: false)
	*/
	readOnly?: boolean;
	/**
	* Split ratio 0.0-1.0 (default: 0.5)
	*/
	ratio?: number;
	/**
	* Split direction: "horizontal" or "vertical"
	*/
	direction?: string;
	/**
	* Panel ID to split from
	*/
	panelId?: string;
	/**
	* Show line numbers in gutter (default: true)
	*/
	showLineNumbers?: boolean;
	/**
	* Show cursor (default: true)
	*/
	showCursors?: boolean;
	/**
	* Disable text editing (default: false)
	*/
	editingDisabled?: boolean;
	/**
	* Enable line wrapping
	*/
	lineWrap?: boolean;
	/**
	* Place the new buffer before (left/top of) the existing content (default: false)
	*/
	before?: boolean;
	/**
	* Initial content entries with optional properties
	*/
	entries?: Array<TextPropertyEntry>;
};
type CreateVirtualBufferOptions = {
	/**
	* Buffer name (displayed in tabs/title)
	*/
	name: string;
	/**
	* Mode for keybindings (e.g., "git-log", "search-results")
	*/
	mode?: string;
	/**
	* Whether buffer is read-only (default: false)
	*/
	readOnly?: boolean;
	/**
	* Show line numbers in gutter (default: false)
	*/
	showLineNumbers?: boolean;
	/**
	* Show cursor (default: true)
	*/
	showCursors?: boolean;
	/**
	* Disable text editing (default: false)
	*/
	editingDisabled?: boolean;
	/**
	* Hide from tab bar (default: false)
	*/
	hiddenFromTabs?: boolean;
	/**
	* Initial content entries with optional properties
	*/
	entries?: Array<TextPropertyEntry>;
};
type LanguagePackConfig = {
	/**
	* Comment prefix for line comments (e.g., "//" or "#")
	*/
	commentPrefix: string | null;
	/**
	* Block comment start marker (e.g., slash-star)
	*/
	blockCommentStart: string | null;
	/**
	* Block comment end marker (e.g., star-slash)
	*/
	blockCommentEnd: string | null;
	/**
	* Whether to use tabs instead of spaces for indentation
	*/
	useTabs: boolean | null;
	/**
	* Tab size (number of spaces per tab level)
	*/
	tabSize: number | null;
	/**
	* Whether auto-indent is enabled
	*/
	autoIndent: boolean | null;
	/**
	* Whether to show whitespace tab indicators (→) for this language
	* Defaults to true. Set to false for languages like Go/Hare that use tabs for indentation.
	*/
	showWhitespaceTabs: boolean | null;
	/**
	* Formatter configuration
	*/
	formatter: FormatterPackConfig | null;
};
type LspServerPackConfig = {
	/**
	* Command to start the LSP server
	*/
	command: string;
	/**
	* Arguments to pass to the command
	*/
	args: Array<string>;
	/**
	* Whether to auto-start the server when a matching file is opened
	*/
	autoStart: boolean | null;
	/**
	* LSP initialization options
	*/
	initializationOptions: Record<string, unknown> | null;
};
type SpawnResult = {
	/**
	* Complete stdout as string
	*/
	stdout: string;
	/**
	* Complete stderr as string
	*/
	stderr: string;
	/**
	* Process exit code (0 usually means success, -1 if killed)
	*/
	exit_code: number;
};
type TextPropertiesAtCursor = Array<Record<string, unknown>>;
type TsHighlightSpan = {
	start: number;
	end: number;
	color: [number, number, number];
	bold: boolean;
	italic: boolean;
};
type VirtualBufferResult = {
	/**
	* The created buffer ID
	*/
	bufferId: number;
	/**
	* The split ID (if created in a new split)
	*/
	splitId: number | null;
};
/**
* Main editor API interface
*/
interface EditorAPI {
	/**
	* Get the plugin API version. Plugins can check this to verify
	* the editor supports the features they need.
	*/
	apiVersion(): number;
	/**
	* Get the active buffer ID (0 if none)
	*/
	getActiveBufferId(): number;
	/**
	* Get the active split ID
	*/
	getActiveSplitId(): number;
	/**
	* List all open buffers - returns array of BufferInfo objects
	*/
	listBuffers(): BufferInfo[];
	debug(msg: string): void;
	info(msg: string): void;
	warn(msg: string): void;
	error(msg: string): void;
	setStatus(msg: string): void;
	copyToClipboard(text: string): void;
	setClipboard(text: string): void;
	/**
	* Register a command - reads plugin name from __pluginName__ global
	* context is optional - can be omitted, null, undefined, or a string
	*/
	registerCommand(name: string, description: string, handlerName: string, context?: unknown): boolean;
	/**
	* Unregister a command by name
	*/
	unregisterCommand(name: string): boolean;
	/**
	* Set a context (for keybinding conditions)
	*/
	setContext(name: string, active: boolean): boolean;
	/**
	* Execute a built-in action
	*/
	executeAction(actionName: string): boolean;
	/**
	* Translate a string - reads plugin name from __pluginName__ global
	* Args is optional - can be omitted, undefined, null, or an object
	*/
	t(key: string, ...args: unknown[]): string;
	/**
	* Get cursor position in active buffer
	*/
	getCursorPosition(): number;
	/**
	* Get file path for a buffer
	*/
	getBufferPath(bufferId: number): string;
	/**
	* Get buffer length in bytes
	*/
	getBufferLength(bufferId: number): number;
	/**
	* Check if buffer has unsaved changes
	*/
	isBufferModified(bufferId: number): boolean;
	/**
	* Save a buffer to a specific file path
	* Used by :w filename to save unnamed buffers or save-as
	*/
	saveBufferToPath(bufferId: number, path: string): boolean;
	/**
	* Get buffer info by ID
	*/
	getBufferInfo(bufferId: number): BufferInfo | null;
	/**
	* Get primary cursor info for active buffer
	*/
	getPrimaryCursor(): CursorInfo | null;
	/**
	* Get all cursors for active buffer
	*/
	getAllCursors(): CursorInfo[];
	/**
	* Get all cursor positions as byte offsets
	*/
	getAllCursorPositions(): number[];
	/**
	* Get viewport info for active buffer
	*/
	getViewport(): ViewportInfo | null;
	/**
	* Get the line number (0-indexed) of the primary cursor
	*/
	getCursorLine(): number;
	/**
	* Get the byte offset of the start of a line (0-indexed line number)
	* Returns null if the line number is out of range
	*/
	getLineStartPosition(line: number): Promise<number | null>;
	/**
	* Get the byte offset of the end of a line (0-indexed line number)
	* Returns the position after the last character of the line (before newline)
	* Returns null if the line number is out of range
	*/
	getLineEndPosition(line: number): Promise<number | null>;
	/**
	* Get the total number of lines in the active buffer
	* Returns null if buffer not found
	*/
	getBufferLineCount(): Promise<number | null>;
	/**
	* Scroll a split to center a specific line in the viewport
	* Line is 0-indexed (0 = first line)
	*/
	scrollToLineCenter(splitId: number, bufferId: number, line: number): boolean;
	/**
	* Find buffer by file path, returns buffer ID or 0 if not found
	*/
	findBufferByPath(path: string): number;
	/**
	* Get diff between buffer content and last saved version
	*/
	getBufferSavedDiff(bufferId: number): BufferSavedDiff | null;
	/**
	* Insert text at a position in a buffer
	*/
	insertText(bufferId: number, position: number, text: string): boolean;
	/**
	* Delete a range from a buffer
	*/
	deleteRange(bufferId: number, start: number, end: number): boolean;
	/**
	* Insert text at cursor position in active buffer
	*/
	insertAtCursor(text: string): boolean;
	/**
	* Open a file, optionally at a specific line/column
	*/
	openFile(path: string, line: number | null, column: number | null): boolean;
	/**
	* Open a file in a specific split
	*/
	openFileInSplit(splitId: number, path: string, line: number, column: number): boolean;
	/**
	* Show a buffer in the current split
	*/
	showBuffer(bufferId: number): boolean;
	/**
	* Close a buffer
	*/
	closeBuffer(bufferId: number): boolean;
	/**
	* Subscribe to an editor event
	*/
	on(eventName: string, handlerName: string): void;
	/**
	* Unsubscribe from an event
	*/
	off(eventName: string, handlerName: string): void;
	/**
	* Get an environment variable
	*/
	getEnv(name: string): string | null;
	/**
	* Get current working directory
	*/
	getCwd(): string;
	/**
	* Join path components (variadic - accepts multiple string arguments)
	* Always uses forward slashes for cross-platform consistency (like Node.js path.posix.join)
	*/
	pathJoin(...parts: string[]): string;
	/**
	* Get directory name from path
	*/
	pathDirname(path: string): string;
	/**
	* Get file name from path
	*/
	pathBasename(path: string): string;
	/**
	* Get file extension
	*/
	pathExtname(path: string): string;
	/**
	* Check if path is absolute
	*/
	pathIsAbsolute(path: string): boolean;
	/**
	* Convert a file:// URI to a local file path.
	* Handles percent-decoding and Windows drive letters.
	* Returns an empty string if the URI is not a valid file URI.
	*/
	fileUriToPath(uri: string): string;
	/**
	* Convert a local file path to a file:// URI.
	* Handles Windows drive letters and special characters.
	* Returns an empty string if the path cannot be converted.
	*/
	pathToFileUri(path: string): string;
	/**
	* Get the UTF-8 byte length of a JavaScript string.
	* 
	* JS strings are UTF-16 internally, so `str.length` returns the number of
	* UTF-16 code units, not the number of bytes in a UTF-8 encoding.  The
	* editor API uses byte offsets for all buffer positions (overlays, cursor,
	* getBufferText ranges, etc.).  This helper lets plugins convert JS string
	* lengths / regex match indices to the byte offsets the editor expects.
	*/
	utf8ByteLength(text: string): number;
	/**
	* Check if file exists
	*/
	fileExists(path: string): boolean;
	/**
	* Read file contents
	*/
	readFile(path: string): string | null;
	/**
	* Write file contents
	*/
	writeFile(path: string, content: string): boolean;
	/**
	* Read directory contents (returns array of {name, is_file, is_dir})
	*/
	readDir(path: string): DirEntry[];
	/**
	* Get current config as JS object
	*/
	getConfig(): unknown;
	/**
	* Get user config as JS object
	*/
	getUserConfig(): unknown;
	/**
	* Reload configuration from file
	*/
	reloadConfig(): void;
	/**
	* Reload theme registry from disk
	* Call this after installing theme packages or saving new themes
	*/
	reloadThemes(): void;
	/**
	* Register a TextMate grammar file for a language
	* The grammar will be pending until reload_grammars() is called
	*/
	registerGrammar(language: string, grammarPath: string, extensions: string[]): boolean;
	/**
	* Register language configuration (comment prefix, indentation, formatter)
	*/
	registerLanguageConfig(language: string, config: LanguagePackConfig): boolean;
	/**
	* Register an LSP server for a language
	*/
	registerLspServer(language: string, config: LspServerPackConfig): boolean;
	/**
	* Reload the grammar registry to apply registered grammars
	* Call this after registering one or more grammars
	*/
	reloadGrammars(): void;
	/**
	* Get config directory path
	*/
	getConfigDir(): string;
	/**
	* Get themes directory path
	*/
	getThemesDir(): string;
	/**
	* Apply a theme by name
	*/
	applyTheme(themeName: string): boolean;
	/**
	* Get theme schema as JS object
	*/
	getThemeSchema(): unknown;
	/**
	* Get list of builtin themes as JS object
	*/
	getBuiltinThemes(): unknown;
	/**
	* Delete a custom theme (alias for deleteThemeSync)
	*/
	deleteTheme(name: string): boolean;
	/**
	* Get file stat information
	*/
	fileStat(path: string): unknown;
	/**
	* Check if a background process is still running
	*/
	isProcessRunning(processId: number): boolean;
	/**
	* Kill a process by ID (alias for killBackgroundProcess)
	*/
	killProcess(processId: number): boolean;
	/**
	* Translate a key for a specific plugin
	*/
	pluginTranslate(pluginName: string, key: string, args?: Record<string, unknown>): string;
	/**
	* Create a composite buffer (async)
	* 
	* Uses typed CreateCompositeBufferOptions - serde validates field names at runtime
	* via `deny_unknown_fields` attribute
	*/
	createCompositeBuffer(opts: TsCreateCompositeBufferOptions): Promise<number>;
	/**
	* Update alignment hunks for a composite buffer
	* 
	* Uses typed Vec<CompositeHunk> - serde validates field names at runtime
	*/
	updateCompositeAlignment(bufferId: number, hunks: TsCompositeHunk[]): boolean;
	/**
	* Close a composite buffer
	*/
	closeCompositeBuffer(bufferId: number): boolean;
	/**
	* Request syntax highlights for a buffer range (async)
	*/
	getHighlights(bufferId: number, start: number, end: number): Promise<TsHighlightSpan[]>;
	/**
	* Add an overlay with styling options
	* 
	* Colors can be specified as RGB arrays `[r, g, b]` or theme key strings.
	* Theme keys are resolved at render time, so overlays update with theme changes.
	* 
	* Theme key examples: "ui.status_bar_fg", "editor.selection_bg", "syntax.keyword"
	* 
	* Options: fg, bg (RGB array or theme key string), bold, italic, underline,
	* strikethrough, extend_to_line_end (all booleans, default false).
	* 
	* Example usage in TypeScript:
	* ```typescript
	* editor.addOverlay(bufferId, "my-namespace", 0, 10, {
	* fg: "syntax.keyword",           // theme key
	* bg: [40, 40, 50],               // RGB array
	* bold: true,
	* strikethrough: true,
	* });
	* ```
	*/
	addOverlay(bufferId: number, namespace: string, start: number, end: number, options: Record<string, unknown>): boolean;
	/**
	* Clear all overlays in a namespace
	*/
	clearNamespace(bufferId: number, namespace: string): boolean;
	/**
	* Clear all overlays from a buffer
	*/
	clearAllOverlays(bufferId: number): boolean;
	/**
	* Clear all overlays that overlap with a byte range
	*/
	clearOverlaysInRange(bufferId: number, start: number, end: number): boolean;
	/**
	* Remove an overlay by its handle
	*/
	removeOverlay(bufferId: number, handle: string): boolean;
	/**
	* Add a conceal range that hides or replaces a byte range during rendering
	*/
	addConceal(bufferId: number, namespace: string, start: number, end: number, replacement: string | null): boolean;
	/**
	* Clear all conceal ranges in a namespace
	*/
	clearConcealNamespace(bufferId: number, namespace: string): boolean;
	/**
	* Clear all conceal ranges that overlap with a byte range
	*/
	clearConcealsInRange(bufferId: number, start: number, end: number): boolean;
	/**
	* Add a soft break point for marker-based line wrapping
	*/
	addSoftBreak(bufferId: number, namespace: string, position: number, indent: number): boolean;
	/**
	* Clear all soft breaks in a namespace
	*/
	clearSoftBreakNamespace(bufferId: number, namespace: string): boolean;
	/**
	* Clear all soft breaks that fall within a byte range
	*/
	clearSoftBreaksInRange(bufferId: number, start: number, end: number): boolean;
	/**
	* Submit a view transform for a buffer/split
	* 
	* Accepts tokens in the simple format:
	* {kind: "text"|"newline"|"space"|"break", text: "...", sourceOffset: N, style?: {...}}
	* 
	* Also accepts the TypeScript-defined format for backwards compatibility:
	* {kind: {Text: "..."} | "Newline" | "Space" | "Break", source_offset: N, style?: {...}}
	*/
	submitViewTransform(bufferId: number, splitId: number | null, start: number, end: number, tokens: Record<string, unknown>[], layoutHints?: Record<string, unknown>): boolean;
	/**
	* Clear view transform for a buffer/split
	*/
	clearViewTransform(bufferId: number, splitId: number | null): boolean;
	/**
	* Set layout hints (compose width, column guides) for a buffer/split
	* without going through the view_transform pipeline.
	*/
	setLayoutHints(bufferId: number, splitId: number | null, hints: LayoutHints): boolean;
	/**
	* Set file explorer decorations for a namespace
	*/
	setFileExplorerDecorations(namespace: string, decorations: Record<string, unknown>[]): boolean;
	/**
	* Clear file explorer decorations for a namespace
	*/
	clearFileExplorerDecorations(namespace: string): boolean;
	/**
	* Add virtual text (inline text that doesn't exist in the buffer)
	*/
	addVirtualText(bufferId: number, virtualTextId: string, position: number, text: string, r: number, g: number, b: number, before: boolean, useBg: boolean): boolean;
	/**
	* Remove a virtual text by ID
	*/
	removeVirtualText(bufferId: number, virtualTextId: string): boolean;
	/**
	* Remove virtual texts whose ID starts with the given prefix
	*/
	removeVirtualTextsByPrefix(bufferId: number, prefix: string): boolean;
	/**
	* Clear all virtual texts from a buffer
	*/
	clearVirtualTexts(bufferId: number): boolean;
	/**
	* Clear all virtual texts in a namespace
	*/
	clearVirtualTextNamespace(bufferId: number, namespace: string): boolean;
	/**
	* Add a virtual line (full line above/below a position)
	*/
	addVirtualLine(bufferId: number, position: number, text: string, fgR: number, fgG: number, fgB: number, bgR: number, bgG: number, bgB: number, above: boolean, namespace: string, priority: number): boolean;
	/**
	* Show a prompt and wait for user input (async)
	* Returns the user input or null if cancelled
	*/
	prompt(label: string, initialValue: string): Promise<string | null>;
	/**
	* Start an interactive prompt
	*/
	startPrompt(label: string, promptType: string): boolean;
	/**
	* Start a prompt with initial value
	*/
	startPromptWithInitial(label: string, promptType: string, initialValue: string): boolean;
	/**
	* Set suggestions for the current prompt
	* 
	* Uses typed Vec<Suggestion> - serde validates field names at runtime
	*/
	setPromptSuggestions(suggestions: PromptSuggestion[]): boolean;
	setPromptInputSync(sync: boolean): boolean;
	/**
	* Define a buffer mode (takes bindings as array of [key, command] pairs)
	*/
	defineMode(name: string, parent: string | null, bindingsArr: string[][], readOnly?: boolean): boolean;
	/**
	* Set the global editor mode
	*/
	setEditorMode(mode: string | null): boolean;
	/**
	* Get the current editor mode
	*/
	getEditorMode(): string | null;
	/**
	* Close a split
	*/
	closeSplit(splitId: number): boolean;
	/**
	* Set the buffer displayed in a split
	*/
	setSplitBuffer(splitId: number, bufferId: number): boolean;
	/**
	* Focus a specific split
	*/
	focusSplit(splitId: number): boolean;
	/**
	* Set scroll position of a split
	*/
	setSplitScroll(splitId: number, topByte: number): boolean;
	/**
	* Set the ratio of a split (0.0 to 1.0, 0.5 = equal)
	*/
	setSplitRatio(splitId: number, ratio: number): boolean;
	/**
	* Set a label on a split (e.g., "sidebar")
	*/
	setSplitLabel(splitId: number, label: string): boolean;
	/**
	* Remove a label from a split
	*/
	clearSplitLabel(splitId: number): boolean;
	/**
	* Find a split by label (async)
	*/
	getSplitByLabel(label: string): Promise<number | null>;
	/**
	* Distribute all splits evenly
	*/
	distributeSplitsEvenly(): boolean;
	/**
	* Set cursor position in a buffer
	*/
	setBufferCursor(bufferId: number, position: number): boolean;
	/**
	* Set a line indicator in the gutter
	*/
	setLineIndicator(bufferId: number, line: number, namespace: string, symbol: string, r: number, g: number, b: number, priority: number): boolean;
	/**
	* Batch set line indicators in the gutter
	*/
	setLineIndicators(bufferId: number, lines: number[], namespace: string, symbol: string, r: number, g: number, b: number, priority: number): boolean;
	/**
	* Clear line indicators in a namespace
	*/
	clearLineIndicators(bufferId: number, namespace: string): boolean;
	/**
	* Enable or disable line numbers for a buffer
	*/
	setLineNumbers(bufferId: number, enabled: boolean): boolean;
	/**
	* Set the view mode for a buffer ("source" or "compose")
	*/
	setViewMode(bufferId: number, mode: string): boolean;
	/**
	* Enable or disable line wrapping for a buffer/split
	*/
	setLineWrap(bufferId: number, splitId: number | null, enabled: boolean): boolean;
	/**
	* Set plugin-managed per-buffer view state (write-through to snapshot + command for persistence)
	*/
	setViewState(bufferId: number, key: string, value: unknown): boolean;
	/**
	* Get plugin-managed per-buffer view state (reads from snapshot)
	*/
	getViewState(bufferId: number, key: string): unknown;
	/**
	* Create a scroll sync group for anchor-based synchronized scrolling
	*/
	createScrollSyncGroup(groupId: number, leftSplit: number, rightSplit: number): boolean;
	/**
	* Set sync anchors for a scroll sync group
	*/
	setScrollSyncAnchors(groupId: number, anchors: number[][]): boolean;
	/**
	* Remove a scroll sync group
	*/
	removeScrollSyncGroup(groupId: number): boolean;
	/**
	* Execute multiple actions in sequence
	* 
	* Takes typed ActionSpec array - serde validates field names at runtime
	*/
	executeActions(actions: ActionSpec[]): boolean;
	/**
	* Show an action popup
	* 
	* Takes a typed ActionPopupOptions struct - serde validates field names at runtime
	*/
	showActionPopup(opts: ActionPopupOptions): boolean;
	/**
	* Disable LSP for a specific language
	*/
	disableLspForLanguage(language: string): boolean;
	/**
	* Set the workspace root URI for a specific language's LSP server
	* This allows plugins to specify project roots (e.g., directory containing .csproj)
	*/
	setLspRootUri(language: string, uri: string): boolean;
	/**
	* Get all diagnostics from LSP
	*/
	getAllDiagnostics(): JsDiagnostic[];
	/**
	* Get registered event handlers for an event
	*/
	getHandlers(eventName: string): string[];
	/**
	* Create a virtual buffer in current split (async, returns buffer and split IDs)
	*/
	createVirtualBuffer(opts: CreateVirtualBufferOptions): Promise<VirtualBufferResult>;
	/**
	* Create a virtual buffer in a new split (async, returns buffer and split IDs)
	*/
	createVirtualBufferInSplit(opts: CreateVirtualBufferInSplitOptions): Promise<VirtualBufferResult>;
	/**
	* Create a virtual buffer in an existing split (async, returns buffer and split IDs)
	*/
	createVirtualBufferInExistingSplit(opts: CreateVirtualBufferInExistingSplitOptions): Promise<VirtualBufferResult>;
	/**
	* Set virtual buffer content (takes array of entry objects)
	* 
	* Note: entries should be TextPropertyEntry[] - uses manual parsing for HashMap support
	*/
	setVirtualBufferContent(bufferId: number, entriesArr: Record<string, unknown>[]): boolean;
	/**
	* Get text properties at cursor position (returns JS array)
	*/
	getTextPropertiesAtCursor(bufferId: number): TextPropertiesAtCursor;
	/**
	* Spawn a process (async, returns request_id)
	*/
	spawnProcess(command: string, args: string[], cwd?: string): ProcessHandle<SpawnResult>;
	/**
	* Wait for a process to complete and get its result (async)
	*/
	spawnProcessWait(processId: number): Promise<SpawnResult>;
	/**
	* Get buffer text range (async, returns request_id)
	*/
	getBufferText(bufferId: number, start: number, end: number): Promise<string>;
	/**
	* Delay/sleep (async, returns request_id)
	*/
	delay(durationMs: number): Promise<void>;
	/**
	* Send LSP request (async, returns request_id)
	*/
	sendLspRequest(language: string, method: string, params: Record<string, unknown> | null): Promise<unknown>;
	/**
	* Spawn a background process (async, returns request_id which is also process_id)
	*/
	spawnBackgroundProcess(command: string, args: string[], cwd?: string): ProcessHandle<BackgroundProcessResult>;
	/**
	* Kill a background process
	*/
	killBackgroundProcess(processId: number): boolean;
	/**
	* Create a new terminal in a split (async, returns TerminalResult)
	*/
	createTerminal(opts?: CreateTerminalOptions): Promise<TerminalResult>;
	/**
	* Send input data to a terminal
	*/
	sendTerminalInput(terminalId: number, data: string): boolean;
	/**
	* Close a terminal
	*/
	closeTerminal(terminalId: number): boolean;
	/**
	* Force refresh of line display
	*/
	refreshLines(bufferId: number): boolean;
	/**
	* Get the current locale
	*/
	getCurrentLocale(): string;
	/**
	* Load a plugin from a file path (async)
	*/
	loadPlugin(path: string): Promise<boolean>;
	/**
	* Unload a plugin by name (async)
	*/
	unloadPlugin(name: string): Promise<boolean>;
	/**
	* Reload a plugin by name (async)
	*/
	reloadPlugin(name: string): Promise<boolean>;
	/**
	* List all loaded plugins (async)
	* Returns array of { name: string, path: string, enabled: boolean }
	*/
	listPlugins(): Promise<Array<{
		name: string;
		path: string;
		enabled: boolean;
	}>>;
}
