#!/data/data/com.termux/files/usr/bin/env perl

# jamo-normalize.pl
#
# Copyright (c) 2003-2015 Dohyun Kim <nomos at ktug org>
#
# This work may be distributed and/or modified under the
# conditions of the LaTeX Project Public License, either version 1.3c
# of this license or (at your option) any later version.
# The latest version of this license is in
#  http://www.latex-project.org/lppl.txt
# and version 1.3c or later is part of all distributions of LaTeX
# version 2006/05/20 or later.
#
#   written by Dohyun Kim <nomos at ktug org>
#   public domain
#

#use strict;
#use warnings;

my %OPT;

while (@ARGV) {
  my $opt = shift @ARGV;
  if    ($opt =~ /-b/i) { $OPT{boundary}    = 1 }
  elsif ($opt =~ /-p/i) { $OPT{topua}       = 1 }
  elsif ($opt =~ /-d/i) { $OPT{decompose}   = 1 }
  elsif ($opt =~ /-o/i) { $OPT{frompua}     = 1 }
  elsif ($opt =~ /-t/i) { $OPT{latintm}     = 1 }
  elsif ($opt =~ /-r/i) { $OPT{reordertm}   = 1 }
  elsif ($opt =~ /-i/i) { $OPT{normalhanja} = 1 }
  elsif ($opt =~ /-c/i) { $OPT{compatjamo}  = 1 }
  else  {
    (my $prog = $0) =~ s/.*[\/\\]//;
    print
    "Usage: $prog [options] < in_file > out_file\n\n",
    "  Translate Hangul Jamo sequence to Hangul syllables\n\n",
    "  -b :  insert ZWS between syllable blocks (not for practical use)\n",
    "  -c :  convert conjoining Jamo to compatibility Jamo\n",
    "  -d :  decomposition only, and no further recomposition\n",
    "  -i :  convert compatibility Hanja to normal Hanja\n",
    "  -o :  decompose PUA Old Hangul syllables to Jamo sequence\n",
    "  -p :  compose Jamo sequence to PUA Old Hangul syllables\n",
    "  -r :  reorder Hangul Tone Marks to the first of syllable block\n",
    "        (not for practical use)\n",
    "  -t :  convert U+00B7 or U+003A to Hangul Tone Marks\n";
    exit;
  }
}

##### variables #####

my $cho       = "\x{1100}-\x{115F}\x{A960}-\x{A97C}";
my $jung      = "\x{1160}-\x{11A7}\x{D7B0}-\x{D7C6}";
my $jong      = "\x{11A8}-\x{11FF}\x{D7CB}-\x{D7FB}";
my $tmrk      = "\x{302E}\x{302F}";
my $boundary  = "\x{200B}";
my $syllblock = "[$cho][$jung][$jong]?[$tmrk]?";
my $jamos     = "[$cho$jung$jong$tmrk]";

my %jamo2cjamo = (
  0x1100 => 0x3131, 0x1101 => 0x3132, 0x1102 => 0x3134, 0x1103 => 0x3137,
  0x1104 => 0x3138, 0x1105 => 0x3139, 0x1106 => 0x3141, 0x1107 => 0x3142,
  0x1108 => 0x3143, 0x1109 => 0x3145, 0x110A => 0x3146, 0x110B => 0x3147,
  0x110C => 0x3148, 0x110D => 0x3149, 0x110E => 0x314A, 0x110F => 0x314B,
  0x1110 => 0x314C, 0x1111 => 0x314D, 0x1112 => 0x314E, 0x1114 => 0x3165,
  0x1115 => 0x3166, 0x111A => 0x3140, 0x111C => 0x316E, 0x111D => 0x3171,
  0x111E => 0x3172, 0x1120 => 0x3173, 0x1121 => 0x3144, 0x1122 => 0x3174,
  0x1123 => 0x3175, 0x1127 => 0x3176, 0x1129 => 0x3177, 0x112B => 0x3178,
  0x112C => 0x3179, 0x112D => 0x317A, 0x112E => 0x317B, 0x112F => 0x317C,
  0x1132 => 0x317D, 0x1136 => 0x317E, 0x1140 => 0x317F,
# 0x1145 => 0x3182, 0x1146 => 0x3183,
  0x1147 => 0x3180, 0x114C => 0x3181, 0x1157 => 0x3184, 0x1158 => 0x3185,
  0x1159 => 0x3186, 0x115B => 0x3167, 0x115C => 0x3135, 0x115D => 0x3136,
  0x1161 => 0x314F, 0x1162 => 0x3150, 0x1163 => 0x3151, 0x1164 => 0x3152,
  0x1165 => 0x3153, 0x1166 => 0x3154, 0x1167 => 0x3155, 0x1168 => 0x3156,
  0x1169 => 0x3157, 0x116A => 0x3158, 0x116B => 0x3159, 0x116C => 0x315A,
  0x116D => 0x315B, 0x116E => 0x315C, 0x116F => 0x315D, 0x1170 => 0x315E,
  0x1171 => 0x315F, 0x1172 => 0x3160, 0x1173 => 0x3161, 0x1174 => 0x3162,
  0x1175 => 0x3163, 0x1184 => 0x3187, 0x1185 => 0x3188, 0x1188 => 0x3189,
  0x1191 => 0x318A, 0x1192 => 0x318B, 0x1194 => 0x318C, 0x119E => 0x318D,
  0x11A1 => 0x318E,
# 0x11A8 => 0x3131, 0x11A9 => 0x3132, 0x11AA => 0x3133, 0x11AB => 0x3134,
# 0x11AC => 0x3135, 0x11AD => 0x3136, 0x11AE => 0x3137, 0x11AF => 0x3139,
# 0x11B0 => 0x313A, 0x11B1 => 0x313B, 0x11B2 => 0x313C, 0x11B3 => 0x313D,
# 0x11B4 => 0x313E, 0x11B5 => 0x313F, 0x11B6 => 0x3140, 0x11B7 => 0x3141,
# 0x11B8 => 0x3142, 0x11B9 => 0x3144, 0x11BA => 0x3145, 0x11BB => 0x3146,
# 0x11BC => 0x3147, 0x11BD => 0x3148, 0x11BE => 0x314A, 0x11BF => 0x314B,
# 0x11C0 => 0x314C, 0x11C1 => 0x314D, 0x11C2 => 0x314E, 0x11C6 => 0x3166,
# 0x11C7 => 0x3167, 0x11C8 => 0x3168, 0x11CC => 0x3169, 0x11CE => 0x316A,
# 0x11D3 => 0x316B, 0x11D7 => 0x316C, 0x11D9 => 0x316D, 0x11DC => 0x316E,
# 0x11DD => 0x316F, 0x11DF => 0x3170, 0x11E2 => 0x3171, 0x11E6 => 0x3178,
# 0x11E7 => 0x317A, 0x11E8 => 0x317C, 0x11EA => 0x317D, 0x11EB => 0x317F,
# 0x11EE => 0x3180, 0x11F0 => 0x3181, 0x11F1 => 0x3182, 0x11F2 => 0x3183,
# 0x11F4 => 0x3184, 0x11F9 => 0x3186, 0x11FF => 0x3165,
  0xA964 => 0x313A, 0xA966 => 0x316A, 0xA968 => 0x313B, 0xA969 => 0x313C,
  0xA96C => 0x313D, 0xA971 => 0x316F,
# 0xD7CD => 0x3138, 0xD7E3 => 0x3173, 0xD7E6 => 0x3143, 0xD7E7 => 0x3175,
# 0xD7E8 => 0x3176, 0xD7EF => 0x317E, 0xD7F9 => 0x3149,
);

my @HYpuaJamo;
if ($OPT{frompua} or $OPT{topua}) { @HYpuaJamo = arr_hypua2jamo(); }

##### main routine #####

binmode (STDIN,":utf8");
binmode (STDOUT,":utf8");

while (<>) {
  print STDERR ".";
  &syllable2jamo;
  if ($OPT{frompua}) {
    &hypua2jamo;
    &hypuasingle2jamo;
  }
  &compose_jamo;
  &insert_filler;
  &ascii2tonemark if $OPT{latintm};
  &insert_boundary if $OPT{boundary};
  &reorder_tonemark if $OPT{reordertm};
  unless ($OPT{decompose}) {
    &jamo2hypua if $OPT{topua};
    &jamo2syllable;
    &jamo2jamocomp if $OPT{compatjamo};
  }
  &normalize_hanja if $OPT{normalhanja};
  print;
}

print STDERR "\n";

##### subroutines #####

sub reorder_tonemark {
  my @blocks = split /($syllblock)/;
  for my $i (0 .. $#blocks) {
    if ($i % 2 == 0) {
      $blocks[$i] =~ s/([$tmrk])/$1\x{25CC}/g;
    }
    else {
      $blocks[$i] =~ s/([$cho$jung$jong]+)([$tmrk])$/$2$1/;
    }
  }
  $_ = join '',@blocks;
}

sub ascii2tonemark {
  s/\x{B7}([$cho][$jung][$jong]?)(?![$tmrk])/$1\x{302E}/g;
  s/\x{3A}([$cho][$jung][$jong]?)(?![$tmrk])/$1\x{302F}/g;
}

sub hypua2jamo {
  for my $i (0 .. $#HYpuaJamo) {
    my $puachar = chr $HYpuaJamo[$i][0];
    my $jamoseq = "";
    for my $j (1 .. $#{$HYpuaJamo[$i]}) {
      $jamoseq .= chr $HYpuaJamo[$i][$j];
    }
    s/$puachar/$jamoseq/g;
  }
}

sub jamo2hypua {
  for my $i (0 .. $#HYpuaJamo) {
    my $puachar = chr $HYpuaJamo[$i][0];
    my $jamoseq = "";
    for my $j (1 .. $#{$HYpuaJamo[$i]}) {
      $jamoseq .= chr $HYpuaJamo[$i][$j];
    }
    s/$jamoseq/$puachar/g;
  }
}

sub insert_boundary {
  chomp;
  my @parts = split /($syllblock)/, $_;
  my @newparts;
  for my $i (0 .. $#parts) {
    if ($i % 2 == 0) {
      next unless $parts[$i];
      my @subparts = split /($jamos)/, $parts[$i];
      my @newsubparts;
      for my $i (0 .. $#subparts) {
        push @newsubparts, $subparts[$i] if $subparts[$i];
      }
      push @newparts, join ($boundary, @newsubparts);
    }
    else {
      push @newparts, $parts[$i];
    }
  }
  $_ = join $boundary, @newparts;
  $_ .= "\n";
}

sub insert_filler {
  s/(?<![$jung])([$jong])/\x{115F}\x{1160}$1/g;
  s/(?<![$cho])([$jung])/\x{115F}$1/g;
  s/([$cho])(?![$jung])/$1\x{1160}/g;
}

sub syllable2jamo { # no decomposition of control sequence
  my @nocs = split /(\\[\x{AC00}-\x{D7A3}]+)/;
  for (my $i=0; $i<@nocs; $i=$i+2) {
    $nocs[$i] =~ s/([\x{AC00}-\x{D7A3}])/do_syllable2jamo($1)/ge;
  }
  $_ = join '', @nocs;
}

sub do_syllable2jamo {
  my $syl = ord shift;
  my $LC = ($syl - 0xac00) / (21 * 28) + 0x1100;
  my $MV = (($syl - 0xac00) % (21 * 28)) / 28 + 0x1161;
  my $TC = ($syl - 0xac00) % 28 + 0x11a7;
  my $result = chr($LC).chr($MV);
  $result .= chr($TC) unless $TC == 0x11a7;
  return $result;
}

sub jamo2syllable {
  # 초성.중성.종성
  s/([\x{1100}-\x{1112}])
  ([\x{1161}-\x{1175}])
  ([\x{11A8}-\x{11C2}])
  /do_jamo2syllable(ord $1, ord $2, ord $3)/gex;
  # 초성.중성
  s/([\x{1100}-\x{1112}])
  ([\x{1161}-\x{1175}])
  (?![$jong])
  /do_jamo2syllable(ord $1, ord $2, 0)/gex;
}

sub do_jamo2syllable {
  my ($LC,$MV,$TC) = @_;
  my $syllable = ($LC - 0x1100) * 21;
  $syllable = ($syllable + $MV - 0x1161) * 28;
  $syllable += $TC - 0x11a7 if $TC;
  $syllable += 0xac00;
  return chr $syllable;
}

sub jamo2jamocomp {
  s/([$cho])\x{1160}/do_jamo2jamocomp($1)/ge;
  s/\x{115F}([$jung])(?![$jong])/do_jamo2jamocomp($1)/ge;
  # s/\x{115F}\x{1160}([$jong])/do_jamo2jamocomp($1)/ge;
}

sub do_jamo2jamocomp {
  my $jamo = shift;
  my $cjamo = $jamo2cjamo{ord $jamo};
  return chr $cjamo if $cjamo;
  return "$jamo\x{1160}" if $jamo =~ /[$cho]/;
  return "\x{115F}$jamo" if $jamo =~ /[$jung]/;
  return "\x{115F}\x{1160}$jamo" if $jamo =~ /[$jong]/;
}

sub normalize_hanja {
  s/\x{F900}/\x{8C48}\x{FE00}/g;
  s/\x{F901}/\x{66F4}\x{FE00}/g;
  s/\x{F902}/\x{8ECA}\x{FE00}/g;
  s/\x{F903}/\x{8CC8}\x{FE00}/g;
  s/\x{F904}/\x{6ED1}\x{FE00}/g;
  s/\x{F905}/\x{4E32}\x{FE00}/g;
  s/\x{F906}/\x{53E5}\x{FE00}/g;
  s/\x{F907}/\x{9F9C}\x{FE00}/g;
  s/\x{F908}/\x{9F9C}\x{FE01}/g;
  s/\x{F909}/\x{5951}\x{FE00}/g;
  s/\x{F90A}/\x{91D1}\x{FE00}/g;
  s/\x{F90B}/\x{5587}\x{FE00}/g;
  s/\x{F90C}/\x{5948}\x{FE00}/g;
  s/\x{F90D}/\x{61F6}\x{FE00}/g;
  s/\x{F90E}/\x{7669}\x{FE00}/g;
  s/\x{F90F}/\x{7F85}\x{FE00}/g;
  s/\x{F910}/\x{863F}\x{FE00}/g;
  s/\x{F911}/\x{87BA}\x{FE00}/g;
  s/\x{F912}/\x{88F8}\x{FE00}/g;
  s/\x{F913}/\x{908F}\x{FE00}/g;
  s/\x{F914}/\x{6A02}\x{FE00}/g;
  s/\x{F915}/\x{6D1B}\x{FE00}/g;
  s/\x{F916}/\x{70D9}\x{FE00}/g;
  s/\x{F917}/\x{73DE}\x{FE00}/g;
  s/\x{F918}/\x{843D}\x{FE00}/g;
  s/\x{F919}/\x{916A}\x{FE00}/g;
  s/\x{F91A}/\x{99F1}\x{FE00}/g;
  s/\x{F91B}/\x{4E82}\x{FE00}/g;
  s/\x{F91C}/\x{5375}\x{FE00}/g;
  s/\x{F91D}/\x{6B04}\x{FE00}/g;
  s/\x{F91E}/\x{721B}\x{FE00}/g;
  s/\x{F91F}/\x{862D}\x{FE00}/g;
  s/\x{F920}/\x{9E1E}\x{FE00}/g;
  s/\x{F921}/\x{5D50}\x{FE00}/g;
  s/\x{F922}/\x{6FEB}\x{FE00}/g;
  s/\x{F923}/\x{85CD}\x{FE00}/g;
  s/\x{F924}/\x{8964}\x{FE00}/g;
  s/\x{F925}/\x{62C9}\x{FE00}/g;
  s/\x{F926}/\x{81D8}\x{FE00}/g;
  s/\x{F927}/\x{881F}\x{FE00}/g;
  s/\x{F928}/\x{5ECA}\x{FE00}/g;
  s/\x{F929}/\x{6717}\x{FE00}/g;
  s/\x{F92A}/\x{6D6A}\x{FE00}/g;
  s/\x{F92B}/\x{72FC}\x{FE00}/g;
  s/\x{F92C}/\x{90CE}\x{FE00}/g;
  s/\x{F92D}/\x{4F86}\x{FE00}/g;
  s/\x{F92E}/\x{51B7}\x{FE00}/g;
  s/\x{F92F}/\x{52DE}\x{FE00}/g;
  s/\x{F930}/\x{64C4}\x{FE00}/g;
  s/\x{F931}/\x{6AD3}\x{FE00}/g;
  s/\x{F932}/\x{7210}\x{FE00}/g;
  s/\x{F933}/\x{76E7}\x{FE00}/g;
  s/\x{F934}/\x{8001}\x{FE00}/g;
  s/\x{F935}/\x{8606}\x{FE00}/g;
  s/\x{F936}/\x{865C}\x{FE00}/g;
  s/\x{F937}/\x{8DEF}\x{FE00}/g;
  s/\x{F938}/\x{9732}\x{FE00}/g;
  s/\x{F939}/\x{9B6F}\x{FE00}/g;
  s/\x{F93A}/\x{9DFA}\x{FE00}/g;
  s/\x{F93B}/\x{788C}\x{FE00}/g;
  s/\x{F93C}/\x{797F}\x{FE00}/g;
  s/\x{F93D}/\x{7DA0}\x{FE00}/g;
  s/\x{F93E}/\x{83C9}\x{FE00}/g;
  s/\x{F93F}/\x{9304}\x{FE00}/g;
  s/\x{F940}/\x{9E7F}\x{FE00}/g;
  s/\x{F941}/\x{8AD6}\x{FE00}/g;
  s/\x{F942}/\x{58DF}\x{FE00}/g;
  s/\x{F943}/\x{5F04}\x{FE00}/g;
  s/\x{F944}/\x{7C60}\x{FE00}/g;
  s/\x{F945}/\x{807E}\x{FE00}/g;
  s/\x{F946}/\x{7262}\x{FE00}/g;
  s/\x{F947}/\x{78CA}\x{FE00}/g;
  s/\x{F948}/\x{8CC2}\x{FE00}/g;
  s/\x{F949}/\x{96F7}\x{FE00}/g;
  s/\x{F94A}/\x{58D8}\x{FE00}/g;
  s/\x{F94B}/\x{5C62}\x{FE00}/g;
  s/\x{F94C}/\x{6A13}\x{FE00}/g;
  s/\x{F94D}/\x{6DDA}\x{FE00}/g;
  s/\x{F94E}/\x{6F0F}\x{FE00}/g;
  s/\x{F94F}/\x{7D2F}\x{FE00}/g;
  s/\x{F950}/\x{7E37}\x{FE00}/g;
  s/\x{F951}/\x{964B}\x{FE00}/g;
  s/\x{F952}/\x{52D2}\x{FE00}/g;
  s/\x{F953}/\x{808B}\x{FE00}/g;
  s/\x{F954}/\x{51DC}\x{FE00}/g;
  s/\x{F955}/\x{51CC}\x{FE00}/g;
  s/\x{F956}/\x{7A1C}\x{FE00}/g;
  s/\x{F957}/\x{7DBE}\x{FE00}/g;
  s/\x{F958}/\x{83F1}\x{FE00}/g;
  s/\x{F959}/\x{9675}\x{FE00}/g;
  s/\x{F95A}/\x{8B80}\x{FE00}/g;
  s/\x{F95B}/\x{62CF}\x{FE00}/g;
  s/\x{F95C}/\x{6A02}\x{FE01}/g;
  s/\x{F95D}/\x{8AFE}\x{FE00}/g;
  s/\x{F95E}/\x{4E39}\x{FE00}/g;
  s/\x{F95F}/\x{5BE7}\x{FE00}/g;
  s/\x{F960}/\x{6012}\x{FE00}/g;
  s/\x{F961}/\x{7387}\x{FE00}/g;
  s/\x{F962}/\x{7570}\x{FE00}/g;
  s/\x{F963}/\x{5317}\x{FE00}/g;
  s/\x{F964}/\x{78FB}\x{FE00}/g;
  s/\x{F965}/\x{4FBF}\x{FE00}/g;
  s/\x{F966}/\x{5FA9}\x{FE00}/g;
  s/\x{F967}/\x{4E0D}\x{FE00}/g;
  s/\x{F968}/\x{6CCC}\x{FE00}/g;
  s/\x{F969}/\x{6578}\x{FE00}/g;
  s/\x{F96A}/\x{7D22}\x{FE00}/g;
  s/\x{F96B}/\x{53C3}\x{FE00}/g;
  s/\x{F96C}/\x{585E}\x{FE00}/g;
  s/\x{F96D}/\x{7701}\x{FE00}/g;
  s/\x{F96E}/\x{8449}\x{FE00}/g;
  s/\x{F96F}/\x{8AAA}\x{FE00}/g;
  s/\x{F970}/\x{6BBA}\x{FE00}/g;
  s/\x{F971}/\x{8FB0}\x{FE00}/g;
  s/\x{F972}/\x{6C88}\x{FE00}/g;
  s/\x{F973}/\x{62FE}\x{FE00}/g;
  s/\x{F974}/\x{82E5}\x{FE00}/g;
  s/\x{F975}/\x{63A0}\x{FE00}/g;
  s/\x{F976}/\x{7565}\x{FE00}/g;
  s/\x{F977}/\x{4EAE}\x{FE00}/g;
  s/\x{F978}/\x{5169}\x{FE00}/g;
  s/\x{F979}/\x{51C9}\x{FE00}/g;
  s/\x{F97A}/\x{6881}\x{FE00}/g;
  s/\x{F97B}/\x{7CE7}\x{FE00}/g;
  s/\x{F97C}/\x{826F}\x{FE00}/g;
  s/\x{F97D}/\x{8AD2}\x{FE00}/g;
  s/\x{F97E}/\x{91CF}\x{FE00}/g;
  s/\x{F97F}/\x{52F5}\x{FE00}/g;
  s/\x{F980}/\x{5442}\x{FE00}/g;
  s/\x{F981}/\x{5973}\x{FE00}/g;
  s/\x{F982}/\x{5EEC}\x{FE00}/g;
  s/\x{F983}/\x{65C5}\x{FE00}/g;
  s/\x{F984}/\x{6FFE}\x{FE00}/g;
  s/\x{F985}/\x{792A}\x{FE00}/g;
  s/\x{F986}/\x{95AD}\x{FE00}/g;
  s/\x{F987}/\x{9A6A}\x{FE00}/g;
  s/\x{F988}/\x{9E97}\x{FE00}/g;
  s/\x{F989}/\x{9ECE}\x{FE00}/g;
  s/\x{F98A}/\x{529B}\x{FE00}/g;
  s/\x{F98B}/\x{66C6}\x{FE00}/g;
  s/\x{F98C}/\x{6B77}\x{FE00}/g;
  s/\x{F98D}/\x{8F62}\x{FE00}/g;
  s/\x{F98E}/\x{5E74}\x{FE00}/g;
  s/\x{F98F}/\x{6190}\x{FE00}/g;
  s/\x{F990}/\x{6200}\x{FE00}/g;
  s/\x{F991}/\x{649A}\x{FE00}/g;
  s/\x{F992}/\x{6F23}\x{FE00}/g;
  s/\x{F993}/\x{7149}\x{FE00}/g;
  s/\x{F994}/\x{7489}\x{FE00}/g;
  s/\x{F995}/\x{79CA}\x{FE00}/g;
  s/\x{F996}/\x{7DF4}\x{FE00}/g;
  s/\x{F997}/\x{806F}\x{FE00}/g;
  s/\x{F998}/\x{8F26}\x{FE00}/g;
  s/\x{F999}/\x{84EE}\x{FE00}/g;
  s/\x{F99A}/\x{9023}\x{FE00}/g;
  s/\x{F99B}/\x{934A}\x{FE00}/g;
  s/\x{F99C}/\x{5217}\x{FE00}/g;
  s/\x{F99D}/\x{52A3}\x{FE00}/g;
  s/\x{F99E}/\x{54BD}\x{FE00}/g;
  s/\x{F99F}/\x{70C8}\x{FE00}/g;
  s/\x{F9A0}/\x{88C2}\x{FE00}/g;
  s/\x{F9A1}/\x{8AAA}\x{FE01}/g;
  s/\x{F9A2}/\x{5EC9}\x{FE00}/g;
  s/\x{F9A3}/\x{5FF5}\x{FE00}/g;
  s/\x{F9A4}/\x{637B}\x{FE00}/g;
  s/\x{F9A5}/\x{6BAE}\x{FE00}/g;
  s/\x{F9A6}/\x{7C3E}\x{FE00}/g;
  s/\x{F9A7}/\x{7375}\x{FE00}/g;
  s/\x{F9A8}/\x{4EE4}\x{FE00}/g;
  s/\x{F9A9}/\x{56F9}\x{FE00}/g;
  s/\x{F9AA}/\x{5BE7}\x{FE01}/g;
  s/\x{F9AB}/\x{5DBA}\x{FE00}/g;
  s/\x{F9AC}/\x{601C}\x{FE00}/g;
  s/\x{F9AD}/\x{73B2}\x{FE00}/g;
  s/\x{F9AE}/\x{7469}\x{FE00}/g;
  s/\x{F9AF}/\x{7F9A}\x{FE00}/g;
  s/\x{F9B0}/\x{8046}\x{FE00}/g;
  s/\x{F9B1}/\x{9234}\x{FE00}/g;
  s/\x{F9B2}/\x{96F6}\x{FE00}/g;
  s/\x{F9B3}/\x{9748}\x{FE00}/g;
  s/\x{F9B4}/\x{9818}\x{FE00}/g;
  s/\x{F9B5}/\x{4F8B}\x{FE00}/g;
  s/\x{F9B6}/\x{79AE}\x{FE00}/g;
  s/\x{F9B7}/\x{91B4}\x{FE00}/g;
  s/\x{F9B8}/\x{96B8}\x{FE00}/g;
  s/\x{F9B9}/\x{60E1}\x{FE00}/g;
  s/\x{F9BA}/\x{4E86}\x{FE00}/g;
  s/\x{F9BB}/\x{50DA}\x{FE00}/g;
  s/\x{F9BC}/\x{5BEE}\x{FE00}/g;
  s/\x{F9BD}/\x{5C3F}\x{FE00}/g;
  s/\x{F9BE}/\x{6599}\x{FE00}/g;
  s/\x{F9BF}/\x{6A02}\x{FE02}/g;
  s/\x{F9C0}/\x{71CE}\x{FE00}/g;
  s/\x{F9C1}/\x{7642}\x{FE00}/g;
  s/\x{F9C2}/\x{84FC}\x{FE00}/g;
  s/\x{F9C3}/\x{907C}\x{FE00}/g;
  s/\x{F9C4}/\x{9F8D}\x{FE00}/g;
  s/\x{F9C5}/\x{6688}\x{FE00}/g;
  s/\x{F9C6}/\x{962E}\x{FE00}/g;
  s/\x{F9C7}/\x{5289}\x{FE00}/g;
  s/\x{F9C8}/\x{677B}\x{FE00}/g;
  s/\x{F9C9}/\x{67F3}\x{FE00}/g;
  s/\x{F9CA}/\x{6D41}\x{FE00}/g;
  s/\x{F9CB}/\x{6E9C}\x{FE00}/g;
  s/\x{F9CC}/\x{7409}\x{FE00}/g;
  s/\x{F9CD}/\x{7559}\x{FE00}/g;
  s/\x{F9CE}/\x{786B}\x{FE00}/g;
  s/\x{F9CF}/\x{7D10}\x{FE00}/g;
  s/\x{F9D0}/\x{985E}\x{FE00}/g;
  s/\x{F9D1}/\x{516D}\x{FE00}/g;
  s/\x{F9D2}/\x{622E}\x{FE00}/g;
  s/\x{F9D3}/\x{9678}\x{FE00}/g;
  s/\x{F9D4}/\x{502B}\x{FE00}/g;
  s/\x{F9D5}/\x{5D19}\x{FE00}/g;
  s/\x{F9D6}/\x{6DEA}\x{FE00}/g;
  s/\x{F9D7}/\x{8F2A}\x{FE00}/g;
  s/\x{F9D8}/\x{5F8B}\x{FE00}/g;
  s/\x{F9D9}/\x{6144}\x{FE00}/g;
  s/\x{F9DA}/\x{6817}\x{FE00}/g;
  s/\x{F9DB}/\x{7387}\x{FE01}/g;
  s/\x{F9DC}/\x{9686}\x{FE00}/g;
  s/\x{F9DD}/\x{5229}\x{FE00}/g;
  s/\x{F9DE}/\x{540F}\x{FE00}/g;
  s/\x{F9DF}/\x{5C65}\x{FE00}/g;
  s/\x{F9E0}/\x{6613}\x{FE00}/g;
  s/\x{F9E1}/\x{674E}\x{FE00}/g;
  s/\x{F9E2}/\x{68A8}\x{FE00}/g;
  s/\x{F9E3}/\x{6CE5}\x{FE00}/g;
  s/\x{F9E4}/\x{7406}\x{FE00}/g;
  s/\x{F9E5}/\x{75E2}\x{FE00}/g;
  s/\x{F9E6}/\x{7F79}\x{FE00}/g;
  s/\x{F9E7}/\x{88CF}\x{FE00}/g;
  s/\x{F9E8}/\x{88E1}\x{FE00}/g;
  s/\x{F9E9}/\x{91CC}\x{FE00}/g;
  s/\x{F9EA}/\x{96E2}\x{FE00}/g;
  s/\x{F9EB}/\x{533F}\x{FE00}/g;
  s/\x{F9EC}/\x{6EBA}\x{FE00}/g;
  s/\x{F9ED}/\x{541D}\x{FE00}/g;
  s/\x{F9EE}/\x{71D0}\x{FE00}/g;
  s/\x{F9EF}/\x{7498}\x{FE00}/g;
  s/\x{F9F0}/\x{85FA}\x{FE00}/g;
  s/\x{F9F1}/\x{96A3}\x{FE00}/g;
  s/\x{F9F2}/\x{9C57}\x{FE00}/g;
  s/\x{F9F3}/\x{9E9F}\x{FE00}/g;
  s/\x{F9F4}/\x{6797}\x{FE00}/g;
  s/\x{F9F5}/\x{6DCB}\x{FE00}/g;
  s/\x{F9F6}/\x{81E8}\x{FE00}/g;
  s/\x{F9F7}/\x{7ACB}\x{FE00}/g;
  s/\x{F9F8}/\x{7B20}\x{FE00}/g;
  s/\x{F9F9}/\x{7C92}\x{FE00}/g;
  s/\x{F9FA}/\x{72C0}\x{FE00}/g;
  s/\x{F9FB}/\x{7099}\x{FE00}/g;
  s/\x{F9FC}/\x{8B58}\x{FE00}/g;
  s/\x{F9FD}/\x{4EC0}\x{FE00}/g;
  s/\x{F9FE}/\x{8336}\x{FE00}/g;
  s/\x{F9FF}/\x{523A}\x{FE00}/g;
  s/\x{FA00}/\x{5207}\x{FE00}/g;
  s/\x{FA01}/\x{5EA6}\x{FE00}/g;
  s/\x{FA02}/\x{62D3}\x{FE00}/g;
  s/\x{FA03}/\x{7CD6}\x{FE00}/g;
  s/\x{FA04}/\x{5B85}\x{FE00}/g;
  s/\x{FA05}/\x{6D1E}\x{FE00}/g;
  s/\x{FA06}/\x{66B4}\x{FE00}/g;
  s/\x{FA07}/\x{8F3B}\x{FE00}/g;
  s/\x{FA08}/\x{884C}\x{FE00}/g;
  s/\x{FA09}/\x{964D}\x{FE00}/g;
  s/\x{FA0A}/\x{898B}\x{FE00}/g;
  s/\x{FA0B}/\x{5ED3}\x{FE00}/g;
  s/\x{FA2E}/\x{90DE}\x{FE00}/g;
  s/\x{FA2F}/\x{96B7}\x{FE00}/g;
}

sub compose_jamo {
  s/\x{1105}\x{1100}\x{1100}/\x{A965}/g;
  s/\x{1105}\x{1103}\x{1103}/\x{A967}/g;
  s/\x{1105}\x{1107}\x{1107}/\x{A96A}/g;
  s/\x{1105}\x{1107}\x{110B}/\x{A96B}/g;
  s/\x{1107}\x{1107}\x{110B}/\x{112C}/g;
  s/\x{1107}\x{1109}\x{1100}/\x{1122}/g;
  s/\x{1107}\x{1109}\x{1103}/\x{1123}/g;
  s/\x{1107}\x{1109}\x{1107}/\x{1124}/g;
  s/\x{1107}\x{1109}\x{1109}/\x{1125}/g;
  s/\x{1107}\x{1109}\x{110C}/\x{1126}/g;
  s/\x{1107}\x{1109}\x{1110}/\x{A972}/g;
  s/\x{1109}\x{1107}\x{1100}/\x{1133}/g;
  s/\x{1109}\x{1109}\x{1107}/\x{A975}/g;
  s/\x{1109}\x{1109}\x{1109}/\x{1134}/g;
  s/\x{110C}\x{110C}\x{1112}/\x{A978}/g;
  s/\x{1100}\x{1100}/\x{1101}/g;
  s/\x{1100}\x{1103}/\x{115A}/g;
  s/\x{1102}\x{1100}/\x{1113}/g;
  s/\x{1102}\x{1102}/\x{1114}/g;
  s/\x{1102}\x{1103}/\x{1115}/g;
  s/\x{1102}\x{1107}/\x{1116}/g;
  s/\x{1102}\x{1109}/\x{115B}/g;
  s/\x{1102}\x{110C}/\x{115C}/g;
  s/\x{1102}\x{1112}/\x{115D}/g;
  s/\x{1103}\x{1100}/\x{1117}/g;
  s/\x{1103}\x{1103}/\x{1104}/g;
  s/\x{1103}\x{1105}/\x{115E}/g;
  s/\x{1103}\x{1106}/\x{A960}/g;
  s/\x{1103}\x{1107}/\x{A961}/g;
  s/\x{1103}\x{1109}/\x{A962}/g;
  s/\x{1103}\x{110C}/\x{A963}/g;
  s/\x{1105}\x{1100}/\x{A964}/g;
  s/\x{1105}\x{1101}/\x{A965}/g;
  s/\x{1105}\x{1102}/\x{1118}/g;
  s/\x{1105}\x{1103}/\x{A966}/g;
  s/\x{1105}\x{1104}/\x{A967}/g;
  s/\x{1105}\x{1105}/\x{1119}/g;
  s/\x{1105}\x{1106}/\x{A968}/g;
  s/\x{1105}\x{1107}/\x{A969}/g;
  s/\x{1105}\x{1108}/\x{A96A}/g;
  s/\x{1105}\x{1109}/\x{A96C}/g;
  s/\x{1105}\x{110B}/\x{111B}/g;
  s/\x{1105}\x{110C}/\x{A96D}/g;
  s/\x{1105}\x{110F}/\x{A96E}/g;
  s/\x{1105}\x{1112}/\x{111A}/g;
  s/\x{1105}\x{112B}/\x{A96B}/g;
  s/\x{1106}\x{1100}/\x{A96F}/g;
  s/\x{1106}\x{1103}/\x{A970}/g;
  s/\x{1106}\x{1107}/\x{111C}/g;
  s/\x{1106}\x{1109}/\x{A971}/g;
  s/\x{1106}\x{110B}/\x{111D}/g;
  s/\x{1107}\x{1100}/\x{111E}/g;
  s/\x{1107}\x{1102}/\x{111F}/g;
  s/\x{1107}\x{1103}/\x{1120}/g;
  s/\x{1107}\x{1107}/\x{1108}/g;
  s/\x{1107}\x{1109}/\x{1121}/g;
  s/\x{1107}\x{110A}/\x{1125}/g;
  s/\x{1107}\x{110B}/\x{112B}/g;
  s/\x{1107}\x{110C}/\x{1127}/g;
  s/\x{1107}\x{110E}/\x{1128}/g;
  s/\x{1107}\x{110F}/\x{A973}/g;
  s/\x{1107}\x{1110}/\x{1129}/g;
  s/\x{1107}\x{1111}/\x{112A}/g;
  s/\x{1107}\x{1112}/\x{A974}/g;
  s/\x{1107}\x{112B}/\x{112C}/g;
  s/\x{1107}\x{112D}/\x{1122}/g;
  s/\x{1107}\x{112F}/\x{1123}/g;
  s/\x{1107}\x{1132}/\x{1124}/g;
  s/\x{1107}\x{1136}/\x{1126}/g;
  s/\x{1107}\x{1139}/\x{A972}/g;
  s/\x{1108}\x{110B}/\x{112C}/g;
  s/\x{1109}\x{1100}/\x{112D}/g;
  s/\x{1109}\x{1102}/\x{112E}/g;
  s/\x{1109}\x{1103}/\x{112F}/g;
  s/\x{1109}\x{1105}/\x{1130}/g;
  s/\x{1109}\x{1106}/\x{1131}/g;
  s/\x{1109}\x{1107}/\x{1132}/g;
  s/\x{1109}\x{1109}/\x{110A}/g;
  s/\x{1109}\x{110A}/\x{1134}/g;
  s/\x{1109}\x{110B}/\x{1135}/g;
  s/\x{1109}\x{110C}/\x{1136}/g;
  s/\x{1109}\x{110E}/\x{1137}/g;
  s/\x{1109}\x{110F}/\x{1138}/g;
  s/\x{1109}\x{1110}/\x{1139}/g;
  s/\x{1109}\x{1111}/\x{113A}/g;
  s/\x{1109}\x{1112}/\x{113B}/g;
  s/\x{1109}\x{111E}/\x{1133}/g;
  s/\x{1109}\x{1132}/\x{A975}/g;
  s/\x{110A}\x{1107}/\x{A975}/g;
  s/\x{110A}\x{1109}/\x{1134}/g;
  s/\x{110B}\x{1100}/\x{1141}/g;
  s/\x{110B}\x{1103}/\x{1142}/g;
  s/\x{110B}\x{1105}/\x{A976}/g;
  s/\x{110B}\x{1106}/\x{1143}/g;
  s/\x{110B}\x{1107}/\x{1144}/g;
  s/\x{110B}\x{1109}/\x{1145}/g;
  s/\x{110B}\x{110B}/\x{1147}/g;
  s/\x{110B}\x{110C}/\x{1148}/g;
  s/\x{110B}\x{110E}/\x{1149}/g;
  s/\x{110B}\x{1110}/\x{114A}/g;
  s/\x{110B}\x{1111}/\x{114B}/g;
  s/\x{110B}\x{1112}/\x{A977}/g;
  s/\x{110B}\x{1140}/\x{1146}/g;
  s/\x{110C}\x{110B}/\x{114D}/g;
  s/\x{110C}\x{110C}/\x{110D}/g;
  s/\x{110D}\x{1112}/\x{A978}/g;
  s/\x{110E}\x{110F}/\x{1152}/g;
  s/\x{110E}\x{1112}/\x{1153}/g;
  s/\x{1110}\x{1110}/\x{A979}/g;
  s/\x{1111}\x{1107}/\x{1156}/g;
  s/\x{1111}\x{110B}/\x{1157}/g;
  s/\x{1111}\x{1112}/\x{A97A}/g;
  s/\x{1112}\x{1109}/\x{A97B}/g;
  s/\x{1112}\x{1112}/\x{1158}/g;
  s/\x{1121}\x{1100}/\x{1122}/g;
  s/\x{1121}\x{1103}/\x{1123}/g;
  s/\x{1121}\x{1107}/\x{1124}/g;
  s/\x{1121}\x{1109}/\x{1125}/g;
  s/\x{1121}\x{110C}/\x{1126}/g;
  s/\x{1121}\x{1110}/\x{A972}/g;
  s/\x{1132}\x{1100}/\x{1133}/g;
  s/\x{113C}\x{113C}/\x{113D}/g;
  s/\x{113E}\x{113E}/\x{113F}/g;
  s/\x{114E}\x{114E}/\x{114F}/g;
  s/\x{1150}\x{1150}/\x{1151}/g;
  s/\x{1159}\x{1159}/\x{A97C}/g;
  s/\x{A964}\x{1100}/\x{A965}/g;
  s/\x{A966}\x{1103}/\x{A967}/g;
  s/\x{A969}\x{1107}/\x{A96A}/g;
  s/\x{A969}\x{110B}/\x{A96B}/g; # newly added
  s/\x{1169}\x{1161}\x{1175}/\x{116B}/g;
  s/\x{1169}\x{1163}\x{1175}/\x{11A7}/g;
  s/\x{1169}\x{1165}\x{1175}/\x{1180}/g;
  s/\x{1169}\x{1167}\x{1175}/\x{1181}/g;
  s/\x{1169}\x{1169}\x{1175}/\x{D7B1}/g;
  s/\x{116D}\x{1161}\x{1175}/\x{D7B3}/g;
  s/\x{116D}\x{1163}\x{1175}/\x{1185}/g;
  s/\x{116E}\x{1161}\x{1175}/\x{118A}/g;
  s/\x{116E}\x{1165}\x{1173}/\x{118B}/g;
  s/\x{116E}\x{1165}\x{1175}/\x{1170}/g;
  s/\x{116E}\x{1167}\x{1175}/\x{118C}/g;
  s/\x{116E}\x{1175}\x{1175}/\x{D7B6}/g;
  s/\x{1172}\x{1161}\x{1175}/\x{D7B7}/g;
  s/\x{1172}\x{1165}\x{1175}/\x{1190}/g;
  s/\x{1172}\x{1167}\x{1175}/\x{1192}/g;
  s/\x{1173}\x{1165}\x{1175}/\x{D7BB}/g;
  s/\x{1173}\x{1175}\x{116E}/\x{1197}/g;
  s/\x{1175}\x{1163}\x{1169}/\x{D7BD}/g;
  s/\x{1175}\x{1163}\x{1175}/\x{D7BE}/g;
  s/\x{1175}\x{1167}\x{1175}/\x{D7C0}/g;
  s/\x{1175}\x{1169}\x{1175}/\x{D7C1}/g;
  s/\x{119E}\x{1165}\x{1175}/\x{D7C6}/g;
  s/\x{1161}\x{1169}/\x{1176}/g;
  s/\x{1161}\x{116E}/\x{1177}/g;
  s/\x{1161}\x{1173}/\x{11A3}/g;
  s/\x{1161}\x{1175}/\x{1162}/g;
  s/\x{1163}\x{1169}/\x{1178}/g;
  s/\x{1163}\x{116D}/\x{1179}/g;
  s/\x{1163}\x{116E}/\x{11A4}/g;
  s/\x{1163}\x{1175}/\x{1164}/g;
  s/\x{1165}\x{1169}/\x{117A}/g;
  s/\x{1165}\x{116E}/\x{117B}/g;
  s/\x{1165}\x{1173}/\x{117C}/g;
  s/\x{1165}\x{1175}/\x{1166}/g;
  s/\x{1167}\x{1163}/\x{11A5}/g;
  s/\x{1167}\x{1169}/\x{117D}/g;
  s/\x{1167}\x{116E}/\x{117E}/g;
  s/\x{1167}\x{1175}/\x{1168}/g;
  s/\x{1169}\x{1161}/\x{116A}/g;
  s/\x{1169}\x{1162}/\x{116B}/g;
  s/\x{1169}\x{1163}/\x{11A6}/g;
  s/\x{1169}\x{1164}/\x{11A7}/g;
  s/\x{1169}\x{1165}/\x{117F}/g;
  s/\x{1169}\x{1166}/\x{1180}/g;
  s/\x{1169}\x{1167}/\x{D7B0}/g;
  s/\x{1169}\x{1168}/\x{1181}/g;
  s/\x{1169}\x{1169}/\x{1182}/g;
  s/\x{1169}\x{116C}/\x{D7B1}/g;
  s/\x{1169}\x{116E}/\x{1183}/g;
  s/\x{1169}\x{1175}/\x{116C}/g;
  s/\x{116A}\x{1175}/\x{116B}/g;
  s/\x{116D}\x{1161}/\x{D7B2}/g;
  s/\x{116D}\x{1162}/\x{D7B3}/g;
  s/\x{116D}\x{1163}/\x{1184}/g;
  s/\x{116D}\x{1164}/\x{1185}/g;
  s/\x{116D}\x{1165}/\x{D7B4}/g;
  s/\x{116D}\x{1167}/\x{1186}/g;
  s/\x{116D}\x{1169}/\x{1187}/g;
  s/\x{116D}\x{1175}/\x{1188}/g;
  s/\x{116E}\x{1161}/\x{1189}/g;
  s/\x{116E}\x{1162}/\x{118A}/g;
  s/\x{116E}\x{1165}/\x{116F}/g;
  s/\x{116E}\x{1166}/\x{1170}/g;
  s/\x{116E}\x{1167}/\x{D7B5}/g;
  s/\x{116E}\x{1168}/\x{118C}/g;
  s/\x{116E}\x{116E}/\x{118D}/g;
  s/\x{116E}\x{1175}/\x{1171}/g;
  s/\x{116E}\x{117C}/\x{118B}/g;
  s/\x{116E}\x{D7C4}/\x{D7B6}/g;
  s/\x{116F}\x{1173}/\x{118B}/g;
  s/\x{116F}\x{1175}/\x{1170}/g;
  s/\x{1171}\x{1175}/\x{D7B6}/g;
  s/\x{1172}\x{1161}/\x{118E}/g;
  s/\x{1172}\x{1162}/\x{D7B7}/g;
  s/\x{1172}\x{1165}/\x{118F}/g;
  s/\x{1172}\x{1166}/\x{1190}/g;
  s/\x{1172}\x{1167}/\x{1191}/g;
  s/\x{1172}\x{1168}/\x{1192}/g;
  s/\x{1172}\x{1169}/\x{D7B8}/g;
  s/\x{1172}\x{116E}/\x{1193}/g;
  s/\x{1172}\x{1175}/\x{1194}/g;
  s/\x{1173}\x{1161}/\x{D7B9}/g;
  s/\x{1173}\x{1165}/\x{D7BA}/g;
  s/\x{1173}\x{1166}/\x{D7BB}/g;
  s/\x{1173}\x{1169}/\x{D7BC}/g;
  s/\x{1173}\x{116E}/\x{1195}/g;
  s/\x{1173}\x{1173}/\x{1196}/g;
  s/\x{1173}\x{1175}/\x{1174}/g;
  s/\x{1173}\x{119B}/\x{1197}/g;
  s/\x{1174}\x{116E}/\x{1197}/g;
  s/\x{1175}\x{1161}/\x{1198}/g;
  s/\x{1175}\x{1163}/\x{1199}/g;
  s/\x{1175}\x{1164}/\x{D7BE}/g;
  s/\x{1175}\x{1167}/\x{D7BF}/g;
  s/\x{1175}\x{1168}/\x{D7C0}/g;
  s/\x{1175}\x{1169}/\x{119A}/g;
  s/\x{1175}\x{116C}/\x{D7C1}/g;
  s/\x{1175}\x{116D}/\x{D7C2}/g;
  s/\x{1175}\x{116E}/\x{119B}/g;
  s/\x{1175}\x{1172}/\x{D7C3}/g;
  s/\x{1175}\x{1173}/\x{119C}/g;
  s/\x{1175}\x{1175}/\x{D7C4}/g;
  s/\x{1175}\x{1178}/\x{D7BD}/g;
  s/\x{1175}\x{119E}/\x{119D}/g;
  s/\x{117F}\x{1175}/\x{1180}/g;
  s/\x{1182}\x{1175}/\x{D7B1}/g;
  s/\x{1184}\x{1175}/\x{1185}/g;
  s/\x{1189}\x{1175}/\x{118A}/g;
  s/\x{118E}\x{1175}/\x{D7B7}/g; # newly added
  s/\x{118F}\x{1175}/\x{1190}/g;
  s/\x{1191}\x{1175}/\x{1192}/g;
  s/\x{1199}\x{1169}/\x{D7BD}/g;
  s/\x{1199}\x{1175}/\x{D7BE}/g; # newly added
  s/\x{119A}\x{1175}/\x{D7C1}/g;
  s/\x{119E}\x{1161}/\x{D7C5}/g;
  s/\x{119E}\x{1165}/\x{119F}/g;
  s/\x{119E}\x{1166}/\x{D7C6}/g;
  s/\x{119E}\x{116E}/\x{11A0}/g;
  s/\x{119E}\x{1175}/\x{11A1}/g;
  s/\x{119E}\x{119E}/\x{11A2}/g;
  s/\x{119F}\x{1175}/\x{D7C6}/g; # newly added
  s/\x{11A6}\x{1175}/\x{11A7}/g; # newly added
  s/\x{D7B0}\x{1175}/\x{1181}/g; # newly added
  s/\x{D7B2}\x{1175}/\x{D7B3}/g; # newly added
  s/\x{D7B5}\x{1175}/\x{118C}/g; # newly added
  s/\x{D7BA}\x{1175}/\x{D7BB}/g; # newly added
  s/\x{D7BF}\x{1175}/\x{D7C0}/g; # newly added
  s/\x{11A8}\x{11BA}\x{11A8}/\x{11C4}/g;
  s/\x{11AE}\x{11AE}\x{11B8}/\x{D7CE}/g;
  s/\x{11AE}\x{11BA}\x{11A8}/\x{D7D1}/g;
  s/\x{11AF}\x{11A8}\x{11A8}/\x{D7D5}/g;
  s/\x{11AF}\x{11A8}\x{11BA}/\x{11CC}/g;
  s/\x{11AF}\x{11A8}\x{11C2}/\x{D7D6}/g;
  s/\x{11AF}\x{11AE}\x{11C2}/\x{11CF}/g;
  s/\x{11AF}\x{11AF}\x{11BF}/\x{D7D7}/g;
  s/\x{11AF}\x{11B7}\x{11A8}/\x{11D1}/g;
  s/\x{11AF}\x{11B7}\x{11BA}/\x{11D2}/g;
  s/\x{11AF}\x{11B7}\x{11C2}/\x{D7D8}/g;
  s/\x{11AF}\x{11B8}\x{11AE}/\x{D7D9}/g;
  s/\x{11AF}\x{11B8}\x{11BA}/\x{11D3}/g;
  s/\x{11AF}\x{11B8}\x{11BC}/\x{11D5}/g;
  s/\x{11AF}\x{11B8}\x{11C1}/\x{D7DA}/g;
  s/\x{11AF}\x{11B8}\x{11C2}/\x{11D4}/g;
  s/\x{11AF}\x{11BA}\x{11BA}/\x{11D6}/g;
  s/\x{11AF}\x{11F9}\x{11C2}/\x{D7DC}/g;
  s/\x{11B7}\x{11AB}\x{11AB}/\x{D7DF}/g;
  s/\x{11B7}\x{11B8}\x{11BA}/\x{D7E1}/g;
  s/\x{11B7}\x{11BA}\x{11BA}/\x{11DE}/g;
  s/\x{11B8}\x{11AF}\x{11C1}/\x{D7E4}/g;
  s/\x{11B8}\x{11BA}\x{11AE}/\x{D7E7}/g;
  s/\x{11BA}\x{11B8}\x{11BC}/\x{D7EB}/g;
  s/\x{11BA}\x{11BA}\x{11A8}/\x{D7EC}/g;
  s/\x{11BA}\x{11BA}\x{11AE}/\x{D7ED}/g;
  s/\x{11BC}\x{11A8}\x{11A8}/\x{11ED}/g; # legacy enc
  s/\x{11BD}\x{11B8}\x{11B8}/\x{D7F8}/g;
  s/\x{11EB}\x{11B8}\x{11BC}/\x{D7F4}/g;
  s/\x{11F0}\x{11A8}\x{11A8}/\x{11ED}/g;
  s/\x{11A8}\x{11A8}/\x{11A9}/g;
  s/\x{11A8}\x{11AB}/\x{11FA}/g;
  s/\x{11A8}\x{11AF}/\x{11C3}/g;
  s/\x{11A8}\x{11B8}/\x{11FB}/g;
  s/\x{11A8}\x{11BA}/\x{11AA}/g;
  s/\x{11A8}\x{11BE}/\x{11FC}/g;
  s/\x{11A8}\x{11BF}/\x{11FD}/g;
  s/\x{11A8}\x{11C2}/\x{11FE}/g;
  s/\x{11A8}\x{11E7}/\x{11C4}/g;
  s/\x{11AA}\x{11A8}/\x{11C4}/g;
  s/\x{11AB}\x{11A8}/\x{11C5}/g;
  s/\x{11AB}\x{11AB}/\x{11FF}/g;
  s/\x{11AB}\x{11AE}/\x{11C6}/g;
  s/\x{11AB}\x{11AF}/\x{D7CB}/g;
  s/\x{11AB}\x{11BA}/\x{11C7}/g;
  s/\x{11AB}\x{11BD}/\x{11AC}/g;
  s/\x{11AB}\x{11BE}/\x{D7CC}/g;
  s/\x{11AB}\x{11C0}/\x{11C9}/g;
  s/\x{11AB}\x{11C2}/\x{11AD}/g;
  s/\x{11AB}\x{11EB}/\x{11C8}/g;
  s/\x{11AE}\x{11A8}/\x{11CA}/g;
  s/\x{11AE}\x{11AE}/\x{D7CD}/g;
  s/\x{11AE}\x{11AF}/\x{11CB}/g;
  s/\x{11AE}\x{11B8}/\x{D7CF}/g;
  s/\x{11AE}\x{11BA}/\x{D7D0}/g;
  s/\x{11AE}\x{11BD}/\x{D7D2}/g;
  s/\x{11AE}\x{11BE}/\x{D7D3}/g;
  s/\x{11AE}\x{11C0}/\x{D7D4}/g;
  s/\x{11AE}\x{11E7}/\x{D7D1}/g;
  s/\x{11AE}\x{D7CF}/\x{D7CE}/g;
  s/\x{11AF}\x{11A8}/\x{11B0}/g;
  s/\x{11AF}\x{11A9}/\x{D7D5}/g;
  s/\x{11AF}\x{11AA}/\x{11CC}/g;
  s/\x{11AF}\x{11AB}/\x{11CD}/g;
  s/\x{11AF}\x{11AE}/\x{11CE}/g;
  s/\x{11AF}\x{11AF}/\x{11D0}/g;
  s/\x{11AF}\x{11B7}/\x{11B1}/g;
  s/\x{11AF}\x{11B8}/\x{11B2}/g;
  s/\x{11AF}\x{11B9}/\x{11D3}/g;
  s/\x{11AF}\x{11BA}/\x{11B3}/g;
  s/\x{11AF}\x{11BB}/\x{11D6}/g;
  s/\x{11AF}\x{11BC}/\x{D7DD}/g;
  s/\x{11AF}\x{11BF}/\x{11D8}/g;
  s/\x{11AF}\x{11C0}/\x{11B4}/g;
  s/\x{11AF}\x{11C1}/\x{11B5}/g;
  s/\x{11AF}\x{11C2}/\x{11B6}/g;
  s/\x{11AF}\x{11D8}/\x{D7D7}/g;
  s/\x{11AF}\x{11DA}/\x{11D1}/g;
  s/\x{11AF}\x{11DD}/\x{11D2}/g;
  s/\x{11AF}\x{11E1}/\x{D7D8}/g;
  s/\x{11AF}\x{11E4}/\x{D7DA}/g;
  s/\x{11AF}\x{11E5}/\x{11D4}/g;
  s/\x{11AF}\x{11E6}/\x{11D5}/g;
  s/\x{11AF}\x{11EB}/\x{11D7}/g;
  s/\x{11AF}\x{11F0}/\x{D7DB}/g;
  s/\x{11AF}\x{11F9}/\x{11D9}/g;
  s/\x{11AF}\x{11FE}/\x{D7D6}/g;
  s/\x{11AF}\x{D7E3}/\x{D7D9}/g;
  s/\x{11B0}\x{11A8}/\x{D7D5}/g;
  s/\x{11B0}\x{11BA}/\x{11CC}/g;
  s/\x{11B0}\x{11C2}/\x{D7D6}/g;
  s/\x{11B1}\x{11A8}/\x{11D1}/g;
  s/\x{11B1}\x{11BA}/\x{11D2}/g;
  s/\x{11B1}\x{11C2}/\x{D7D8}/g;
  s/\x{11B2}\x{11AE}/\x{D7D9}/g;
  s/\x{11B2}\x{11BA}/\x{11D3}/g;
  s/\x{11B2}\x{11BC}/\x{11D5}/g;
  s/\x{11B2}\x{11C1}/\x{D7DA}/g;
  s/\x{11B2}\x{11C2}/\x{11D4}/g;
  s/\x{11B3}\x{11BA}/\x{11D6}/g;
  s/\x{11B7}\x{11A8}/\x{11DA}/g;
  s/\x{11B7}\x{11AB}/\x{D7DE}/g;
  s/\x{11B7}\x{11AF}/\x{11DB}/g;
  s/\x{11B7}\x{11B7}/\x{D7E0}/g;
  s/\x{11B7}\x{11B8}/\x{11DC}/g;
  s/\x{11B7}\x{11B9}/\x{D7E1}/g;
  s/\x{11B7}\x{11BA}/\x{11DD}/g;
  s/\x{11B7}\x{11BB}/\x{11DE}/g;
  s/\x{11B7}\x{11BC}/\x{11E2}/g;
  s/\x{11B7}\x{11BD}/\x{D7E2}/g;
  s/\x{11B7}\x{11BE}/\x{11E0}/g;
  s/\x{11B7}\x{11C2}/\x{11E1}/g;
  s/\x{11B7}\x{11EB}/\x{11DF}/g;
  s/\x{11B7}\x{11FF}/\x{D7DF}/g;
  s/\x{11B8}\x{11AE}/\x{D7E3}/g;
  s/\x{11B8}\x{11AF}/\x{11E3}/g;
  s/\x{11B8}\x{11B5}/\x{D7E4}/g;
  s/\x{11B8}\x{11B7}/\x{D7E5}/g;
  s/\x{11B8}\x{11B8}/\x{D7E6}/g;
  s/\x{11B8}\x{11BA}/\x{11B9}/g;
  s/\x{11B8}\x{11BC}/\x{11E6}/g;
  s/\x{11B8}\x{11BD}/\x{D7E8}/g;
  s/\x{11B8}\x{11BE}/\x{D7E9}/g;
  s/\x{11B8}\x{11C1}/\x{11E4}/g;
  s/\x{11B8}\x{11C2}/\x{11E5}/g;
  s/\x{11B8}\x{11E8}/\x{D7E7}/g;
  s/\x{11B9}\x{11AE}/\x{D7E7}/g;
  s/\x{11BA}\x{11A8}/\x{11E7}/g;
  s/\x{11BA}\x{11AE}/\x{11E8}/g;
  s/\x{11BA}\x{11AF}/\x{11E9}/g;
  s/\x{11BA}\x{11B7}/\x{D7EA}/g;
  s/\x{11BA}\x{11B8}/\x{11EA}/g;
  s/\x{11BA}\x{11BA}/\x{11BB}/g;
  s/\x{11BA}\x{11BD}/\x{D7EF}/g;
  s/\x{11BA}\x{11BE}/\x{D7F0}/g;
  s/\x{11BA}\x{11C0}/\x{D7F1}/g;
  s/\x{11BA}\x{11C2}/\x{D7F2}/g;
  s/\x{11BA}\x{11E6}/\x{D7EB}/g;
  s/\x{11BA}\x{11E7}/\x{D7EC}/g;
  s/\x{11BA}\x{11E8}/\x{D7ED}/g;
  s/\x{11BA}\x{11EB}/\x{D7EE}/g;
  s/\x{11BB}\x{11A8}/\x{D7EC}/g;
  s/\x{11BB}\x{11AE}/\x{D7ED}/g;
  s/\x{11BC}\x{11A8}/\x{11EC}/g; # legacy enc
  s/\x{11BC}\x{11A9}/\x{11ED}/g; # legacy enc
  s/\x{11BC}\x{11B7}/\x{D7F5}/g; # legacy enc ㅇㅁ(ms)
  s/\x{11BC}\x{11BA}/\x{11F1}/g; # legacy enc ㅇᆺ (ms)
  s/\x{11BC}\x{11BC}/\x{11EE}/g; # legacy enc
  s/\x{11BC}\x{11BF}/\x{11EF}/g; # legacy enc
  s/\x{11BC}\x{11C2}/\x{D7F6}/g; # legacy enc ㅇᇂ (ms)
  s/\x{11BD}\x{11B8}/\x{D7F7}/g;
  s/\x{11BD}\x{11BD}/\x{D7F9}/g;
  s/\x{11BD}\x{D7E6}/\x{D7F8}/g;
  s/\x{11C1}\x{11B8}/\x{11F3}/g;
  s/\x{11C1}\x{11BA}/\x{D7FA}/g;
  s/\x{11C1}\x{11BC}/\x{11F4}/g;
  s/\x{11C1}\x{11C0}/\x{D7FB}/g;
  s/\x{11C2}\x{11AB}/\x{11F5}/g;
  s/\x{11C2}\x{11AF}/\x{11F6}/g;
  s/\x{11C2}\x{11B7}/\x{11F7}/g;
  s/\x{11C2}\x{11B8}/\x{11F8}/g;
  s/\x{11CE}\x{11C2}/\x{11CF}/g;
  s/\x{11D0}\x{11BF}/\x{D7D7}/g;
  s/\x{11D9}\x{11C2}/\x{D7DC}/g;
  s/\x{11DC}\x{11BA}/\x{D7E1}/g;
  s/\x{11DD}\x{11BA}/\x{11DE}/g;
  s/\x{11E3}\x{11C1}/\x{D7E4}/g;
  s/\x{11EA}\x{11BC}/\x{D7EB}/g;
  s/\x{11EB}\x{11B8}/\x{D7F3}/g;
  s/\x{11EB}\x{11E6}/\x{D7F4}/g;
  s/\x{11EC}\x{11A8}/\x{11ED}/g;
  s/\x{11F0}\x{11A8}/\x{11EC}/g;
  s/\x{11F0}\x{11A9}/\x{11ED}/g;
  s/\x{11F0}\x{11B7}/\x{D7F5}/g;
  s/\x{11F0}\x{11BA}/\x{11F1}/g;
  s/\x{11F0}\x{11BF}/\x{11EF}/g;
  s/\x{11F0}\x{11C2}/\x{D7F6}/g;
  s/\x{11F0}\x{11EB}/\x{11F2}/g;
  s/\x{11F0}\x{11F0}/\x{11EE}/g;
  s/\x{D7CD}\x{11B8}/\x{D7CE}/g;
  s/\x{D7D0}\x{11A8}/\x{D7D1}/g;
  s/\x{D7DE}\x{11AB}/\x{D7DF}/g;
  s/\x{D7F3}\x{11BC}/\x{D7F4}/g;
  s/\x{D7F7}\x{11B8}/\x{D7F8}/g;
}

sub hypuasingle2jamo {
  s/\x{F784}/\x{115F}/g;
  s/\x{F785}/\x{1100}/g;
  s/\x{F786}/\x{1101}/g;
  s/\x{F787}/\x{115A}/g;
  s/\x{F788}/\x{1102}/g;
  s/\x{F789}/\x{1113}/g;
  s/\x{F78A}/\x{1114}/g;
  s/\x{F78B}/\x{1115}/g;
  s/\x{F78C}/\x{1116}/g;
  s/\x{F78D}/\x{115B}/g;
  s/\x{F78E}/\x{115C}/g;
  s/\x{F78F}/\x{115D}/g;
  s/\x{F790}/\x{1103}/g;
  s/\x{F791}/\x{1117}/g;
  s/\x{F792}/\x{1104}/g;
  s/\x{F793}/\x{115E}/g;
  s/\x{F794}/\x{A960}/g;
  s/\x{F795}/\x{A961}/g;
  s/\x{F796}/\x{A962}/g;
  s/\x{F797}/\x{A963}/g;
  s/\x{F798}/\x{1105}/g;
  s/\x{F799}/\x{A964}/g;
  s/\x{F79A}/\x{A965}/g;
  s/\x{F79B}/\x{1118}/g;
  s/\x{F79C}/\x{A966}/g;
  s/\x{F79D}/\x{A967}/g;
  s/\x{F79E}/\x{1119}/g;
  s/\x{F79F}/\x{A968}/g;
  s/\x{F7A0}/\x{A969}/g;
  s/\x{F7A1}/\x{A96A}/g;
  s/\x{F7A2}/\x{A96B}/g;
  s/\x{F7A3}/\x{A96C}/g;
  s/\x{F7A4}/\x{A96D}/g;
  s/\x{F7A5}/\x{A96E}/g;
  s/\x{F7A6}/\x{111A}/g;
  s/\x{F7A7}/\x{111B}/g;
  s/\x{F7A8}/\x{1106}/g;
  s/\x{F7A9}/\x{A96F}/g;
  s/\x{F7AA}/\x{A970}/g;
  s/\x{F7AB}/\x{111C}/g;
  s/\x{F7AC}/\x{A971}/g;
  s/\x{F7AD}/\x{111D}/g;
  s/\x{F7AE}/\x{1107}/g;
  s/\x{F7AF}/\x{111E}/g;
  s/\x{F7B0}/\x{111F}/g;
  s/\x{F7B1}/\x{1120}/g;
  s/\x{F7B2}/\x{1108}/g;
  s/\x{F7B3}/\x{1121}/g;
  s/\x{F7B4}/\x{1122}/g;
  s/\x{F7B5}/\x{1123}/g;
  s/\x{F7B6}/\x{1124}/g;
  s/\x{F7B7}/\x{1125}/g;
  s/\x{F7B8}/\x{1126}/g;
  s/\x{F7B9}/\x{A972}/g;
  s/\x{F7BA}/\x{1127}/g;
  s/\x{F7BB}/\x{1128}/g;
  s/\x{F7BC}/\x{A973}/g;
  s/\x{F7BD}/\x{1129}/g;
  s/\x{F7BE}/\x{112A}/g;
  s/\x{F7BF}/\x{A974}/g;
  s/\x{F7C0}/\x{112B}/g;
  s/\x{F7C1}/\x{112C}/g;
  s/\x{F7C2}/\x{1109}/g;
  s/\x{F7C3}/\x{112D}/g;
  s/\x{F7C4}/\x{112E}/g;
  s/\x{F7C5}/\x{112F}/g;
  s/\x{F7C6}/\x{1130}/g;
  s/\x{F7C7}/\x{1131}/g;
  s/\x{F7C8}/\x{1132}/g;
  s/\x{F7C9}/\x{1133}/g;
  s/\x{F7CA}/\x{110A}/g;
  s/\x{F7CB}/\x{A975}/g;
  s/\x{F7CC}/\x{1134}/g;
  s/\x{F7CD}/\x{1135}/g;
  s/\x{F7CE}/\x{1136}/g;
  s/\x{F7CF}/\x{1137}/g;
  s/\x{F7D0}/\x{1138}/g;
  s/\x{F7D1}/\x{1139}/g;
  s/\x{F7D2}/\x{113A}/g;
  s/\x{F7D3}/\x{113B}/g;
  s/\x{F7D4}/\x{113C}/g;
  s/\x{F7D5}/\x{113D}/g;
  s/\x{F7D6}/\x{113E}/g;
  s/\x{F7D7}/\x{113F}/g;
  s/\x{F7D8}/\x{1140}/g;
  s/\x{F7D9}/\x{110B}/g;
  s/\x{F7DA}/\x{1141}/g;
  s/\x{F7DB}/\x{1142}/g;
  s/\x{F7DC}/\x{A976}/g;
  s/\x{F7DD}/\x{1143}/g;
  s/\x{F7DE}/\x{1144}/g;
  s/\x{F7DF}/\x{1145}/g;
  s/\x{F7E0}/\x{1146}/g;
  s/\x{F7E1}/\x{1147}/g;
  s/\x{F7E2}/\x{1148}/g;
  s/\x{F7E3}/\x{1149}/g;
  s/\x{F7E4}/\x{114A}/g;
  s/\x{F7E5}/\x{114B}/g;
  s/\x{F7E6}/\x{A977}/g;
  s/\x{F7E7}/\x{114C}/g;
  s/\x{F7E8}/\x{110C}/g;
  s/\x{F7E9}/\x{114D}/g;
  s/\x{F7EA}/\x{110D}/g;
  s/\x{F7EB}/\x{A978}/g;
  s/\x{F7EC}/\x{114E}/g;
  s/\x{F7ED}/\x{114F}/g;
  s/\x{F7EE}/\x{1150}/g;
  s/\x{F7EF}/\x{1151}/g;
  s/\x{F7F0}/\x{110E}/g;
  s/\x{F7F1}/\x{1152}/g;
  s/\x{F7F2}/\x{1153}/g;
  s/\x{F7F3}/\x{1154}/g;
  s/\x{F7F4}/\x{1155}/g;
  s/\x{F7F5}/\x{110F}/g;
  s/\x{F7F6}/\x{1110}/g;
  s/\x{F7F7}/\x{A979}/g;
  s/\x{F7F8}/\x{1111}/g;
  s/\x{F7F9}/\x{1156}/g;
  s/\x{F7FA}/\x{A97A}/g;
  s/\x{F7FB}/\x{1157}/g;
  s/\x{F7FC}/\x{1112}/g;
  s/\x{F7FD}/\x{A97B}/g;
  s/\x{F7FE}/\x{1158}/g;
  s/\x{F7FF}/\x{1159}/g;
  s/\x{F800}/\x{A97C}/g;
  s/\x{F806}/\x{1160}/g;
  s/\x{F807}/\x{1161}/g;
  s/\x{F808}/\x{1176}/g;
  s/\x{F809}/\x{1177}/g;
  s/\x{F80A}/\x{11A3}/g;
  s/\x{F80B}/\x{1162}/g;
  s/\x{F80C}/\x{1163}/g;
  s/\x{F80D}/\x{1178}/g;
  s/\x{F80E}/\x{1179}/g;
  s/\x{F80F}/\x{11A4}/g;
  s/\x{F810}/\x{1164}/g;
  s/\x{F811}/\x{1165}/g;
  s/\x{F812}/\x{117A}/g;
  s/\x{F813}/\x{117B}/g;
  s/\x{F814}/\x{117C}/g;
  s/\x{F815}/\x{1166}/g;
  s/\x{F816}/\x{1167}/g;
  s/\x{F817}/\x{11A5}/g;
  s/\x{F818}/\x{117D}/g;
  s/\x{F819}/\x{117E}/g;
  s/\x{F81A}/\x{1168}/g;
  s/\x{F81B}/\x{1169}/g;
  s/\x{F81C}/\x{116A}/g;
  s/\x{F81D}/\x{116B}/g;
  s/\x{F81E}/\x{11A6}/g;
  s/\x{F81F}/\x{11A7}/g;
  s/\x{F820}/\x{117F}/g;
  s/\x{F821}/\x{1180}/g;
  s/\x{F822}/\x{D7B0}/g;
  s/\x{F823}/\x{1181}/g;
  s/\x{F824}/\x{1182}/g;
  s/\x{F825}/\x{D7B1}/g;
  s/\x{F826}/\x{1183}/g;
  s/\x{F827}/\x{116C}/g;
  s/\x{F828}/\x{116D}/g;
  s/\x{F829}/\x{D7B2}/g;
  s/\x{F82A}/\x{D7B3}/g;
  s/\x{F82B}/\x{1184}/g;
  s/\x{F82C}/\x{1185}/g;
  s/\x{F82D}/\x{D7B4}/g;
  s/\x{F82E}/\x{1186}/g;
  s/\x{F82F}/\x{1187}/g;
  s/\x{F830}/\x{1188}/g;
  s/\x{F831}/\x{116E}/g;
  s/\x{F832}/\x{1189}/g;
  s/\x{F833}/\x{118A}/g;
  s/\x{F834}/\x{116F}/g;
  s/\x{F835}/\x{118B}/g;
  s/\x{F836}/\x{1170}/g;
  s/\x{F837}/\x{D7B5}/g;
  s/\x{F838}/\x{118C}/g;
  s/\x{F839}/\x{118D}/g;
  s/\x{F83A}/\x{1171}/g;
  s/\x{F83B}/\x{D7B6}/g;
  s/\x{F83C}/\x{1172}/g;
  s/\x{F83D}/\x{118E}/g;
  s/\x{F83E}/\x{D7B7}/g;
  s/\x{F83F}/\x{118F}/g;
  s/\x{F840}/\x{1190}/g;
  s/\x{F841}/\x{1191}/g;
  s/\x{F842}/\x{1192}/g;
  s/\x{F843}/\x{D7B8}/g;
  s/\x{F844}/\x{1193}/g;
  s/\x{F845}/\x{1194}/g;
  s/\x{F846}/\x{1173}/g;
  s/\x{F847}/\x{D7B9}/g;
  s/\x{F848}/\x{D7BA}/g;
  s/\x{F849}/\x{D7BB}/g;
  s/\x{F84A}/\x{D7BC}/g;
  s/\x{F84B}/\x{1195}/g;
  s/\x{F84C}/\x{1196}/g;
  s/\x{F84D}/\x{1174}/g;
  s/\x{F84E}/\x{1197}/g;
  s/\x{F84F}/\x{1175}/g;
  s/\x{F850}/\x{1198}/g;
  s/\x{F851}/\x{1199}/g;
  s/\x{F852}/\x{D7BD}/g;
  s/\x{F853}/\x{D7BE}/g;
  s/\x{F854}/\x{D7BF}/g;
  s/\x{F855}/\x{D7C0}/g;
  s/\x{F856}/\x{119A}/g;
  s/\x{F857}/\x{D7C1}/g;
  s/\x{F858}/\x{D7C2}/g;
  s/\x{F859}/\x{119B}/g;
  s/\x{F85A}/\x{D7C3}/g;
  s/\x{F85B}/\x{119C}/g;
  s/\x{F85C}/\x{D7C4}/g;
  s/\x{F85D}/\x{119D}/g;
  s/\x{F85E}/\x{119E}/g;
  s/\x{F85F}/\x{D7C5}/g;
  s/\x{F860}/\x{119F}/g;
  s/\x{F861}/\x{D7C6}/g;
  s/\x{F862}/\x{11A0}/g;
  s/\x{F863}/\x{11A1}/g;
  s/\x{F864}/\x{11A2}/g;
  s/\x{F86A}//g;
  s/\x{F86B}/\x{11A8}/g;
  s/\x{F86C}/\x{11A9}/g;
  s/\x{F86D}/\x{11FA}/g;
  s/\x{F86E}/\x{11C3}/g;
  s/\x{F86F}/\x{11FB}/g;
  s/\x{F870}/\x{11AA}/g;
  s/\x{F871}/\x{11C4}/g;
  s/\x{F872}/\x{11FC}/g;
  s/\x{F873}/\x{11FD}/g;
  s/\x{F874}/\x{11FE}/g;
  s/\x{F875}/\x{11AB}/g;
  s/\x{F876}/\x{11C5}/g;
  s/\x{F877}/\x{11FF}/g;
  s/\x{F878}/\x{11C6}/g;
  s/\x{F879}/\x{D7CB}/g;
  s/\x{F87A}/\x{11C7}/g;
  s/\x{F87B}/\x{11C8}/g;
  s/\x{F87C}/\x{11AC}/g;
  s/\x{F87D}/\x{D7CC}/g;
  s/\x{F87E}/\x{11C9}/g;
  s/\x{F87F}/\x{11AD}/g;
  s/\x{F880}/\x{11AE}/g;
  s/\x{F881}/\x{11CA}/g;
  s/\x{F882}/\x{D7CD}/g;
  s/\x{F883}/\x{D7CE}/g;
  s/\x{F884}/\x{11CB}/g;
  s/\x{F885}/\x{D7CF}/g;
  s/\x{F886}/\x{D7D0}/g;
  s/\x{F887}/\x{D7D1}/g;
  s/\x{F888}/\x{D7D2}/g;
  s/\x{F889}/\x{D7D3}/g;
  s/\x{F88A}/\x{D7D4}/g;
  s/\x{F88B}/\x{11AF}/g;
  s/\x{F88C}/\x{11B0}/g;
  s/\x{F88D}/\x{D7D5}/g;
  s/\x{F88E}/\x{11CC}/g;
  s/\x{F88F}/\x{D7D6}/g;
  s/\x{F890}/\x{11CD}/g;
  s/\x{F891}/\x{11CE}/g;
  s/\x{F892}/\x{11CF}/g;
  s/\x{F893}/\x{11D0}/g;
  s/\x{F894}/\x{D7D7}/g;
  s/\x{F895}/\x{11B1}/g;
  s/\x{F896}/\x{11D1}/g;
  s/\x{F897}/\x{11D2}/g;
  s/\x{F898}/\x{D7D8}/g;
  s/\x{F899}/\x{11B2}/g;
  s/\x{F89A}/\x{D7D9}/g;
  s/\x{F89B}/\x{11D3}/g;
  s/\x{F89C}/\x{D7DA}/g;
  s/\x{F89D}/\x{11D4}/g;
  s/\x{F89E}/\x{11D5}/g;
  s/\x{F89F}/\x{11B3}/g;
  s/\x{F8A0}/\x{11D6}/g;
  s/\x{F8A1}/\x{11D7}/g;
  s/\x{F8A2}/\x{D7DB}/g;
  s/\x{F8A3}/\x{11D8}/g;
  s/\x{F8A4}/\x{11B4}/g;
  s/\x{F8A5}/\x{11B5}/g;
  s/\x{F8A6}/\x{11B6}/g;
  s/\x{F8A7}/\x{11D9}/g;
  s/\x{F8A8}/\x{D7DC}/g;
  s/\x{F8A9}/\x{D7DD}/g;
  s/\x{F8AA}/\x{11B7}/g;
  s/\x{F8AB}/\x{11DA}/g;
  s/\x{F8AC}/\x{D7DE}/g;
  s/\x{F8AD}/\x{D7DF}/g;
  s/\x{F8AE}/\x{11DB}/g;
  s/\x{F8AF}/\x{D7E0}/g;
  s/\x{F8B0}/\x{11DC}/g;
  s/\x{F8B1}/\x{D7E1}/g;
  s/\x{F8B2}/\x{11DD}/g;
  s/\x{F8B3}/\x{11DE}/g;
  s/\x{F8B4}/\x{11DF}/g;
  s/\x{F8B5}/\x{D7E2}/g;
  s/\x{F8B6}/\x{11E0}/g;
  s/\x{F8B7}/\x{11E1}/g;
  s/\x{F8B8}/\x{11E2}/g;
  s/\x{F8B9}/\x{11B8}/g;
  s/\x{F8BA}/\x{D7E3}/g;
  s/\x{F8BB}/\x{11E3}/g;
  s/\x{F8BC}/\x{D7E4}/g;
  s/\x{F8BD}/\x{D7E5}/g;
  s/\x{F8BE}/\x{D7E6}/g;
  s/\x{F8BF}/\x{11B9}/g;
  s/\x{F8C0}/\x{D7E7}/g;
  s/\x{F8C1}/\x{D7E8}/g;
  s/\x{F8C2}/\x{D7E9}/g;
  s/\x{F8C3}/\x{11E4}/g;
  s/\x{F8C4}/\x{11E5}/g;
  s/\x{F8C5}/\x{11E6}/g;
  s/\x{F8C6}/\x{11BA}/g;
  s/\x{F8C7}/\x{11E7}/g;
  s/\x{F8C8}/\x{11E8}/g;
  s/\x{F8C9}/\x{11E9}/g;
  s/\x{F8CA}/\x{D7EA}/g;
  s/\x{F8CB}/\x{11EA}/g;
  s/\x{F8CC}/\x{D7EB}/g;
  s/\x{F8CD}/\x{11BB}/g;
  s/\x{F8CE}/\x{D7EC}/g;
  s/\x{F8CF}/\x{D7ED}/g;
  s/\x{F8D0}/\x{D7EE}/g;
  s/\x{F8D1}/\x{D7EF}/g;
  s/\x{F8D2}/\x{D7F0}/g;
  s/\x{F8D3}/\x{D7F1}/g;
  s/\x{F8D4}/\x{D7F2}/g;
  s/\x{F8D5}/\x{11EB}/g;
  s/\x{F8D6}/\x{D7F3}/g;
  s/\x{F8D7}/\x{D7F4}/g;
  s/\x{F8D8}/\x{11BC}/g;
  s/\x{F8D9}/\x{11EC}/g;
  s/\x{F8DA}/\x{11ED}/g;
  s/\x{F8DB}/\x{D7F5}/g;
  s/\x{F8DC}/\x{11F1}/g;
  s/\x{F8DD}/\x{11EE}/g;
  s/\x{F8DE}/\x{11EF}/g;
  s/\x{F8DF}/\x{D7F6}/g;
  s/\x{F8E0}/\x{11F0}/g;
  s/\x{F8E1}/\x{11EC}/g;
  s/\x{F8E2}/\x{11F1}/g;
  s/\x{F8E3}/\x{11F2}/g;
  s/\x{F8E4}/\x{11EF}/g;
  s/\x{F8E5}/\x{D7F6}/g;
  s/\x{F8E6}/\x{11BD}/g;
  s/\x{F8E7}/\x{D7F7}/g;
  s/\x{F8E8}/\x{D7F8}/g;
  s/\x{F8E9}/\x{D7F9}/g;
  s/\x{F8EA}/\x{11BE}/g;
  s/\x{F8EB}/\x{11BF}/g;
  s/\x{F8EC}/\x{11C0}/g;
  s/\x{F8ED}/\x{11C1}/g;
  s/\x{F8EE}/\x{11F3}/g;
  s/\x{F8EF}/\x{D7FA}/g;
  s/\x{F8F0}/\x{D7FB}/g;
  s/\x{F8F1}/\x{11F4}/g;
  s/\x{F8F2}/\x{11C2}/g;
  s/\x{F8F3}/\x{11F5}/g;
  s/\x{F8F4}/\x{11F6}/g;
  s/\x{F8F5}/\x{11F7}/g;
  s/\x{F8F6}/\x{11F8}/g;
  s/\x{F8F7}/\x{11F9}/g;
}

sub arr_hypua2jamo {
  return (
    [ 0xE0BC, 0x115F,0x1161,0x11AE ],
    [ 0xE0BD, 0x115F,0x1161,0xD7CD ],
    [ 0xE0BE, 0x115F,0x1161,0x11AF ],
    [ 0xE0BF, 0x115F,0x1161,0x11B0 ],
    [ 0xE0C0, 0x115F,0x1161,0x11B1 ],
    [ 0xE0C1, 0x115F,0x1161,0x11B2 ],
    [ 0xE0C2, 0x115F,0x1161,0x11BA ],
    [ 0xE0C3, 0x115F,0x1161,0x11BB ],
    [ 0xE0C4, 0x115F,0x1161,0x11BD ],
    [ 0xE0C5, 0x115F,0x1161,0xD7F9 ],
    [ 0xE0C7, 0x115F,0x11A3,0x11AE ],
    [ 0xE0C8, 0x115F,0x1163,0x11AB ],
    [ 0xE0C9, 0x115F,0x1165,0x11AF ],
    [ 0xE0CA, 0x115F,0x1165,0x11B8 ],
    [ 0xE0CB, 0x115F,0x116E,0x11AB ],
    [ 0xE0CC, 0x115F,0x1175,0x11A8 ],
    [ 0xE0CD, 0x115F,0x1175,0x11AB ],
    [ 0xE0CE, 0x115F,0x1175,0x11AF ],
    [ 0xE0CF, 0x115F,0x1175,0x11BA ],
    [ 0xE0D0, 0x115F,0x1175,0x11BC ],
    [ 0xE0D1, 0x115F,0x119E,0x11AF ],
    [ 0xE0D2, 0x1100,0x1161,0x11FF ],
    [ 0xE0D3, 0x1100,0x1161,0x11C7 ],
    [ 0xE0D4, 0x1100,0x1161,0x11CA ],
    [ 0xE0D5, 0x1100,0x1161,0x11CD ],
    [ 0xE0D6, 0x1100,0x1161,0x11CE ],
    [ 0xE0D7, 0x1100,0x1161,0x11D0 ],
    [ 0xE0D8, 0x1100,0x1161,0xD7DB ],
    [ 0xE0D9, 0x1100,0x1161,0x11D9 ],
    [ 0xE0DA, 0x1100,0x1161,0x11DD ],
    [ 0xE0DB, 0x1100,0x1161,0x11E1 ],
    [ 0xE0DC, 0x1100,0x1161,0x11E2 ],
    [ 0xE0DD, 0x1100,0x1161,0x11E4 ],
    [ 0xE0DE, 0x1100,0x1161,0x11E5 ],
    [ 0xE0DF, 0x1100,0x1161,0x11E6 ],
    [ 0xE0E0, 0x1100,0x1161,0x11E7 ],
    [ 0xE0E1, 0x1100,0x1161,0x11EA ],
    [ 0xE0E2, 0x1100,0x1161,0xD7F0 ],
    [ 0xE0E3, 0x1100,0x1161,0xD7F1 ],
    [ 0xE0E4, 0x1100,0x1161,0x11EB ],
    [ 0xE0E5, 0x1100,0x1161,0x11F1 ],
    [ 0xE0E6, 0x1100,0x1161,0x11F0 ],
    [ 0xE0E7, 0x1100,0x1161,0x11F1 ],
    [ 0xE0E8, 0x1100,0x1161,0x11F9 ],
    [ 0xE0EB, 0x1100,0x1162,0x11CD ],
    [ 0xE0EC, 0x1100,0x1162,0xD7DB ],
    [ 0xE0ED, 0x1100,0x1162,0x11F0 ],
    [ 0xE0EE, 0x1100,0x1163,0x11E2 ],
    [ 0xE0EF, 0x1100,0x1163,0x11E6 ],
    [ 0xE0F0, 0x1100,0x1163,0x11F0 ],
    [ 0xE0F1, 0x1100,0x1163,0x11F9 ],
    [ 0xE0F3, 0x1100,0x1165,0x11C7 ],
    [ 0xE0F4, 0x1100,0x1165,0x11D0 ],
    [ 0xE0F5, 0x1100,0x1165,0x11D9 ],
    [ 0xE0F6, 0x1100,0x1165,0x11DD ],
    [ 0xE0F7, 0x1100,0x1165,0x11E2 ],
    [ 0xE0F8, 0x1100,0x1165,0x11E3 ],
    [ 0xE0F9, 0x1100,0x1165,0x11E6 ],
    [ 0xE0FA, 0x1100,0x1165,0x11E7 ],
    [ 0xE0FB, 0x1100,0x1165,0xD7F0 ],
    [ 0xE0FC, 0x1100,0x1165,0xD7F2 ],
    [ 0xE0FD, 0x1100,0x1165,0x11EB ],
    [ 0xE0FE, 0x1100,0x1165,0x11F0 ],
    [ 0xE0FF, 0x1100,0x1165,0x11F9 ],
    [ 0xE101, 0x1100,0x117C,0x11AB ],
    [ 0xE102, 0x1100,0x117C,0x11AF ],
    [ 0xE103, 0x1100,0x117C,0x11B7 ],
    [ 0xE104, 0x1100,0x1166,0x11F0 ],
    [ 0xE105, 0x1100,0x1167,0xD7D4 ],
    [ 0xE106, 0x1100,0x1167,0x11D9 ],
    [ 0xE107, 0x1100,0x1167,0x11E2 ],
    [ 0xE108, 0x1100,0x1167,0xD7E9 ],
    [ 0xE109, 0x1100,0x1167,0x11E7 ],
    [ 0xE10A, 0x1100,0x1167,0xD7F0 ],
    [ 0xE10B, 0x1100,0x1167,0xD7F1 ],
    [ 0xE10C, 0x1100,0x1167,0xD7F2 ],
    [ 0xE10D, 0x1100,0x1167,0x11EB ],
    [ 0xE10E, 0x1100,0x1167,0x11F0 ],
    [ 0xE10F, 0x1100,0x1167,0x11F1 ],
    [ 0xE110, 0x1100,0x1167,0x11F9 ],
    [ 0xE111, 0x1100,0x1168,0x11F0 ],
    [ 0xE112, 0x1100,0x1169,0x11D0 ],
    [ 0xE113, 0x1100,0x1169,0xD7DA ],
    [ 0xE114, 0x1100,0x1169,0x11D6 ],
    [ 0xE115, 0x1100,0x1169,0x11D9 ],
    [ 0xE116, 0x1100,0x1169,0x11DA ],
    [ 0xE117, 0x1100,0x1169,0x11DD ],
    [ 0xE118, 0x1100,0x1169,0x11E2 ],
    [ 0xE119, 0x1100,0x1169,0x11E6 ],
    [ 0xE11A, 0x1100,0x1169,0xD7F0 ],
    [ 0xE11B, 0x1100,0x1169,0x11EB ],
    [ 0xE11C, 0x1100,0x1169,0x11F0 ],
    [ 0xE11D, 0x1100,0x1169,0x11F1 ],
    [ 0xE11E, 0x1100,0x1169,0x11F7 ],
    [ 0xE11F, 0x1100,0x1169,0x11F8 ],
    [ 0xE120, 0x1100,0x116A,0x11C7 ],
    [ 0xE121, 0x1100,0x116A,0x11D9 ],
    [ 0xE122, 0x1100,0x116A,0x11F0 ],
    [ 0xE123, 0x1100,0x116A,0x11F9 ],
    [ 0xE125, 0x1100,0x11A6,0x11BC ],
    [ 0xE127, 0x1100,0x117F,0x11AB ],
    [ 0xE128, 0x1100,0x117F,0x11AF ],
    [ 0xE12A, 0x1100,0x1182,0x11AF ],
    [ 0xE12B, 0x1100,0x116C,0x11F0 ],
    [ 0xE12C, 0x1100,0x116D,0x11D9 ],
    [ 0xE12D, 0x1100,0x116D,0x11E2 ],
    [ 0xE12E, 0x1100,0x116D,0x11F0 ],
    [ 0xE131, 0x1100,0xD7B3,0x11AF ],
    [ 0xE134, 0x1100,0x1185,0x11AF ],
    [ 0xE135, 0x1100,0x1185,0x11B0 ],
    [ 0xE137, 0x1100,0x1188,0x11AB ],
    [ 0xE138, 0x1100,0x116E,0x11FD ],
    [ 0xE139, 0x1100,0x116E,0x11C7 ],
    [ 0xE13A, 0x1100,0x116E,0x11CD ],
    [ 0xE13B, 0x1100,0x116E,0x11D0 ],
    [ 0xE13C, 0x1100,0x116E,0x11D9 ],
    [ 0xE13D, 0x1100,0x116E,0x11DA ],
    [ 0xE13E, 0x1100,0x116E,0x11DD ],
    [ 0xE13F, 0x1100,0x116E,0x11E2 ],
    [ 0xE140, 0x1100,0x116E,0x11E6 ],
    [ 0xE141, 0x1100,0x116E,0x11EC ],
    [ 0xE142, 0x1100,0x116E,0x11F0 ],
    [ 0xE143, 0x1100,0x116E,0x11EC ],
    [ 0xE144, 0x1100,0x116E,0x11F6 ],
    [ 0xE145, 0x1100,0x116E,0x11F9 ],
    [ 0xE147, 0x1100,0x1189,0x11AB ],
    [ 0xE148, 0x1100,0x1189,0x11B7 ],
    [ 0xE149, 0x1100,0x1189,0x11BC ],
    [ 0xE14B, 0x1100,0x116F,0x11C7 ],
    [ 0xE14C, 0x1100,0x116F,0x11D9 ],
    [ 0xE14D, 0x1100,0x116F,0x11F9 ],
    [ 0xE14F, 0x1100,0x118B,0x11AB ],
    [ 0xE153, 0x1100,0x118D,0x11AF ],
    [ 0xE154, 0x1100,0x1171,0x11F0 ],
    [ 0xE155, 0x1100,0x1171,0x11F9 ],
    [ 0xE156, 0x1100,0x1172,0x11D9 ],
    [ 0xE157, 0x1100,0x1172,0x11E2 ],
    [ 0xE158, 0x1100,0x1172,0x11F0 ],
    [ 0xE159, 0x1100,0x1172,0x11F9 ],
    [ 0xE15B, 0x1100,0x118E,0x11AB ],
    [ 0xE15D, 0x1100,0x118F,0x11AB ],
    [ 0xE15F, 0x1100,0x1191,0x11A8 ],
    [ 0xE160, 0x1100,0x1191,0x11AB ],
    [ 0xE161, 0x1100,0x1191,0x11AF ],
    [ 0xE162, 0x1100,0x1191,0x11D9 ],
    [ 0xE163, 0x1100,0x1191,0x11BC ],
    [ 0xE164, 0x1100,0x1191,0x11F0 ],
    [ 0xE165, 0x1100,0x1191,0x11F9 ],
    [ 0xE167, 0x1100,0x1192,0x11AF ],
    [ 0xE168, 0x1100,0x1192,0x11B0 ],
    [ 0xE169, 0x1100,0x1192,0x11BC ],
    [ 0xE16B, 0x1100,0x1194,0x11A8 ],
    [ 0xE16C, 0x1100,0x1194,0x11AF ],
    [ 0xE16D, 0x1100,0x1194,0x11D9 ],
    [ 0xE16E, 0x1100,0x1194,0x11BC ],
    [ 0xE16F, 0x1100,0x1194,0x11F0 ],
    [ 0xE170, 0x1100,0x1173,0x11C7 ],
    [ 0xE171, 0x1100,0x1173,0xD7CC ],
    [ 0xE172, 0x1100,0x1173,0x11C9 ],
    [ 0xE173, 0x1100,0x1173,0xD7D3 ],
    [ 0xE174, 0x1100,0x1173,0xD7D4 ],
    [ 0xE175, 0x1100,0x1173,0x11CD ],
    [ 0xE176, 0x1100,0x1173,0x11D0 ],
    [ 0xE177, 0x1100,0x1173,0x11D9 ],
    [ 0xE178, 0x1100,0x1173,0x11DA ],
    [ 0xE179, 0x1100,0x1173,0x11DD ],
    [ 0xE17A, 0x1100,0x1173,0x11DF ],
    [ 0xE17B, 0x1100,0x1173,0x11E1 ],
    [ 0xE17C, 0x1100,0x1173,0x11E2 ],
    [ 0xE17D, 0x1100,0x1173,0xD7F0 ],
    [ 0xE17E, 0x1100,0x1173,0x11EB ],
    [ 0xE17F, 0x1100,0x1173,0x11F0 ],
    [ 0xE180, 0x1100,0x1173,0x11F9 ],
    [ 0xE185, 0x1100,0x1174,0x11D9 ],
    [ 0xE186, 0x1100,0x1174,0x11F9 ],
    [ 0xE187, 0x1100,0x1175,0x11CA ],
    [ 0xE188, 0x1100,0x1175,0xD7CD ],
    [ 0xE189, 0x1100,0x1175,0x11CD ],
    [ 0xE18A, 0x1100,0x1175,0x11D0 ],
    [ 0xE18B, 0x1100,0x1175,0x11D9 ],
    [ 0xE18C, 0x1100,0x1175,0xD7DC ],
    [ 0xE18D, 0x1100,0x1175,0x11E0 ],
    [ 0xE18E, 0x1100,0x1175,0x11E2 ],
    [ 0xE18F, 0x1100,0x1175,0xD7E6 ],
    [ 0xE190, 0x1100,0x1175,0xD7E8 ],
    [ 0xE191, 0x1100,0x1175,0x11E4 ],
    [ 0xE192, 0x1100,0x1175,0x11E5 ],
    [ 0xE193, 0x1100,0x1175,0x11E6 ],
    [ 0xE194, 0x1100,0x1175,0x11E7 ],
    [ 0xE195, 0x1100,0x1175,0xD7EC ],
    [ 0xE196, 0x1100,0x1175,0xD7F0 ],
    [ 0xE197, 0x1100,0x1175,0x11EB ],
    [ 0xE198, 0x1100,0x1175,0x11F0 ],
    [ 0xE199, 0x1100,0x1175,0x11F9 ],
    [ 0xE19C, 0x1100,0x1199,0x11AB ],
    [ 0xE19D, 0x1100,0x1199,0x11BC ],
    [ 0xE1A3, 0x1100,0xD7C3,0x11AB ],
    [ 0xE1A8, 0x1100,0x119E,0x11A8 ],
    [ 0xE1A9, 0x1100,0x119E,0x11A9 ],
    [ 0xE1AA, 0x1100,0x119E,0x11AB ],
    [ 0xE1AB, 0x1100,0x119E,0x11AE ],
    [ 0xE1AC, 0x1100,0x119E,0xD7CF ],
    [ 0xE1AD, 0x1100,0x119E,0x11AF ],
    [ 0xE1AE, 0x1100,0x119E,0x11B0 ],
    [ 0xE1AF, 0x1100,0x119E,0x11CD ],
    [ 0xE1B0, 0x1100,0x119E,0x11D0 ],
    [ 0xE1B1, 0x1100,0x119E,0x11B1 ],
    [ 0xE1B2, 0x1100,0x119E,0x11B2 ],
    [ 0xE1B3, 0x1100,0x119E,0x11D3 ],
    [ 0xE1B4, 0x1100,0x119E,0xD7DA ],
    [ 0xE1B5, 0x1100,0x119E,0x11D5 ],
    [ 0xE1B6, 0x1100,0x119E,0x11B3 ],
    [ 0xE1B7, 0x1100,0x119E,0x11B5 ],
    [ 0xE1B8, 0x1100,0x119E,0x11B6 ],
    [ 0xE1B9, 0x1100,0x119E,0x11D9 ],
    [ 0xE1BA, 0x1100,0x119E,0x11B7 ],
    [ 0xE1BB, 0x1100,0x119E,0x11E2 ],
    [ 0xE1BC, 0x1100,0x119E,0x11B8 ],
    [ 0xE1BD, 0x1100,0x119E,0x11E3 ],
    [ 0xE1BE, 0x1100,0x119E,0x11E6 ],
    [ 0xE1BF, 0x1100,0x119E,0x11BA ],
    [ 0xE1C0, 0x1100,0x119E,0x11E7 ],
    [ 0xE1C1, 0x1100,0x119E,0x11EA ],
    [ 0xE1C2, 0x1100,0x119E,0x11EB ],
    [ 0xE1C3, 0x1100,0x119E,0x11BC ],
    [ 0xE1C4, 0x1100,0x119E,0x11F0 ],
    [ 0xE1C5, 0x1100,0x119E,0x11F1 ],
    [ 0xE1C6, 0x1100,0x119E,0x11BD ],
    [ 0xE1C7, 0x1100,0x119E,0x11BE ],
    [ 0xE1C8, 0x1100,0x119E,0x11C0 ],
    [ 0xE1C9, 0x1100,0x119E,0x11C2 ],
    [ 0xE1CB, 0x1100,0x119F,0x11AB ],
    [ 0xE1CC, 0x1100,0x119F,0x11BC ],
    [ 0xE1CE, 0x1100,0x11A1,0x11A8 ],
    [ 0xE1CF, 0x1100,0x11A1,0x11AB ],
    [ 0xE1D0, 0x1100,0x11A1,0x11AF ],
    [ 0xE1D1, 0x1100,0x11A1,0x11B7 ],
    [ 0xE1D2, 0x1100,0x11A1,0x11B8 ],
    [ 0xE1D3, 0x1100,0x11A1,0x11BA ],
    [ 0xE1D4, 0x1100,0x11A1,0x11BC ],
    [ 0xE1D5, 0x1100,0x11A1,0x11F0 ],
    [ 0xE1D6, 0x1100,0x11A1,0x11F1 ],
    [ 0xE1D7, 0x1101,0x1161,0x11F0 ],
    [ 0xE1D9, 0x1101,0x1163,0x11E2 ],
    [ 0xE1DA, 0x1101,0x1165,0x11D9 ],
    [ 0xE1DB, 0x1101,0x1165,0x11E2 ],
    [ 0xE1DC, 0x1101,0x1165,0x11E6 ],
    [ 0xE1DD, 0x1101,0x1165,0x11EC ],
    [ 0xE1DE, 0x1101,0x1165,0x11ED ],
    [ 0xE1E0, 0x1101,0x1167,0x11F0 ],
    [ 0xE1E1, 0x1101,0x1169,0x11E2 ],
    [ 0xE1E2, 0x1101,0x1169,0x11F0 ],
    [ 0xE1E3, 0x1101,0x116A,0x11F0 ],
    [ 0xE1E4, 0x1101,0x116D,0x11E2 ],
    [ 0xE1E5, 0x1101,0x116D,0x11F0 ],
    [ 0xE1E7, 0x1101,0x116E,0x11D9 ],
    [ 0xE1E8, 0x1101,0x116E,0x11DA ],
    [ 0xE1E9, 0x1101,0x116E,0x11E2 ],
    [ 0xE1EA, 0x1101,0x116E,0x11F0 ],
    [ 0xE1EC, 0x1101,0x116F,0x11D9 ],
    [ 0xE1ED, 0x1101,0x116F,0x11F0 ],
    [ 0xE1F0, 0x1101,0x1172,0x11D9 ],
    [ 0xE1F1, 0x1101,0x1172,0x11E2 ],
    [ 0xE1F2, 0x1101,0x1172,0x11F0 ],
    [ 0xE1F4, 0x1101,0x1191,0x11AB ],
    [ 0xE1F5, 0x1101,0x1191,0x11BC ],
    [ 0xE1F6, 0x1101,0x1191,0x11F0 ],
    [ 0xE1F7, 0x1101,0x1191,0x11F9 ],
    [ 0xE1F9, 0x1101,0x1194,0x11BC ],
    [ 0xE1FA, 0x1101,0x1173,0x11D9 ],
    [ 0xE1FB, 0x1101,0x1173,0x11F0 ],
    [ 0xE1FC, 0x1101,0x1173,0x11F6 ],
    [ 0xE1FD, 0x1101,0x1174,0x11F0 ],
    [ 0xE1FE, 0x1101,0x1175,0x11E2 ],
    [ 0xE1FF, 0x1101,0x1175,0x11F0 ],
    [ 0xE200, 0x1101,0x1175,0x11F9 ],
    [ 0xE206, 0x1102,0x1161,0x11C4 ],
    [ 0xE207, 0x1102,0x1161,0x11C7 ],
    [ 0xE208, 0x1102,0x1161,0xD7CF ],
    [ 0xE209, 0x1102,0x1161,0x11CD ],
    [ 0xE20A, 0x1102,0x1161,0x11D0 ],
    [ 0xE20B, 0x1102,0x1161,0x11D9 ],
    [ 0xE20C, 0x1102,0x1161,0x11DA ],
    [ 0xE20D, 0x1102,0x1161,0x11DC ],
    [ 0xE20E, 0x1102,0x1161,0x11DD ],
    [ 0xE20F, 0x1102,0x1161,0x11E2 ],
    [ 0xE210, 0x1102,0x1161,0x11E6 ],
    [ 0xE211, 0x1102,0x1161,0x11E7 ],
    [ 0xE212, 0x1102,0x1161,0x11EA ],
    [ 0xE213, 0x1102,0x1161,0xD7EF ],
    [ 0xE214, 0x1102,0x1161,0xD7F2 ],
    [ 0xE215, 0x1102,0x1161,0x11EB ],
    [ 0xE216, 0x1102,0x1161,0x11EC ],
    [ 0xE217, 0x1102,0x1161,0x11F0 ],
    [ 0xE218, 0x1102,0x1161,0xD7F7 ],
    [ 0xE219, 0x1102,0x1161,0xD7F8 ],
    [ 0xE21A, 0x1102,0x1161,0x11F5 ],
    [ 0xE21B, 0x1102,0x1161,0x11F9 ],
    [ 0xE21E, 0x1102,0x1162,0xD7CF ],
    [ 0xE21F, 0x1102,0x1162,0x11D9 ],
    [ 0xE220, 0x1102,0x1162,0xD7E3 ],
    [ 0xE221, 0x1102,0x1162,0x11E3 ],
    [ 0xE222, 0x1102,0x1163,0x11E2 ],
    [ 0xE223, 0x1102,0x1163,0x11F0 ],
    [ 0xE225, 0x1102,0x1165,0x11D0 ],
    [ 0xE226, 0x1102,0x1165,0x11DA ],
    [ 0xE227, 0x1102,0x1165,0x11DB ],
    [ 0xE228, 0x1102,0x1165,0x11DD ],
    [ 0xE229, 0x1102,0x1165,0x11E3 ],
    [ 0xE22A, 0x1102,0x1165,0x11E5 ],
    [ 0xE22B, 0x1102,0x1165,0x11E9 ],
    [ 0xE22C, 0x1102,0x1165,0xD7F2 ],
    [ 0xE22D, 0x1102,0x1165,0x11F0 ],
    [ 0xE22F, 0x1102,0x117B,0x11B7 ],
    [ 0xE231, 0x1102,0x117A,0x11A8 ],
    [ 0xE231, 0x1102,0x117C,0x11A8 ],
    [ 0xE232, 0x1102,0x117A,0x11C2 ],
    [ 0xE232, 0x1102,0x117C,0x11C2 ],
    [ 0xE233, 0x1102,0x1167,0x11FD ],
    [ 0xE234, 0x1102,0x1167,0x11FE ],
    [ 0xE235, 0x1102,0x1167,0x11C5 ],
    [ 0xE236, 0x1102,0x1167,0x11FF ],
    [ 0xE237, 0x1102,0x1167,0x11C7 ],
    [ 0xE238, 0x1102,0x1167,0x11D9 ],
    [ 0xE239, 0x1102,0x1167,0x11E7 ],
    [ 0xE23A, 0x1102,0x1167,0xD7F2 ],
    [ 0xE23B, 0x1102,0x1167,0x11F0 ],
    [ 0xE23C, 0x1102,0x1167,0x11F9 ],
    [ 0xE23D, 0x1102,0x1169,0x11CD ],
    [ 0xE23E, 0x1102,0x1169,0x11D0 ],
    [ 0xE23F, 0x1102,0x1169,0x11D9 ],
    [ 0xE240, 0x1102,0x1169,0x11E2 ],
    [ 0xE241, 0x1102,0x1169,0x11E4 ],
    [ 0xE242, 0x1102,0x1169,0x11E5 ],
    [ 0xE243, 0x1102,0x1169,0x11E6 ],
    [ 0xE244, 0x1102,0x1169,0xD7F2 ],
    [ 0xE245, 0x1102,0x1169,0x11EB ],
    [ 0xE246, 0x1102,0x1169,0x11F0 ],
    [ 0xE247, 0x1102,0x1169,0xD7F6 ],
    [ 0xE248, 0x1102,0x1169,0x11F3 ],
    [ 0xE249, 0x1102,0x116A,0x11D9 ],
    [ 0xE24B, 0x1102,0x116C,0x11F0 ],
    [ 0xE24C, 0x1102,0x116D,0x11D9 ],
    [ 0xE24D, 0x1102,0x116D,0x11E2 ],
    [ 0xE24E, 0x1102,0x116D,0x11F0 ],
    [ 0xE250, 0x1102,0xD7B2,0x11BC ],
    [ 0xE252, 0x1102,0x116E,0x11C6 ],
    [ 0xE253, 0x1102,0x116E,0x11C7 ],
    [ 0xE254, 0x1102,0x116E,0x11C8 ],
    [ 0xE255, 0x1102,0x116E,0x11D0 ],
    [ 0xE256, 0x1102,0x116E,0x11DD ],
    [ 0xE257, 0x1102,0x116E,0x11E2 ],
    [ 0xE258, 0x1102,0x116E,0x11E6 ],
    [ 0xE259, 0x1102,0x116E,0x11F5 ],
    [ 0xE25B, 0x1102,0x1189,0x11AB ],
    [ 0xE25E, 0x1102,0x1171,0x11EB ],
    [ 0xE260, 0x1102,0x1172,0x11D9 ],
    [ 0xE261, 0x1102,0x1172,0x11E2 ],
    [ 0xE262, 0x1102,0x1172,0x11F0 ],
    [ 0xE265, 0x1102,0x1194,0x11AB ],
    [ 0xE266, 0x1102,0x1194,0x11BC ],
    [ 0xE267, 0x1102,0x1173,0x11C3 ],
    [ 0xE268, 0x1102,0x1173,0x11CD ],
    [ 0xE269, 0x1102,0x1173,0x11D9 ],
    [ 0xE26A, 0x1102,0x1173,0x11F0 ],
    [ 0xE26B, 0x1102,0x1173,0xD7F6 ],
    [ 0xE26D, 0x1102,0x1174,0x11F0 ],
    [ 0xE26F, 0x1102,0x1175,0x11C3 ],
    [ 0xE270, 0x1102,0x1175,0x11CE ],
    [ 0xE271, 0x1102,0x1175,0x11D0 ],
    [ 0xE272, 0x1102,0x1175,0xD7D9 ],
    [ 0xE273, 0x1102,0x1175,0x11D9 ],
    [ 0xE274, 0x1102,0x1175,0x11DD ],
    [ 0xE275, 0x1102,0x1175,0x11E2 ],
    [ 0xE276, 0x1102,0x1175,0x11E4 ],
    [ 0xE277, 0x1102,0x1175,0x11E5 ],
    [ 0xE278, 0x1102,0x1175,0x11E6 ],
    [ 0xE279, 0x1102,0x1175,0x11EB ],
    [ 0xE27A, 0x1102,0x1175,0x11F0 ],
    [ 0xE27B, 0x1102,0x1175,0x11F9 ],
    [ 0xE27E, 0x1102,0x1199,0x11AB ],
    [ 0xE27F, 0x1102,0x1199,0x11BC ],
    [ 0xE284, 0x1102,0x119E,0x11A8 ],
    [ 0xE285, 0x1102,0x119E,0x11AB ],
    [ 0xE286, 0x1102,0x119E,0x11AE ],
    [ 0xE287, 0x1102,0x119E,0xD7D3 ],
    [ 0xE288, 0x1102,0x119E,0x11AF ],
    [ 0xE289, 0x1102,0x119E,0x11B0 ],
    [ 0xE28A, 0x1102,0x119E,0x11B1 ],
    [ 0xE28B, 0x1102,0x119E,0x11B2 ],
    [ 0xE28C, 0x1102,0x119E,0x11B3 ],
    [ 0xE28D, 0x1102,0x119E,0x11D7 ],
    [ 0xE28E, 0x1102,0x119E,0x11B6 ],
    [ 0xE28F, 0x1102,0x119E,0x11D9 ],
    [ 0xE290, 0x1102,0x119E,0x11B7 ],
    [ 0xE291, 0x1102,0x119E,0x11B8 ],
    [ 0xE292, 0x1102,0x119E,0x11BA ],
    [ 0xE293, 0x1102,0x119E,0xD7F0 ],
    [ 0xE294, 0x1102,0x119E,0x11EB ],
    [ 0xE295, 0x1102,0x119E,0x11BC ],
    [ 0xE296, 0x1102,0x119E,0x11F0 ],
    [ 0xE297, 0x1102,0x119E,0x11BD ],
    [ 0xE298, 0x1102,0x119E,0x11BE ],
    [ 0xE29A, 0x1102,0x11A1,0x11A8 ],
    [ 0xE29B, 0x1102,0x11A1,0x11AB ],
    [ 0xE29C, 0x1102,0x11A1,0x11AE ],
    [ 0xE29D, 0x1102,0x11A1,0x11AF ],
    [ 0xE29E, 0x1102,0x11A1,0x11B7 ],
    [ 0xE29F, 0x1102,0x11A1,0x11B8 ],
    [ 0xE2A0, 0x1102,0x11A1,0x11BA ],
    [ 0xE2A1, 0x1102,0x11A1,0x11E8 ],
    [ 0xE2A2, 0x1102,0x11A1,0x11BC ],
    [ 0xE2A3, 0x1102,0x11A1,0x11F0 ],
    [ 0xE2A4, 0x1102,0x11A1,0x11C2 ],
    [ 0xE2AC, 0x1113,0x1168,0x11AB ],
    [ 0xE2C2, 0x1114,0x1163,0x11BC ],
    [ 0xE2C8, 0x1114,0x1173,0x11AB ],
    [ 0xE2CA, 0x1114,0x119E,0x11AB ],
    [ 0xE2CC, 0x1115,0x1161,0x11AB ],
    [ 0xE2D1, 0x1115,0x1166,0x11AB ],
    [ 0xE2D5, 0x1115,0x1169,0x11AB ],
    [ 0xE307, 0x115D,0x1163,0x11BC ],
    [ 0xE30A, 0x115D,0x1172,0x11AB ],
    [ 0xE30C, 0x115D,0x1175,0x11BC ],
    [ 0xE30D, 0x1103,0x1161,0x11C7 ],
    [ 0xE30E, 0x1103,0x1161,0x11CA ],
    [ 0xE30F, 0x1103,0x1161,0xD7CD ],
    [ 0xE310, 0x1103,0x1161,0x11CD ],
    [ 0xE311, 0x1103,0x1161,0x11D0 ],
    [ 0xE312, 0x1103,0x1161,0x11D9 ],
    [ 0xE313, 0x1103,0x1161,0x11DB ],
    [ 0xE314, 0x1103,0x1161,0x11DD ],
    [ 0xE315, 0x1103,0x1161,0x11E2 ],
    [ 0xE316, 0x1103,0x1161,0x11E5 ],
    [ 0xE317, 0x1103,0x1161,0x11E6 ],
    [ 0xE318, 0x1103,0x1161,0x11E7 ],
    [ 0xE319, 0x1103,0x1161,0xD7F2 ],
    [ 0xE31A, 0x1103,0x1161,0x11EB ],
    [ 0xE31B, 0x1103,0x1161,0x11F0 ],
    [ 0xE31C, 0x1103,0x1161,0x11F1 ],
    [ 0xE31D, 0x1103,0x1161,0xD7F6 ],
    [ 0xE31E, 0x1103,0x1161,0x11F9 ],
    [ 0xE320, 0x1103,0x1176,0x11AF ],
    [ 0xE321, 0x1103,0x1162,0x11F0 ],
    [ 0xE322, 0x1103,0x1163,0x11E2 ],
    [ 0xE323, 0x1103,0x1163,0x11F0 ],
    [ 0xE324, 0x1103,0x1163,0x11F1 ],
    [ 0xE326, 0x1103,0x1165,0x11DD ],
    [ 0xE327, 0x1103,0x1165,0x11E4 ],
    [ 0xE328, 0x1103,0x1165,0x11E6 ],
    [ 0xE329, 0x1103,0x1165,0x11EB ],
    [ 0xE32A, 0x1103,0x1165,0x11F0 ],
    [ 0xE32D, 0x1103,0x117B,0x11AF ],
    [ 0xE32F, 0x1103,0x117C,0x11AF ],
    [ 0xE330, 0x1103,0x117C,0x11B7 ],
    [ 0xE331, 0x1103,0x1166,0x11C6 ],
    [ 0xE332, 0x1103,0x1166,0xD7E3 ],
    [ 0xE333, 0x1103,0x1167,0x11D9 ],
    [ 0xE334, 0x1103,0x1167,0x11E2 ],
    [ 0xE335, 0x1103,0x1167,0x11F0 ],
    [ 0xE336, 0x1103,0x1167,0x11F9 ],
    [ 0xE337, 0x1103,0x1169,0x11C6 ],
    [ 0xE338, 0x1103,0x1169,0x11CA ],
    [ 0xE339, 0x1103,0x1169,0xD7D4 ],
    [ 0xE33A, 0x1103,0x1169,0x11D0 ],
    [ 0xE33B, 0x1103,0x1169,0x11D6 ],
    [ 0xE33C, 0x1103,0x1169,0x11D8 ],
    [ 0xE33D, 0x1103,0x1169,0x11D9 ],
    [ 0xE33E, 0x1103,0x1169,0x11E2 ],
    [ 0xE33F, 0x1103,0x1169,0x11E6 ],
    [ 0xE340, 0x1103,0x1169,0x11E7 ],
    [ 0xE341, 0x1103,0x1169,0xD7EC ],
    [ 0xE342, 0x1103,0x1169,0xD7F2 ],
    [ 0xE343, 0x1103,0x1169,0x11EB ],
    [ 0xE344, 0x1103,0x1169,0x11F0 ],
    [ 0xE345, 0x1103,0x1169,0x11F1 ],
    [ 0xE346, 0x1103,0x116A,0x11D9 ],
    [ 0xE347, 0x1103,0x116A,0x11F0 ],
    [ 0xE349, 0x1103,0x1182,0x11B7 ],
    [ 0xE34A, 0x1103,0x116D,0x11E2 ],
    [ 0xE34B, 0x1103,0x116D,0xD7F5 ],
    [ 0xE34C, 0x1103,0x116D,0x11F0 ],
    [ 0xE34E, 0x1103,0x1188,0x11AB ],
    [ 0xE34F, 0x1103,0x1188,0x11AF ],
    [ 0xE350, 0x1103,0x116E,0x11CE ],
    [ 0xE351, 0x1103,0x116E,0x11D0 ],
    [ 0xE352, 0x1103,0x116E,0x11D9 ],
    [ 0xE353, 0x1103,0x116E,0x11DA ],
    [ 0xE354, 0x1103,0x116E,0x11E2 ],
    [ 0xE355, 0x1103,0x116E,0x11F0 ],
    [ 0xE357, 0x1103,0x1189,0x11AB ],
    [ 0xE358, 0x1103,0x116F,0x11F9 ],
    [ 0xE35B, 0x1103,0x1171,0x11E6 ],
    [ 0xE35C, 0x1103,0x1171,0x11F9 ],
    [ 0xE35D, 0x1103,0x1172,0x11D9 ],
    [ 0xE35E, 0x1103,0x1172,0x11E2 ],
    [ 0xE35F, 0x1103,0x1172,0x11F0 ],
    [ 0xE360, 0x1103,0x1172,0x11F1 ],
    [ 0xE362, 0x1103,0x1191,0x11AB ],
    [ 0xE363, 0x1103,0x1191,0x11AF ],
    [ 0xE364, 0x1103,0x1191,0x11D9 ],
    [ 0xE365, 0x1103,0x1191,0x11BC ],
    [ 0xE367, 0x1103,0x1192,0x11BC ],
    [ 0xE369, 0x1103,0x1194,0x11AB ],
    [ 0xE36A, 0x1103,0x1194,0x11BC ],
    [ 0xE36B, 0x1103,0x1173,0x11CD ],
    [ 0xE36C, 0x1103,0x1173,0x11CE ],
    [ 0xE36D, 0x1103,0x1173,0x11D0 ],
    [ 0xE36E, 0x1103,0x1173,0x11D9 ],
    [ 0xE36F, 0x1103,0x1173,0x11DA ],
    [ 0xE370, 0x1103,0x1173,0x11E2 ],
    [ 0xE371, 0x1103,0x1173,0x11E3 ],
    [ 0xE372, 0x1103,0x1173,0x11E6 ],
    [ 0xE373, 0x1103,0x1173,0x11EB ],
    [ 0xE374, 0x1103,0x1173,0x11F0 ],
    [ 0xE375, 0x1103,0x1173,0x11F1 ],
    [ 0xE378, 0x1103,0x1174,0x11D9 ],
    [ 0xE379, 0x1103,0x1174,0x11F9 ],
    [ 0xE37A, 0x1103,0x1175,0x11D0 ],
    [ 0xE37B, 0x1103,0x1175,0x11D9 ],
    [ 0xE37C, 0x1103,0x1175,0x11E2 ],
    [ 0xE37D, 0x1103,0x1175,0x11E4 ],
    [ 0xE37E, 0x1103,0x1175,0x11E5 ],
    [ 0xE37F, 0x1103,0x1175,0xD7F1 ],
    [ 0xE380, 0x1103,0x1175,0x11EB ],
    [ 0xE381, 0x1103,0x1175,0x11F0 ],
    [ 0xE382, 0x1103,0x1175,0x11F9 ],
    [ 0xE385, 0x1103,0x1199,0x11AB ],
    [ 0xE386, 0x1103,0x1199,0x11BC ],
    [ 0xE38B, 0x1103,0x119E,0x11A8 ],
    [ 0xE38C, 0x1103,0x119E,0x11C3 ],
    [ 0xE38D, 0x1103,0x119E,0x11AB ],
    [ 0xE38E, 0x1103,0x119E,0x11AE ],
    [ 0xE38F, 0x1103,0x119E,0x11AF ],
    [ 0xE390, 0x1103,0x119E,0x11B0 ],
    [ 0xE391, 0x1103,0x119E,0x11CC ],
    [ 0xE392, 0x1103,0x119E,0x11B1 ],
    [ 0xE393, 0x1103,0x119E,0x11B2 ],
    [ 0xE394, 0x1103,0x119E,0x11B3 ],
    [ 0xE395, 0x1103,0x119E,0x11B6 ],
    [ 0xE396, 0x1103,0x119E,0x11D9 ],
    [ 0xE397, 0x1103,0x119E,0x11B7 ],
    [ 0xE398, 0x1103,0x119E,0x11DA ],
    [ 0xE399, 0x1103,0x119E,0xD7DE ],
    [ 0xE39A, 0x1103,0x119E,0xD7DF ],
    [ 0xE39B, 0x1103,0x119E,0x11B8 ],
    [ 0xE39C, 0x1103,0x119E,0x11E3 ],
    [ 0xE39D, 0x1103,0x119E,0x11E6 ],
    [ 0xE39E, 0x1103,0x119E,0x11BA ],
    [ 0xE39F, 0x1103,0x119E,0x11E7 ],
    [ 0xE3A0, 0x1103,0x119E,0x11E8 ],
    [ 0xE3A1, 0x1103,0x119E,0x11E9 ],
    [ 0xE3A2, 0x1103,0x119E,0x11EA ],
    [ 0xE3A3, 0x1103,0x119E,0x11EB ],
    [ 0xE3A4, 0x1103,0x119E,0x11BC ],
    [ 0xE3A5, 0x1103,0x119E,0x11F0 ],
    [ 0xE3A6, 0x1103,0x119E,0x11BD ],
    [ 0xE3A7, 0x1103,0x119E,0x11BE ],
    [ 0xE3A9, 0x1103,0x11A1,0x11A8 ],
    [ 0xE3AA, 0x1103,0x11A1,0x11AB ],
    [ 0xE3AB, 0x1103,0x11A1,0x11AE ],
    [ 0xE3AC, 0x1103,0x11A1,0x11AF ],
    [ 0xE3AD, 0x1103,0x11A1,0x11D9 ],
    [ 0xE3AE, 0x1103,0x11A1,0x11B7 ],
    [ 0xE3AF, 0x1103,0x11A1,0x11B8 ],
    [ 0xE3B0, 0x1103,0x11A1,0x11BA ],
    [ 0xE3B1, 0x1103,0x11A1,0x11EB ],
    [ 0xE3B2, 0x1103,0x11A1,0x11BC ],
    [ 0xE3B3, 0x1103,0x11A1,0x11F0 ],
    [ 0xE3B5, 0x1117,0x1173,0x11AF ],
    [ 0xE3B7, 0x1117,0x1175,0x11AF ],
    [ 0xE3B8, 0x1104,0x1161,0x11D0 ],
    [ 0xE3B9, 0x1104,0x1161,0x11D8 ],
    [ 0xE3BA, 0x1104,0x1161,0x11D9 ],
    [ 0xE3BB, 0x1104,0x1161,0x11E2 ],
    [ 0xE3BC, 0x1104,0x1161,0x11E6 ],
    [ 0xE3BD, 0x1104,0x1161,0x11F0 ],
    [ 0xE3BE, 0x1104,0x1161,0x11F9 ],
    [ 0xE3BF, 0x1104,0x1163,0x11E2 ],
    [ 0xE3C0, 0x1104,0x1163,0x11F0 ],
    [ 0xE3C3, 0x1104,0x117C,0x11AF ],
    [ 0xE3C4, 0x1104,0x117C,0x11B2 ],
    [ 0xE3C5, 0x1104,0x1167,0x11D9 ],
    [ 0xE3C6, 0x1104,0x1167,0x11F0 ],
    [ 0xE3C7, 0x1104,0x1169,0x11D9 ],
    [ 0xE3C8, 0x1104,0x1169,0x11E2 ],
    [ 0xE3C9, 0x1104,0x1169,0x11F0 ],
    [ 0xE3CA, 0x1104,0x1169,0x11F9 ],
    [ 0xE3CB, 0x1104,0x116A,0x11D9 ],
    [ 0xE3CC, 0x1104,0x116A,0x11F0 ],
    [ 0xE3CE, 0x1104,0x116D,0x11E2 ],
    [ 0xE3CF, 0x1104,0x116E,0x11E2 ],
    [ 0xE3D0, 0x1104,0x116E,0x11F0 ],
    [ 0xE3D1, 0x1104,0x116E,0x11F9 ],
    [ 0xE3D4, 0x1104,0x1171,0x11F9 ],
    [ 0xE3D5, 0x1104,0x1172,0x11D9 ],
    [ 0xE3D6, 0x1104,0x1172,0x11E2 ],
    [ 0xE3D7, 0x1104,0x1172,0x11F0 ],
    [ 0xE3D9, 0x1104,0x1191,0x11AB ],
    [ 0xE3DB, 0x1104,0x1194,0x11BC ],
    [ 0xE3DC, 0x1104,0x1173,0xD7CD ],
    [ 0xE3DD, 0x1104,0x1173,0x11E2 ],
    [ 0xE3DE, 0x1104,0x1173,0x11F0 ],
    [ 0xE3DF, 0x1104,0x1173,0x11F9 ],
    [ 0xE3E0, 0x1104,0x1174,0x11F9 ],
    [ 0xE3E1, 0x1104,0x1175,0x11D9 ],
    [ 0xE3E2, 0x1104,0x1175,0x11F0 ],
    [ 0xE3E3, 0x1104,0x1175,0x11F9 ],
    [ 0xE3E6, 0x1104,0x119E,0x11A8 ],
    [ 0xE3E7, 0x1104,0x119E,0x11AB ],
    [ 0xE3E8, 0x1104,0x119E,0x11AF ],
    [ 0xE3E9, 0x1104,0x119E,0x11B7 ],
    [ 0xE3EA, 0x1104,0x119E,0x11BA ],
    [ 0xE3EC, 0x1104,0x11A1,0x11A8 ],
    [ 0xE3ED, 0x1104,0x11A1,0x11AF ],
    [ 0xE3EE, 0x1104,0x11A1,0x11BC ],
    [ 0xE3EF, 0x1104,0x11A1,0x11F0 ],
    [ 0xE3F1, 0x115E,0x1175,0x11BC ],
    [ 0xE3F2, 0x115E,0x1175,0x11EF ],
    [ 0xE3F4, 0xA960,0x1161,0x11AF ],
    [ 0xE3F8, 0xA961,0x116E,0x11AF ],
    [ 0xE3FA, 0xA962,0x1172,0x11AF ],
    [ 0xE3FB, 0xA962,0x1172,0x11BC ],
    [ 0xE3FE, 0xA962,0x1175,0x11A8 ],
    [ 0xE400, 0xA963,0x1163,0x11BC ],
    [ 0xE402, 0x1105,0x1161,0x11D0 ],
    [ 0xE403, 0x1105,0x1161,0x11D9 ],
    [ 0xE404, 0x1105,0x1161,0xD7E0 ],
    [ 0xE405, 0x1105,0x1161,0x11DD ],
    [ 0xE406, 0x1105,0x1161,0x11E2 ],
    [ 0xE407, 0x1105,0x1161,0x11E6 ],
    [ 0xE408, 0x1105,0x1161,0x11E7 ],
    [ 0xE409, 0x1105,0x1161,0xD7EC ],
    [ 0xE40A, 0x1105,0x1161,0x11EB ],
    [ 0xE40B, 0x1105,0x1161,0x11F0 ],
    [ 0xE40C, 0x1105,0x1161,0x11EC ],
    [ 0xE40D, 0x1105,0x1161,0x11F5 ],
    [ 0xE40E, 0x1105,0x1161,0x11F9 ],
    [ 0xE410, 0x1105,0x1176,0x11AF ],
    [ 0xE412, 0x1105,0x1162,0x11F0 ],
    [ 0xE413, 0x1105,0x1163,0x11E2 ],
    [ 0xE414, 0x1105,0x1163,0x11F0 ],
    [ 0xE416, 0x1105,0x1165,0xD7E0 ],
    [ 0xE417, 0x1105,0x1165,0x11E6 ],
    [ 0xE418, 0x1105,0x1165,0xD7F2 ],
    [ 0xE419, 0x1105,0x1165,0x11F0 ],
    [ 0xE41A, 0x1105,0x1165,0x11F5 ],
    [ 0xE41D, 0x1105,0x117B,0x11AF ],
    [ 0xE41E, 0x1105,0x1166,0x11F0 ],
    [ 0xE41F, 0x1105,0x1167,0x11C7 ],
    [ 0xE420, 0x1105,0x1167,0x11D9 ],
    [ 0xE421, 0x1105,0x1167,0x11E2 ],
    [ 0xE422, 0x1105,0x1167,0x11E6 ],
    [ 0xE423, 0x1105,0x1167,0x11F0 ],
    [ 0xE424, 0x1105,0x1167,0x11F9 ],
    [ 0xE425, 0x1105,0x1169,0xD7D0 ],
    [ 0xE426, 0x1105,0x1169,0x11D9 ],
    [ 0xE427, 0x1105,0x1169,0x11DD ],
    [ 0xE428, 0x1105,0x1169,0x11E2 ],
    [ 0xE429, 0x1105,0x1169,0xD7E5 ],
    [ 0xE42A, 0x1105,0x1169,0x11E6 ],
    [ 0xE42B, 0x1105,0x1169,0xD7F0 ],
    [ 0xE42C, 0x1105,0x1169,0x11F0 ],
    [ 0xE42D, 0x1105,0x1169,0x11F1 ],
    [ 0xE42E, 0x1105,0x116A,0x11D9 ],
    [ 0xE42F, 0x1105,0x116A,0x11F0 ],
    [ 0xE431, 0x1105,0x116D,0x11D9 ],
    [ 0xE432, 0x1105,0x116D,0x11DD ],
    [ 0xE433, 0x1105,0x116D,0x11E2 ],
    [ 0xE434, 0x1105,0x116D,0x11F0 ],
    [ 0xE438, 0x1105,0x116E,0x11D9 ],
    [ 0xE439, 0x1105,0x116E,0x11DD ],
    [ 0xE43A, 0x1105,0x116E,0x11E2 ],
    [ 0xE43B, 0x1105,0x116E,0x11E4 ],
    [ 0xE43C, 0x1105,0x116E,0x11F0 ],
    [ 0xE43D, 0x1105,0x116E,0x11F9 ],
    [ 0xE43F, 0x1105,0x1189,0x11AB ],
    [ 0xE443, 0x1105,0x1172,0x11D9 ],
    [ 0xE444, 0x1105,0x1172,0x11E2 ],
    [ 0xE445, 0x1105,0x1172,0x11F0 ],
    [ 0xE446, 0x1105,0x1172,0x11F9 ],
    [ 0xE448, 0x1105,0x1191,0x11AB ],
    [ 0xE449, 0x1105,0x1191,0x11AF ],
    [ 0xE44A, 0x1105,0x1191,0x11D9 ],
    [ 0xE44C, 0x1105,0x1194,0x11BC ],
    [ 0xE44D, 0x1105,0x1173,0xD7D3 ],
    [ 0xE44E, 0x1105,0x1173,0x11D0 ],
    [ 0xE44F, 0x1105,0x1173,0x11D9 ],
    [ 0xE450, 0x1105,0x1173,0xD7E0 ],
    [ 0xE451, 0x1105,0x1173,0x11DD ],
    [ 0xE452, 0x1105,0x1173,0x11E2 ],
    [ 0xE453, 0x1105,0x1173,0x11E6 ],
    [ 0xE454, 0x1105,0x1173,0x11F0 ],
    [ 0xE457, 0x1105,0x1174,0x11F9 ],
    [ 0xE458, 0x1105,0x1175,0x11C7 ],
    [ 0xE459, 0x1105,0x1175,0x11D9 ],
    [ 0xE45A, 0x1105,0x1175,0x11DD ],
    [ 0xE45B, 0x1105,0x1175,0x11E2 ],
    [ 0xE45C, 0x1105,0x1175,0x11E6 ],
    [ 0xE45D, 0x1105,0x1175,0x11F0 ],
    [ 0xE45E, 0x1105,0x1175,0x11F9 ],
    [ 0xE461, 0x1105,0x1199,0x11AB ],
    [ 0xE462, 0x1105,0x1199,0x11BC ],
    [ 0xE467, 0x1105,0x119A,0x11AF ],
    [ 0xE46A, 0x1105,0x119B,0x11AF ],
    [ 0xE46D, 0x1105,0x119E,0x11A8 ],
    [ 0xE46E, 0x1105,0x119E,0x11AB ],
    [ 0xE46F, 0x1105,0x119E,0x11AE ],
    [ 0xE470, 0x1105,0x119E,0x11AF ],
    [ 0xE471, 0x1105,0x119E,0x11B0 ],
    [ 0xE472, 0x1105,0x119E,0x11B1 ],
    [ 0xE473, 0x1105,0x119E,0x11B3 ],
    [ 0xE474, 0x1105,0x119E,0x11D7 ],
    [ 0xE475, 0x1105,0x119E,0x11B6 ],
    [ 0xE476, 0x1105,0x119E,0x11D9 ],
    [ 0xE477, 0x1105,0x119E,0x11B7 ],
    [ 0xE478, 0x1105,0x119E,0xD7E0 ],
    [ 0xE479, 0x1105,0x119E,0x11DC ],
    [ 0xE47A, 0x1105,0x119E,0x11DD ],
    [ 0xE47B, 0x1105,0x119E,0x11B8 ],
    [ 0xE47C, 0x1105,0x119E,0x11B9 ],
    [ 0xE47D, 0x1105,0x119E,0x11E6 ],
    [ 0xE47E, 0x1105,0x119E,0x11BA ],
    [ 0xE47F, 0x1105,0x119E,0x11BB ],
    [ 0xE480, 0x1105,0x119E,0x11EB ],
    [ 0xE481, 0x1105,0x119E,0x11F0 ],
    [ 0xE482, 0x1105,0x119E,0x11BD ],
    [ 0xE483, 0x1105,0x119E,0x11BE ],
    [ 0xE484, 0x1105,0x119E,0x11C0 ],
    [ 0xE485, 0x1105,0x119E,0x11C2 ],
    [ 0xE488, 0x1105,0x11A1,0x11A8 ],
    [ 0xE489, 0x1105,0x11A1,0x11AB ],
    [ 0xE48A, 0x1105,0x11A1,0x11AE ],
    [ 0xE48B, 0x1105,0x11A1,0x11AF ],
    [ 0xE48C, 0x1105,0x11A1,0x11B7 ],
    [ 0xE48D, 0x1105,0x11A1,0x11BA ],
    [ 0xE48E, 0x1105,0x11A1,0x11BC ],
    [ 0xE48F, 0x1105,0x11A1,0x11F0 ],
    [ 0xE492, 0xA964,0x1169,0x11BA ],
    [ 0xE495, 0x1118,0x1161,0x11A8 ],
    [ 0xE496, 0x1118,0x1161,0x11AB ],
    [ 0xE497, 0x1118,0x1161,0x11BC ],
    [ 0xE49A, 0x1118,0x1166,0x11AF ],
    [ 0xE49B, 0x1118,0x1166,0x11B8 ],
    [ 0xE49D, 0x1118,0x1169,0x11A8 ],
    [ 0xE49E, 0x1118,0x1169,0x11AB ],
    [ 0xE49F, 0x1118,0x1169,0x11B7 ],
    [ 0xE4A0, 0x1118,0x1169,0x11BA ],
    [ 0xE4A2, 0x1118,0x116D,0x11AB ],
    [ 0xE4A5, 0x1118,0x1172,0x11A8 ],
    [ 0xE4A8, 0x1118,0x1174,0x11A8 ],
    [ 0xE4A9, 0x1118,0x1174,0x11AB ],
    [ 0xE4AA, 0x1118,0x1174,0x11B8 ],
    [ 0xE4AC, 0x1118,0x119E,0x11AB ],
    [ 0xE4AD, 0x1118,0x119E,0x11BC ],
    [ 0xE4AF, 0x1118,0x11A1,0x11A8 ],
    [ 0xE4B0, 0x1118,0x11A1,0x11B7 ],
    [ 0xE4B3, 0x1119,0x1161,0x11A8 ],
    [ 0xE4B4, 0x1119,0x1161,0x11AB ],
    [ 0xE4B5, 0x1119,0x1161,0x11BC ],
    [ 0xE4B8, 0x1119,0x1162,0x11AB ],
    [ 0xE4BA, 0x1119,0x1163,0x11BC ],
    [ 0xE4BD, 0x1119,0x1165,0x11AB ],
    [ 0xE4BE, 0x1119,0x1165,0x11BC ],
    [ 0xE4C1, 0x1119,0x1166,0x11A8 ],
    [ 0xE4C3, 0x1119,0x1167,0x11A8 ],
    [ 0xE4C5, 0x1119,0x1168,0x11AB ],
    [ 0xE4C7, 0x1119,0x1169,0x11BC ],
    [ 0xE4CA, 0x1119,0x116E,0x11AB ],
    [ 0xE4CB, 0x1119,0x116E,0x11BC ],
    [ 0xE4CD, 0x1119,0x1189,0x11AB ],
    [ 0xE4D5, 0x1119,0x1173,0x11AF ],
    [ 0xE4D6, 0x1119,0x1173,0x11D0 ],
    [ 0xE4D7, 0x1119,0x1173,0x11BC ],
    [ 0xE4D9, 0x1119,0x1175,0x11AB ],
    [ 0xE4DA, 0x1119,0x1175,0x11BC ],
    [ 0xE4DD, 0xA968,0x1161,0x11AB ],
    [ 0xE4E3, 0xA96C,0x1169,0x11A8 ],
    [ 0xE4E8, 0xA96D,0x1172,0x11AF ],
    [ 0xE4EB, 0x111A,0x1161,0x11AB ],
    [ 0xE4ED, 0x111A,0x1163,0x11BC ],
    [ 0xE4F1, 0x1106,0x1161,0x11FE ],
    [ 0xE4F2, 0x1106,0x1161,0x11C6 ],
    [ 0xE4F3, 0x1106,0x1161,0x11C7 ],
    [ 0xE4F4, 0x1106,0x1161,0xD7D4 ],
    [ 0xE4F5, 0x1106,0x1161,0x11CD ],
    [ 0xE4F6, 0x1106,0x1161,0x11D0 ],
    [ 0xE4F7, 0x1106,0x1161,0x11D8 ],
    [ 0xE4F8, 0x1106,0x1161,0x11D9 ],
    [ 0xE4F9, 0x1106,0x1161,0x11E2 ],
    [ 0xE4FA, 0x1106,0x1161,0x11E6 ],
    [ 0xE4FB, 0x1106,0x1161,0x11E8 ],
    [ 0xE4FC, 0x1106,0x1161,0xD7ED ],
    [ 0xE4FD, 0x1106,0x1161,0xD7F0 ],
    [ 0xE4FE, 0x1106,0x1161,0xD7F1 ],
    [ 0xE4FF, 0x1106,0x1161,0x11EB ],
    [ 0xE500, 0x1106,0x1161,0x11F0 ],
    [ 0xE501, 0x1106,0x1161,0x11F9 ],
    [ 0xE504, 0x1106,0x1162,0x11F0 ],
    [ 0xE505, 0x1106,0x1163,0x11E2 ],
    [ 0xE507, 0x1106,0x1165,0x11C6 ],
    [ 0xE508, 0x1106,0x1165,0x11CD ],
    [ 0xE509, 0x1106,0x1165,0x11D0 ],
    [ 0xE50A, 0x1106,0x1165,0x11D9 ],
    [ 0xE50B, 0x1106,0x1165,0x11F0 ],
    [ 0xE50C, 0x1106,0x1165,0x11F9 ],
    [ 0xE50F, 0x1106,0x117C,0x11AF ],
    [ 0xE510, 0x1106,0x1166,0x11D9 ],
    [ 0xE511, 0x1106,0x1167,0x11D9 ],
    [ 0xE512, 0x1106,0x1167,0x11E2 ],
    [ 0xE513, 0x1106,0x1167,0x11EB ],
    [ 0xE514, 0x1106,0x1167,0x11F0 ],
    [ 0xE515, 0x1106,0x1167,0xD7F6 ],
    [ 0xE516, 0x1106,0x1167,0x11F9 ],
    [ 0xE517, 0x1106,0x1169,0x11C6 ],
    [ 0xE518, 0x1106,0x1169,0x11D0 ],
    [ 0xE519, 0x1106,0x1169,0x11D9 ],
    [ 0xE51A, 0x1106,0x1169,0x11DD ],
    [ 0xE51B, 0x1106,0x1169,0x11E2 ],
    [ 0xE51C, 0x1106,0x1169,0xD7F0 ],
    [ 0xE51D, 0x1106,0x1169,0x11EB ],
    [ 0xE51E, 0x1106,0x1169,0x11F0 ],
    [ 0xE520, 0x1106,0x117F,0x11AB ],
    [ 0xE524, 0x1106,0x116D,0x11E2 ],
    [ 0xE527, 0x1106,0x116E,0x11C6 ],
    [ 0xE528, 0x1106,0x116E,0x11C7 ],
    [ 0xE529, 0x1106,0x116E,0x11D0 ],
    [ 0xE52A, 0x1106,0x116E,0x11D9 ],
    [ 0xE52B, 0x1106,0x116E,0x11DD ],
    [ 0xE52C, 0x1106,0x116E,0x11E2 ],
    [ 0xE52D, 0x1106,0x116E,0x11E7 ],
    [ 0xE52E, 0x1106,0x116E,0xD7EC ],
    [ 0xE52F, 0x1106,0x116E,0xD7F1 ],
    [ 0xE530, 0x1106,0x116E,0x11EB ],
    [ 0xE531, 0x1106,0x116E,0x11EC ],
    [ 0xE532, 0x1106,0x116E,0x11ED ],
    [ 0xE533, 0x1106,0x116E,0x11F0 ],
    [ 0xE534, 0x1106,0x116E,0x11F9 ],
    [ 0xE535, 0x1106,0x116F,0x11F9 ],
    [ 0xE539, 0x1106,0x1171,0x11F9 ],
    [ 0xE53A, 0x1106,0x1172,0x11E2 ],
    [ 0xE53C, 0x1106,0x1173,0x11C4 ],
    [ 0xE53D, 0x1106,0x1173,0x11D0 ],
    [ 0xE53E, 0x1106,0x1173,0x11D9 ],
    [ 0xE53F, 0x1106,0x1173,0x11E2 ],
    [ 0xE540, 0x1106,0x1173,0x11E7 ],
    [ 0xE541, 0x1106,0x1173,0xD7F2 ],
    [ 0xE542, 0x1106,0x1173,0x11EB ],
    [ 0xE543, 0x1106,0x1173,0x11F0 ],
    [ 0xE546, 0x1106,0x1196,0x11B7 ],
    [ 0xE547, 0x1106,0x1174,0x11D9 ],
    [ 0xE548, 0x1106,0x1174,0x11E6 ],
    [ 0xE549, 0x1106,0x1174,0x11F9 ],
    [ 0xE54A, 0x1106,0x1175,0xD7CF ],
    [ 0xE54B, 0x1106,0x1175,0xD7D3 ],
    [ 0xE54C, 0x1106,0x1175,0xD7D4 ],
    [ 0xE54D, 0x1106,0x1175,0x11CD ],
    [ 0xE54E, 0x1106,0x1175,0x11D0 ],
    [ 0xE54F, 0x1106,0x1175,0x11D9 ],
    [ 0xE550, 0x1106,0x1175,0x11E6 ],
    [ 0xE551, 0x1106,0x1175,0x11E7 ],
    [ 0xE552, 0x1106,0x1175,0xD7F0 ],
    [ 0xE553, 0x1106,0x1175,0xD7F1 ],
    [ 0xE554, 0x1106,0x1175,0xD7F2 ],
    [ 0xE555, 0x1106,0x1175,0x11EB ],
    [ 0xE556, 0x1106,0x1175,0x11F0 ],
    [ 0xE557, 0x1106,0x1175,0x11F6 ],
    [ 0xE558, 0x1106,0x1175,0x11F9 ],
    [ 0xE55B, 0x1106,0x1199,0x11AB ],
    [ 0xE55F, 0x1106,0x119C,0x11BC ],
    [ 0xE561, 0x1106,0x119E,0x11A8 ],
    [ 0xE562, 0x1106,0x119E,0x11C3 ],
    [ 0xE563, 0x1106,0x119E,0x11AB ],
    [ 0xE564, 0x1106,0x119E,0x11AE ],
    [ 0xE565, 0x1106,0x119E,0xD7D3 ],
    [ 0xE566, 0x1106,0x119E,0x11AF ],
    [ 0xE567, 0x1106,0x119E,0x11B0 ],
    [ 0xE568, 0x1106,0x119E,0x11D0 ],
    [ 0xE569, 0x1106,0x119E,0x11B2 ],
    [ 0xE56A, 0x1106,0x119E,0x11D5 ],
    [ 0xE56B, 0x1106,0x119E,0x11B3 ],
    [ 0xE56C, 0x1106,0x119E,0x11B6 ],
    [ 0xE56D, 0x1106,0x119E,0x11D9 ],
    [ 0xE56E, 0x1106,0x119E,0x11B7 ],
    [ 0xE56F, 0x1106,0x119E,0x11B8 ],
    [ 0xE570, 0x1106,0x119E,0x11BA ],
    [ 0xE571, 0x1106,0x119E,0xD7F0 ],
    [ 0xE572, 0x1106,0x119E,0x11EB ],
    [ 0xE573, 0x1106,0x119E,0x11BC ],
    [ 0xE574, 0x1106,0x119E,0x11F0 ],
    [ 0xE575, 0x1106,0x119E,0x11BD ],
    [ 0xE576, 0x1106,0x119E,0x11BE ],
    [ 0xE577, 0x1106,0x119E,0x11C0 ],
    [ 0xE579, 0x1106,0x119F,0x11AB ],
    [ 0xE57A, 0x1106,0x119F,0x11BC ],
    [ 0xE57C, 0x1106,0x11A1,0x11A8 ],
    [ 0xE57D, 0x1106,0x11A1,0x11A9 ],
    [ 0xE57E, 0x1106,0x11A1,0x11AB ],
    [ 0xE57F, 0x1106,0x11A1,0x11AE ],
    [ 0xE580, 0x1106,0x11A1,0x11AF ],
    [ 0xE581, 0x1106,0x11A1,0x11B3 ],
    [ 0xE582, 0x1106,0x11A1,0x11D9 ],
    [ 0xE583, 0x1106,0x11A1,0x11B7 ],
    [ 0xE584, 0x1106,0x11A1,0x11B8 ],
    [ 0xE585, 0x1106,0x11A1,0x11E6 ],
    [ 0xE586, 0x1106,0x11A1,0x11BA ],
    [ 0xE587, 0x1106,0x11A1,0x11BC ],
    [ 0xE588, 0x1106,0x11A1,0x11F0 ],
    [ 0xE589, 0x1106,0x11A1,0x11BD ],
    [ 0xE58A, 0x1106,0x11A1,0x11BE ],
    [ 0xE58B, 0x1106,0x11A1,0x11C2 ],
    [ 0xE591, 0xA970,0x1167,0x11AF ],
    [ 0xE594, 0x111C,0x1161,0x11AB ],
    [ 0xE595, 0x111C,0x1161,0x11AE ],
    [ 0xE59C, 0x111C,0x1168,0x11AB ],
    [ 0xE59D, 0x111C,0x1168,0x11AE ],
    [ 0xE59F, 0x111C,0x1169,0x11A8 ],
    [ 0xE5A0, 0x111C,0x1169,0x11AB ],
    [ 0xE5A1, 0x111C,0x1169,0x11AE ],
    [ 0xE5A7, 0x111C,0x116E,0x11AB ],
    [ 0xE5A8, 0x111C,0x116E,0x11BC ],
    [ 0xE5B3, 0x111C,0x1175,0x11AB ],
    [ 0xE5B8, 0xA971,0x1169,0x11A8 ],
    [ 0xE5BA, 0x111D,0x1161,0x11AB ],
    [ 0xE5BB, 0x111D,0x1161,0x11BC ],
    [ 0xE5BC, 0x111D,0x1161,0x11F0 ],
    [ 0xE5BF, 0x111D,0x116A,0x11AB ],
    [ 0xE5C0, 0x111D,0x116A,0x11F9 ],
    [ 0xE5C2, 0x111D,0x116E,0x11AB ],
    [ 0xE5C3, 0x111D,0x116E,0x11BC ],
    [ 0xE5C4, 0x111D,0x116E,0x11F9 ],
    [ 0xE5C7, 0x111D,0x1172,0x11AB ],
    [ 0xE5C9, 0x111D,0x1173,0x11AB ],
    [ 0xE5CA, 0x111D,0x1173,0x11F9 ],
    [ 0xE5CB, 0x111D,0xD7B9,0x11AB ],
    [ 0xE5CD, 0x111D,0x1175,0x11BC ],
    [ 0xE5CE, 0x1107,0x1161,0x11C3 ],
    [ 0xE5CF, 0x1107,0x1161,0x11FC ],
    [ 0xE5D0, 0x1107,0x1161,0xD7CB ],
    [ 0xE5D1, 0x1107,0x1161,0x11C7 ],
    [ 0xE5D2, 0x1107,0x1161,0x11C9 ],
    [ 0xE5D3, 0x1107,0x1161,0x11CA ],
    [ 0xE5D4, 0x1107,0x1161,0xD7D0 ],
    [ 0xE5D5, 0x1107,0x1161,0xD7D1 ],
    [ 0xE5D6, 0x1107,0x1161,0xD7D4 ],
    [ 0xE5D7, 0x1107,0x1161,0x11D0 ],
    [ 0xE5D8, 0x1107,0x1161,0x11D9 ],
    [ 0xE5D9, 0x1107,0x1161,0xD7E0 ],
    [ 0xE5DA, 0x1107,0x1161,0x11DC ],
    [ 0xE5DB, 0x1107,0x1161,0x11DD ],
    [ 0xE5DC, 0x1107,0x1161,0x11E2 ],
    [ 0xE5DD, 0x1107,0x1161,0x11E3 ],
    [ 0xE5DE, 0x1107,0x1161,0x11E6 ],
    [ 0xE5DF, 0x1107,0x1161,0x11E7 ],
    [ 0xE5E0, 0x1107,0x1161,0x11E8 ],
    [ 0xE5E1, 0x1107,0x1161,0x11EA ],
    [ 0xE5E2, 0x1107,0x1161,0xD7F0 ],
    [ 0xE5E3, 0x1107,0x1161,0x11EB ],
    [ 0xE5E4, 0x1107,0x1161,0x11F1 ],
    [ 0xE5E5, 0x1107,0x1161,0x11F0 ],
    [ 0xE5E6, 0x1107,0x1161,0x11F1 ],
    [ 0xE5E7, 0x1107,0x1161,0x11F6 ],
    [ 0xE5E8, 0x1107,0x1161,0x11F9 ],
    [ 0xE5EA, 0x1107,0x1176,0x11AF ],
    [ 0xE5EC, 0x1107,0x1177,0x11AF ],
    [ 0xE5ED, 0x1107,0x1162,0x11F0 ],
    [ 0xE5EE, 0x1107,0x1163,0x11E2 ],
    [ 0xE5F0, 0x1107,0x1165,0x11C7 ],
    [ 0xE5F1, 0x1107,0x1165,0x11CD ],
    [ 0xE5F2, 0x1107,0x1165,0x11D9 ],
    [ 0xE5F3, 0x1107,0x1165,0x11EB ],
    [ 0xE5F4, 0x1107,0x1165,0x11F0 ],
    [ 0xE5F6, 0x1107,0x117B,0x11AF ],
    [ 0xE5F8, 0x1107,0x117C,0x11A8 ],
    [ 0xE5F9, 0x1107,0x117C,0x11AF ],
    [ 0xE5FA, 0x1107,0x117C,0x11B7 ],
    [ 0xE5FB, 0x1107,0x117C,0x11BC ],
    [ 0xE5FC, 0x1107,0x1167,0x11D9 ],
    [ 0xE5FD, 0x1107,0x1167,0x11E2 ],
    [ 0xE5FE, 0x1107,0x1167,0xD7F1 ],
    [ 0xE5FF, 0x1107,0x1167,0x11F0 ],
    [ 0xE600, 0x1107,0x1167,0x11F1 ],
    [ 0xE601, 0x1107,0x1167,0x11F9 ],
    [ 0xE602, 0x1107,0x1169,0x11D0 ],
    [ 0xE603, 0x1107,0x1169,0x11D9 ],
    [ 0xE604, 0x1107,0x1169,0xD7E0 ],
    [ 0xE605, 0x1107,0x1169,0x11DC ],
    [ 0xE606, 0x1107,0x1169,0x11DD ],
    [ 0xE607, 0x1107,0x1169,0x11E2 ],
    [ 0xE608, 0x1107,0x1169,0x11E7 ],
    [ 0xE609, 0x1107,0x1169,0xD7EC ],
    [ 0xE60A, 0x1107,0x1169,0x11EB ],
    [ 0xE60B, 0x1107,0x1169,0x11F0 ],
    [ 0xE60E, 0x1107,0x116C,0x11D9 ],
    [ 0xE60F, 0x1107,0x116C,0xD7F0 ],
    [ 0xE610, 0x1107,0x116D,0x11D9 ],
    [ 0xE611, 0x1107,0x116D,0x11E2 ],
    [ 0xE612, 0x1107,0x116D,0x11F0 ],
    [ 0xE615, 0x1107,0x116E,0x11C7 ],
    [ 0xE616, 0x1107,0x116E,0x11D0 ],
    [ 0xE617, 0x1107,0x116E,0x11D1 ],
    [ 0xE618, 0x1107,0x116E,0x11D2 ],
    [ 0xE619, 0x1107,0x116E,0x11D9 ],
    [ 0xE61A, 0x1107,0x116E,0x11E2 ],
    [ 0xE61B, 0x1107,0x116E,0x11E6 ],
    [ 0xE61C, 0x1107,0x116E,0x11E7 ],
    [ 0xE61D, 0x1107,0x116E,0xD7F0 ],
    [ 0xE61E, 0x1107,0x116E,0x11EB ],
    [ 0xE61F, 0x1107,0x116E,0x11EF ],
    [ 0xE620, 0x1107,0x116E,0x11F0 ],
    [ 0xE621, 0x1107,0x116E,0x11F9 ],
    [ 0xE622, 0x1107,0x116F,0x11F9 ],
    [ 0xE626, 0x1107,0x1171,0x11D9 ],
    [ 0xE627, 0x1107,0x1171,0x11F9 ],
    [ 0xE628, 0x1107,0x1172,0x11D9 ],
    [ 0xE629, 0x1107,0x1172,0x11E2 ],
    [ 0xE62B, 0x1107,0x1173,0x11C3 ],
    [ 0xE62C, 0x1107,0x1173,0xD7D4 ],
    [ 0xE62D, 0x1107,0x1173,0x11D0 ],
    [ 0xE62E, 0x1107,0x1173,0x11D9 ],
    [ 0xE62F, 0x1107,0x1173,0x11DD ],
    [ 0xE630, 0x1107,0x1173,0x11E7 ],
    [ 0xE631, 0x1107,0x1173,0xD7F1 ],
    [ 0xE632, 0x1107,0x1173,0x11EB ],
    [ 0xE633, 0x1107,0x1173,0x11F0 ],
    [ 0xE634, 0x1107,0x1173,0x11F9 ],
    [ 0xE636, 0x1107,0x1174,0x11F9 ],
    [ 0xE637, 0x1107,0x1175,0x11CA ],
    [ 0xE638, 0x1107,0x1175,0xD7D0 ],
    [ 0xE639, 0x1107,0x1175,0x11D9 ],
    [ 0xE63A, 0x1107,0x1175,0x11E2 ],
    [ 0xE63B, 0x1107,0x1175,0x11E7 ],
    [ 0xE63C, 0x1107,0x1175,0xD7EE ],
    [ 0xE63D, 0x1107,0x1175,0xD7F0 ],
    [ 0xE63E, 0x1107,0x1175,0xD7F2 ],
    [ 0xE63F, 0x1107,0x1175,0x11EB ],
    [ 0xE640, 0x1107,0x1175,0x11F0 ],
    [ 0xE641, 0x1107,0x1175,0x11F1 ],
    [ 0xE642, 0x1107,0x1175,0x11F9 ],
    [ 0xE645, 0x1107,0x1199,0x11AB ],
    [ 0xE64B, 0x1107,0x119E,0x11A8 ],
    [ 0xE64C, 0x1107,0x119E,0x11C3 ],
    [ 0xE64D, 0x1107,0x119E,0x11AB ],
    [ 0xE64E, 0x1107,0x119E,0x11AE ],
    [ 0xE64F, 0x1107,0x119E,0x11AF ],
    [ 0xE650, 0x1107,0x119E,0x11B0 ],
    [ 0xE651, 0x1107,0x119E,0x11D0 ],
    [ 0xE652, 0x1107,0x119E,0x11B1 ],
    [ 0xE653, 0x1107,0x119E,0x11B2 ],
    [ 0xE654, 0x1107,0x119E,0x11D5 ],
    [ 0xE655, 0x1107,0x119E,0x11B3 ],
    [ 0xE656, 0x1107,0x119E,0x11B4 ],
    [ 0xE657, 0x1107,0x119E,0x11B6 ],
    [ 0xE658, 0x1107,0x119E,0x11D9 ],
    [ 0xE659, 0x1107,0x119E,0x11B7 ],
    [ 0xE65A, 0x1107,0x119E,0x11B8 ],
    [ 0xE65B, 0x1107,0x119E,0x11E3 ],
    [ 0xE65C, 0x1107,0x119E,0x11BA ],
    [ 0xE65D, 0x1107,0x119E,0xD7F1 ],
    [ 0xE65E, 0x1107,0x119E,0x11EB ],
    [ 0xE65F, 0x1107,0x119E,0x11F0 ],
    [ 0xE660, 0x1107,0x119E,0x11BD ],
    [ 0xE661, 0x1107,0x119E,0x11BE ],
    [ 0xE662, 0x1107,0x119E,0x11C0 ],
    [ 0xE664, 0x1107,0x119F,0x11AB ],
    [ 0xE665, 0x1107,0x119F,0x11AF ],
    [ 0xE666, 0x1107,0x119F,0x11BC ],
    [ 0xE669, 0x1107,0x11A1,0x11A8 ],
    [ 0xE66A, 0x1107,0x11A1,0x11AB ],
    [ 0xE66B, 0x1107,0x11A1,0x11AE ],
    [ 0xE66C, 0x1107,0x11A1,0x11AF ],
    [ 0xE66D, 0x1107,0x11A1,0x11B0 ],
    [ 0xE66E, 0x1107,0x11A1,0x11D9 ],
    [ 0xE66F, 0x1107,0x11A1,0x11B7 ],
    [ 0xE670, 0x1107,0x11A1,0x11B8 ],
    [ 0xE671, 0x1107,0x11A1,0x11BA ],
    [ 0xE672, 0x1107,0x11A1,0x11BC ],
    [ 0xE673, 0x1107,0x11A1,0x11F0 ],
    [ 0xE674, 0x1107,0x11A1,0x11F1 ],
    [ 0xE675, 0x1107,0x11A1,0x11BD ],
    [ 0xE676, 0x1107,0x11A1,0x11BE ],
    [ 0xE677, 0x1107,0x11A1,0x11C0 ],
    [ 0xE678, 0x1107,0x11A1,0x11C2 ],
    [ 0xE67A, 0x111E,0x1161,0x11A8 ],
    [ 0xE67B, 0x111E,0x1161,0x11B7 ],
    [ 0xE67D, 0x111E,0x1162,0x11AB ],
    [ 0xE67F, 0x111E,0x1165,0x11BA ],
    [ 0xE681, 0x111E,0x1166,0x11AF ],
    [ 0xE683, 0x111E,0x1167,0x11AB ],
    [ 0xE684, 0x111E,0x1167,0x11BA ],
    [ 0xE686, 0x111E,0x1168,0x11AF ],
    [ 0xE688, 0x111E,0x1169,0x11BA ],
    [ 0xE68A, 0x111E,0x116C,0x11BA ],
    [ 0xE68C, 0x111E,0x116E,0x11AF ],
    [ 0xE68D, 0x111E,0x116E,0x11B7 ],
    [ 0xE68E, 0x111E,0x116E,0x11BA ],
    [ 0xE693, 0x111E,0x1173,0x11AB ],
    [ 0xE694, 0x111E,0x1173,0x11AE ],
    [ 0xE695, 0x111E,0x1173,0x11AF ],
    [ 0xE696, 0x111E,0x1173,0x11BA ],
    [ 0xE698, 0x111E,0x1174,0x11BA ],
    [ 0xE69A, 0x111E,0x1175,0x11AF ],
    [ 0xE69B, 0x111E,0x1175,0x11B7 ],
    [ 0xE69C, 0x111E,0x1175,0x11BA ],
    [ 0xE69E, 0x111E,0x119E,0x11AF ],
    [ 0xE6A0, 0x111E,0x11A1,0x11AB ],
    [ 0xE6A1, 0x111E,0x11A1,0x11BA ],
    [ 0xE6A4, 0x1120,0x1161,0x11A8 ],
    [ 0xE6A5, 0x1120,0x1161,0x11AB ],
    [ 0xE6A6, 0x1120,0x1161,0x11AF ],
    [ 0xE6A7, 0x1120,0x1161,0x11B7 ],
    [ 0xE6A8, 0x1120,0x1161,0x11B8 ],
    [ 0xE6A9, 0x1120,0x1161,0x11BA ],
    [ 0xE6AD, 0x1120,0x1165,0x11A8 ],
    [ 0xE6AE, 0x1120,0x1165,0x11AB ],
    [ 0xE6AF, 0x1120,0x1165,0x11AE ],
    [ 0xE6B0, 0x1120,0x1165,0x11AF ],
    [ 0xE6B1, 0x1120,0x1165,0x11B2 ],
    [ 0xE6B2, 0x1120,0x1165,0x11D5 ],
    [ 0xE6B3, 0x1120,0x1165,0x11B7 ],
    [ 0xE6B4, 0x1120,0x1165,0x11BA ],
    [ 0xE6B6, 0x1120,0x1166,0x11AB ],
    [ 0xE6B7, 0x1120,0x1166,0x11BA ],
    [ 0xE6B9, 0x1120,0x1167,0x11BA ],
    [ 0xE6BB, 0x1120,0x1168,0x11BA ],
    [ 0xE6BD, 0x1120,0x1169,0x11A8 ],
    [ 0xE6BE, 0x1120,0x1169,0x11AF ],
    [ 0xE6BF, 0x1120,0x1169,0x11B7 ],
    [ 0xE6C0, 0x1120,0x1169,0x11BC ],
    [ 0xE6C4, 0x1120,0x116C,0x11BA ],
    [ 0xE6C7, 0x1120,0x116E,0x11A8 ],
    [ 0xE6C8, 0x1120,0x116E,0x11AF ],
    [ 0xE6C9, 0x1120,0x116E,0x11B2 ],
    [ 0xE6CA, 0x1120,0x116E,0x11B7 ],
    [ 0xE6CB, 0x1120,0x116E,0x11BA ],
    [ 0xE6CC, 0x1120,0x116E,0x11BC ],
    [ 0xE6CD, 0x1120,0x116E,0x11F0 ],
    [ 0xE6D3, 0x1120,0x1171,0x11AF ],
    [ 0xE6D4, 0x1120,0x1171,0x11BA ],
    [ 0xE6D8, 0x1120,0x1173,0x11A8 ],
    [ 0xE6D9, 0x1120,0x1173,0x11AB ],
    [ 0xE6DA, 0x1120,0x1173,0x11AE ],
    [ 0xE6DB, 0x1120,0x1173,0x11AF ],
    [ 0xE6DC, 0x1120,0x1173,0x11B0 ],
    [ 0xE6DD, 0x1120,0x1173,0x11B2 ],
    [ 0xE6DE, 0x1120,0x1173,0x11B3 ],
    [ 0xE6DF, 0x1120,0x1173,0x11B6 ],
    [ 0xE6E0, 0x1120,0x1173,0x11B7 ],
    [ 0xE6E1, 0x1120,0x1173,0x11BA ],
    [ 0xE6E2, 0x1120,0x1173,0x11BC ],
    [ 0xE6E3, 0x1120,0x1173,0x11BD ],
    [ 0xE6E5, 0x1120,0x1174,0x11AF ],
    [ 0xE6E6, 0x1120,0x1174,0x11BA ],
    [ 0xE6E8, 0x1120,0x1175,0x11A8 ],
    [ 0xE6E9, 0x1120,0x1175,0x11AB ],
    [ 0xE6EA, 0x1120,0x1175,0x11AF ],
    [ 0xE6EB, 0x1120,0x1175,0x11D0 ],
    [ 0xE6EC, 0x1120,0x1175,0x11B7 ],
    [ 0xE6ED, 0x1120,0x1175,0x11BA ],
    [ 0xE6F0, 0x1120,0x119E,0x11AB ],
    [ 0xE6F1, 0x1120,0x119E,0x11AE ],
    [ 0xE6F2, 0x1120,0x119E,0x11AF ],
    [ 0xE6F3, 0x1120,0x119E,0x11B7 ],
    [ 0xE6F4, 0x1120,0x119E,0x11BA ],
    [ 0xE6F6, 0x1120,0x11A1,0x11AB ],
    [ 0xE6F7, 0x1120,0x11A1,0x11BA ],
    [ 0xE6F8, 0x1108,0x1161,0x11D0 ],
    [ 0xE6F9, 0x1108,0x1161,0x11D9 ],
    [ 0xE6FA, 0x1108,0x1161,0x11E2 ],
    [ 0xE6FB, 0x1108,0x1161,0x11E6 ],
    [ 0xE6FC, 0x1108,0x1161,0x11F0 ],
    [ 0xE6FD, 0x1108,0x1161,0x11F9 ],
    [ 0xE6FE, 0x1108,0x1163,0x11E2 ],
    [ 0xE6FF, 0x1108,0x1163,0x11F0 ],
    [ 0xE700, 0x1108,0x1165,0x11D9 ],
    [ 0xE701, 0x1108,0x1165,0x11F9 ],
    [ 0xE702, 0x1108,0x1167,0x11D9 ],
    [ 0xE703, 0x1108,0x1167,0x11E2 ],
    [ 0xE704, 0x1108,0x1167,0x11F0 ],
    [ 0xE705, 0x1108,0x1167,0x11F9 ],
    [ 0xE706, 0x1108,0x1169,0x11D9 ],
    [ 0xE707, 0x1108,0x1169,0x11E2 ],
    [ 0xE708, 0x1108,0x1169,0x11F0 ],
    [ 0xE709, 0x1108,0x116D,0x11E2 ],
    [ 0xE70A, 0x1108,0x116E,0x11D9 ],
    [ 0xE70B, 0x1108,0x116E,0x11E2 ],
    [ 0xE70C, 0x1108,0x116E,0x11F0 ],
    [ 0xE70D, 0x1108,0x116E,0x11F9 ],
    [ 0xE70E, 0x1108,0x116F,0x11F9 ],
    [ 0xE710, 0x1108,0xD7B5,0x11F9 ],
    [ 0xE711, 0x1108,0x1172,0x11E2 ],
    [ 0xE712, 0x1108,0x1173,0x11D9 ],
    [ 0xE713, 0x1108,0x1173,0x11E2 ],
    [ 0xE714, 0x1108,0x1173,0xD7E6 ],
    [ 0xE715, 0x1108,0x1173,0x11F0 ],
    [ 0xE716, 0x1108,0x1173,0x11F9 ],
    [ 0xE717, 0x1108,0x1174,0x11F0 ],
    [ 0xE718, 0x1108,0x1174,0x11F9 ],
    [ 0xE719, 0x1108,0x1175,0x11D9 ],
    [ 0xE71A, 0x1108,0x1175,0x11E2 ],
    [ 0xE71B, 0x1108,0x1175,0x11F0 ],
    [ 0xE71C, 0x1108,0x1175,0x11F9 ],
    [ 0xE71E, 0x1108,0x119E,0x11AF ],
    [ 0xE71F, 0x1108,0x119E,0x11B3 ],
    [ 0xE720, 0x1108,0x119E,0x11D9 ],
    [ 0xE722, 0x1108,0x11A1,0x11A8 ],
    [ 0xE723, 0x1108,0x11A1,0x11BC ],
    [ 0xE724, 0x1108,0x11A1,0x11F0 ],
    [ 0xE726, 0x1121,0x1161,0x11A8 ],
    [ 0xE727, 0x1121,0x1161,0x11AB ],
    [ 0xE728, 0x1121,0x1161,0x11AE ],
    [ 0xE729, 0x1121,0x1161,0x11AF ],
    [ 0xE72A, 0x1121,0x1161,0x11B2 ],
    [ 0xE72B, 0x1121,0x1161,0x11B7 ],
    [ 0xE72C, 0x1121,0x1161,0x11B8 ],
    [ 0xE72D, 0x1121,0x1161,0x11BA ],
    [ 0xE72E, 0x1121,0x1161,0xD7F1 ],
    [ 0xE72F, 0x1121,0x1161,0xD7F2 ],
    [ 0xE730, 0x1121,0x1161,0x11BC ],
    [ 0xE731, 0x1121,0x1161,0x11F0 ],
    [ 0xE732, 0x1121,0x1161,0x11C2 ],
    [ 0xE734, 0x1121,0x1162,0x11B8 ],
    [ 0xE736, 0x1121,0x1163,0x11BC ],
    [ 0xE737, 0x1121,0x1163,0x11F0 ],
    [ 0xE738, 0x1121,0x1163,0x11C1 ],
    [ 0xE739, 0x1121,0x1163,0x11C2 ],
    [ 0xE73C, 0x1121,0x1165,0x11A8 ],
    [ 0xE73D, 0x1121,0x1165,0x11AB ],
    [ 0xE73E, 0x1121,0x1165,0x11AF ],
    [ 0xE73F, 0x1121,0x1165,0x11B7 ],
    [ 0xE740, 0x1121,0x1165,0x11BA ],
    [ 0xE742, 0x1121,0x1166,0x11AB ],
    [ 0xE743, 0x1121,0x1166,0x11AF ],
    [ 0xE745, 0x1121,0x1167,0x11A8 ],
    [ 0xE746, 0x1121,0x1167,0x11AB ],
    [ 0xE747, 0x1121,0x1167,0x11AF ],
    [ 0xE748, 0x1121,0x1167,0x11B7 ],
    [ 0xE749, 0x1121,0x1167,0x11BA ],
    [ 0xE74B, 0x1121,0x1168,0x11AB ],
    [ 0xE74D, 0x1121,0x1169,0x11A8 ],
    [ 0xE74E, 0x1121,0x1169,0x11AB ],
    [ 0xE74F, 0x1121,0x1169,0x11AE ],
    [ 0xE750, 0x1121,0x1169,0x11AF ],
    [ 0xE751, 0x1121,0x1169,0x11B7 ],
    [ 0xE752, 0x1121,0x1169,0x11BA ],
    [ 0xE753, 0x1121,0x1169,0x11BC ],
    [ 0xE754, 0x1121,0x1169,0x11F0 ],
    [ 0xE756, 0x1121,0x116A,0x11BC ],
    [ 0xE757, 0x1121,0x116A,0x11F0 ],
    [ 0xE758, 0x1121,0x116A,0x11F9 ],
    [ 0xE75A, 0x1121,0x116B,0x11BA ],
    [ 0xE75E, 0x1121,0x116E,0x11A8 ],
    [ 0xE75F, 0x1121,0x116E,0x11AB ],
    [ 0xE760, 0x1121,0x116E,0x11AF ],
    [ 0xE761, 0x1121,0x116E,0x11D9 ],
    [ 0xE762, 0x1121,0x116E,0x11B7 ],
    [ 0xE763, 0x1121,0x116E,0x11BA ],
    [ 0xE764, 0x1121,0x116E,0x11BD ],
    [ 0xE765, 0x1121,0x116E,0x11BE ],
    [ 0xE76C, 0x1121,0x1172,0x11B8 ],
    [ 0xE76D, 0x1121,0x1172,0x11BA ],
    [ 0xE76E, 0x1121,0x1172,0x11BD ],
    [ 0xE771, 0x1121,0x1173,0x11A8 ],
    [ 0xE772, 0x1121,0x1173,0x11AB ],
    [ 0xE773, 0x1121,0x1173,0x11AE ],
    [ 0xE774, 0x1121,0x1173,0x11AF ],
    [ 0xE775, 0x1121,0x1173,0x11B6 ],
    [ 0xE776, 0x1121,0x1173,0x11D9 ],
    [ 0xE777, 0x1121,0x1173,0x11B7 ],
    [ 0xE778, 0x1121,0x1173,0x11BA ],
    [ 0xE779, 0x1121,0x1173,0x11BE ],
    [ 0xE77B, 0x1121,0x1174,0x11A8 ],
    [ 0xE77D, 0x1121,0x1175,0x11A8 ],
    [ 0xE77E, 0x1121,0x1175,0x11AB ],
    [ 0xE77F, 0x1121,0x1175,0x11AF ],
    [ 0xE780, 0x1121,0x1175,0x11B7 ],
    [ 0xE781, 0x1121,0x1175,0x11B8 ],
    [ 0xE782, 0x1121,0x1175,0x11BA ],
    [ 0xE783, 0x1121,0x1175,0x11BC ],
    [ 0xE784, 0x1121,0x1175,0x11F0 ],
    [ 0xE787, 0x1121,0x119E,0x11AB ],
    [ 0xE788, 0x1121,0x119E,0x11AE ],
    [ 0xE789, 0x1121,0x119E,0x11AF ],
    [ 0xE78A, 0x1121,0x119E,0x11B3 ],
    [ 0xE78B, 0x1121,0x119E,0x11D9 ],
    [ 0xE78C, 0x1121,0x119E,0x11B7 ],
    [ 0xE78D, 0x1121,0x119E,0x11B8 ],
    [ 0xE78E, 0x1121,0x119E,0x11BA ],
    [ 0xE78F, 0x1121,0x119E,0xD7F2 ],
    [ 0xE790, 0x1121,0x119E,0x11F0 ],
    [ 0xE791, 0x1121,0x119E,0x11C2 ],
    [ 0xE793, 0x1121,0x11A1,0x11BA ],
    [ 0xE796, 0x1122,0x1162,0x11AB ],
    [ 0xE797, 0x1122,0x1162,0x11BA ],
    [ 0xE799, 0x1122,0x1165,0x11B8 ],
    [ 0xE79A, 0x1122,0x1165,0x11E6 ],
    [ 0xE79B, 0x1122,0x1165,0x11BA ],
    [ 0xE79D, 0x1122,0x1166,0x11AF ],
    [ 0xE79F, 0x1122,0x1167,0x11A8 ],
    [ 0xE7A0, 0x1122,0x1167,0x11BA ],
    [ 0xE7A2, 0x1122,0x1168,0x11BA ],
    [ 0xE7A5, 0x1122,0x116C,0x11BA ],
    [ 0xE7A7, 0x1122,0x116E,0x11AB ],
    [ 0xE7A8, 0x1122,0x116E,0x11AF ],
    [ 0xE7A9, 0x1122,0x116E,0x11B7 ],
    [ 0xE7AA, 0x1122,0x116E,0x11BA ],
    [ 0xE7AE, 0x1122,0x1171,0x11AB ],
    [ 0xE7B0, 0x1122,0x1173,0x11AB ],
    [ 0xE7B1, 0x1122,0x1173,0x11AF ],
    [ 0xE7B2, 0x1122,0x1173,0x11B7 ],
    [ 0xE7B3, 0x1122,0x1173,0x11BA ],
    [ 0xE7B5, 0x1122,0x1174,0x11AB ],
    [ 0xE7B6, 0x1122,0x1174,0x11AF ],
    [ 0xE7B7, 0x1122,0x1174,0x11B7 ],
    [ 0xE7B8, 0x1122,0x1174,0x11BA ],
    [ 0xE7BA, 0x1122,0x1175,0x11AF ],
    [ 0xE7BB, 0x1122,0x1175,0x11B7 ],
    [ 0xE7BD, 0x1122,0x119E,0x11AB ],
    [ 0xE7BF, 0x1123,0x1161,0x11B8 ],
    [ 0xE7C2, 0x1123,0x1165,0x11AF ],
    [ 0xE7C3, 0x1123,0x1165,0x11B2 ],
    [ 0xE7C4, 0x1123,0x1165,0x11BA ],
    [ 0xE7C6, 0x1123,0x1167,0x11BA ],
    [ 0xE7CA, 0x1123,0x116E,0x11AF ],
    [ 0xE7CB, 0x1123,0x116E,0x11B7 ],
    [ 0xE7CD, 0x1123,0x1172,0x11B7 ],
    [ 0xE7CF, 0x1123,0x1173,0x11AB ],
    [ 0xE7D0, 0x1123,0x1173,0x11BA ],
    [ 0xE7D3, 0x1123,0x1175,0x11AF ],
    [ 0xE7D5, 0x1123,0x119E,0x11AF ],
    [ 0xE7DD, 0x1126,0x119E,0x11AB ],
    [ 0xE7E0, 0x1127,0x1161,0x11A8 ],
    [ 0xE7E1, 0x1127,0x1161,0x11AB ],
    [ 0xE7E2, 0x1127,0x1161,0x11B7 ],
    [ 0xE7E3, 0x1127,0x1161,0x11BA ],
    [ 0xE7E4, 0x1127,0x1161,0x11BC ],
    [ 0xE7E5, 0x1127,0x1161,0x11F0 ],
    [ 0xE7E8, 0x1127,0x1163,0x11A8 ],
    [ 0xE7E9, 0x1127,0x1163,0x11BC ],
    [ 0xE7EA, 0x1127,0x1163,0x11F0 ],
    [ 0xE7EC, 0x1127,0x1165,0x11A8 ],
    [ 0xE7F0, 0x1127,0x1169,0x11A8 ],
    [ 0xE7F1, 0x1127,0x1169,0x11AB ],
    [ 0xE7F2, 0x1127,0x1169,0x11B7 ],
    [ 0xE7F3, 0x1127,0x1169,0x11BA ],
    [ 0xE7F4, 0x1127,0x1169,0x11BC ],
    [ 0xE7F5, 0x1127,0x1169,0x11F0 ],
    [ 0xE7F6, 0x1127,0x1169,0x11BD ],
    [ 0xE7F7, 0x1127,0x1169,0x11BE ],
    [ 0xE7F9, 0x1127,0x116A,0x11BC ],
    [ 0xE7FA, 0x1127,0x116A,0x11F0 ],
    [ 0xE7FD, 0x1127,0x116C,0x11AB ],
    [ 0xE7FE, 0x1127,0x116C,0x11AF ],
    [ 0xE801, 0x1127,0x116E,0x11A8 ],
    [ 0xE802, 0x1127,0x116E,0x11AF ],
    [ 0xE803, 0x1127,0x116E,0x11B7 ],
    [ 0xE806, 0x1127,0x1173,0x11A8 ],
    [ 0xE807, 0x1127,0x1173,0x11AB ],
    [ 0xE808, 0x1127,0x1173,0x11AE ],
    [ 0xE809, 0x1127,0x1173,0x11AF ],
    [ 0xE80A, 0x1127,0x1173,0x11B7 ],
    [ 0xE80B, 0x1127,0x1173,0x11BA ],
    [ 0xE80C, 0x1127,0x1173,0x11BD ],
    [ 0xE80E, 0x1127,0x1174,0x11BC ],
    [ 0xE80F, 0x1127,0x1174,0x11F0 ],
    [ 0xE811, 0x1127,0x1175,0x11A8 ],
    [ 0xE812, 0x1127,0x1175,0x11AB ],
    [ 0xE813, 0x1127,0x1175,0x11AE ],
    [ 0xE814, 0x1127,0x1175,0x11AF ],
    [ 0xE815, 0x1127,0x1175,0x11B7 ],
    [ 0xE816, 0x1127,0x1175,0x11BA ],
    [ 0xE817, 0x1127,0x1175,0xD7F2 ],
    [ 0xE818, 0x1127,0x1175,0x11BC ],
    [ 0xE819, 0x1127,0x1175,0x11F0 ],
    [ 0xE81B, 0x1127,0x119E,0x11A8 ],
    [ 0xE81C, 0x1127,0x119E,0x11AB ],
    [ 0xE81D, 0x1127,0x119E,0x11AF ],
    [ 0xE81E, 0x1127,0x119E,0x11B7 ],
    [ 0xE820, 0x1127,0x11A1,0x11AB ],
    [ 0xE821, 0x1127,0x11A1,0x11AF ],
    [ 0xE822, 0x1127,0x11A1,0x11BC ],
    [ 0xE824, 0x1128,0x1169,0x11BC ],
    [ 0xE826, 0x1128,0x119E,0x11AF ],
    [ 0xE82D, 0x1129,0x1165,0x11A8 ],
    [ 0xE82E, 0x1129,0x1165,0x11AF ],
    [ 0xE82F, 0x1129,0x1165,0x11B2 ],
    [ 0xE830, 0x1129,0x1165,0x11D5 ],
    [ 0xE831, 0x1129,0x1165,0x11B7 ],
    [ 0xE832, 0x1129,0x1165,0x11BA ],
    [ 0xE835, 0x1129,0x1169,0x11B7 ],
    [ 0xE836, 0x1129,0x1169,0x11B8 ],
    [ 0xE838, 0x1129,0x116E,0x11A8 ],
    [ 0xE839, 0x1129,0x116E,0x11FB ],
    [ 0xE83B, 0x1129,0x1171,0x11AF ],
    [ 0xE83C, 0x1129,0x1171,0x11B7 ],
    [ 0xE83E, 0x1129,0x1173,0x11AB ],
    [ 0xE83F, 0x1129,0x1173,0x11AE ],
    [ 0xE840, 0x1129,0x1173,0x11BA ],
    [ 0xE842, 0x1129,0x1174,0x11B7 ],
    [ 0xE844, 0x1129,0x1175,0x11AB ],
    [ 0xE845, 0x1129,0x1175,0x11AF ],
    [ 0xE846, 0x1129,0x1175,0x11B7 ],
    [ 0xE848, 0x1129,0x119E,0x11AB ],
    [ 0xE849, 0x1129,0x119E,0x11AF ],
    [ 0xE84C, 0x112A,0x1161,0x11AB ],
    [ 0xE84E, 0x112A,0x1169,0x11AB ],
    [ 0xE851, 0x112A,0x1173,0x11AF ],
    [ 0xE854, 0x112A,0x119E,0x11AF ],
    [ 0xE856, 0xA974,0x1161,0x11AB ],
    [ 0xE858, 0x112B,0x1161,0x11A8 ],
    [ 0xE859, 0x112B,0x1161,0x11AB ],
    [ 0xE85A, 0x112B,0x1161,0x11AE ],
    [ 0xE85B, 0x112B,0x1161,0x11AF ],
    [ 0xE85C, 0x112B,0x1161,0x11B0 ],
    [ 0xE85D, 0x112B,0x1161,0x11D9 ],
    [ 0xE85E, 0x112B,0x1161,0x11B7 ],
    [ 0xE85F, 0x112B,0x1161,0x11BA ],
    [ 0xE860, 0x112B,0x1161,0x11EB ],
    [ 0xE861, 0x112B,0x1161,0x11BC ],
    [ 0xE862, 0x112B,0x1161,0x11F0 ],
    [ 0xE863, 0x112B,0x1161,0x11C0 ],
    [ 0xE864, 0x112B,0x1161,0x11F9 ],
    [ 0xE866, 0x112B,0x1162,0x11AB ],
    [ 0xE867, 0x112B,0x1162,0x11BA ],
    [ 0xE869, 0x112B,0x1165,0x11AB ],
    [ 0xE86A, 0x112B,0x1165,0x11B7 ],
    [ 0xE86B, 0x112B,0x1165,0x11B8 ],
    [ 0xE86C, 0x112B,0x1165,0x11BC ],
    [ 0xE86E, 0x112B,0x1166,0x11BA ],
    [ 0xE870, 0x112B,0x1167,0x11A8 ],
    [ 0xE871, 0x112B,0x1167,0x11AF ],
    [ 0xE872, 0x112B,0x1167,0x11D9 ],
    [ 0xE873, 0x112B,0x1167,0x11BA ],
    [ 0xE875, 0x112B,0x1168,0x11BA ],
    [ 0xE877, 0x112B,0x1169,0x11AB ],
    [ 0xE878, 0x112B,0x1169,0x11AF ],
    [ 0xE879, 0x112B,0x1169,0x11B3 ],
    [ 0xE87A, 0x112B,0x1169,0x11D9 ],
    [ 0xE87B, 0x112B,0x1169,0x11B7 ],
    [ 0xE87C, 0x112B,0x1169,0x11DD ],
    [ 0xE87D, 0x112B,0x1169,0x11B8 ],
    [ 0xE87E, 0x112B,0x1169,0x11BA ],
    [ 0xE87F, 0x112B,0x1169,0x11BC ],
    [ 0xE881, 0x112B,0x116A,0x11AB ],
    [ 0xE886, 0x112B,0x116E,0x11AB ],
    [ 0xE887, 0x112B,0x116E,0x11AF ],
    [ 0xE888, 0x112B,0x116E,0x11B7 ],
    [ 0xE889, 0x112B,0x116E,0x11B8 ],
    [ 0xE88A, 0x112B,0x116E,0x11BA ],
    [ 0xE88B, 0x112B,0x116E,0x11BC ],
    [ 0xE88C, 0x112B,0x116E,0x11F0 ],
    [ 0xE88D, 0x112B,0x116E,0x11C2 ],
    [ 0xE88E, 0x112B,0x116E,0x11F9 ],
    [ 0xE891, 0x112B,0x1171,0x11AF ],
    [ 0xE892, 0x112B,0x1171,0x11D9 ],
    [ 0xE894, 0x112B,0x1172,0x11F9 ],
    [ 0xE896, 0x112B,0x1173,0x11AB ],
    [ 0xE897, 0x112B,0x1173,0x11C7 ],
    [ 0xE898, 0x112B,0x1173,0x11AF ],
    [ 0xE899, 0x112B,0x1173,0x11B3 ],
    [ 0xE89A, 0x112B,0x1173,0x11D9 ],
    [ 0xE89B, 0x112B,0x1173,0x11B7 ],
    [ 0xE89C, 0x112B,0x1173,0x11E2 ],
    [ 0xE89D, 0x112B,0x1173,0x11BA ],
    [ 0xE89E, 0x112B,0x1173,0x11BC ],
    [ 0xE8A2, 0x112B,0x1175,0x11A8 ],
    [ 0xE8A3, 0x112B,0x1175,0x11AB ],
    [ 0xE8A4, 0x112B,0x1175,0x11AF ],
    [ 0xE8A5, 0x112B,0x1175,0x11D9 ],
    [ 0xE8A6, 0x112B,0x1175,0x11BC ],
    [ 0xE8A8, 0x112B,0x119E,0x11A8 ],
    [ 0xE8A9, 0x112B,0x119E,0x11AB ],
    [ 0xE8AA, 0x112B,0x119E,0x11AF ],
    [ 0xE8AB, 0x112B,0x119E,0x11B0 ],
    [ 0xE8AC, 0x112B,0x119E,0x11B2 ],
    [ 0xE8AD, 0x112B,0x119E,0x11B3 ],
    [ 0xE8AE, 0x112B,0x119E,0x11D9 ],
    [ 0xE8AF, 0x112B,0x119E,0x11B7 ],
    [ 0xE8B0, 0x112B,0x119E,0x11EB ],
    [ 0xE8B1, 0x112B,0x119E,0x11BC ],
    [ 0xE8B3, 0x112B,0x11A1,0x11A8 ],
    [ 0xE8B4, 0x112B,0x11A1,0x11AB ],
    [ 0xE8B5, 0x112B,0x11A1,0x11AF ],
    [ 0xE8B7, 0x112C,0x1161,0x11AB ],
    [ 0xE8B8, 0x112C,0x1161,0x11BC ],
    [ 0xE8B9, 0x112C,0x1161,0x11F0 ],
    [ 0xE8BA, 0x112C,0x1161,0x11F9 ],
    [ 0xE8BC, 0x112C,0x1169,0x11AB ],
    [ 0xE8BE, 0x112C,0x116A,0x11AB ],
    [ 0xE8C0, 0x112C,0x116E,0x11AB ],
    [ 0xE8C1, 0x112C,0x116E,0x11BC ],
    [ 0xE8C2, 0x112C,0x116E,0x11F0 ],
    [ 0xE8C3, 0x112C,0x116E,0x11F9 ],
    [ 0xE8C5, 0x112C,0x1172,0x11AB ],
    [ 0xE8C7, 0x112C,0x1173,0x11AB ],
    [ 0xE8C8, 0x112C,0x1173,0x11B7 ],
    [ 0xE8C9, 0x112C,0x1173,0x11E2 ],
    [ 0xE8CA, 0x112C,0x1173,0x11BC ],
    [ 0xE8CD, 0x1109,0x1161,0x11C7 ],
    [ 0xE8CE, 0x1109,0x1161,0x11CA ],
    [ 0xE8CF, 0x1109,0x1161,0xD7D4 ],
    [ 0xE8D0, 0x1109,0x1161,0x11CD ],
    [ 0xE8D1, 0x1109,0x1161,0x11D0 ],
    [ 0xE8D2, 0x1109,0x1161,0x11D9 ],
    [ 0xE8D3, 0x1109,0x1161,0x11DD ],
    [ 0xE8D4, 0x1109,0x1161,0x11E2 ],
    [ 0xE8D5, 0x1109,0x1161,0x11E7 ],
    [ 0xE8D6, 0x1109,0x1161,0x11E8 ],
    [ 0xE8D7, 0x1109,0x1161,0xD7F2 ],
    [ 0xE8D8, 0x1109,0x1161,0x11EB ],
    [ 0xE8D9, 0x1109,0x1161,0x11F1 ],
    [ 0xE8DA, 0x1109,0x1161,0x11F0 ],
    [ 0xE8DB, 0x1109,0x1161,0x11F1 ],
    [ 0xE8DC, 0x1109,0x1161,0x11F9 ],
    [ 0xE8E0, 0x1109,0x11A3,0x11B7 ],
    [ 0xE8E1, 0x1109,0x1162,0x11E6 ],
    [ 0xE8E2, 0x1109,0x1162,0x11F0 ],
    [ 0xE8E3, 0x1109,0x1163,0x11D9 ],
    [ 0xE8E4, 0x1109,0x1163,0x11E2 ],
    [ 0xE8E5, 0x1109,0x1163,0x11EB ],
    [ 0xE8E6, 0x1109,0x1163,0x11F0 ],
    [ 0xE8E7, 0x1109,0x1163,0x11F1 ],
    [ 0xE8E9, 0x1109,0x1165,0x11C3 ],
    [ 0xE8EA, 0x1109,0x1165,0x11D9 ],
    [ 0xE8EB, 0x1109,0x1165,0x11E5 ],
    [ 0xE8EC, 0x1109,0x1165,0x11E7 ],
    [ 0xE8ED, 0x1109,0x1165,0x11EB ],
    [ 0xE8EE, 0x1109,0x1165,0x11EC ],
    [ 0xE8EF, 0x1109,0x1165,0x11ED ],
    [ 0xE8F0, 0x1109,0x1165,0x11F0 ],
    [ 0xE8F1, 0x1109,0x1165,0x11F9 ],
    [ 0xE8F5, 0x1109,0x117C,0x11A8 ],
    [ 0xE8F6, 0x1109,0x117C,0x11AF ],
    [ 0xE8F7, 0x1109,0x117C,0x11B7 ],
    [ 0xE8F8, 0x1109,0x117C,0x11BA ],
    [ 0xE8F9, 0x1109,0x117C,0x11BB ],
    [ 0xE8FA, 0x1109,0x117C,0x11BC ],
    [ 0xE8FB, 0x1109,0x1166,0xD7FB ],
    [ 0xE8FC, 0x1109,0x1167,0x11D5 ],
    [ 0xE8FD, 0x1109,0x1167,0x11D9 ],
    [ 0xE8FE, 0x1109,0x1167,0x11DD ],
    [ 0xE8FF, 0x1109,0x1167,0x11E2 ],
    [ 0xE900, 0x1109,0x1167,0x11E3 ],
    [ 0xE901, 0x1109,0x1167,0x11E4 ],
    [ 0xE902, 0x1109,0x1167,0x11E5 ],
    [ 0xE903, 0x1109,0x1167,0x11E7 ],
    [ 0xE904, 0x1109,0x1167,0x11F0 ],
    [ 0xE905, 0x1109,0x1167,0x11F1 ],
    [ 0xE906, 0x1109,0x1167,0x11F9 ],
    [ 0xE909, 0x1109,0x1168,0x11F0 ],
    [ 0xE90A, 0x1109,0x1169,0x11C7 ],
    [ 0xE90B, 0x1109,0x1169,0x11D9 ],
    [ 0xE90C, 0x1109,0x1169,0x11E2 ],
    [ 0xE90D, 0x1109,0x1169,0xD7E7 ],
    [ 0xE90E, 0x1109,0x1169,0x11E8 ],
    [ 0xE90F, 0x1109,0x1169,0xD7F0 ],
    [ 0xE910, 0x1109,0x1169,0x11EB ],
    [ 0xE911, 0x1109,0x1169,0x11F0 ],
    [ 0xE912, 0x1109,0x116A,0x11D9 ],
    [ 0xE913, 0x1109,0x116A,0x11F0 ],
    [ 0xE914, 0x1109,0x116A,0x11F9 ],
    [ 0xE915, 0x1109,0x116B,0x11F9 ],
    [ 0xE919, 0x1109,0x116C,0x11F0 ],
    [ 0xE91A, 0x1109,0x116D,0x11D9 ],
    [ 0xE91B, 0x1109,0x116D,0x11E2 ],
    [ 0xE91C, 0x1109,0x116D,0x11F0 ],
    [ 0xE91D, 0x1109,0x116D,0x11F1 ],
    [ 0xE91F, 0x1109,0x1187,0x11AF ],
    [ 0xE921, 0x1109,0x1188,0x11AB ],
    [ 0xE922, 0x1109,0x1188,0x11AF ],
    [ 0xE923, 0x1109,0x1188,0x11BA ],
    [ 0xE924, 0x1109,0x116E,0x11D0 ],
    [ 0xE925, 0x1109,0x116E,0x11D9 ],
    [ 0xE926, 0x1109,0x116E,0x11E2 ],
    [ 0xE927, 0x1109,0x116E,0x11E4 ],
    [ 0xE928, 0x1109,0x116E,0x11E6 ],
    [ 0xE929, 0x1109,0x116E,0x11E7 ],
    [ 0xE92A, 0x1109,0x116E,0xD7F0 ],
    [ 0xE92B, 0x1109,0x116E,0x11EB ],
    [ 0xE92C, 0x1109,0x116E,0x11F0 ],
    [ 0xE92D, 0x1109,0x116E,0x11F9 ],
    [ 0xE92F, 0x1109,0x1189,0x11AB ],
    [ 0xE930, 0x1109,0x1189,0x11B7 ],
    [ 0xE932, 0x1109,0xD7B5,0x11AB ],
    [ 0xE934, 0x1109,0x1171,0x11D9 ],
    [ 0xE935, 0x1109,0x1171,0x11E6 ],
    [ 0xE936, 0x1109,0x1171,0x11E7 ],
    [ 0xE937, 0x1109,0x1172,0x11D9 ],
    [ 0xE938, 0x1109,0x1172,0x11E2 ],
    [ 0xE939, 0x1109,0x1172,0x11E6 ],
    [ 0xE93A, 0x1109,0x1172,0x11F0 ],
    [ 0xE93C, 0x1109,0x118E,0x11AB ],
    [ 0xE93D, 0x1109,0x118E,0x11F0 ],
    [ 0xE942, 0x1109,0x1191,0x11AB ],
    [ 0xE943, 0x1109,0x1191,0x11AF ],
    [ 0xE944, 0x1109,0x1191,0x11D9 ],
    [ 0xE945, 0x1109,0x1191,0x11F0 ],
    [ 0xE946, 0x1109,0x1191,0x11F9 ],
    [ 0xE948, 0x1109,0x1192,0x11BC ],
    [ 0xE94A, 0x1109,0xD7B8,0x11AF ],
    [ 0xE94C, 0x1109,0x1193,0x11AF ],
    [ 0xE94E, 0x1109,0x1194,0x11AB ],
    [ 0xE94F, 0x1109,0x1194,0x11AF ],
    [ 0xE950, 0x1109,0x1194,0x11B7 ],
    [ 0xE951, 0x1109,0x1194,0x11B8 ],
    [ 0xE952, 0x1109,0x1194,0x11BC ],
    [ 0xE953, 0x1109,0x1173,0x11D9 ],
    [ 0xE954, 0x1109,0x1173,0x11E2 ],
    [ 0xE955, 0x1109,0x1173,0x11E3 ],
    [ 0xE956, 0x1109,0x1173,0xD7F0 ],
    [ 0xE957, 0x1109,0x1173,0xD7F2 ],
    [ 0xE958, 0x1109,0x1173,0x11EB ],
    [ 0xE959, 0x1109,0x1173,0x11F1 ],
    [ 0xE95A, 0x1109,0x1173,0xD7F6 ],
    [ 0xE95B, 0x1109,0x1173,0x11F0 ],
    [ 0xE95C, 0x1109,0x1173,0x11F2 ],
    [ 0xE95C, 0x1109,0x1173,0x11F1 ],
    [ 0xE95D, 0x1109,0x1173,0x11F9 ],
    [ 0xE960, 0x1109,0x1174,0x11F0 ],
    [ 0xE961, 0x1109,0x1174,0x11F9 ],
    [ 0xE962, 0x1109,0x1175,0x11FA ],
    [ 0xE963, 0x1109,0x1175,0x11FF ],
    [ 0xE964, 0x1109,0x1175,0x11C7 ],
    [ 0xE965, 0x1109,0x1175,0x11CE ],
    [ 0xE966, 0x1109,0x1175,0x11D0 ],
    [ 0xE967, 0x1109,0x1175,0x11D9 ],
    [ 0xE968, 0x1109,0x1175,0x11DA ],
    [ 0xE969, 0x1109,0x1175,0x11E1 ],
    [ 0xE96A, 0x1109,0x1175,0x11E2 ],
    [ 0xE96B, 0x1109,0x1175,0x11E6 ],
    [ 0xE96C, 0x1109,0x1175,0x11EB ],
    [ 0xE96D, 0x1109,0x1175,0x11F0 ],
    [ 0xE96E, 0x1109,0x1175,0x11F9 ],
    [ 0xE971, 0x1109,0x1199,0x11AB ],
    [ 0xE972, 0x1109,0x1199,0x11BC ],
    [ 0xE97C, 0x1109,0xD7C3,0x11AB ],
    [ 0xE97E, 0x1109,0x119E,0x11A8 ],
    [ 0xE97F, 0x1109,0x119E,0x11AB ],
    [ 0xE980, 0x1109,0x119E,0x11AE ],
    [ 0xE981, 0x1109,0x119E,0x11CA ],
    [ 0xE982, 0x1109,0x119E,0x11AF ],
    [ 0xE983, 0x1109,0x119E,0x11B0 ],
    [ 0xE984, 0x1109,0x119E,0x11D0 ],
    [ 0xE985, 0x1109,0x119E,0x11B1 ],
    [ 0xE986, 0x1109,0x119E,0x11B2 ],
    [ 0xE987, 0x1109,0x119E,0x11D5 ],
    [ 0xE988, 0x1109,0x119E,0x11B3 ],
    [ 0xE989, 0x1109,0x119E,0x11B6 ],
    [ 0xE98A, 0x1109,0x119E,0x11D9 ],
    [ 0xE98B, 0x1109,0x119E,0x11B7 ],
    [ 0xE98C, 0x1109,0x119E,0xD7E0 ],
    [ 0xE98D, 0x1109,0x119E,0x11DD ],
    [ 0xE98E, 0x1109,0x119E,0x11B8 ],
    [ 0xE98F, 0x1109,0x119E,0x11E3 ],
    [ 0xE990, 0x1109,0x119E,0x11E6 ],
    [ 0xE991, 0x1109,0x119E,0x11BA ],
    [ 0xE992, 0x1109,0x119E,0xD7F0 ],
    [ 0xE993, 0x1109,0x119E,0x11EB ],
    [ 0xE994, 0x1109,0x119E,0x11BC ],
    [ 0xE995, 0x1109,0x119E,0x11F0 ],
    [ 0xE996, 0x1109,0x119E,0x11BD ],
    [ 0xE997, 0x1109,0x119E,0x11BE ],
    [ 0xE998, 0x1109,0x119E,0x11C0 ],
    [ 0xE99A, 0x1109,0x11A1,0x11A8 ],
    [ 0xE99B, 0x1109,0x11A1,0x11AB ],
    [ 0xE99C, 0x1109,0x11A1,0x11AF ],
    [ 0xE99D, 0x1109,0x11A1,0x11D9 ],
    [ 0xE99E, 0x1109,0x11A1,0x11B7 ],
    [ 0xE99F, 0x1109,0x11A1,0x11DD ],
    [ 0xE9A0, 0x1109,0x11A1,0x11B8 ],
    [ 0xE9A1, 0x1109,0x11A1,0x11BA ],
    [ 0xE9A2, 0x1109,0x11A1,0x11BC ],
    [ 0xE9A3, 0x1109,0x11A1,0x11F0 ],
    [ 0xE9A4, 0x1109,0x11A1,0x11F1 ],
    [ 0xE9A6, 0x112D,0x1161,0x11A8 ],
    [ 0xE9A7, 0x112D,0x1161,0x11A9 ],
    [ 0xE9A8, 0x112D,0x1161,0x11AA ],
    [ 0xE9A9, 0x112D,0x1161,0x11AB ],
    [ 0xE9AA, 0x112D,0x1161,0x11AF ],
    [ 0xE9AB, 0x112D,0x1161,0x11B6 ],
    [ 0xE9AC, 0x112D,0x1161,0x11B7 ],
    [ 0xE9AD, 0x112D,0x1161,0x11B8 ],
    [ 0xE9AE, 0x112D,0x1161,0x11B9 ],
    [ 0xE9AF, 0x112D,0x1161,0x11E6 ],
    [ 0xE9B0, 0x112D,0x1161,0x11BA ],
    [ 0xE9B1, 0x112D,0x1161,0x11E7 ],
    [ 0xE9B2, 0x112D,0x1161,0x11BC ],
    [ 0xE9B3, 0x112D,0x1161,0x11F0 ],
    [ 0xE9B5, 0x112D,0x1162,0x11A8 ],
    [ 0xE9B6, 0x112D,0x1162,0x11AB ],
    [ 0xE9B7, 0x112D,0x1162,0x11AF ],
    [ 0xE9B8, 0x112D,0x1162,0x11B7 ],
    [ 0xE9B9, 0x112D,0x1162,0x11BA ],
    [ 0xE9BA, 0x112D,0x1162,0x11F0 ],
    [ 0xE9BC, 0x112D,0x1163,0x11A8 ],
    [ 0xE9BE, 0x112D,0x1164,0x11BA ],
    [ 0xE9C0, 0x112D,0x1165,0x11A8 ],
    [ 0xE9C1, 0x112D,0x1165,0x11A9 ],
    [ 0xE9C2, 0x112D,0x1165,0x11AB ],
    [ 0xE9C3, 0x112D,0x1165,0x11AF ],
    [ 0xE9C4, 0x112D,0x1165,0x11B2 ],
    [ 0xE9C5, 0x112D,0x1165,0x11D5 ],
    [ 0xE9C6, 0x112D,0x1165,0x11B7 ],
    [ 0xE9C7, 0x112D,0x1165,0x11B8 ],
    [ 0xE9C8, 0x112D,0x1165,0x11E6 ],
    [ 0xE9C9, 0x112D,0x1165,0x11BA ],
    [ 0xE9CA, 0x112D,0x1165,0x11E7 ],
    [ 0xE9CB, 0x112D,0x1165,0x11EB ],
    [ 0xE9CC, 0x112D,0x1165,0x11F0 ],
    [ 0xE9CE, 0x112D,0x1166,0x11A8 ],
    [ 0xE9CF, 0x112D,0x1166,0x11AB ],
    [ 0xE9D0, 0x112D,0x1166,0x11AF ],
    [ 0xE9D1, 0x112D,0x1166,0x11B7 ],
    [ 0xE9D2, 0x112D,0x1166,0x11B8 ],
    [ 0xE9D3, 0x112D,0x1166,0x11BA ],
    [ 0xE9D4, 0x112D,0x1166,0x11F0 ],
    [ 0xE9D6, 0x112D,0x1167,0x11A8 ],
    [ 0xE9D7, 0x112D,0x1167,0x11AB ],
    [ 0xE9D8, 0x112D,0x1167,0x11AE ],
    [ 0xE9D9, 0x112D,0x1167,0x11AF ],
    [ 0xE9DA, 0x112D,0x1167,0x11B7 ],
    [ 0xE9DB, 0x112D,0x1167,0x11B8 ],
    [ 0xE9DC, 0x112D,0x1167,0x11BA ],
    [ 0xE9DD, 0x112D,0x1167,0x11BC ],
    [ 0xE9DE, 0x112D,0x1167,0x11F0 ],
    [ 0xE9DF, 0x112D,0x1167,0x11C0 ],
    [ 0xE9E2, 0x112D,0x1169,0x11A8 ],
    [ 0xE9E3, 0x112D,0x1169,0x11AA ],
    [ 0xE9E4, 0x112D,0x1169,0x11AB ],
    [ 0xE9E5, 0x112D,0x1169,0x11AE ],
    [ 0xE9E6, 0x112D,0x1169,0x11AF ],
    [ 0xE9E7, 0x112D,0x1169,0x11B3 ],
    [ 0xE9E8, 0x112D,0x1169,0x11B7 ],
    [ 0xE9E9, 0x112D,0x1169,0x11B8 ],
    [ 0xE9EA, 0x112D,0x1169,0x11BA ],
    [ 0xE9EB, 0x112D,0x1169,0xD7F0 ],
    [ 0xE9EC, 0x112D,0x1169,0x11BC ],
    [ 0xE9ED, 0x112D,0x1169,0x11F0 ],
    [ 0xE9EE, 0x112D,0x1169,0x11BD ],
    [ 0xE9EF, 0x112D,0x1169,0x11BE ],
    [ 0xE9F1, 0x112D,0x116A,0x11A8 ],
    [ 0xE9F2, 0x112D,0x116A,0x11AB ],
    [ 0xE9F3, 0x112D,0x116A,0x11AF ],
    [ 0xE9F4, 0x112D,0x116A,0x11BC ],
    [ 0xE9F5, 0x112D,0x116A,0x11F0 ],
    [ 0xE9F7, 0x112D,0x116B,0x11AF ],
    [ 0xE9F8, 0x112D,0x116B,0x11B0 ],
    [ 0xE9F9, 0x112D,0x116B,0x11BC ],
    [ 0xE9FB, 0x112D,0x116C,0x11A8 ],
    [ 0xE9FC, 0x112D,0x116C,0x11AB ],
    [ 0xE9FD, 0x112D,0x116C,0x11AF ],
    [ 0xE9FE, 0x112D,0x116C,0x11B7 ],
    [ 0xE9FF, 0x112D,0x116C,0x11BA ],
    [ 0xEA00, 0x112D,0x116C,0x11F0 ],
    [ 0xEA02, 0x112D,0x116D,0x11AB ],
    [ 0xEA03, 0x112D,0x116D,0x11AF ],
    [ 0xEA04, 0x112D,0x116D,0x11B7 ],
    [ 0xEA06, 0x112D,0x116E,0x11A8 ],
    [ 0xEA07, 0x112D,0x116E,0x11AB ],
    [ 0xEA08, 0x112D,0x116E,0x11AE ],
    [ 0xEA09, 0x112D,0x116E,0x11AF ],
    [ 0xEA0A, 0x112D,0x116E,0x11D9 ],
    [ 0xEA0B, 0x112D,0x116E,0x11B7 ],
    [ 0xEA0C, 0x112D,0x116E,0x11DA ],
    [ 0xEA0D, 0x112D,0x116E,0xD7E0 ],
    [ 0xEA0E, 0x112D,0x116E,0x11DD ],
    [ 0xEA0F, 0x112D,0x116E,0x11B8 ],
    [ 0xEA10, 0x112D,0x116E,0x11BA ],
    [ 0xEA11, 0x112D,0x116E,0x11BC ],
    [ 0xEA12, 0x112D,0x116E,0x11EC ],
    [ 0xEA13, 0x112D,0x116E,0x11F0 ],
    [ 0xEA14, 0x112D,0x116E,0x11BD ],
    [ 0xEA15, 0x112D,0x116E,0x11BE ],
    [ 0xEA17, 0x112D,0x116F,0x11A8 ],
    [ 0xEA18, 0x112D,0x116F,0x11AB ],
    [ 0xEA19, 0x112D,0x116F,0x11AF ],
    [ 0xEA1A, 0x112D,0x116F,0x11B7 ],
    [ 0xEA1B, 0x112D,0x116F,0x11BC ],
    [ 0xEA1C, 0x112D,0x116F,0x11F0 ],
    [ 0xEA1E, 0x112D,0x1170,0x11AB ],
    [ 0xEA1F, 0x112D,0x1170,0x11AF ],
    [ 0xEA23, 0x112D,0x1171,0x11AB ],
    [ 0xEA24, 0x112D,0x1171,0x11AF ],
    [ 0xEA25, 0x112D,0x1171,0x11B7 ],
    [ 0xEA26, 0x112D,0x1171,0x11F0 ],
    [ 0xEA28, 0x112D,0x1172,0x11AB ],
    [ 0xEA29, 0x112D,0x1172,0x11AF ],
    [ 0xEA2A, 0x112D,0x1172,0x11B7 ],
    [ 0xEA2C, 0x112D,0x118F,0x11BC ],
    [ 0xEA2F, 0x112D,0x1173,0x11A8 ],
    [ 0xEA30, 0x112D,0x1173,0x11AB ],
    [ 0xEA31, 0x112D,0x1173,0xD7CC ],
    [ 0xEA32, 0x112D,0x1173,0x11AD ],
    [ 0xEA33, 0x112D,0x1173,0x11AE ],
    [ 0xEA34, 0x112D,0x1173,0xD7D3 ],
    [ 0xEA35, 0x112D,0x1173,0x11AF ],
    [ 0xEA36, 0x112D,0x1173,0x11D0 ],
    [ 0xEA37, 0x112D,0x1173,0x11B3 ],
    [ 0xEA38, 0x112D,0x1173,0x11B4 ],
    [ 0xEA39, 0x112D,0x1173,0x11B6 ],
    [ 0xEA3A, 0x112D,0x1173,0x11D9 ],
    [ 0xEA3B, 0x112D,0x1173,0x11B7 ],
    [ 0xEA3C, 0x112D,0x1173,0x11BA ],
    [ 0xEA3D, 0x112D,0x1173,0xD7F0 ],
    [ 0xEA3E, 0x112D,0x1173,0xD7F1 ],
    [ 0xEA3F, 0x112D,0x1173,0xD7F2 ],
    [ 0xEA40, 0x112D,0x1173,0x11EB ],
    [ 0xEA41, 0x112D,0x1173,0x11BC ],
    [ 0xEA42, 0x112D,0x1173,0x11F0 ],
    [ 0xEA43, 0x112D,0x1173,0x11BE ],
    [ 0xEA44, 0x112D,0x1173,0x11C0 ],
    [ 0xEA46, 0x112D,0x1174,0x11AB ],
    [ 0xEA47, 0x112D,0x1174,0x11AF ],
    [ 0xEA48, 0x112D,0x1174,0x11D0 ],
    [ 0xEA49, 0x112D,0x1174,0x11B7 ],
    [ 0xEA4A, 0x112D,0x1174,0x11BA ],
    [ 0xEA4C, 0x112D,0x1175,0x11A8 ],
    [ 0xEA4D, 0x112D,0x1175,0x11AB ],
    [ 0xEA4E, 0x112D,0x1175,0x11AD ],
    [ 0xEA4F, 0x112D,0x1175,0x11AE ],
    [ 0xEA50, 0x112D,0x1175,0x11AF ],
    [ 0xEA51, 0x112D,0x1175,0x11B6 ],
    [ 0xEA52, 0x112D,0x1175,0x11B7 ],
    [ 0xEA53, 0x112D,0x1175,0x11B8 ],
    [ 0xEA54, 0x112D,0x1175,0x11BA ],
    [ 0xEA55, 0x112D,0x1175,0x11BC ],
    [ 0xEA56, 0x112D,0x1175,0x11C2 ],
    [ 0xEA59, 0x112D,0x119E,0x11A8 ],
    [ 0xEA5A, 0x112D,0x119E,0x11AB ],
    [ 0xEA5B, 0x112D,0x119E,0x11AE ],
    [ 0xEA5C, 0x112D,0x119E,0x11AF ],
    [ 0xEA5D, 0x112D,0x119E,0x11B7 ],
    [ 0xEA5E, 0x112D,0x119E,0x11B8 ],
    [ 0xEA5F, 0x112D,0x119E,0x11BA ],
    [ 0xEA60, 0x112D,0x119E,0x11F0 ],
    [ 0xEA62, 0x112D,0x11A1,0x11A8 ],
    [ 0xEA63, 0x112D,0x11A1,0x11AB ],
    [ 0xEA64, 0x112D,0x11A1,0x11AF ],
    [ 0xEA65, 0x112D,0x11A1,0x11D9 ],
    [ 0xEA66, 0x112D,0x11A1,0x11B7 ],
    [ 0xEA67, 0x112D,0x11A1,0x11B8 ],
    [ 0xEA68, 0x112D,0x11A1,0x11BA ],
    [ 0xEA69, 0x112D,0x11A1,0x11BC ],
    [ 0xEA6B, 0x112E,0x1161,0x11AF ],
    [ 0xEA6D, 0x112E,0x1167,0x11A8 ],
    [ 0xEA72, 0x112F,0x1161,0x11A8 ],
    [ 0xEA73, 0x112F,0x1161,0x11AB ],
    [ 0xEA74, 0x112F,0x1161,0x11AE ],
    [ 0xEA75, 0x112F,0x1161,0x11AF ],
    [ 0xEA76, 0x112F,0x1161,0x11B6 ],
    [ 0xEA77, 0x112F,0x1161,0x11B7 ],
    [ 0xEA78, 0x112F,0x1161,0x11B8 ],
    [ 0xEA79, 0x112F,0x1161,0x11BA ],
    [ 0xEA7A, 0x112F,0x1161,0x11BC ],
    [ 0xEA7B, 0x112F,0x1161,0xD7F6 ],
    [ 0xEA7C, 0x112F,0x1161,0x11F0 ],
    [ 0xEA7D, 0x112F,0x1161,0xD7F6 ],
    [ 0xEA7E, 0x112F,0x1161,0x11C2 ],
    [ 0xEA80, 0x112F,0x1162,0x11A8 ],
    [ 0xEA81, 0x112F,0x1162,0x11AB ],
    [ 0xEA82, 0x112F,0x1162,0x11AF ],
    [ 0xEA83, 0x112F,0x1162,0x11B7 ],
    [ 0xEA84, 0x112F,0x1162,0x11B8 ],
    [ 0xEA85, 0x112F,0x1162,0x11BA ],
    [ 0xEA86, 0x112F,0x1162,0x11BC ],
    [ 0xEA87, 0x112F,0x1162,0x11F0 ],
    [ 0xEA89, 0x112F,0x1163,0x11A8 ],
    [ 0xEA8A, 0x112F,0x1163,0x11BC ],
    [ 0xEA8B, 0x112F,0x1163,0x11F0 ],
    [ 0xEA8E, 0x112F,0x1165,0x11A8 ],
    [ 0xEA8F, 0x112F,0x1165,0x11AB ],
    [ 0xEA90, 0x112F,0x1165,0x11AE ],
    [ 0xEA91, 0x112F,0x1165,0x11AF ],
    [ 0xEA92, 0x112F,0x1165,0x11B0 ],
    [ 0xEA93, 0x112F,0x1165,0x11B2 ],
    [ 0xEA94, 0x112F,0x1165,0x11B7 ],
    [ 0xEA95, 0x112F,0x1165,0x11B8 ],
    [ 0xEA96, 0x112F,0x1165,0x11BA ],
    [ 0xEA97, 0x112F,0x1165,0x11BC ],
    [ 0xEA98, 0x112F,0x1165,0x11F0 ],
    [ 0xEA99, 0x112F,0x1165,0x11BF ],
    [ 0xEA9B, 0x112F,0x117C,0x11BA ],
    [ 0xEA9C, 0x112F,0x117C,0x11BB ],
    [ 0xEA9D, 0x112F,0x117C,0x11BF ],
    [ 0xEA9F, 0x112F,0x1166,0x11A8 ],
    [ 0xEAA0, 0x112F,0x1166,0x11AB ],
    [ 0xEAA1, 0x112F,0x1166,0x11AF ],
    [ 0xEAA2, 0x112F,0x1166,0x11B7 ],
    [ 0xEAA3, 0x112F,0x1166,0x11BA ],
    [ 0xEAA4, 0x112F,0x1166,0x11BC ],
    [ 0xEAA6, 0x112F,0x1167,0x11A8 ],
    [ 0xEAA7, 0x112F,0x1167,0x11AB ],
    [ 0xEAA8, 0x112F,0x1167,0x11BA ],
    [ 0xEAA9, 0x112F,0x1167,0x11BC ],
    [ 0xEAAC, 0x112F,0x1169,0x11A8 ],
    [ 0xEAAD, 0x112F,0x1169,0x11AA ],
    [ 0xEAAE, 0x112F,0x1169,0x11AB ],
    [ 0xEAAF, 0x112F,0x1169,0x11AF ],
    [ 0xEAB0, 0x112F,0x1169,0x11B7 ],
    [ 0xEAB1, 0x112F,0x1169,0x11B8 ],
    [ 0xEAB2, 0x112F,0x1169,0x11BA ],
    [ 0xEAB3, 0x112F,0x1169,0x11BC ],
    [ 0xEAB4, 0x112F,0x1169,0x11F0 ],
    [ 0xEAB5, 0x112F,0x1169,0x11F1 ],
    [ 0xEAB9, 0x112F,0x116C,0x11AB ],
    [ 0xEABA, 0x112F,0x116C,0x11AF ],
    [ 0xEABC, 0x112F,0x116D,0x11A8 ],
    [ 0xEABD, 0x112F,0x116D,0x11BA ],
    [ 0xEABE, 0x112F,0x116D,0x11BC ],
    [ 0xEAC0, 0x112F,0x116E,0x11A8 ],
    [ 0xEAC1, 0x112F,0x116E,0x11AB ],
    [ 0xEAC2, 0x112F,0x116E,0x11AF ],
    [ 0xEAC3, 0x112F,0x116E,0x11B0 ],
    [ 0xEAC4, 0x112F,0x116E,0x11B2 ],
    [ 0xEAC5, 0x112F,0x116E,0x11B7 ],
    [ 0xEAC6, 0x112F,0x116E,0x11BA ],
    [ 0xEAC7, 0x112F,0x116E,0x11BC ],
    [ 0xEAC8, 0x112F,0x116E,0x11F0 ],
    [ 0xEACE, 0x112F,0x1171,0x11AB ],
    [ 0xEACF, 0x112F,0x1171,0x11AF ],
    [ 0xEAD0, 0x112F,0x1171,0x11B7 ],
    [ 0xEAD1, 0x112F,0x1171,0x11BA ],
    [ 0xEAD3, 0x112F,0x1172,0x11A8 ],
    [ 0xEAD4, 0x112F,0x1172,0x11AF ],
    [ 0xEAD5, 0x112F,0x1172,0x11B7 ],
    [ 0xEAD6, 0x112F,0x1172,0x11BC ],
    [ 0xEAD7, 0x112F,0x1172,0x11F0 ],
    [ 0xEADA, 0x112F,0x1173,0x11A8 ],
    [ 0xEADB, 0x112F,0x1173,0x11AB ],
    [ 0xEADC, 0x112F,0x1173,0x11AE ],
    [ 0xEADD, 0x112F,0x1173,0x11AF ],
    [ 0xEADE, 0x112F,0x1173,0x11B2 ],
    [ 0xEADF, 0x112F,0x1173,0x11B6 ],
    [ 0xEAE0, 0x112F,0x1173,0x11B7 ],
    [ 0xEAE1, 0x112F,0x1173,0x11B8 ],
    [ 0xEAE2, 0x112F,0x1173,0x11BA ],
    [ 0xEAE3, 0x112F,0x1173,0x11BC ],
    [ 0xEAE4, 0x112F,0x1173,0x11F0 ],
    [ 0xEAE5, 0x112F,0x1173,0x11F1 ],
    [ 0xEAE6, 0x112F,0x1173,0x11BD ],
    [ 0xEAE8, 0x112F,0x1174,0x11AB ],
    [ 0xEAE9, 0x112F,0x1174,0x11AF ],
    [ 0xEAEA, 0x112F,0x1174,0x11BA ],
    [ 0xEAEB, 0x112F,0x1174,0x11EB ],
    [ 0xEAEC, 0x112F,0x1174,0x11BE ],
    [ 0xEAEE, 0x112F,0x1175,0x11A8 ],
    [ 0xEAEF, 0x112F,0x1175,0x11AB ],
    [ 0xEAF0, 0x112F,0x1175,0x11AE ],
    [ 0xEAF1, 0x112F,0x1175,0x11AF ],
    [ 0xEAF2, 0x112F,0x1175,0x11B7 ],
    [ 0xEAF3, 0x112F,0x1175,0x11BA ],
    [ 0xEAF4, 0x112F,0x1175,0xD7F2 ],
    [ 0xEAF5, 0x112F,0x1175,0x11BC ],
    [ 0xEAF6, 0x112F,0x1175,0x11C2 ],
    [ 0xEAF9, 0x112F,0x119E,0x11A8 ],
    [ 0xEAFA, 0x112F,0x119E,0x11AB ],
    [ 0xEAFB, 0x112F,0x119E,0x11AE ],
    [ 0xEAFC, 0x112F,0x119E,0x11AF ],
    [ 0xEAFD, 0x112F,0x119E,0x11B6 ],
    [ 0xEAFE, 0x112F,0x119E,0x11B7 ],
    [ 0xEAFF, 0x112F,0x119E,0x11B8 ],
    [ 0xEB00, 0x112F,0x119E,0x11BA ],
    [ 0xEB01, 0x112F,0x119E,0x11BC ],
    [ 0xEB02, 0x112F,0x119E,0x11F0 ],
    [ 0xEB03, 0x112F,0x119E,0x11C2 ],
    [ 0xEB05, 0x112F,0x11A1,0x11A8 ],
    [ 0xEB06, 0x112F,0x11A1,0x11AB ],
    [ 0xEB07, 0x112F,0x11A1,0x11AF ],
    [ 0xEB08, 0x112F,0x11A1,0x11B7 ],
    [ 0xEB09, 0x112F,0x11A1,0x11B8 ],
    [ 0xEB0A, 0x112F,0x11A1,0x11BA ],
    [ 0xEB0B, 0x112F,0x11A1,0x11BC ],
    [ 0xEB0C, 0x112F,0x11A1,0x11F0 ],
    [ 0xEB0F, 0x1130,0x1162,0x11AB ],
    [ 0xEB12, 0x1130,0x1165,0x11BA ],
    [ 0xEB15, 0x1130,0x1167,0x11BA ],
    [ 0xEB18, 0x1130,0x1172,0x11AB ],
    [ 0xEB19, 0x1130,0x1172,0x11BC ],
    [ 0xEB1B, 0x1130,0x1173,0x11AB ],
    [ 0xEB1C, 0x1130,0x1173,0x11AF ],
    [ 0xEB1D, 0x1130,0x1173,0x11B7 ],
    [ 0xEB1F, 0x1130,0x1175,0x11AB ],
    [ 0xEB21, 0x1131,0x1161,0x11AF ],
    [ 0xEB24, 0x1131,0x1173,0x11AF ],
    [ 0xEB26, 0x1131,0x119E,0x11AF ],
    [ 0xEB28, 0x1132,0x1161,0x11A8 ],
    [ 0xEB29, 0x1132,0x1161,0x11AB ],
    [ 0xEB2A, 0x1132,0x1161,0x11AE ],
    [ 0xEB2B, 0x1132,0x1161,0x11AF ],
    [ 0xEB2C, 0x1132,0x1161,0x11B0 ],
    [ 0xEB2D, 0x1132,0x1161,0x11D0 ],
    [ 0xEB2E, 0x1132,0x1161,0x11B7 ],
    [ 0xEB2F, 0x1132,0x1161,0x11B8 ],
    [ 0xEB30, 0x1132,0x1161,0x11BA ],
    [ 0xEB31, 0x1132,0x1161,0x11BC ],
    [ 0xEB32, 0x1132,0x1161,0x11F0 ],
    [ 0xEB33, 0x1132,0x1161,0x11C2 ],
    [ 0xEB35, 0x1132,0x1162,0x11A8 ],
    [ 0xEB36, 0x1132,0x1162,0x11AB ],
    [ 0xEB37, 0x1132,0x1162,0x11AF ],
    [ 0xEB38, 0x1132,0x1162,0x11B7 ],
    [ 0xEB39, 0x1132,0x1162,0x11BA ],
    [ 0xEB3A, 0x1132,0x1162,0x11F0 ],
    [ 0xEB3C, 0x1132,0x1163,0x11B7 ],
    [ 0xEB3E, 0x1132,0x1165,0x11A8 ],
    [ 0xEB3F, 0x1132,0x1165,0x11AB ],
    [ 0xEB40, 0x1132,0x1165,0x11AE ],
    [ 0xEB41, 0x1132,0x1165,0x11AF ],
    [ 0xEB42, 0x1132,0x1165,0x11B7 ],
    [ 0xEB43, 0x1132,0x1165,0x11B8 ],
    [ 0xEB44, 0x1132,0x1165,0x11BA ],
    [ 0xEB45, 0x1132,0x1165,0x11BC ],
    [ 0xEB46, 0x1132,0x1165,0x11F0 ],
    [ 0xEB48, 0x1132,0x1166,0x11AB ],
    [ 0xEB49, 0x1132,0x1166,0x11AF ],
    [ 0xEB4A, 0x1132,0x1166,0x11B7 ],
    [ 0xEB4B, 0x1132,0x1166,0x11BA ],
    [ 0xEB4D, 0x1132,0x1167,0x11A8 ],
    [ 0xEB4E, 0x1132,0x1167,0x11AB ],
    [ 0xEB4F, 0x1132,0x1167,0x11AF ],
    [ 0xEB50, 0x1132,0x1167,0x11B7 ],
    [ 0xEB51, 0x1132,0x1167,0x11BA ],
    [ 0xEB52, 0x1132,0x1167,0xD7F2 ],
    [ 0xEB53, 0x1132,0x1167,0x11F0 ],
    [ 0xEB54, 0x1132,0x1167,0x11C2 ],
    [ 0xEB57, 0x1132,0x1169,0x11A8 ],
    [ 0xEB58, 0x1132,0x1169,0x11AB ],
    [ 0xEB59, 0x1132,0x1169,0x11AF ],
    [ 0xEB5A, 0x1132,0x1169,0x11B7 ],
    [ 0xEB5B, 0x1132,0x1169,0x11B8 ],
    [ 0xEB5C, 0x1132,0x1169,0x11BA ],
    [ 0xEB5D, 0x1132,0x1169,0x11E7 ],
    [ 0xEB5E, 0x1132,0x1169,0x11BC ],
    [ 0xEB5F, 0x1132,0x1169,0x11F0 ],
    [ 0xEB60, 0x1132,0x1169,0x11F1 ],
    [ 0xEB61, 0x1132,0x1169,0x11C2 ],
    [ 0xEB67, 0x1132,0x116E,0x11A8 ],
    [ 0xEB68, 0x1132,0x116E,0x11AB ],
    [ 0xEB69, 0x1132,0x116E,0x11AE ],
    [ 0xEB6A, 0x1132,0x116E,0x11AF ],
    [ 0xEB6B, 0x1132,0x116E,0x11B1 ],
    [ 0xEB6C, 0x1132,0x116E,0x11B7 ],
    [ 0xEB6D, 0x1132,0x116E,0x11BA ],
    [ 0xEB6E, 0x1132,0x116E,0x11BC ],
    [ 0xEB6F, 0x1132,0x116E,0x11BD ],
    [ 0xEB70, 0x1132,0x116E,0x11BE ],
    [ 0xEB79, 0x1132,0x1173,0x11A8 ],
    [ 0xEB7A, 0x1132,0x1173,0x11AB ],
    [ 0xEB7B, 0x1132,0x1173,0x11AF ],
    [ 0xEB7C, 0x1132,0x1173,0x11B3 ],
    [ 0xEB7D, 0x1132,0x1173,0x11B7 ],
    [ 0xEB7E, 0x1132,0x1173,0x11BA ],
    [ 0xEB7F, 0x1132,0x1173,0x11BC ],
    [ 0xEB81, 0x1132,0x1174,0x11AF ],
    [ 0xEB83, 0x1132,0x1175,0x11A8 ],
    [ 0xEB84, 0x1132,0x1175,0x11AB ],
    [ 0xEB85, 0x1132,0x1175,0x11AF ],
    [ 0xEB86, 0x1132,0x1175,0x11B7 ],
    [ 0xEB87, 0x1132,0x1175,0x11BA ],
    [ 0xEB88, 0x1132,0x1175,0x11E7 ],
    [ 0xEB89, 0x1132,0x1175,0xD7F2 ],
    [ 0xEB8A, 0x1132,0x1175,0x11F0 ],
    [ 0xEB8B, 0x1132,0x1175,0x11BE ],
    [ 0xEB8C, 0x1132,0x1175,0x11C2 ],
    [ 0xEB8F, 0x1132,0x119E,0x11A8 ],
    [ 0xEB90, 0x1132,0x119E,0x11AB ],
    [ 0xEB91, 0x1132,0x119E,0x11AF ],
    [ 0xEB92, 0x1132,0x119E,0x11D0 ],
    [ 0xEB93, 0x1132,0x119E,0x11B7 ],
    [ 0xEB94, 0x1132,0x119E,0x11B8 ],
    [ 0xEB95, 0x1132,0x119E,0x11BA ],
    [ 0xEB96, 0x1132,0x119E,0x11BC ],
    [ 0xEB98, 0x1132,0x11A1,0x11A8 ],
    [ 0xEB99, 0x1132,0x11A1,0x11AB ],
    [ 0xEB9A, 0x1132,0x11A1,0x11AF ],
    [ 0xEB9B, 0x1132,0x11A1,0x11B7 ],
    [ 0xEB9C, 0x1132,0x11A1,0x11BA ],
    [ 0xEB9D, 0x1132,0x11A1,0x11E7 ],
    [ 0xEB9E, 0x1132,0x11A1,0x11BC ],
    [ 0xEB9F, 0x1132,0x11A1,0x11F0 ],
    [ 0xEBA3, 0x1133,0x116E,0x11AF ],
    [ 0xEBA5, 0x1133,0x1173,0x11AF ],
    [ 0xEBA8, 0x110A,0x1161,0x11E2 ],
    [ 0xEBA9, 0x110A,0x1161,0xD7F2 ],
    [ 0xEBAA, 0x110A,0x1161,0x11F0 ],
    [ 0xEBAB, 0x110A,0x1163,0x11E2 ],
    [ 0xEBAC, 0x110A,0x1163,0x11F0 ],
    [ 0xEBAD, 0x110A,0x1165,0x11F0 ],
    [ 0xEBAE, 0x110A,0x1167,0x11D9 ],
    [ 0xEBAF, 0x110A,0x1167,0x11F0 ],
    [ 0xEBB0, 0x110A,0x1167,0x11F9 ],
    [ 0xEBB1, 0x110A,0x1169,0x11E2 ],
    [ 0xEBB2, 0x110A,0x1169,0x11F0 ],
    [ 0xEBB3, 0x110A,0x116D,0x11E2 ],
    [ 0xEBB4, 0x110A,0x116D,0x11F0 ],
    [ 0xEBB6, 0x110A,0x116E,0x11D9 ],
    [ 0xEBB7, 0x110A,0x116E,0x11E2 ],
    [ 0xEBB8, 0x110A,0x116E,0x11F9 ],
    [ 0xEBBA, 0x110A,0x1189,0x11AB ],
    [ 0xEBBB, 0x110A,0x1189,0x11BC ],
    [ 0xEBBD, 0x110A,0xD7B5,0x11AB ],
    [ 0xEBBF, 0x110A,0x1172,0x11D9 ],
    [ 0xEBC0, 0x110A,0x1172,0x11E2 ],
    [ 0xEBC1, 0x110A,0x1172,0x11F0 ],
    [ 0xEBC2, 0x110A,0x1172,0x11F9 ],
    [ 0xEBC4, 0x110A,0x1191,0x11AB ],
    [ 0xEBC5, 0x110A,0x1191,0x11BC ],
    [ 0xEBC7, 0x110A,0x1192,0x11BC ],
    [ 0xEBC9, 0x110A,0x1194,0x11BC ],
    [ 0xEBCA, 0x110A,0x1194,0x11F0 ],
    [ 0xEBCB, 0x110A,0x1173,0x11D9 ],
    [ 0xEBCC, 0x110A,0x1173,0x11EB ],
    [ 0xEBCD, 0x110A,0x1173,0x11F0 ],
    [ 0xEBCE, 0x110A,0x1175,0x11D9 ],
    [ 0xEBCF, 0x110A,0x1175,0x11E2 ],
    [ 0xEBD0, 0x110A,0x1175,0x11EB ],
    [ 0xEBD1, 0x110A,0x1175,0x11F0 ],
    [ 0xEBD2, 0x110A,0x1175,0x11F9 ],
    [ 0xEBD5, 0x110A,0x119E,0x11A8 ],
    [ 0xEBD6, 0x110A,0x119E,0x11AB ],
    [ 0xEBD7, 0x110A,0x119E,0x11AE ],
    [ 0xEBD8, 0x110A,0x119E,0x11AF ],
    [ 0xEBD9, 0x110A,0x119E,0x11B7 ],
    [ 0xEBDA, 0x110A,0x119E,0x11DD ],
    [ 0xEBDB, 0x110A,0x119E,0x11B8 ],
    [ 0xEBDC, 0x110A,0x119E,0x11E6 ],
    [ 0xEBDD, 0x110A,0x119E,0x11BA ],
    [ 0xEBDE, 0x110A,0x119E,0x11BC ],
    [ 0xEBDF, 0x110A,0x119E,0x11F0 ],
    [ 0xEBE0, 0x110A,0x119E,0x11C1 ],
    [ 0xEBE2, 0x110A,0x11A1,0x11AB ],
    [ 0xEBE3, 0x110A,0x11A1,0x11F0 ],
    [ 0xEBE7, 0x1135,0x1173,0x11AF ],
    [ 0xEBE8, 0x1135,0x1173,0x11B7 ],
    [ 0xEBEA, 0x1135,0x1175,0x11AF ],
    [ 0xEBEC, 0x1136,0x1161,0x11A8 ],
    [ 0xEBED, 0x1136,0x1161,0x11AB ],
    [ 0xEBEE, 0x1136,0x1161,0x11AF ],
    [ 0xEBEF, 0x1136,0x1161,0x11B2 ],
    [ 0xEBF0, 0x1136,0x1161,0x11B7 ],
    [ 0xEBF1, 0x1136,0x1161,0x11B8 ],
    [ 0xEBF2, 0x1136,0x1161,0x11BA ],
    [ 0xEBF3, 0x1136,0x1161,0x11E9 ],
    [ 0xEBF4, 0x1136,0x1161,0x11BC ],
    [ 0xEBF5, 0x1136,0x1161,0x11F0 ],
    [ 0xEBF7, 0x1136,0x1162,0x11A8 ],
    [ 0xEBF8, 0x1136,0x1162,0x11AB ],
    [ 0xEBF9, 0x1136,0x1162,0x11AF ],
    [ 0xEBFA, 0x1136,0x1162,0x11BA ],
    [ 0xEBFB, 0x1136,0x1162,0x11BC ],
    [ 0xEBFC, 0x1136,0x1162,0x11F0 ],
    [ 0xEBFE, 0x1136,0x1163,0x11A8 ],
    [ 0xEBFF, 0x1136,0x1163,0x11BC ],
    [ 0xEC00, 0x1136,0x1163,0x11F0 ],
    [ 0xEC02, 0x1136,0x1165,0x11A8 ],
    [ 0xEC03, 0x1136,0x1165,0x11AB ],
    [ 0xEC04, 0x1136,0x1165,0x11AF ],
    [ 0xEC05, 0x1136,0x1165,0x11B7 ],
    [ 0xEC06, 0x1136,0x1165,0x11B8 ],
    [ 0xEC07, 0x1136,0x1165,0x11BA ],
    [ 0xEC08, 0x1136,0x1165,0x11BC ],
    [ 0xEC0A, 0x1136,0x1166,0x11AB ],
    [ 0xEC0B, 0x1136,0x1166,0x11AF ],
    [ 0xEC0C, 0x1136,0x1166,0x11B8 ],
    [ 0xEC0D, 0x1136,0x1166,0x11BA ],
    [ 0xEC0F, 0x1136,0x1167,0x11A8 ],
    [ 0xEC10, 0x1136,0x1167,0x11AB ],
    [ 0xEC11, 0x1136,0x1167,0x11AF ],
    [ 0xEC12, 0x1136,0x1167,0x11BA ],
    [ 0xEC13, 0x1136,0x1167,0x11F0 ],
    [ 0xEC15, 0x1136,0x1168,0x11AB ],
    [ 0xEC17, 0x1136,0x1169,0x11A8 ],
    [ 0xEC18, 0x1136,0x1169,0x11AB ],
    [ 0xEC19, 0x1136,0x1169,0x11AF ],
    [ 0xEC1A, 0x1136,0x1169,0x11B7 ],
    [ 0xEC1B, 0x1136,0x1169,0x11B8 ],
    [ 0xEC1C, 0x1136,0x1169,0x11BA ],
    [ 0xEC1D, 0x1136,0x1169,0x11BC ],
    [ 0xEC1E, 0x1136,0x1169,0x11BD ],
    [ 0xEC1F, 0x1136,0x1169,0x11BE ],
    [ 0xEC21, 0x1136,0x116A,0x11A8 ],
    [ 0xEC23, 0x1136,0x116C,0x11A8 ],
    [ 0xEC24, 0x1136,0x116C,0x11AB ],
    [ 0xEC25, 0x1136,0x116C,0x11AF ],
    [ 0xEC26, 0x1136,0x116C,0x11B7 ],
    [ 0xEC28, 0x1136,0x116D,0x11A8 ],
    [ 0xEC29, 0x1136,0x116D,0x11AE ],
    [ 0xEC2A, 0x1136,0x116D,0x11B7 ],
    [ 0xEC2B, 0x1136,0x116D,0x11BA ],
    [ 0xEC2C, 0x1136,0x116D,0x11BC ],
    [ 0xEC2E, 0x1136,0x116E,0x11A8 ],
    [ 0xEC2F, 0x1136,0x116E,0x11AB ],
    [ 0xEC30, 0x1136,0x116E,0x11AF ],
    [ 0xEC31, 0x1136,0x116E,0x11B7 ],
    [ 0xEC32, 0x1136,0x116E,0x11B8 ],
    [ 0xEC33, 0x1136,0x116E,0x11BC ],
    [ 0xEC37, 0x1136,0x1171,0x11A8 ],
    [ 0xEC39, 0x1136,0x1172,0x11A8 ],
    [ 0xEC3A, 0x1136,0x1172,0x11AF ],
    [ 0xEC3B, 0x1136,0x1172,0x11B7 ],
    [ 0xEC3D, 0x1136,0x1173,0x11A8 ],
    [ 0xEC3E, 0x1136,0x1173,0x11AB ],
    [ 0xEC3F, 0x1136,0x1173,0x11AF ],
    [ 0xEC40, 0x1136,0x1173,0x11B7 ],
    [ 0xEC41, 0x1136,0x1173,0x11BA ],
    [ 0xEC42, 0x1136,0x1173,0x11BC ],
    [ 0xEC44, 0x1136,0x1174,0x11AF ],
    [ 0xEC46, 0x1136,0x1175,0x11A8 ],
    [ 0xEC47, 0x1136,0x1175,0x11AB ],
    [ 0xEC48, 0x1136,0x1175,0x11AF ],
    [ 0xEC49, 0x1136,0x1175,0x11D9 ],
    [ 0xEC4A, 0x1136,0x1175,0x11B7 ],
    [ 0xEC4B, 0x1136,0x1175,0x11B8 ],
    [ 0xEC4C, 0x1136,0x1175,0x11BA ],
    [ 0xEC4D, 0x1136,0x1175,0xD7F2 ],
    [ 0xEC4E, 0x1136,0x1175,0x11BC ],
    [ 0xEC4F, 0x1136,0x1175,0x11F0 ],
    [ 0xEC50, 0x1136,0x1175,0x11C2 ],
    [ 0xEC52, 0x1136,0x119E,0x11A8 ],
    [ 0xEC53, 0x1136,0x119E,0x11AB ],
    [ 0xEC54, 0x1136,0x119E,0x11AF ],
    [ 0xEC55, 0x1136,0x119E,0x11B7 ],
    [ 0xEC56, 0x1136,0x119E,0x11B8 ],
    [ 0xEC57, 0x1136,0x119E,0x11BA ],
    [ 0xEC58, 0x1136,0x119E,0x11BC ],
    [ 0xEC5A, 0x1136,0x11A1,0x11A8 ],
    [ 0xEC5B, 0x1136,0x11A1,0x11AF ],
    [ 0xEC5C, 0x1136,0x11A1,0x11BC ],
    [ 0xEC5D, 0x1136,0x11A1,0x11F0 ],
    [ 0xEC61, 0x1137,0x119E,0x11AF ],
    [ 0xEC62, 0x1137,0x119E,0x11B8 ],
    [ 0xEC64, 0x1138,0x1161,0x11BC ],
    [ 0xEC66, 0x1138,0x1165,0x11AB ],
    [ 0xEC67, 0x1138,0x1165,0x11AF ],
    [ 0xEC6A, 0x1138,0x1169,0x11AB ],
    [ 0xEC6B, 0x1138,0x1169,0x11BA ],
    [ 0xEC6C, 0x1138,0x1169,0x11BC ],
    [ 0xEC6D, 0x1138,0x1169,0x11F0 ],
    [ 0xEC71, 0x1138,0x116C,0x11BC ],
    [ 0xEC72, 0x1138,0x116C,0x11F0 ],
    [ 0xEC76, 0x1138,0x1173,0x11AB ],
    [ 0xEC7B, 0x1139,0x1161,0x11AF ],
    [ 0xEC7D, 0x1139,0x1165,0x11AF ],
    [ 0xEC80, 0x1139,0x1169,0x11B8 ],
    [ 0xEC82, 0x1139,0x116E,0x11A8 ],
    [ 0xEC85, 0x1139,0x1173,0x11AF ],
    [ 0xEC86, 0x1139,0x1173,0x11B7 ],
    [ 0xEC87, 0x1139,0x1173,0x11BA ],
    [ 0xEC89, 0x1139,0x1175,0x11BF ],
    [ 0xEC8B, 0x1139,0x119E,0x11AF ],
    [ 0xEC8C, 0x1139,0x119E,0x11B7 ],
    [ 0xEC8E, 0x113A,0x1167,0x11AB ],
    [ 0xEC90, 0x113A,0x116E,0x11AB ],
    [ 0xEC93, 0x113A,0x119E,0x11AF ],
    [ 0xEC95, 0x113B,0x1165,0x11AF ],
    [ 0xEC98, 0x113B,0x1173,0x11AF ],
    [ 0xEC9A, 0x113C,0x1161,0x11AB ],
    [ 0xEC9B, 0x113C,0x1161,0x11F0 ],
    [ 0xEC9E, 0x113C,0x1163,0x11B7 ],
    [ 0xEC9F, 0x113C,0x1163,0x11E2 ],
    [ 0xECA0, 0x113C,0x1163,0x11F0 ],
    [ 0xECA2, 0x113C,0x1165,0x11AB ],
    [ 0xECA4, 0x113C,0x1167,0x11AB ],
    [ 0xECA7, 0x113C,0x116E,0x11AB ],
    [ 0xECA8, 0x113C,0x116E,0x11BC ],
    [ 0xECAB, 0x113C,0x1172,0x11AB ],
    [ 0xECAD, 0x113C,0x1191,0x11AB ],
    [ 0xECAF, 0x113C,0x1173,0x11AB ],
    [ 0xECB0, 0x113C,0x1173,0x11EB ],
    [ 0xECB2, 0x113C,0x1175,0x11BC ],
    [ 0xECB4, 0x113D,0x1163,0x11BC ],
    [ 0xECB5, 0x113D,0x1163,0x11F0 ],
    [ 0xECB7, 0x113D,0x1167,0x11AB ],
    [ 0xECBB, 0x113D,0x1172,0x11AB ],
    [ 0xECBC, 0x113D,0x1172,0x11F0 ],
    [ 0xECBE, 0x113D,0x1191,0x11AB ],
    [ 0xECC1, 0x113D,0x1173,0x11EB ],
    [ 0xECC3, 0x113D,0x1175,0x11AB ],
    [ 0xECC4, 0x113D,0x1175,0x11EB ],
    [ 0xECC5, 0x113D,0x1175,0x11F9 ],
    [ 0xECC7, 0x113E,0x1161,0x11AB ],
    [ 0xECC8, 0x113E,0x1161,0x11AE ],
    [ 0xECCB, 0x113E,0x1163,0x11B7 ],
    [ 0xECCC, 0x113E,0x1163,0x11E2 ],
    [ 0xECCE, 0x113E,0x1167,0x11AB ],
    [ 0xECD3, 0x113E,0x116E,0x11A8 ],
    [ 0xECD4, 0x113E,0x116E,0x11BC ],
    [ 0xECD7, 0x113E,0x1172,0x11AB ],
    [ 0xECD9, 0x113E,0x1191,0x11AB ],
    [ 0xECDB, 0x113E,0x1173,0x11EB ],
    [ 0xECDC, 0x113E,0x1173,0x11BC ],
    [ 0xECDE, 0x113E,0x1175,0x11AB ],
    [ 0xECDF, 0x113E,0x1175,0x11B7 ],
    [ 0xECE0, 0x113E,0x1175,0x11E2 ],
    [ 0xECE2, 0x113F,0x1163,0x11F0 ],
    [ 0xECE4, 0x113F,0x1167,0x11AB ],
    [ 0xECE8, 0x113F,0x1172,0x11A8 ],
    [ 0xECE9, 0x113F,0x1172,0x11AB ],
    [ 0xECEA, 0x113F,0x1172,0x11F0 ],
    [ 0xECEC, 0x113F,0x1173,0x11EB ],
    [ 0xECEE, 0x113F,0x1175,0x11AB ],
    [ 0xECEF, 0x113F,0x1175,0x11F9 ],
    [ 0xECF1, 0x1140,0x1161,0x11A8 ],
    [ 0xECF2, 0x1140,0x1161,0x11AB ],
    [ 0xECF3, 0x1140,0x1161,0x11AE ],
    [ 0xECF4, 0x1140,0x1161,0x11AF ],
    [ 0xECF5, 0x1140,0x1161,0x11B7 ],
    [ 0xECF6, 0x1140,0x1161,0x11DD ],
    [ 0xECF7, 0x1140,0x1161,0x11B8 ],
    [ 0xECF8, 0x1140,0x1161,0x11BA ],
    [ 0xECF9, 0x1140,0x1161,0x11BC ],
    [ 0xECFA, 0x1140,0x1161,0x11F0 ],
    [ 0xECFC, 0x1140,0x1162,0x11AB ],
    [ 0xECFD, 0x1140,0x1162,0x11BA ],
    [ 0xECFE, 0x1140,0x1162,0x11EB ],
    [ 0xECFF, 0x1140,0x1162,0x11BC ],
    [ 0xED00, 0x1140,0x1162,0x11F0 ],
    [ 0xED02, 0x1140,0x1163,0x11A8 ],
    [ 0xED03, 0x1140,0x1163,0x11AB ],
    [ 0xED04, 0x1140,0x1163,0x11B7 ],
    [ 0xED05, 0x1140,0x1163,0x11E2 ],
    [ 0xED06, 0x1140,0x1163,0x11B8 ],
    [ 0xED07, 0x1140,0x1163,0x11E6 ],
    [ 0xED08, 0x1140,0x1163,0x11BC ],
    [ 0xED09, 0x1140,0x1163,0x11F0 ],
    [ 0xED0A, 0x1140,0x1163,0x11F9 ],
    [ 0xED0D, 0x1140,0x1165,0x11A8 ],
    [ 0xED0E, 0x1140,0x1165,0x11AB ],
    [ 0xED0F, 0x1140,0x1165,0x11AF ],
    [ 0xED10, 0x1140,0x1165,0x11B7 ],
    [ 0xED11, 0x1140,0x1165,0x11DD ],
    [ 0xED12, 0x1140,0x1165,0x11B8 ],
    [ 0xED13, 0x1140,0x1165,0x11BA ],
    [ 0xED14, 0x1140,0x1165,0x11EB ],
    [ 0xED15, 0x1140,0x1165,0x11BC ],
    [ 0xED16, 0x1140,0x1165,0x11F0 ],
    [ 0xED17, 0x1140,0x1165,0x11C2 ],
    [ 0xED1A, 0x1140,0x1166,0x11AB ],
    [ 0xED1B, 0x1140,0x1166,0x11BA ],
    [ 0xED1D, 0x1140,0x1167,0x11A8 ],
    [ 0xED1E, 0x1140,0x1167,0x11AB ],
    [ 0xED1F, 0x1140,0x1167,0x11AF ],
    [ 0xED20, 0x1140,0x1167,0x11B3 ],
    [ 0xED21, 0x1140,0x1167,0x11D9 ],
    [ 0xED22, 0x1140,0x1167,0x11B7 ],
    [ 0xED23, 0x1140,0x1167,0x11B8 ],
    [ 0xED24, 0x1140,0x1167,0x11BC ],
    [ 0xED25, 0x1140,0x1167,0x11F0 ],
    [ 0xED26, 0x1140,0x1167,0x11F9 ],
    [ 0xED28, 0x1140,0x1168,0x11AB ],
    [ 0xED29, 0x1140,0x1168,0x11AE ],
    [ 0xED2A, 0x1140,0x1168,0x11B7 ],
    [ 0xED2B, 0x1140,0x1168,0x11BA ],
    [ 0xED2C, 0x1140,0x1168,0x11BC ],
    [ 0xED2E, 0x1140,0x1169,0x11AB ],
    [ 0xED2F, 0x1140,0x1169,0x11AF ],
    [ 0xED30, 0x1140,0x1169,0x11D9 ],
    [ 0xED31, 0x1140,0x1169,0x11B7 ],
    [ 0xED32, 0x1140,0x1169,0x11B8 ],
    [ 0xED33, 0x1140,0x1169,0x11EB ],
    [ 0xED34, 0x1140,0x1169,0x11BC ],
    [ 0xED35, 0x1140,0x1169,0x11F0 ],
    [ 0xED3A, 0x1140,0x116D,0x11A8 ],
    [ 0xED3B, 0x1140,0x116D,0x11B7 ],
    [ 0xED3C, 0x1140,0x116D,0x11E2 ],
    [ 0xED3D, 0x1140,0x116D,0x11BA ],
    [ 0xED3E, 0x1140,0x116D,0x11BC ],
    [ 0xED3F, 0x1140,0x116D,0x11F0 ],
    [ 0xED40, 0x1140,0x116D,0x11C2 ],
    [ 0xED42, 0x1140,0x116E,0x11A8 ],
    [ 0xED43, 0x1140,0x116E,0x11AB ],
    [ 0xED44, 0x1140,0x116E,0x11AF ],
    [ 0xED45, 0x1140,0x116E,0x11D9 ],
    [ 0xED46, 0x1140,0x116E,0x11B7 ],
    [ 0xED47, 0x1140,0x116E,0x11DD ],
    [ 0xED48, 0x1140,0x116E,0x11B8 ],
    [ 0xED49, 0x1140,0x116E,0x11BA ],
    [ 0xED4A, 0x1140,0x116E,0x11BC ],
    [ 0xED4B, 0x1140,0x116E,0x11F0 ],
    [ 0xED4C, 0x1140,0x116E,0x11BE ],
    [ 0xED4D, 0x1140,0x116E,0x11C2 ],
    [ 0xED4F, 0x1140,0x1189,0x11AB ],
    [ 0xED55, 0x1140,0x1171,0x11AB ],
    [ 0xED57, 0x1140,0x1172,0x11A8 ],
    [ 0xED58, 0x1140,0x1172,0x11AB ],
    [ 0xED59, 0x1140,0x1172,0x11B7 ],
    [ 0xED5A, 0x1140,0x1172,0x11E2 ],
    [ 0xED5B, 0x1140,0x1172,0x11BA ],
    [ 0xED5C, 0x1140,0x1172,0x11BC ],
    [ 0xED5D, 0x1140,0x1172,0x11F0 ],
    [ 0xED5E, 0x1140,0x1172,0x11BD ],
    [ 0xED5F, 0x1140,0x1172,0x11BE ],
    [ 0xED60, 0x1140,0x1172,0x11F9 ],
    [ 0xED62, 0x1140,0x118F,0x11AB ],
    [ 0xED64, 0x1140,0x1191,0x11AB ],
    [ 0xED65, 0x1140,0x1191,0x11AF ],
    [ 0xED66, 0x1140,0x1191,0x11D9 ],
    [ 0xED68, 0x1140,0x1192,0x11BC ],
    [ 0xED6A, 0x1140,0x1194,0x11BC ],
    [ 0xED6C, 0x1140,0x1173,0x11A8 ],
    [ 0xED6D, 0x1140,0x1173,0x11AB ],
    [ 0xED6E, 0x1140,0x1173,0x11AF ],
    [ 0xED6F, 0x1140,0x1173,0x11B1 ],
    [ 0xED70, 0x1140,0x1173,0x11B3 ],
    [ 0xED71, 0x1140,0x1173,0x11B6 ],
    [ 0xED72, 0x1140,0x1173,0x11D9 ],
    [ 0xED73, 0x1140,0x1173,0x11B7 ],
    [ 0xED74, 0x1140,0x1173,0x11DD ],
    [ 0xED75, 0x1140,0x1173,0x11E0 ],
    [ 0xED76, 0x1140,0x1173,0x11B8 ],
    [ 0xED77, 0x1140,0x1173,0x11BA ],
    [ 0xED78, 0x1140,0x1173,0xD7F0 ],
    [ 0xED79, 0x1140,0x1173,0x11EB ],
    [ 0xED7A, 0x1140,0x1173,0x11F0 ],
    [ 0xED7B, 0x1140,0x1173,0x11BE ],
    [ 0xED7D, 0x1140,0x1174,0x11AB ],
    [ 0xED7E, 0x1140,0x1174,0x11BA ],
    [ 0xED80, 0x1140,0x1175,0x11A8 ],
    [ 0xED81, 0x1140,0x1175,0x11AB ],
    [ 0xED82, 0x1140,0x1175,0x11AE ],
    [ 0xED83, 0x1140,0x1175,0x11AF ],
    [ 0xED84, 0x1140,0x1175,0x11B3 ],
    [ 0xED85, 0x1140,0x1175,0x11D9 ],
    [ 0xED86, 0x1140,0x1175,0x11B7 ],
    [ 0xED87, 0x1140,0x1175,0x11B8 ],
    [ 0xED88, 0x1140,0x1175,0x11E6 ],
    [ 0xED89, 0x1140,0x1175,0x11BA ],
    [ 0xED8A, 0x1140,0x1175,0x11EB ],
    [ 0xED8B, 0x1140,0x1175,0x11BC ],
    [ 0xED8C, 0x1140,0x1175,0x11F0 ],
    [ 0xED8D, 0x1140,0x1175,0x11F9 ],
    [ 0xED90, 0x1140,0x119E,0x11A8 ],
    [ 0xED91, 0x1140,0x119E,0x11AB ],
    [ 0xED92, 0x1140,0x119E,0x11AE ],
    [ 0xED93, 0x1140,0x119E,0x11AF ],
    [ 0xED94, 0x1140,0x119E,0x11B0 ],
    [ 0xED95, 0x1140,0x119E,0x11B3 ],
    [ 0xED96, 0x1140,0x119E,0x11B6 ],
    [ 0xED97, 0x1140,0x119E,0x11D9 ],
    [ 0xED98, 0x1140,0x119E,0x11B7 ],
    [ 0xED99, 0x1140,0x119E,0x11DD ],
    [ 0xED9A, 0x1140,0x119E,0x11B8 ],
    [ 0xED9B, 0x1140,0x119E,0x11E6 ],
    [ 0xED9C, 0x1140,0x119E,0x11BA ],
    [ 0xED9D, 0x1140,0x119E,0x11EB ],
    [ 0xED9E, 0x1140,0x119E,0x11BC ],
    [ 0xED9F, 0x1140,0x119E,0x11BD ],
    [ 0xEDA1, 0x1140,0x11A1,0x11AB ],
    [ 0xEDA2, 0x1140,0x11A1,0x11AF ],
    [ 0xEDA3, 0x1140,0x11A1,0x11B8 ],
    [ 0xEDA4, 0x1140,0x11A1,0x11BA ],
    [ 0xEDA5, 0x110B,0x1161,0x11C7 ],
    [ 0xEDA6, 0x110B,0x1161,0x11C9 ],
    [ 0xEDA7, 0x110B,0x1161,0x11CD ],
    [ 0xEDA8, 0x110B,0x1161,0x11D0 ],
    [ 0xEDA9, 0x110B,0x1161,0x11D3 ],
    [ 0xEDAA, 0x110B,0x1161,0xD7DA ],
    [ 0xEDAB, 0x110B,0x1161,0x11D4 ],
    [ 0xEDAC, 0x110B,0x1161,0x11D6 ],
    [ 0xEDAD, 0x110B,0x1161,0x11D9 ],
    [ 0xEDAE, 0x110B,0x1161,0x11DD ],
    [ 0xEDAF, 0x110B,0x1161,0x11E1 ],
    [ 0xEDB0, 0x110B,0x1161,0x11E2 ],
    [ 0xEDB1, 0x110B,0x1161,0x11E3 ],
    [ 0xEDB2, 0x110B,0x1161,0xD7E4 ],
    [ 0xEDB3, 0x110B,0x1161,0x11E4 ],
    [ 0xEDB4, 0x110B,0x1161,0x11E5 ],
    [ 0xEDB5, 0x110B,0x1161,0x11E6 ],
    [ 0xEDB6, 0x110B,0x1161,0x11EA ],
    [ 0xEDB7, 0x110B,0x1161,0x11EB ],
    [ 0xEDB8, 0x110B,0x1161,0x11F0 ],
    [ 0xEDB9, 0x110B,0x1161,0x11F5 ],
    [ 0xEDBB, 0x110B,0x1176,0x11AF ],
    [ 0xEDBC, 0x110B,0x1162,0x11EB ],
    [ 0xEDBD, 0x110B,0x1162,0x11F0 ],
    [ 0xEDBE, 0x110B,0x1163,0x11D9 ],
    [ 0xEDBF, 0x110B,0x1163,0x11DC ],
    [ 0xEDC0, 0x110B,0x1163,0x11DD ],
    [ 0xEDC1, 0x110B,0x1163,0x11E2 ],
    [ 0xEDC2, 0x110B,0x1163,0x11E6 ],
    [ 0xEDC3, 0x110B,0x1163,0x11EA ],
    [ 0xEDC4, 0x110B,0x1163,0x11F1 ],
    [ 0xEDC5, 0x110B,0x1163,0x11F0 ],
    [ 0xEDC6, 0x110B,0x1163,0x11F5 ],
    [ 0xEDC7, 0x110B,0x1163,0x11F9 ],
    [ 0xEDCA, 0x110B,0x1165,0x11C3 ],
    [ 0xEDCB, 0x110B,0x1165,0x11C6 ],
    [ 0xEDCC, 0x110B,0x1165,0x11C7 ],
    [ 0xEDCD, 0x110B,0x1165,0x11CD ],
    [ 0xEDCE, 0x110B,0x1165,0x11D0 ],
    [ 0xEDCF, 0x110B,0x1165,0x11D9 ],
    [ 0xEDD0, 0x110B,0x1165,0x11DD ],
    [ 0xEDD1, 0x110B,0x1165,0x11DE ],
    [ 0xEDD2, 0x110B,0x1165,0x11E4 ],
    [ 0xEDD3, 0x110B,0x1165,0x11E6 ],
    [ 0xEDD4, 0x110B,0x1165,0xD7EF ],
    [ 0xEDD5, 0x110B,0x1165,0x11EB ],
    [ 0xEDD6, 0x110B,0x1165,0x11F0 ],
    [ 0xEDD7, 0x110B,0x1165,0xD7FA ],
    [ 0xEDDA, 0x110B,0x117C,0x11AB ],
    [ 0xEDDB, 0x110B,0x117C,0x11AF ],
    [ 0xEDDC, 0x110B,0x117C,0x11B7 ],
    [ 0xEDDD, 0x110B,0x117C,0x11B8 ],
    [ 0xEDDE, 0x110B,0x117C,0x11B9 ],
    [ 0xEDDF, 0x110B,0x117C,0x11BA ],
    [ 0xEDE0, 0x110B,0x117C,0x11BC ],
    [ 0xEDE1, 0x110B,0x1166,0x11C6 ],
    [ 0xEDE2, 0x110B,0x1166,0x11EB ],
    [ 0xEDE3, 0x110B,0x1166,0x11F0 ],
    [ 0xEDE4, 0x110B,0x1167,0x11C4 ],
    [ 0xEDE5, 0x110B,0x1167,0x11FD ],
    [ 0xEDE6, 0x110B,0x1167,0x11C6 ],
    [ 0xEDE7, 0x110B,0x1167,0x11C7 ],
    [ 0xEDE8, 0x110B,0x1167,0x11C9 ],
    [ 0xEDE9, 0x110B,0x1167,0xD7D4 ],
    [ 0xEDEA, 0x110B,0x1167,0x11D0 ],
    [ 0xEDEB, 0x110B,0x1167,0x11D5 ],
    [ 0xEDEC, 0x110B,0x1167,0x11D9 ],
    [ 0xEDED, 0x110B,0x1167,0x11E2 ],
    [ 0xEDEE, 0x110B,0x1167,0x11E3 ],
    [ 0xEDEF, 0x110B,0x1167,0xD7E6 ],
    [ 0xEDF0, 0x110B,0x1167,0x11E6 ],
    [ 0xEDF1, 0x110B,0x1167,0x11E7 ],
    [ 0xEDF2, 0x110B,0x1167,0x11EA ],
    [ 0xEDF3, 0x110B,0x1167,0xD7F1 ],
    [ 0xEDF4, 0x110B,0x1167,0xD7F2 ],
    [ 0xEDF5, 0x110B,0x1167,0x11EB ],
    [ 0xEDF6, 0x110B,0x1167,0x11EC ],
    [ 0xEDF7, 0x110B,0x1167,0x11ED ],
    [ 0xEDF8, 0x110B,0x1167,0x11F0 ],
    [ 0xEDF9, 0x110B,0x1167,0x11F9 ],
    [ 0xEDFC, 0x110B,0x1169,0x11C7 ],
    [ 0xEDFD, 0x110B,0x1169,0xD7D5 ],
    [ 0xEDFE, 0x110B,0x1169,0xD7D6 ],
    [ 0xEDFF, 0x110B,0x1169,0x11CD ],
    [ 0xEE00, 0x110B,0x1169,0x11D0 ],
    [ 0xEE01, 0x110B,0x1169,0x11D6 ],
    [ 0xEE02, 0x110B,0x1169,0x11D9 ],
    [ 0xEE03, 0x110B,0x1169,0x11DD ],
    [ 0xEE04, 0x110B,0x1169,0x11E2 ],
    [ 0xEE05, 0x110B,0x1169,0xD7F0 ],
    [ 0xEE06, 0x110B,0x1169,0x11EB ],
    [ 0xEE07, 0x110B,0x1169,0x11F0 ],
    [ 0xEE08, 0x110B,0x1169,0x11F1 ],
    [ 0xEE09, 0x110B,0x116A,0xD7CF ],
    [ 0xEE0A, 0x110B,0x116A,0xD7D4 ],
    [ 0xEE0B, 0x110B,0x116A,0x11DD ],
    [ 0xEE0C, 0x110B,0x116A,0x11EA ],
    [ 0xEE0D, 0x110B,0x116A,0x11F0 ],
    [ 0xEE0E, 0x110B,0x116A,0x11F9 ],
    [ 0xEE10, 0x110B,0x117F,0x11AB ],
    [ 0xEE11, 0x110B,0x117F,0x11AF ],
    [ 0xEE13, 0x110B,0x1182,0x11AF ],
    [ 0xEE14, 0x110B,0x116C,0x11D0 ],
    [ 0xEE15, 0x110B,0x116C,0x11D9 ],
    [ 0xEE16, 0x110B,0x116C,0x11F0 ],
    [ 0xEE17, 0x110B,0x116D,0x11D9 ],
    [ 0xEE18, 0x110B,0x116D,0x11E2 ],
    [ 0xEE19, 0x110B,0x116D,0x11F0 ],
    [ 0xEE1C, 0x110B,0xD7B3,0x11AB ],
    [ 0xEE1E, 0x110B,0xD7B4,0x11AB ],
    [ 0xEE21, 0x110B,0x1188,0x11AF ],
    [ 0xEE22, 0x110B,0x116E,0x11C9 ],
    [ 0xEE23, 0x110B,0x116E,0xD7D3 ],
    [ 0xEE24, 0x110B,0x116E,0x11CD ],
    [ 0xEE25, 0x110B,0x116E,0x11D0 ],
    [ 0xEE26, 0x110B,0x116E,0x11D9 ],
    [ 0xEE27, 0x110B,0x116E,0x11DD ],
    [ 0xEE28, 0x110B,0x116E,0x11E0 ],
    [ 0xEE29, 0x110B,0x116E,0x11E1 ],
    [ 0xEE2A, 0x110B,0x116E,0x11E2 ],
    [ 0xEE2B, 0x110B,0x116E,0x11E4 ],
    [ 0xEE2C, 0x110B,0x116E,0x11EA ],
    [ 0xEE2D, 0x110B,0x116E,0xD7EB ],
    [ 0xEE2E, 0x110B,0x116E,0xD7F0 ],
    [ 0xEE2F, 0x110B,0x116E,0x11EB ],
    [ 0xEE30, 0x110B,0x116E,0xD7F3 ],
    [ 0xEE31, 0x110B,0x116E,0xD7F4 ],
    [ 0xEE32, 0x110B,0x116E,0x11F0 ],
    [ 0xEE33, 0x110B,0x116E,0x11F9 ],
    [ 0xEE35, 0x110B,0x1189,0x11AB ],
    [ 0xEE36, 0x110B,0x1189,0x11BC ],
    [ 0xEE37, 0x110B,0x1189,0x11F0 ],
    [ 0xEE39, 0x110B,0x116F,0x11D8 ],
    [ 0xEE3A, 0x110B,0x116F,0x11D9 ],
    [ 0xEE3B, 0x110B,0x116F,0x11EB ],
    [ 0xEE3C, 0x110B,0x116F,0x11F0 ],
    [ 0xEE3D, 0x110B,0x116F,0x11F1 ],
    [ 0xEE3F, 0x110B,0x118B,0x11AB ],
    [ 0xEE42, 0x110B,0x1171,0x11D9 ],
    [ 0xEE43, 0x110B,0x1171,0x11F0 ],
    [ 0xEE44, 0x110B,0x1172,0x11D9 ],
    [ 0xEE45, 0x110B,0x1172,0x11E2 ],
    [ 0xEE46, 0x110B,0x1172,0x11F0 ],
    [ 0xEE47, 0x110B,0x1172,0x11F9 ],
    [ 0xEE49, 0x110B,0x118E,0x11AB ],
    [ 0xEE4A, 0x110B,0x118E,0x11B7 ],
    [ 0xEE4D, 0x110B,0x1190,0x11AB ],
    [ 0xEE4F, 0x110B,0x1191,0x11A8 ],
    [ 0xEE50, 0x110B,0x1191,0x11AB ],
    [ 0xEE51, 0x110B,0x1191,0x11AF ],
    [ 0xEE52, 0x110B,0x1191,0x11D9 ],
    [ 0xEE53, 0x110B,0x1191,0x11F0 ],
    [ 0xEE54, 0x110B,0x1191,0x11F9 ],
    [ 0xEE56, 0x110B,0x1192,0x11BC ],
    [ 0xEE57, 0x110B,0x1192,0x11F0 ],
    [ 0xEE5A, 0x110B,0x1194,0x11AB ],
    [ 0xEE5B, 0x110B,0x1194,0x11BC ],
    [ 0xEE5C, 0x110B,0x1194,0x11F0 ],
    [ 0xEE5D, 0x110B,0x1173,0x11FF ],
    [ 0xEE5E, 0x110B,0x1173,0x11C7 ],
    [ 0xEE5F, 0x110B,0x1173,0xD7CD ],
    [ 0xEE60, 0x110B,0x1173,0x11D0 ],
    [ 0xEE61, 0x110B,0x1173,0xD7DA ],
    [ 0xEE62, 0x110B,0x1173,0x11D4 ],
    [ 0xEE63, 0x110B,0x1173,0x11D9 ],
    [ 0xEE64, 0x110B,0x1173,0x11DD ],
    [ 0xEE65, 0x110B,0x1173,0x11E2 ],
    [ 0xEE66, 0x110B,0x1173,0x11E3 ],
    [ 0xEE67, 0x110B,0x1173,0xD7E6 ],
    [ 0xEE68, 0x110B,0x1173,0x11E4 ],
    [ 0xEE69, 0x110B,0x1173,0xD7F0 ],
    [ 0xEE6A, 0x110B,0x1173,0x11EB ],
    [ 0xEE6B, 0x110B,0x1173,0x11F0 ],
    [ 0xEE6C, 0x110B,0x1173,0xD7F9 ],
    [ 0xEE6D, 0x110B,0x1173,0x11F9 ],
    [ 0xEE70, 0x110B,0x1196,0x11AB ],
    [ 0xEE71, 0x110B,0x1196,0x11BC ],
    [ 0xEE72, 0x110B,0x1196,0x11F0 ],
    [ 0xEE73, 0x110B,0x1196,0x11F9 ],
    [ 0xEE74, 0x110B,0x1174,0x11D9 ],
    [ 0xEE75, 0x110B,0x1174,0x11F9 ],
    [ 0xEE76, 0x110B,0x1175,0x11C7 ],
    [ 0xEE77, 0x110B,0x1175,0x11CD ],
    [ 0xEE78, 0x110B,0x1175,0x11D0 ],
    [ 0xEE79, 0x110B,0x1175,0x11D9 ],
    [ 0xEE7A, 0x110B,0x1175,0x11E2 ],
    [ 0xEE7B, 0x110B,0x1175,0x11E5 ],
    [ 0xEE7C, 0x110B,0x1175,0x11E6 ],
    [ 0xEE7D, 0x110B,0x1175,0x11E7 ],
    [ 0xEE7E, 0x110B,0x1175,0x11E9 ],
    [ 0xEE7F, 0x110B,0x1175,0x11EA ],
    [ 0xEE80, 0x110B,0x1175,0x11F0 ],
    [ 0xEE81, 0x110B,0x1175,0x11F9 ],
    [ 0xEE87, 0x110B,0x119C,0x11AB ],
    [ 0xEE89, 0x110B,0x119E,0x11A8 ],
    [ 0xEE8A, 0x110B,0x119E,0x11AB ],
    [ 0xEE8B, 0x110B,0x119E,0x11AE ],
    [ 0xEE8C, 0x110B,0x119E,0x11AF ],
    [ 0xEE8D, 0x110B,0x119E,0x11B0 ],
    [ 0xEE8E, 0x110B,0x119E,0x11B1 ],
    [ 0xEE8F, 0x110B,0x119E,0x11B2 ],
    [ 0xEE90, 0x110B,0x119E,0x11B3 ],
    [ 0xEE91, 0x110B,0x119E,0x11B6 ],
    [ 0xEE92, 0x110B,0x119E,0x11D9 ],
    [ 0xEE93, 0x110B,0x119E,0x11B7 ],
    [ 0xEE94, 0x110B,0x119E,0x11DD ],
    [ 0xEE95, 0x110B,0x119E,0x11B8 ],
    [ 0xEE96, 0x110B,0x119E,0x11B9 ],
    [ 0xEE97, 0x110B,0x119E,0x11BA ],
    [ 0xEE98, 0x110B,0x119E,0x11EB ],
    [ 0xEE99, 0x110B,0x119E,0x11BC ],
    [ 0xEE9A, 0x110B,0x119E,0x11F0 ],
    [ 0xEE9B, 0x110B,0x119E,0x11BD ],
    [ 0xEE9C, 0x110B,0x119E,0x11BE ],
    [ 0xEE9D, 0x110B,0x119E,0x11C0 ],
    [ 0xEE9F, 0x110B,0x119F,0x11AB ],
    [ 0xEEA0, 0x110B,0x119F,0x11AF ],
    [ 0xEEA1, 0x110B,0x119F,0x11B2 ],
    [ 0xEEA2, 0x110B,0x119F,0x11B7 ],
    [ 0xEEA3, 0x110B,0x119F,0x11BC ],
    [ 0xEEA5, 0x110B,0x11A1,0x11A8 ],
    [ 0xEEA6, 0x110B,0x11A1,0x11AB ],
    [ 0xEEA7, 0x110B,0x11A1,0x11AF ],
    [ 0xEEA8, 0x110B,0x11A1,0x11B7 ],
    [ 0xEEA9, 0x110B,0x11A1,0x11B8 ],
    [ 0xEEAA, 0x110B,0x11A1,0x11BA ],
    [ 0xEEAB, 0x110B,0x11A1,0x11EB ],
    [ 0xEEAC, 0x110B,0x11A1,0x11BC ],
    [ 0xEEAD, 0x110B,0x11A1,0x11F0 ],
    [ 0xEEAF, 0x110B,0x11A2,0x11AF ],
    [ 0xEEB1, 0x1141,0x1161,0x11A8 ],
    [ 0xEEB4, 0x1141,0x1163,0x11AB ],
    [ 0xEEB9, 0x1141,0x1168,0x11AB ],
    [ 0xEEBB, 0x1141,0x1169,0x11A8 ],
    [ 0xEEBC, 0x1141,0x1169,0x11AB ],
    [ 0xEEBE, 0x1141,0x116A,0x11AB ],
    [ 0xEEC3, 0x1141,0x116E,0x11AB ],
    [ 0xEECE, 0x1141,0x1175,0x11AB ],
    [ 0xEED3, 0x1142,0x1161,0x11AB ],
    [ 0xEED5, 0x1142,0x1165,0x11AF ],
    [ 0xEED7, 0x1142,0x1166,0x11AB ],
    [ 0xEED9, 0x1142,0x1169,0x11AB ],
    [ 0xEEDA, 0x1142,0x1169,0x11AF ],
    [ 0xEEDC, 0x1142,0x1173,0x11B7 ],
    [ 0xEEDF, 0x1142,0x1175,0x11BC ],
    [ 0xEEE1, 0x1142,0x119E,0x11B7 ],
    [ 0xEEE9, 0x1144,0x1161,0x11AB ],
    [ 0xEEEA, 0x1144,0x1161,0x11AF ],
    [ 0xEEEB, 0x1144,0x1161,0x11B7 ],
    [ 0xEEED, 0x1144,0x1162,0x11AF ],
    [ 0xEEEE, 0x1144,0x1162,0x11B7 ],
    [ 0xEEF0, 0x1144,0x1165,0x11AF ],
    [ 0xEEF1, 0x1144,0x1165,0x11C0 ],
    [ 0xEEF3, 0x1144,0x1166,0x11AB ],
    [ 0xEEF4, 0x1144,0x1166,0x11AF ],
    [ 0xEEF5, 0x1144,0x1166,0x11C0 ],
    [ 0xEEF8, 0x1144,0x1168,0x11AB ],
    [ 0xEEFA, 0x1144,0x1169,0x11AB ],
    [ 0xEEFB, 0x1144,0x1169,0x11AF ],
    [ 0xEEFC, 0x1144,0x1169,0x11C0 ],
    [ 0xEEFE, 0x1144,0x116A,0x11B7 ],
    [ 0xEF01, 0x1144,0x116E,0x11AB ],
    [ 0xEF02, 0x1144,0x116E,0x11AF ],
    [ 0xEF05, 0x1144,0x1170,0x11AF ],
    [ 0xEF07, 0x1144,0x1171,0x11BC ],
    [ 0xEF0A, 0x1144,0x1173,0x11AB ],
    [ 0xEF0B, 0x1144,0x1173,0x11AF ],
    [ 0xEF0E, 0x1144,0x1175,0x11A8 ],
    [ 0xEF0F, 0x1144,0x1175,0x11AB ],
    [ 0xEF10, 0x1144,0x1175,0x11AF ],
    [ 0xEF11, 0x1144,0x1175,0x11BC ],
    [ 0xEF12, 0x1144,0x1175,0x11BF ],
    [ 0xEF16, 0x1145,0x1161,0x11AB ],
    [ 0xEF17, 0x1145,0x1161,0x11BC ],
    [ 0xEF1B, 0x1145,0x1165,0x11AB ],
    [ 0xEF1C, 0x1145,0x1165,0x11AF ],
    [ 0xEF1D, 0x1145,0x1165,0x11B7 ],
    [ 0xEF1E, 0x1145,0x1165,0x11BC ],
    [ 0xEF23, 0x1145,0x1168,0x11AB ],
    [ 0xEF25, 0x1145,0x1169,0x11AB ],
    [ 0xEF27, 0x1145,0x116E,0x11AB ],
    [ 0xEF29, 0x1145,0x1189,0x11AB ],
    [ 0xEF2A, 0x1145,0x1189,0x11BC ],
    [ 0xEF30, 0x1145,0x118E,0x11AB ],
    [ 0xEF35, 0x1145,0x1173,0x11AB ],
    [ 0xEF37, 0x1145,0x1174,0x11AB ],
    [ 0xEF39, 0x1145,0x1175,0x11AF ],
    [ 0xEF3B, 0x1145,0x119E,0x11AF ],
    [ 0xEF57, 0x1147,0x1163,0x11AB ],
    [ 0xEF59, 0x1147,0x1167,0x11AB ],
    [ 0xEF5A, 0x1147,0x1167,0x11BA ],
    [ 0xEF5C, 0x1147,0x1168,0x11BA ],
    [ 0xEF5E, 0x1147,0x116D,0x11AB ],
    [ 0xEF5F, 0x1147,0x116D,0x11B7 ],
    [ 0xEF60, 0x1147,0x116D,0x11E2 ],
    [ 0xEF61, 0x1147,0x116D,0x11BA ],
    [ 0xEF65, 0x1147,0x1172,0x11AB ],
    [ 0xEF66, 0x1147,0x1172,0x11B7 ],
    [ 0xEF68, 0x1147,0x118E,0x11AB ],
    [ 0xEF6A, 0x1147,0x1175,0x11AB ],
    [ 0xEF6C, 0x1148,0x1161,0x11AB ],
    [ 0xEF6D, 0x1148,0x1161,0x11BC ],
    [ 0xEF71, 0x1148,0x1165,0x11AB ],
    [ 0xEF72, 0x1148,0x1165,0x11B7 ],
    [ 0xEF73, 0x1148,0x1165,0x11BC ],
    [ 0xEF78, 0x1148,0x116E,0x11AB ],
    [ 0xEF79, 0x1148,0x116E,0x11BC ],
    [ 0xEF7B, 0x1148,0x1189,0x11AB ],
    [ 0xEF84, 0x1149,0x1161,0x11AB ],
    [ 0xEF85, 0x1149,0x1161,0x11BC ],
    [ 0xEF88, 0x1149,0x1165,0x11AB ],
    [ 0xEF89, 0x1149,0x1165,0x11BC ],
    [ 0xEF8B, 0x1149,0x116E,0x11AB ],
    [ 0xEF8C, 0x1149,0x116E,0x11BC ],
    [ 0xEF8E, 0x1149,0x1189,0x11AB ],
    [ 0xEF93, 0x1149,0x118E,0x11AB ],
    [ 0xEF98, 0x114B,0x1161,0x11A8 ],
    [ 0xEF99, 0x114B,0x1161,0x11AB ],
    [ 0xEF9A, 0x114B,0x1161,0x11AF ],
    [ 0xEF9B, 0x114B,0x1161,0x11D8 ],
    [ 0xEF9C, 0x114B,0x1161,0x11BF ],
    [ 0xEF9E, 0x114B,0x1162,0x11AB ],
    [ 0xEF9F, 0x114B,0x1162,0x11BF ],
    [ 0xEFA1, 0x114B,0x1165,0x11AB ],
    [ 0xEFA2, 0x114B,0x1165,0x11AF ],
    [ 0xEFA3, 0x114B,0x1165,0x11B7 ],
    [ 0xEFA4, 0x114B,0x1165,0x11BC ],
    [ 0xEFA6, 0x114B,0x1166,0x11A8 ],
    [ 0xEFA7, 0x114B,0x1166,0x11AB ],
    [ 0xEFA8, 0x114B,0x1166,0x11AF ],
    [ 0xEFA9, 0x114B,0x1166,0x11BF ],
    [ 0xEFAB, 0x114B,0x1169,0x11AB ],
    [ 0xEFAC, 0x114B,0x1169,0x11AF ],
    [ 0xEFAD, 0x114B,0x1169,0x11B1 ],
    [ 0xEFAE, 0x114B,0x1169,0x11D8 ],
    [ 0xEFAF, 0x114B,0x1169,0x11B7 ],
    [ 0xEFB0, 0x114B,0x1169,0x11BF ],
    [ 0xEFB3, 0x114B,0x116B,0x11AB ],
    [ 0xEFB5, 0x114B,0x116E,0x11AB ],
    [ 0xEFB6, 0x114B,0x116E,0x11AF ],
    [ 0xEFB7, 0x114B,0x116E,0x11BF ],
    [ 0xEFB8, 0x114B,0x116E,0x11C0 ],
    [ 0xEFBA, 0x114B,0x1189,0x11AB ],
    [ 0xEFBC, 0x114B,0x116F,0x11AB ],
    [ 0xEFBE, 0x114B,0x1170,0x11AB ],
    [ 0xEFC0, 0x114B,0x1171,0x11A8 ],
    [ 0xEFC1, 0x114B,0x1171,0x11AB ],
    [ 0xEFC2, 0x114B,0x1171,0x11AF ],
    [ 0xEFC4, 0x114B,0x1172,0x11B7 ],
    [ 0xEFC6, 0x114B,0x1173,0x11AF ],
    [ 0xEFC8, 0x114B,0x1175,0x11AB ],
    [ 0xEFC9, 0x114B,0x1175,0x11AF ],
    [ 0xEFCA, 0x114B,0x1175,0x11BC ],
    [ 0xEFCB, 0x114B,0x1175,0x11BF ],
    [ 0xEFCC, 0x114B,0x1175,0x11C0 ],
    [ 0xEFCD, 0x114B,0x1175,0x11C1 ],
    [ 0xEFD3, 0x114C,0x1161,0x11A8 ],
    [ 0xEFD4, 0x114C,0x1161,0x11AB ],
    [ 0xEFD5, 0x114C,0x1161,0x11AF ],
    [ 0xEFD6, 0x114C,0x1161,0x11D9 ],
    [ 0xEFD7, 0x114C,0x1161,0x11B7 ],
    [ 0xEFD8, 0x114C,0x1161,0x11B8 ],
    [ 0xEFD9, 0x114C,0x1161,0x11EB ],
    [ 0xEFDA, 0x114C,0x1161,0x11BC ],
    [ 0xEFDB, 0x114C,0x1161,0x11F0 ],
    [ 0xEFDC, 0x114C,0x1161,0x11F9 ],
    [ 0xEFDE, 0x114C,0x1162,0x11BA ],
    [ 0xEFDF, 0x114C,0x1162,0x11BC ],
    [ 0xEFE1, 0x114C,0x1163,0x11A8 ],
    [ 0xEFE2, 0x114C,0x1163,0x11AB ],
    [ 0xEFE3, 0x114C,0x1163,0x11BC ],
    [ 0xEFE4, 0x114C,0x1163,0x11F0 ],
    [ 0xEFE6, 0x114C,0x1164,0x11BA ],
    [ 0xEFE8, 0x114C,0x1165,0x11A8 ],
    [ 0xEFE9, 0x114C,0x1165,0x11AB ],
    [ 0xEFEA, 0x114C,0x1165,0x11AF ],
    [ 0xEFEB, 0x114C,0x1165,0x11D9 ],
    [ 0xEFEC, 0x114C,0x1165,0x11B7 ],
    [ 0xEFED, 0x114C,0x1165,0x11B8 ],
    [ 0xEFEE, 0x114C,0x1165,0x11BA ],
    [ 0xEFEF, 0x114C,0x1165,0x11EB ],
    [ 0xEFF0, 0x114C,0x1165,0x11BC ],
    [ 0xEFF1, 0x114C,0x1165,0x11F0 ],
    [ 0xEFF3, 0x114C,0x1166,0x11AB ],
    [ 0xEFF4, 0x114C,0x1166,0x11BA ],
    [ 0xEFF5, 0x114C,0x1166,0x11EB ],
    [ 0xEFF6, 0x114C,0x1166,0x11BC ],
    [ 0xEFF7, 0x114C,0x1166,0x11F0 ],
    [ 0xEFF9, 0x114C,0x1167,0x11A8 ],
    [ 0xEFFA, 0x114C,0x1167,0x11AB ],
    [ 0xEFFB, 0x114C,0x1167,0x11AF ],
    [ 0xEFFC, 0x114C,0x1167,0x11B7 ],
    [ 0xEFFD, 0x114C,0x1167,0x11E2 ],
    [ 0xEFFE, 0x114C,0x1167,0x11B8 ],
    [ 0xEFFF, 0x114C,0x1167,0x11BA ],
    [ 0xF100, 0x114C,0x1167,0x11BC ],
    [ 0xF101, 0x114C,0x1167,0x11F0 ],
    [ 0xF102, 0x114C,0x1167,0x11C1 ],
    [ 0xF103, 0x114C,0x1167,0x11F4 ],
    [ 0xF105, 0x114C,0x1168,0x11BA ],
    [ 0xF106, 0x114C,0x1168,0x11BC ],
    [ 0xF108, 0x114C,0x1169,0x11A8 ],
    [ 0xF109, 0x114C,0x1169,0x11AB ],
    [ 0xF10A, 0x114C,0x1169,0x11AF ],
    [ 0xF10B, 0x114C,0x1169,0x11B3 ],
    [ 0xF10C, 0x114C,0x1169,0x11D9 ],
    [ 0xF10D, 0x114C,0x1169,0x11B7 ],
    [ 0xF10E, 0x114C,0x1169,0x11E2 ],
    [ 0xF10F, 0x114C,0x1169,0x11BA ],
    [ 0xF110, 0x114C,0x1169,0x11BC ],
    [ 0xF111, 0x114C,0x1169,0x11F0 ],
    [ 0xF113, 0x114C,0x116A,0x11A8 ],
    [ 0xF114, 0x114C,0x116A,0x11AB ],
    [ 0xF115, 0x114C,0x116A,0x11AF ],
    [ 0xF116, 0x114C,0x116A,0x11D9 ],
    [ 0xF117, 0x114C,0x116A,0x11B7 ],
    [ 0xF118, 0x114C,0x116A,0x11DD ],
    [ 0xF119, 0x114C,0x116A,0x11BC ],
    [ 0xF11A, 0x114C,0x116A,0x11F0 ],
    [ 0xF11C, 0x114C,0x116B,0x11BC ],
    [ 0xF11E, 0x114C,0x116C,0x11AB ],
    [ 0xF11F, 0x114C,0x116C,0x11BC ],
    [ 0xF120, 0x114C,0x116C,0x11F0 ],
    [ 0xF122, 0x114C,0x116D,0x11B7 ],
    [ 0xF123, 0x114C,0x116D,0x11E2 ],
    [ 0xF124, 0x114C,0x116D,0x11F0 ],
    [ 0xF127, 0x114C,0x1185,0x11BC ],
    [ 0xF129, 0x114C,0x116E,0x11A8 ],
    [ 0xF12A, 0x114C,0x116E,0x11AB ],
    [ 0xF12B, 0x114C,0x116E,0x11AF ],
    [ 0xF12C, 0x114C,0x116E,0x11D9 ],
    [ 0xF12D, 0x114C,0x116E,0x11B7 ],
    [ 0xF12E, 0x114C,0x116E,0x11E2 ],
    [ 0xF12F, 0x114C,0x116E,0x11BC ],
    [ 0xF131, 0x114C,0x116F,0x11AB ],
    [ 0xF132, 0x114C,0x116F,0x11AF ],
    [ 0xF133, 0x114C,0x116F,0x11D9 ],
    [ 0xF134, 0x114C,0x116F,0x11BC ],
    [ 0xF135, 0x114C,0x116F,0x11F0 ],
    [ 0xF138, 0x114C,0x1171,0x11AF ],
    [ 0xF139, 0x114C,0x1171,0x11D9 ],
    [ 0xF13A, 0x114C,0x1171,0x11BC ],
    [ 0xF13B, 0x114C,0x1171,0x11F0 ],
    [ 0xF13D, 0x114C,0x1172,0x11A8 ],
    [ 0xF13E, 0x114C,0x1172,0x11AB ],
    [ 0xF13F, 0x114C,0x1172,0x11AF ],
    [ 0xF140, 0x114C,0x1172,0x11D9 ],
    [ 0xF141, 0x114C,0x1172,0x11B7 ],
    [ 0xF142, 0x114C,0x1172,0x11E2 ],
    [ 0xF144, 0x114C,0x1191,0x11A8 ],
    [ 0xF145, 0x114C,0x1191,0x11AB ],
    [ 0xF146, 0x114C,0x1191,0x11F0 ],
    [ 0xF147, 0x114C,0x1191,0x11F9 ],
    [ 0xF14A, 0x114C,0x1173,0x11A8 ],
    [ 0xF14B, 0x114C,0x1173,0x11AB ],
    [ 0xF14C, 0x114C,0x1173,0x11AF ],
    [ 0xF14D, 0x114C,0x1173,0x11D9 ],
    [ 0xF14E, 0x114C,0x1173,0x11B7 ],
    [ 0xF14F, 0x114C,0x1173,0x11B8 ],
    [ 0xF150, 0x114C,0x1173,0x11BC ],
    [ 0xF151, 0x114C,0x1173,0x11F0 ],
    [ 0xF153, 0x114C,0x1174,0x11BC ],
    [ 0xF155, 0x114C,0x1175,0x11AB ],
    [ 0xF156, 0x114C,0x1175,0x11AE ],
    [ 0xF157, 0x114C,0x1175,0x11AF ],
    [ 0xF158, 0x114C,0x1175,0x11BA ],
    [ 0xF159, 0x114C,0x1175,0x11EB ],
    [ 0xF15A, 0x114C,0x1175,0x11BC ],
    [ 0xF15B, 0x114C,0x1175,0x11F0 ],
    [ 0xF15D, 0x114C,0x119E,0x11A8 ],
    [ 0xF15E, 0x114C,0x119E,0x11AB ],
    [ 0xF15F, 0x114C,0x119E,0x11AE ],
    [ 0xF160, 0x114C,0x119E,0x11AF ],
    [ 0xF161, 0x114C,0x119E,0x11BA ],
    [ 0xF163, 0x114C,0x11A1,0x11A8 ],
    [ 0xF164, 0x110C,0x1161,0x11C7 ],
    [ 0xF165, 0x110C,0x1161,0x11CD ],
    [ 0xF166, 0x110C,0x1161,0x11D0 ],
    [ 0xF167, 0x110C,0x1161,0x11D9 ],
    [ 0xF168, 0x110C,0x1161,0x11DA ],
    [ 0xF169, 0x110C,0x1161,0x11DD ],
    [ 0xF16A, 0x110C,0x1161,0x11E2 ],
    [ 0xF16B, 0x110C,0x1161,0xD7E6 ],
    [ 0xF16C, 0x110C,0x1161,0x11E6 ],
    [ 0xF16D, 0x110C,0x1161,0x11EB ],
    [ 0xF16E, 0x110C,0x1161,0x11F0 ],
    [ 0xF16F, 0x110C,0x1161,0x11EC ],
    [ 0xF170, 0x110C,0x1161,0x11F1 ],
    [ 0xF171, 0x110C,0x1161,0x11F9 ],
    [ 0xF173, 0x110C,0x1162,0x11F0 ],
    [ 0xF174, 0x110C,0x1163,0x11C7 ],
    [ 0xF175, 0x110C,0x1163,0x11CD ],
    [ 0xF176, 0x110C,0x1163,0x11D0 ],
    [ 0xF177, 0x110C,0x1163,0x11E2 ],
    [ 0xF178, 0x110C,0x1163,0x11EB ],
    [ 0xF179, 0x110C,0x1163,0x11F1 ],
    [ 0xF17A, 0x110C,0x1163,0x11F0 ],
    [ 0xF17B, 0x110C,0x1163,0x11F1 ],
    [ 0xF17E, 0x110C,0x11A4,0x11AF ],
    [ 0xF17F, 0x110C,0x1165,0x11FE ],
    [ 0xF180, 0x110C,0x1165,0x11CD ],
    [ 0xF181, 0x110C,0x1165,0x11D0 ],
    [ 0xF182, 0x110C,0x1165,0x11D9 ],
    [ 0xF183, 0x110C,0x1165,0x11DD ],
    [ 0xF184, 0x110C,0x1165,0x11E4 ],
    [ 0xF185, 0x110C,0x1165,0xD7F2 ],
    [ 0xF186, 0x110C,0x1165,0x11EB ],
    [ 0xF187, 0x110C,0x1165,0x11F0 ],
    [ 0xF188, 0x110C,0x1165,0xD7F6 ],
    [ 0xF189, 0x110C,0x1165,0x11F9 ],
    [ 0xF18C, 0x110C,0x117C,0x11A8 ],
    [ 0xF18D, 0x110C,0x117C,0x11AB ],
    [ 0xF18E, 0x110C,0x117C,0x11BC ],
    [ 0xF18F, 0x110C,0x117C,0x11F9 ],
    [ 0xF190, 0x110C,0x1167,0x11C7 ],
    [ 0xF191, 0x110C,0x1167,0x11CA ],
    [ 0xF192, 0x110C,0x1167,0xD7CD ],
    [ 0xF193, 0x110C,0x1167,0xD7CE ],
    [ 0xF194, 0x110C,0x1167,0xD7D2 ],
    [ 0xF195, 0x110C,0x1167,0x11D0 ],
    [ 0xF196, 0x110C,0x1167,0x11D9 ],
    [ 0xF197, 0x110C,0x1167,0x11E2 ],
    [ 0xF198, 0x110C,0x1167,0x11EB ],
    [ 0xF199, 0x110C,0x1167,0x11F0 ],
    [ 0xF19A, 0x110C,0x1167,0x11F9 ],
    [ 0xF19C, 0x110C,0x11A5,0x11BC ],
    [ 0xF19F, 0x110C,0x1169,0x11C7 ],
    [ 0xF1A0, 0x110C,0x1169,0x11D9 ],
    [ 0xF1A1, 0x110C,0x1169,0x11DD ],
    [ 0xF1A2, 0x110C,0x1169,0x11E2 ],
    [ 0xF1A3, 0x110C,0x1169,0x11EB ],
    [ 0xF1A4, 0x110C,0x1169,0x11F0 ],
    [ 0xF1A5, 0x110C,0x116A,0x11D9 ],
    [ 0xF1A6, 0x110C,0x116A,0x11E2 ],
    [ 0xF1A7, 0x110C,0x116A,0x11F0 ],
    [ 0xF1A8, 0x110C,0x116D,0x11D9 ],
    [ 0xF1A9, 0x110C,0x116D,0x11E2 ],
    [ 0xF1AA, 0x110C,0x116D,0x11E6 ],
    [ 0xF1AB, 0x110C,0x116D,0x11F1 ],
    [ 0xF1AC, 0x110C,0x116D,0x11F0 ],
    [ 0xF1AD, 0x110C,0x116D,0x11F1 ],
    [ 0xF1B0, 0x110C,0x1188,0x11AF ],
    [ 0xF1B1, 0x110C,0x1188,0x11D9 ],
    [ 0xF1B2, 0x110C,0x116E,0x11C7 ],
    [ 0xF1B3, 0x110C,0x116E,0x11CD ],
    [ 0xF1B4, 0x110C,0x116E,0x11D9 ],
    [ 0xF1B5, 0x110C,0x116E,0x11DD ],
    [ 0xF1B6, 0x110C,0x116E,0x11E2 ],
    [ 0xF1B7, 0x110C,0x116E,0x11EB ],
    [ 0xF1B8, 0x110C,0x116E,0x11F0 ],
    [ 0xF1B9, 0x110C,0x116E,0x11F9 ],
    [ 0xF1BB, 0x110C,0x1189,0x11BC ],
    [ 0xF1BE, 0x110C,0xD7B5,0x11AB ],
    [ 0xF1BF, 0x110C,0xD7B5,0x11AF ],
    [ 0xF1C0, 0x110C,0xD7B5,0x11D9 ],
    [ 0xF1C2, 0x110C,0x1172,0x11D9 ],
    [ 0xF1C3, 0x110C,0x1172,0x11E2 ],
    [ 0xF1C4, 0x110C,0x1172,0x11F0 ],
    [ 0xF1C5, 0x110C,0x1172,0x11EC ],
    [ 0xF1C6, 0x110C,0x1172,0x11F9 ],
    [ 0xF1C8, 0x110C,0x118E,0x11AB ],
    [ 0xF1CA, 0x110C,0x118F,0x11AB ],
    [ 0xF1CB, 0x110C,0x118F,0x11AF ],
    [ 0xF1CE, 0x110C,0x1191,0x11AB ],
    [ 0xF1CF, 0x110C,0x1191,0x11AF ],
    [ 0xF1D0, 0x110C,0x1191,0x11D9 ],
    [ 0xF1D1, 0x110C,0x1191,0x11F9 ],
    [ 0xF1D3, 0x110C,0x1192,0x11AB ],
    [ 0xF1D4, 0x110C,0x1192,0x11B8 ],
    [ 0xF1D5, 0x110C,0x1192,0x11BC ],
    [ 0xF1D7, 0x110C,0x1194,0x11A8 ],
    [ 0xF1D8, 0x110C,0x1194,0x11AB ],
    [ 0xF1D9, 0x110C,0x1194,0x11AF ],
    [ 0xF1DA, 0x110C,0x1194,0x11D9 ],
    [ 0xF1DB, 0x110C,0x1194,0x11BA ],
    [ 0xF1DC, 0x110C,0x1194,0x11BC ],
    [ 0xF1DD, 0x110C,0x1173,0x11D0 ],
    [ 0xF1DE, 0x110C,0x1173,0x11D9 ],
    [ 0xF1DF, 0x110C,0x1173,0x11E2 ],
    [ 0xF1E0, 0x110C,0x1173,0x11E6 ],
    [ 0xF1E1, 0x110C,0x1173,0x11EB ],
    [ 0xF1E2, 0x110C,0x1173,0x11F0 ],
    [ 0xF1E5, 0x110C,0x1174,0x11F0 ],
    [ 0xF1E6, 0x110C,0x1174,0x11F9 ],
    [ 0xF1E7, 0x110C,0x1175,0x11FE ],
    [ 0xF1E8, 0x110C,0x1175,0xD7D3 ],
    [ 0xF1E9, 0x110C,0x1175,0x11CD ],
    [ 0xF1EA, 0x110C,0x1175,0x11D9 ],
    [ 0xF1EB, 0x110C,0x1175,0x11DD ],
    [ 0xF1EC, 0x110C,0x1175,0x11E2 ],
    [ 0xF1ED, 0x110C,0x1175,0x11E6 ],
    [ 0xF1EE, 0x110C,0x1175,0xD7F0 ],
    [ 0xF1EF, 0x110C,0x1175,0xD7F1 ],
    [ 0xF1F0, 0x110C,0x1175,0xD7F2 ],
    [ 0xF1F1, 0x110C,0x1175,0x11EB ],
    [ 0xF1F2, 0x110C,0x1175,0x11F0 ],
    [ 0xF1F3, 0x110C,0x1175,0x11F9 ],
    [ 0xF1F6, 0x110C,0x1199,0x11AB ],
    [ 0xF1FA, 0x110C,0x119A,0x11AF ],
    [ 0xF1FD, 0x110C,0x119E,0x11A8 ],
    [ 0xF1FE, 0x110C,0x119E,0x11AB ],
    [ 0xF1FF, 0x110C,0x119E,0x11AE ],
    [ 0xF200, 0x110C,0x119E,0x11AF ],
    [ 0xF201, 0x110C,0x119E,0x11CD ],
    [ 0xF202, 0x110C,0x119E,0x11D0 ],
    [ 0xF203, 0x110C,0x119E,0x11D9 ],
    [ 0xF204, 0x110C,0x119E,0x11B7 ],
    [ 0xF205, 0x110C,0x119E,0x11DA ],
    [ 0xF206, 0x110C,0x119E,0x11DD ],
    [ 0xF207, 0x110C,0x119E,0x11B8 ],
    [ 0xF208, 0x110C,0x119E,0x11E6 ],
    [ 0xF209, 0x110C,0x119E,0x11BA ],
    [ 0xF20A, 0x110C,0x119E,0x11EB ],
    [ 0xF20B, 0x110C,0x119E,0x11BC ],
    [ 0xF20C, 0x110C,0x119E,0x11F0 ],
    [ 0xF20D, 0x110C,0x119E,0x11BD ],
    [ 0xF20E, 0x110C,0x119E,0x11BE ],
    [ 0xF20F, 0x110C,0x119E,0x11C0 ],
    [ 0xF211, 0x110C,0xD7C5,0x11BC ],
    [ 0xF213, 0x110C,0x11A1,0x11A8 ],
    [ 0xF214, 0x110C,0x11A1,0x11AB ],
    [ 0xF215, 0x110C,0x11A1,0x11AE ],
    [ 0xF216, 0x110C,0x11A1,0x11AF ],
    [ 0xF217, 0x110C,0x11A1,0x11B7 ],
    [ 0xF218, 0x110C,0x11A1,0x11B8 ],
    [ 0xF219, 0x110C,0x11A1,0x11BA ],
    [ 0xF21A, 0x110C,0x11A1,0x11BC ],
    [ 0xF21B, 0x110C,0x11A1,0x11F0 ],
    [ 0xF21D, 0x114D,0x1173,0x11AF ],
    [ 0xF21F, 0x114D,0x1175,0x11AF ],
    [ 0xF220, 0x110D,0x1161,0x11D0 ],
    [ 0xF221, 0x110D,0x1161,0x11D9 ],
    [ 0xF222, 0x110D,0x1161,0x11E2 ],
    [ 0xF223, 0x110D,0x1161,0xD7E6 ],
    [ 0xF224, 0x110D,0x1161,0x11F0 ],
    [ 0xF225, 0x110D,0x1161,0x11F9 ],
    [ 0xF226, 0x110D,0x1163,0x11E2 ],
    [ 0xF227, 0x110D,0x1163,0x11E6 ],
    [ 0xF228, 0x110D,0x1163,0x11F0 ],
    [ 0xF229, 0x110D,0x1165,0x11F0 ],
    [ 0xF22A, 0x110D,0x1165,0x11F9 ],
    [ 0xF22B, 0x110D,0x1167,0x11D9 ],
    [ 0xF22C, 0x110D,0x1167,0x11F0 ],
    [ 0xF22D, 0x110D,0x1169,0x11D9 ],
    [ 0xF22E, 0x110D,0x1169,0x11E2 ],
    [ 0xF22F, 0x110D,0x1169,0x11F0 ],
    [ 0xF230, 0x110D,0x1169,0xD7F9 ],
    [ 0xF231, 0x110D,0x1169,0x11F9 ],
    [ 0xF232, 0x110D,0x116A,0x11F0 ],
    [ 0xF234, 0x110D,0x116C,0x11F9 ],
    [ 0xF235, 0x110D,0x116D,0x11E2 ],
    [ 0xF236, 0x110D,0x116D,0x11F0 ],
    [ 0xF237, 0x110D,0x116E,0x11D9 ],
    [ 0xF238, 0x110D,0x116E,0x11E2 ],
    [ 0xF239, 0x110D,0x116E,0x11F0 ],
    [ 0xF23A, 0x110D,0x116E,0x11F9 ],
    [ 0xF23B, 0x110D,0x1171,0x11F9 ],
    [ 0xF23C, 0x110D,0x1172,0x11D9 ],
    [ 0xF23D, 0x110D,0x1172,0x11E2 ],
    [ 0xF23F, 0x110D,0x1191,0x11AB ],
    [ 0xF240, 0x110D,0x1191,0x11AF ],
    [ 0xF241, 0x110D,0x1191,0x11D9 ],
    [ 0xF242, 0x110D,0x1191,0x11F9 ],
    [ 0xF244, 0x110D,0x1194,0x11BC ],
    [ 0xF245, 0x110D,0x1173,0x11E2 ],
    [ 0xF246, 0x110D,0x1173,0x11EB ],
    [ 0xF247, 0x110D,0x1173,0x11F0 ],
    [ 0xF248, 0x110D,0x1173,0xD7F9 ],
    [ 0xF249, 0x110D,0x1174,0x11F0 ],
    [ 0xF24A, 0x110D,0x1174,0x11F9 ],
    [ 0xF24B, 0x110D,0x1175,0x11D9 ],
    [ 0xF24C, 0x110D,0x1175,0x11E2 ],
    [ 0xF24D, 0x110D,0x1175,0x11EB ],
    [ 0xF24E, 0x110D,0x1175,0x11F0 ],
    [ 0xF24F, 0x110D,0x1175,0x11F9 ],
    [ 0xF251, 0x110D,0x119E,0x11A8 ],
    [ 0xF252, 0x110D,0x119E,0x11AB ],
    [ 0xF253, 0x110D,0x119E,0x11AF ],
    [ 0xF254, 0x110D,0x119E,0x11B7 ],
    [ 0xF255, 0x110D,0x119E,0x11B8 ],
    [ 0xF256, 0x110D,0x119E,0x11E6 ],
    [ 0xF257, 0x110D,0x119E,0x11BA ],
    [ 0xF258, 0x110D,0x119E,0x11BC ],
    [ 0xF259, 0x110D,0x119E,0x11F0 ],
    [ 0xF25B, 0x110D,0x11A1,0x11A8 ],
    [ 0xF25C, 0x110D,0x11A1,0x11AB ],
    [ 0xF25D, 0x110D,0x11A1,0x11AF ],
    [ 0xF25E, 0x110D,0x11A1,0x11D9 ],
    [ 0xF25F, 0x110D,0x11A1,0x11B7 ],
    [ 0xF260, 0x110D,0x11A1,0x11BC ],
    [ 0xF261, 0x110D,0x11A1,0x11F0 ],
    [ 0xF262, 0x110D,0x11A1,0x11F9 ],
    [ 0xF265, 0xA978,0x1166,0x11AF ],
    [ 0xF266, 0xA978,0x1166,0x11D0 ],
    [ 0xF267, 0xA978,0x1166,0xD7D7 ],
    [ 0xF269, 0x114E,0x1161,0x11AB ],
    [ 0xF26A, 0x114E,0x1161,0x11B7 ],
    [ 0xF26B, 0x114E,0x1161,0x11E2 ],
    [ 0xF26E, 0x114E,0x1163,0x11BC ],
    [ 0xF270, 0x114E,0x1165,0x11AB ],
    [ 0xF272, 0x114E,0x1167,0x11AB ],
    [ 0xF273, 0x114E,0x1167,0x11F9 ],
    [ 0xF27A, 0x114E,0x116E,0x11AB ],
    [ 0xF27B, 0x114E,0x116E,0x11BC ],
    [ 0xF27E, 0x114E,0x1172,0x11AB ],
    [ 0xF27F, 0x114E,0x1172,0x11BC ],
    [ 0xF281, 0x114E,0x1191,0x11AB ],
    [ 0xF283, 0x114E,0x1173,0x11AB ],
    [ 0xF284, 0x114E,0x1173,0x11BA ],
    [ 0xF285, 0x114E,0x1173,0x11EB ],
    [ 0xF287, 0x114E,0x1175,0x11AB ],
    [ 0xF288, 0x114E,0x1175,0x11B7 ],
    [ 0xF289, 0x114E,0x1175,0x11E2 ],
    [ 0xF28A, 0x114E,0x1175,0x11EB ],
    [ 0xF28C, 0x114F,0x1161,0x11AB ],
    [ 0xF28D, 0x114F,0x1161,0x11B7 ],
    [ 0xF28E, 0x114F,0x1161,0x11E2 ],
    [ 0xF28F, 0x114F,0x1161,0x11F0 ],
    [ 0xF292, 0x114F,0x1163,0x11B7 ],
    [ 0xF293, 0x114F,0x1163,0x11F0 ],
    [ 0xF295, 0x114F,0x1165,0x11AB ],
    [ 0xF297, 0x114F,0x1167,0x11AB ],
    [ 0xF29C, 0x114F,0x1172,0x11F0 ],
    [ 0xF29E, 0x114F,0x1191,0x11AB ],
    [ 0xF2A0, 0x114F,0x1173,0x11EB ],
    [ 0xF2A1, 0x114F,0x1173,0x11F0 ],
    [ 0xF2A3, 0x114F,0x1175,0x11AB ],
    [ 0xF2A7, 0x1150,0x1161,0x11AB ],
    [ 0xF2A8, 0x1150,0x1161,0x11AE ],
    [ 0xF2A9, 0x1150,0x1161,0x11B7 ],
    [ 0xF2AA, 0x1150,0x1161,0x11E2 ],
    [ 0xF2AB, 0x1150,0x1161,0x11B8 ],
    [ 0xF2AE, 0x1150,0x1163,0x11B7 ],
    [ 0xF2AF, 0x1150,0x1163,0x11E2 ],
    [ 0xF2B0, 0x1150,0x1163,0x11F0 ],
    [ 0xF2B2, 0x1150,0x1167,0x11AB ],
    [ 0xF2B5, 0x1150,0x116A,0x11BC ],
    [ 0xF2B7, 0x1150,0x116E,0x11BC ],
    [ 0xF2B8, 0x1150,0x116E,0x11F9 ],
    [ 0xF2BA, 0x1150,0x1172,0x11A8 ],
    [ 0xF2BB, 0x1150,0x1172,0x11AB ],
    [ 0xF2BC, 0x1150,0x1172,0x11F0 ],
    [ 0xF2BE, 0x1150,0x1191,0x11AB ],
    [ 0xF2C0, 0x1150,0x1173,0x11B7 ],
    [ 0xF2C1, 0x1150,0x1173,0x11EB ],
    [ 0xF2C3, 0x1150,0x1175,0x11AB ],
    [ 0xF2C4, 0x1150,0x1175,0x11EB ],
    [ 0xF2C5, 0x1150,0x1175,0x11F0 ],
    [ 0xF2C7, 0x1151,0x1161,0x11AB ],
    [ 0xF2C8, 0x1151,0x1161,0x11C2 ],
    [ 0xF2CB, 0x1151,0x1163,0x11B7 ],
    [ 0xF2CC, 0x1151,0x1163,0x11E2 ],
    [ 0xF2CD, 0x1151,0x1163,0x11B8 ],
    [ 0xF2CE, 0x1151,0x1163,0x11F0 ],
    [ 0xF2D0, 0x1151,0x1167,0x11AB ],
    [ 0xF2D4, 0x1151,0x1172,0x11F0 ],
    [ 0xF2D6, 0x1151,0x1191,0x11AB ],
    [ 0xF2D8, 0x1151,0x1173,0x11B7 ],
    [ 0xF2D9, 0x1151,0x1173,0x11E2 ],
    [ 0xF2DB, 0x1151,0x1175,0x11AB ],
    [ 0xF2DC, 0x1151,0x1175,0x11B7 ],
    [ 0xF2DD, 0x1151,0x1175,0x11EB ],
    [ 0xF2DE, 0x1151,0x1175,0x11F0 ],
    [ 0xF2DF, 0x110E,0x1161,0x11D9 ],
    [ 0xF2E0, 0x110E,0x1161,0x11E2 ],
    [ 0xF2E1, 0x110E,0x1161,0x11E6 ],
    [ 0xF2E2, 0x110E,0x1161,0x11F0 ],
    [ 0xF2E3, 0x110E,0x1161,0x11F9 ],
    [ 0xF2E5, 0x110E,0x1163,0x11D9 ],
    [ 0xF2E6, 0x110E,0x1163,0x11E6 ],
    [ 0xF2E7, 0x110E,0x1163,0x11F0 ],
    [ 0xF2E8, 0x110E,0x1163,0x11F1 ],
    [ 0xF2EA, 0x110E,0x1178,0x11AF ],
    [ 0xF2EC, 0x110E,0x1165,0x11EB ],
    [ 0xF2ED, 0x110E,0x1165,0x11F0 ],
    [ 0xF2EF, 0x110E,0x117C,0x11AB ],
    [ 0xF2F0, 0x110E,0x1167,0xD7CD ],
    [ 0xF2F1, 0x110E,0x1167,0xD7CE ],
    [ 0xF2F2, 0x110E,0x1167,0xD7CF ],
    [ 0xF2F3, 0x110E,0x1167,0x11D9 ],
    [ 0xF2F4, 0x110E,0x1167,0x11E2 ],
    [ 0xF2F5, 0x110E,0x1167,0x11F0 ],
    [ 0xF2F6, 0x110E,0x1167,0x11F1 ],
    [ 0xF2F7, 0x110E,0x1167,0x11F9 ],
    [ 0xF2FA, 0x110E,0x1169,0x11D9 ],
    [ 0xF2FB, 0x110E,0x1169,0x11E2 ],
    [ 0xF2FC, 0x110E,0x1169,0xD7E8 ],
    [ 0xF2FD, 0x110E,0x1169,0x11EB ],
    [ 0xF2FE, 0x110E,0x1169,0x11F0 ],
    [ 0xF2FF, 0x110E,0x116A,0x11D9 ],
    [ 0xF300, 0x110E,0x116A,0x11F0 ],
    [ 0xF302, 0x110E,0x116D,0x11E2 ],
    [ 0xF303, 0x110E,0x116D,0x11F0 ],
    [ 0xF306, 0x110E,0x1188,0x11AB ],
    [ 0xF307, 0x110E,0x116E,0x11D9 ],
    [ 0xF308, 0x110E,0x116E,0x11E0 ],
    [ 0xF309, 0x110E,0x116E,0x11E2 ],
    [ 0xF30A, 0x110E,0x116E,0x11F0 ],
    [ 0xF30B, 0x110E,0x116E,0x11F9 ],
    [ 0xF30D, 0x110E,0x1189,0x11AB ],
    [ 0xF30E, 0x110E,0x1189,0x11BC ],
    [ 0xF310, 0x110E,0x116F,0x11F9 ],
    [ 0xF313, 0x110E,0x1171,0x11D9 ],
    [ 0xF314, 0x110E,0x1171,0x11F9 ],
    [ 0xF315, 0x110E,0x1172,0x11D9 ],
    [ 0xF316, 0x110E,0x1172,0x11E2 ],
    [ 0xF317, 0x110E,0x1172,0x11F0 ],
    [ 0xF318, 0x110E,0x1172,0x11F9 ],
    [ 0xF31A, 0x110E,0x118E,0x11AB ],
    [ 0xF31C, 0x110E,0x118F,0x11B7 ],
    [ 0xF31F, 0x110E,0x1191,0x11AB ],
    [ 0xF320, 0x110E,0x1191,0x11AF ],
    [ 0xF321, 0x110E,0x1191,0x11D9 ],
    [ 0xF322, 0x110E,0x1191,0x11B7 ],
    [ 0xF324, 0x110E,0x1192,0x11BC ],
    [ 0xF326, 0x110E,0x1194,0x11AB ],
    [ 0xF327, 0x110E,0x1194,0x11AF ],
    [ 0xF328, 0x110E,0x1194,0x11BC ],
    [ 0xF329, 0x110E,0x1173,0xD7DB ],
    [ 0xF32A, 0x110E,0x1173,0x11D9 ],
    [ 0xF32B, 0x110E,0x1173,0xD7DD ],
    [ 0xF32C, 0x110E,0x1173,0x11E2 ],
    [ 0xF32D, 0x110E,0x1173,0x11EB ],
    [ 0xF32E, 0x110E,0x1173,0x11F0 ],
    [ 0xF331, 0x110E,0x1174,0x11F9 ],
    [ 0xF332, 0x110E,0x1175,0x11D9 ],
    [ 0xF333, 0x110E,0x1175,0x11E2 ],
    [ 0xF334, 0x110E,0x1175,0xD7E3 ],
    [ 0xF335, 0x110E,0x1175,0x11E6 ],
    [ 0xF336, 0x110E,0x1175,0x11EB ],
    [ 0xF337, 0x110E,0x1175,0x11F0 ],
    [ 0xF338, 0x110E,0x1175,0x11F9 ],
    [ 0xF33B, 0x110E,0x1199,0x11AB ],
    [ 0xF33C, 0x110E,0x1199,0x11BC ],
    [ 0xF342, 0x110E,0x119E,0x11A8 ],
    [ 0xF343, 0x110E,0x119E,0x11AB ],
    [ 0xF344, 0x110E,0x119E,0x11AE ],
    [ 0xF345, 0x110E,0x119E,0x11AF ],
    [ 0xF346, 0x110E,0x119E,0x11CE ],
    [ 0xF347, 0x110E,0x119E,0x11B1 ],
    [ 0xF348, 0x110E,0x119E,0x11B3 ],
    [ 0xF349, 0x110E,0x119E,0x11B6 ],
    [ 0xF34A, 0x110E,0x119E,0x11D9 ],
    [ 0xF34B, 0x110E,0x119E,0x11B7 ],
    [ 0xF34C, 0x110E,0x119E,0x11DD ],
    [ 0xF34D, 0x110E,0x119E,0x11B8 ],
    [ 0xF34E, 0x110E,0x119E,0x11BA ],
    [ 0xF34F, 0x110E,0x119E,0x11BB ],
    [ 0xF350, 0x110E,0x119E,0x11BC ],
    [ 0xF351, 0x110E,0x119E,0x11BD ],
    [ 0xF352, 0x110E,0x119E,0x11BE ],
    [ 0xF354, 0x110E,0x11A1,0x11A8 ],
    [ 0xF355, 0x110E,0x11A1,0x11AB ],
    [ 0xF356, 0x110E,0x11A1,0x11AF ],
    [ 0xF357, 0x110E,0x11A1,0x11B7 ],
    [ 0xF358, 0x110E,0x11A1,0x11BA ],
    [ 0xF359, 0x110E,0x11A1,0x11BC ],
    [ 0xF35A, 0x110E,0x11A1,0x11F0 ],
    [ 0xF35E, 0x1153,0x1175,0x11AF ],
    [ 0xF360, 0x1154,0x1161,0x11AB ],
    [ 0xF364, 0x1154,0x1165,0x11AB ],
    [ 0xF366, 0x1154,0x1167,0x11AB ],
    [ 0xF36D, 0x1154,0x116E,0x11AB ],
    [ 0xF36E, 0x1154,0x116E,0x11BC ],
    [ 0xF371, 0x1154,0x1172,0x11AB ],
    [ 0xF373, 0x1154,0x1191,0x11AB ],
    [ 0xF376, 0x1154,0x1175,0x11AB ],
    [ 0xF377, 0x1154,0x1175,0x11EB ],
    [ 0xF378, 0x1154,0x1175,0x11F0 ],
    [ 0xF379, 0x1154,0x1175,0x11F9 ],
    [ 0xF37B, 0x1155,0x1161,0x11AB ],
    [ 0xF37C, 0x1155,0x1161,0x11B7 ],
    [ 0xF37D, 0x1155,0x1161,0x11E2 ],
    [ 0xF37E, 0x1155,0x1161,0x11F9 ],
    [ 0xF381, 0x1155,0x1163,0x11F0 ],
    [ 0xF383, 0x1155,0x1167,0x11AB ],
    [ 0xF386, 0x1155,0x116A,0x11AB ],
    [ 0xF38B, 0x1155,0x1172,0x11A8 ],
    [ 0xF38C, 0x1155,0x1172,0x11AB ],
    [ 0xF38D, 0x1155,0x1172,0x11BC ],
    [ 0xF38E, 0x1155,0x1172,0x11F0 ],
    [ 0xF390, 0x1155,0x1191,0x11AB ],
    [ 0xF392, 0x1155,0x1173,0x11AB ],
    [ 0xF393, 0x1155,0x1173,0x11EB ],
    [ 0xF395, 0x1155,0x1175,0x11AB ],
    [ 0xF396, 0x1155,0x1175,0x11B7 ],
    [ 0xF397, 0x1155,0x1175,0x11E2 ],
    [ 0xF398, 0x1155,0x1175,0x11EB ],
    [ 0xF399, 0x1155,0x1175,0x11F9 ],
    [ 0xF39B, 0x110F,0x1161,0x11D0 ],
    [ 0xF39C, 0x110F,0x1161,0x11D9 ],
    [ 0xF39D, 0x110F,0x1161,0x11E2 ],
    [ 0xF39E, 0x110F,0x1161,0x11E6 ],
    [ 0xF39F, 0x110F,0x1161,0x11F0 ],
    [ 0xF3A1, 0x110F,0x1163,0x11E2 ],
    [ 0xF3A2, 0x110F,0x1163,0x11F0 ],
    [ 0xF3A3, 0x110F,0x1163,0x11F9 ],
    [ 0xF3A5, 0x110F,0x1165,0x11D9 ],
    [ 0xF3A6, 0x110F,0x1165,0x11F9 ],
    [ 0xF3A9, 0x110F,0x1167,0x11D9 ],
    [ 0xF3AA, 0x110F,0x1167,0x11E2 ],
    [ 0xF3AB, 0x110F,0x1167,0x11F0 ],
    [ 0xF3AC, 0x110F,0x1167,0x11F9 ],
    [ 0xF3AE, 0x110F,0x1169,0x11D9 ],
    [ 0xF3AF, 0x110F,0x1169,0x11E2 ],
    [ 0xF3B0, 0x110F,0x1169,0x11F0 ],
    [ 0xF3B1, 0x110F,0x116A,0x11D9 ],
    [ 0xF3B2, 0x110F,0x116A,0x11F0 ],
    [ 0xF3B4, 0x110F,0x116C,0x11F0 ],
    [ 0xF3B5, 0x110F,0x116D,0x11D9 ],
    [ 0xF3B6, 0x110F,0x116D,0x11E2 ],
    [ 0xF3B8, 0x110F,0x116E,0x11D9 ],
    [ 0xF3B9, 0x110F,0x116E,0x11E2 ],
    [ 0xF3BA, 0x110F,0x116E,0x11F0 ],
    [ 0xF3BB, 0x110F,0x116E,0x11F9 ],
    [ 0xF3BD, 0x110F,0x1189,0x11AB ],
    [ 0xF3BE, 0x110F,0x1189,0x11BC ],
    [ 0xF3C0, 0x110F,0x116F,0x11D9 ],
    [ 0xF3C1, 0x110F,0x116F,0x11F9 ],
    [ 0xF3C3, 0x110F,0xD7B5,0x11AB ],
    [ 0xF3C5, 0x110F,0x1172,0x11F0 ],
    [ 0xF3C6, 0x110F,0x1172,0x11F9 ],
    [ 0xF3CA, 0x110F,0x1191,0x11A8 ],
    [ 0xF3CB, 0x110F,0x1191,0x11AB ],
    [ 0xF3CC, 0x110F,0x1191,0x11AF ],
    [ 0xF3CD, 0x110F,0x1191,0x11D9 ],
    [ 0xF3CE, 0x110F,0x1191,0x11F0 ],
    [ 0xF3CF, 0x110F,0x1191,0x11F9 ],
    [ 0xF3D1, 0x110F,0x1192,0x11BC ],
    [ 0xF3D3, 0x110F,0x1194,0x11BC ],
    [ 0xF3D4, 0x110F,0x1173,0x11D9 ],
    [ 0xF3D5, 0x110F,0x1173,0x11E2 ],
    [ 0xF3D6, 0x110F,0x1173,0x11F0 ],
    [ 0xF3D9, 0x110F,0x1174,0x11F9 ],
    [ 0xF3DA, 0x110F,0x1175,0x11D9 ],
    [ 0xF3DB, 0x110F,0x1175,0x11F0 ],
    [ 0xF3DC, 0x110F,0x1175,0x11F9 ],
    [ 0xF3DF, 0x110F,0x1199,0x11AB ],
    [ 0xF3E0, 0x110F,0x1199,0x11BC ],
    [ 0xF3E7, 0x110F,0x119B,0x11AF ],
    [ 0xF3E9, 0x110F,0xD7C3,0x11AB ],
    [ 0xF3EB, 0x110F,0x119E,0x11A8 ],
    [ 0xF3EC, 0x110F,0x119E,0x11AB ],
    [ 0xF3ED, 0x110F,0x119E,0x11AE ],
    [ 0xF3EE, 0x110F,0x119E,0x11AF ],
    [ 0xF3EF, 0x110F,0x119E,0x11B7 ],
    [ 0xF3F0, 0x110F,0x119E,0x11B8 ],
    [ 0xF3F1, 0x110F,0x119E,0x11BA ],
    [ 0xF3F2, 0x110F,0x119E,0x11C0 ],
    [ 0xF3F4, 0x110F,0x11A1,0x11A8 ],
    [ 0xF3F5, 0x110F,0x11A1,0x11AB ],
    [ 0xF3F6, 0x110F,0x11A1,0x11AF ],
    [ 0xF3F7, 0x110F,0x11A1,0x11B7 ],
    [ 0xF3F8, 0x110F,0x11A1,0x11BA ],
    [ 0xF3F9, 0x110F,0x11A1,0x11BC ],
    [ 0xF3FA, 0x110F,0x11A1,0x11F0 ],
    [ 0xF3FB, 0x1110,0x1161,0x11D9 ],
    [ 0xF3FC, 0x1110,0x1161,0x11E2 ],
    [ 0xF3FD, 0x1110,0x1161,0x11F0 ],
    [ 0xF3FE, 0x1110,0x1161,0x11F9 ],
    [ 0xF401, 0x1110,0x1163,0x11E2 ],
    [ 0xF402, 0x1110,0x1163,0x11F0 ],
    [ 0xF404, 0x1110,0x1165,0x11DD ],
    [ 0xF405, 0x1110,0x1165,0x11E6 ],
    [ 0xF406, 0x1110,0x1165,0x11F0 ],
    [ 0xF407, 0x1110,0x1165,0x11F1 ],
    [ 0xF409, 0x1110,0x117B,0x11AE ],
    [ 0xF40A, 0x1110,0x117B,0x11AF ],
    [ 0xF40B, 0x1110,0x1167,0x11D9 ],
    [ 0xF40C, 0x1110,0x1167,0x11F0 ],
    [ 0xF40D, 0x1110,0x1167,0x11F9 ],
    [ 0xF40E, 0x1110,0x1168,0x11F0 ],
    [ 0xF40F, 0x1110,0x1169,0x11D0 ],
    [ 0xF410, 0x1110,0x1169,0x11D9 ],
    [ 0xF411, 0x1110,0x1169,0x11E2 ],
    [ 0xF412, 0x1110,0x1169,0xD7E5 ],
    [ 0xF413, 0x1110,0x1169,0x11E7 ],
    [ 0xF414, 0x1110,0x1169,0x11F0 ],
    [ 0xF415, 0x1110,0x116A,0x11D9 ],
    [ 0xF416, 0x1110,0x116A,0x11F0 ],
    [ 0xF418, 0x1110,0x1182,0x11AF ],
    [ 0xF419, 0x1110,0x116D,0x11D9 ],
    [ 0xF41A, 0x1110,0x116D,0x11E2 ],
    [ 0xF41B, 0x1110,0x116D,0x11F0 ],
    [ 0xF41D, 0x1110,0x116E,0x11D0 ],
    [ 0xF41E, 0x1110,0x116E,0x11D9 ],
    [ 0xF41F, 0x1110,0x116E,0x11E2 ],
    [ 0xF420, 0x1110,0x116E,0x11F0 ],
    [ 0xF421, 0x1110,0x116E,0x11F9 ],
    [ 0xF423, 0x1110,0x1189,0x11AB ],
    [ 0xF424, 0x1110,0x116F,0x11F9 ],
    [ 0xF428, 0x1110,0x1172,0x11D9 ],
    [ 0xF429, 0x1110,0x1172,0x11E2 ],
    [ 0xF42A, 0x1110,0x1172,0x11F0 ],
    [ 0xF42C, 0x1110,0x1173,0x11D0 ],
    [ 0xF42D, 0x1110,0x1173,0x11D7 ],
    [ 0xF42E, 0x1110,0x1173,0x11D9 ],
    [ 0xF42F, 0x1110,0x1173,0x11E2 ],
    [ 0xF430, 0x1110,0x1173,0x11F0 ],
    [ 0xF433, 0x1110,0x1174,0x11F9 ],
    [ 0xF434, 0x1110,0x1175,0x11D9 ],
    [ 0xF435, 0x1110,0x1175,0x11E6 ],
    [ 0xF436, 0x1110,0x1175,0x11F0 ],
    [ 0xF439, 0x1110,0x1199,0x11AB ],
    [ 0xF43D, 0x1110,0x119E,0x11A8 ],
    [ 0xF43E, 0x1110,0x119E,0x11AB ],
    [ 0xF43F, 0x1110,0x119E,0x11AE ],
    [ 0xF440, 0x1110,0x119E,0x11AF ],
    [ 0xF441, 0x1110,0x119E,0x11B0 ],
    [ 0xF442, 0x1110,0x119E,0x11B3 ],
    [ 0xF443, 0x1110,0x119E,0x11B7 ],
    [ 0xF444, 0x1110,0x119E,0x11B8 ],
    [ 0xF445, 0x1110,0x119E,0x11BA ],
    [ 0xF446, 0x1110,0x119E,0x11BC ],
    [ 0xF447, 0x1110,0x119E,0x11BD ],
    [ 0xF449, 0x1110,0x11A1,0x11A8 ],
    [ 0xF44A, 0x1110,0x11A1,0x11AB ],
    [ 0xF44B, 0x1110,0x11A1,0x11AF ],
    [ 0xF44C, 0x1110,0x11A1,0x11B7 ],
    [ 0xF44D, 0x1110,0x11A1,0x11BA ],
    [ 0xF44E, 0x1110,0x11A1,0x11BC ],
    [ 0xF44F, 0x1110,0x11A1,0x11F0 ],
    [ 0xF450, 0x1110,0x11A1,0x11F9 ],
    [ 0xF453, 0xA979,0x119E,0x11AF ],
    [ 0xF454, 0x1111,0x1161,0x11D0 ],
    [ 0xF455, 0x1111,0x1161,0x11D8 ],
    [ 0xF456, 0x1111,0x1161,0x11D9 ],
    [ 0xF457, 0x1111,0x1161,0x11E2 ],
    [ 0xF458, 0x1111,0x1161,0x11E6 ],
    [ 0xF459, 0x1111,0x1161,0x11E7 ],
    [ 0xF45A, 0x1111,0x1161,0x11F0 ],
    [ 0xF45B, 0x1111,0x1161,0x11F9 ],
    [ 0xF45D, 0x1111,0x1162,0x11F9 ],
    [ 0xF45E, 0x1111,0x1163,0x11E2 ],
    [ 0xF460, 0x1111,0x1165,0x11D0 ],
    [ 0xF461, 0x1111,0x1165,0x11F0 ],
    [ 0xF464, 0x1111,0x117C,0x11AF ],
    [ 0xF465, 0x1111,0x1166,0x11C7 ],
    [ 0xF466, 0x1111,0x1167,0x11C7 ],
    [ 0xF467, 0x1111,0x1167,0x11D9 ],
    [ 0xF468, 0x1111,0x1167,0x11E2 ],
    [ 0xF469, 0x1111,0x1167,0x11F0 ],
    [ 0xF46A, 0x1111,0x1167,0x11F9 ],
    [ 0xF46B, 0x1111,0x1169,0x11D9 ],
    [ 0xF46C, 0x1111,0x1169,0x11E2 ],
    [ 0xF46D, 0x1111,0x1169,0x11F0 ],
    [ 0xF46F, 0x1111,0x116D,0x11E2 ],
    [ 0xF470, 0x1111,0x116D,0x11E6 ],
    [ 0xF472, 0x1111,0x116E,0x11D0 ],
    [ 0xF473, 0x1111,0x116E,0x11D9 ],
    [ 0xF474, 0x1111,0x116E,0x11E2 ],
    [ 0xF475, 0x1111,0x116E,0x11F0 ],
    [ 0xF476, 0x1111,0x116E,0x11F1 ],
    [ 0xF477, 0x1111,0x116F,0x11F9 ],
    [ 0xF47A, 0x1111,0x1171,0x11F9 ],
    [ 0xF47B, 0x1111,0x1172,0x11E2 ],
    [ 0xF47C, 0x1111,0x1172,0x11F0 ],
    [ 0xF47E, 0x1111,0x118E,0x11AB ],
    [ 0xF480, 0x1111,0x1173,0x11D9 ],
    [ 0xF481, 0x1111,0x1173,0x11F0 ],
    [ 0xF483, 0x1111,0x1174,0x11F0 ],
    [ 0xF484, 0x1111,0x1174,0x11F9 ],
    [ 0xF485, 0x1111,0x1175,0x11D9 ],
    [ 0xF486, 0x1111,0x1175,0x11F0 ],
    [ 0xF487, 0x1111,0x1175,0x11F9 ],
    [ 0xF48A, 0x1111,0x1199,0x11AB ],
    [ 0xF48F, 0x1111,0x119E,0x11A8 ],
    [ 0xF490, 0x1111,0x119E,0x11AB ],
    [ 0xF491, 0x1111,0x119E,0x11AE ],
    [ 0xF492, 0x1111,0x119E,0x11AF ],
    [ 0xF493, 0x1111,0x119E,0x11B0 ],
    [ 0xF494, 0x1111,0x119E,0x11B3 ],
    [ 0xF495, 0x1111,0x119E,0x11B6 ],
    [ 0xF496, 0x1111,0x119E,0x11B7 ],
    [ 0xF497, 0x1111,0x119E,0x11B8 ],
    [ 0xF498, 0x1111,0x119E,0x11BA ],
    [ 0xF499, 0x1111,0x119E,0x11E7 ],
    [ 0xF49A, 0x1111,0x119E,0x11BC ],
    [ 0xF49B, 0x1111,0x119E,0x11BD ],
    [ 0xF49C, 0x1111,0x119E,0x11BE ],
    [ 0xF49D, 0x1111,0x119E,0x11C0 ],
    [ 0xF49F, 0x1111,0x119F,0x11AB ],
    [ 0xF4A1, 0x1111,0x11A1,0x11A8 ],
    [ 0xF4A2, 0x1111,0x11A1,0x11AB ],
    [ 0xF4A3, 0x1111,0x11A1,0x11AF ],
    [ 0xF4A4, 0x1111,0x11A1,0x11B7 ],
    [ 0xF4A5, 0x1111,0x11A1,0x11B8 ],
    [ 0xF4A6, 0x1111,0x11A1,0x11BA ],
    [ 0xF4A7, 0x1111,0x11A1,0x11BC ],
    [ 0xF4A8, 0x1111,0x11A1,0x11F0 ],
    [ 0xF4AA, 0x1156,0x1161,0x11AB ],
    [ 0xF4AC, 0xA97A,0x1175,0x11AB ],
    [ 0xF4AE, 0x1157,0x116E,0x11AB ],
    [ 0xF4AF, 0x1112,0x1161,0x11CE ],
    [ 0xF4B0, 0x1112,0x1161,0x11CF ],
    [ 0xF4B1, 0x1112,0x1161,0x11D0 ],
    [ 0xF4B2, 0x1112,0x1161,0xD7D8 ],
    [ 0xF4B3, 0x1112,0x1161,0x11D9 ],
    [ 0xF4B4, 0x1112,0x1161,0xD7E2 ],
    [ 0xF4B5, 0x1112,0x1161,0x11E2 ],
    [ 0xF4B6, 0x1112,0x1161,0x11F0 ],
    [ 0xF4B7, 0x1112,0x1161,0x11F1 ],
    [ 0xF4BA, 0x1112,0x1162,0x11EB ],
    [ 0xF4BB, 0x1112,0x1162,0x11F0 ],
    [ 0xF4BC, 0x1112,0x1162,0xD7F6 ],
    [ 0xF4BD, 0x1112,0x1163,0x11C7 ],
    [ 0xF4BE, 0x1112,0x1163,0x11E2 ],
    [ 0xF4BF, 0x1112,0x1163,0x11F0 ],
    [ 0xF4C0, 0x1112,0x1163,0x11F9 ],
    [ 0xF4C2, 0x1112,0x1164,0x11F0 ],
    [ 0xF4C3, 0x1112,0x1165,0x11D0 ],
    [ 0xF4C4, 0x1112,0x1165,0x11D9 ],
    [ 0xF4C5, 0x1112,0x1165,0x11F0 ],
    [ 0xF4C6, 0x1112,0x1165,0x11F1 ],
    [ 0xF4C7, 0x1112,0x117C,0x11F9 ],
    [ 0xF4C7, 0x1112,0x1165,0x11F9 ],
    [ 0xF4CB, 0x1112,0x117C,0x11AB ],
    [ 0xF4CC, 0x1112,0x117C,0x11B7 ],
    [ 0xF4CD, 0x1112,0x1167,0x11D9 ],
    [ 0xF4CE, 0x1112,0x1167,0x11E2 ],
    [ 0xF4CF, 0x1112,0x1167,0x11F0 ],
    [ 0xF4D0, 0x1112,0x1167,0x11F9 ],
    [ 0xF4D1, 0x1112,0x1168,0x11D9 ],
    [ 0xF4D2, 0x1112,0x1169,0x11C7 ],
    [ 0xF4D3, 0x1112,0x1169,0x11CD ],
    [ 0xF4D4, 0x1112,0x1169,0x11D9 ],
    [ 0xF4D5, 0x1112,0x1169,0x11DA ],
    [ 0xF4D6, 0x1112,0x1169,0x11DD ],
    [ 0xF4D7, 0x1112,0x1169,0x11E2 ],
    [ 0xF4D8, 0x1112,0x1169,0x11EB ],
    [ 0xF4D9, 0x1112,0x1169,0x11F0 ],
    [ 0xF4DA, 0x1112,0x1169,0x11F1 ],
    [ 0xF4DB, 0x1112,0x116A,0x11D9 ],
    [ 0xF4DC, 0x1112,0x116A,0x11F0 ],
    [ 0xF4DD, 0x1112,0x116A,0x11F1 ],
    [ 0xF4DE, 0x1112,0x116B,0x11F9 ],
    [ 0xF4E1, 0x1112,0x117F,0x11AB ],
    [ 0xF4E2, 0x1112,0x117F,0x11F9 ],
    [ 0xF4E5, 0x1112,0x116C,0x11F0 ],
    [ 0xF4E6, 0x1112,0x116C,0x11F9 ],
    [ 0xF4E7, 0x1112,0x116D,0x11D9 ],
    [ 0xF4E8, 0x1112,0x116D,0x11E2 ],
    [ 0xF4EA, 0x1112,0xD7B2,0x11AB ],
    [ 0xF4EC, 0x1112,0x1184,0x11BC ],
    [ 0xF4EF, 0x1112,0x1188,0x11AB ],
    [ 0xF4F0, 0x1112,0x116E,0x11D9 ],
    [ 0xF4F1, 0x1112,0x116E,0x11DD ],
    [ 0xF4F2, 0x1112,0x116E,0x11E2 ],
    [ 0xF4F3, 0x1112,0x116E,0x11EB ],
    [ 0xF4F4, 0x1112,0x116E,0x11F0 ],
    [ 0xF4F5, 0x1112,0x116E,0x11F9 ],
    [ 0xF4F7, 0x1112,0x1189,0x11AB ],
    [ 0xF4F8, 0x1112,0x1189,0x11BC ],
    [ 0xF4FA, 0x1112,0x116F,0x11D9 ],
    [ 0xF4FB, 0x1112,0x116F,0x11F0 ],
    [ 0xF4FC, 0x1112,0x116F,0x11F9 ],
    [ 0xF4FE, 0x1112,0x118B,0x11AB ],
    [ 0xF4FF, 0x1112,0x118B,0x11AF ],
    [ 0xF502, 0x1112,0x1171,0x11F0 ],
    [ 0xF503, 0x1112,0x1171,0x11F9 ],
    [ 0xF504, 0x1112,0x1172,0x11D9 ],
    [ 0xF505, 0x1112,0x1172,0x11E2 ],
    [ 0xF506, 0x1112,0x1172,0x11F0 ],
    [ 0xF508, 0x1112,0x118E,0x11AB ],
    [ 0xF50A, 0x1112,0x1191,0x11A8 ],
    [ 0xF50B, 0x1112,0x1191,0x11AB ],
    [ 0xF50C, 0x1112,0x1191,0x11AF ],
    [ 0xF50D, 0x1112,0x1191,0x11D9 ],
    [ 0xF50E, 0x1112,0x1191,0x11BC ],
    [ 0xF50F, 0x1112,0x1191,0x11F0 ],
    [ 0xF510, 0x1112,0x1191,0x11F9 ],
    [ 0xF512, 0x1112,0x1192,0x11AF ],
    [ 0xF513, 0x1112,0x1192,0x11BC ],
    [ 0xF515, 0x1112,0x1194,0x11BC ],
    [ 0xF516, 0x1112,0x1173,0x11C3 ],
    [ 0xF517, 0x1112,0x1173,0xD7D4 ],
    [ 0xF518, 0x1112,0x1173,0x11D0 ],
    [ 0xF519, 0x1112,0x1173,0x11D9 ],
    [ 0xF51A, 0x1112,0x1173,0x11E2 ],
    [ 0xF51B, 0x1112,0x1173,0x11F0 ],
    [ 0xF520, 0x1112,0x1174,0x11EB ],
    [ 0xF521, 0x1112,0x1174,0x11F0 ],
    [ 0xF522, 0x1112,0x1174,0x11F9 ],
    [ 0xF523, 0x1112,0x1175,0x11FF ],
    [ 0xF524, 0x1112,0x1175,0x11D9 ],
    [ 0xF525, 0x1112,0x1175,0x11DC ],
    [ 0xF526, 0x1112,0x1175,0xD7E1 ],
    [ 0xF527, 0x1112,0x1175,0x11DD ],
    [ 0xF528, 0x1112,0x1175,0x11DE ],
    [ 0xF529, 0x1112,0x1175,0x11E2 ],
    [ 0xF52A, 0x1112,0x1175,0x11EE ],
    [ 0xF52B, 0x1112,0x1175,0x11F0 ],
    [ 0xF52C, 0x1112,0x1175,0x11F1 ],
    [ 0xF52F, 0x1112,0x1199,0x11AB ],
    [ 0xF530, 0x1112,0x1199,0x11BC ],
    [ 0xF536, 0x1112,0xD7C3,0x11AB ],
    [ 0xF538, 0x1112,0x119E,0x11A8 ],
    [ 0xF539, 0x1112,0x119E,0x11C3 ],
    [ 0xF53A, 0x1112,0x119E,0x11AB ],
    [ 0xF53B, 0x1112,0x119E,0x11AE ],
    [ 0xF53C, 0x1112,0x119E,0x11AF ],
    [ 0xF53D, 0x1112,0x119E,0x11B0 ],
    [ 0xF53E, 0x1112,0x119E,0x11CD ],
    [ 0xF53F, 0x1112,0x119E,0x11CE ],
    [ 0xF540, 0x1112,0x119E,0x11D0 ],
    [ 0xF541, 0x1112,0x119E,0x11B1 ],
    [ 0xF542, 0x1112,0x119E,0x11B3 ],
    [ 0xF543, 0x1112,0x119E,0x11B6 ],
    [ 0xF544, 0x1112,0x119E,0x11D9 ],
    [ 0xF545, 0x1112,0x119E,0x11B7 ],
    [ 0xF546, 0x1112,0x119E,0x11B8 ],
    [ 0xF547, 0x1112,0x119E,0x11E6 ],
    [ 0xF548, 0x1112,0x119E,0x11BA ],
    [ 0xF549, 0x1112,0x119E,0x11BC ],
    [ 0xF54A, 0x1112,0x119E,0x11F0 ],
    [ 0xF54B, 0x1112,0x119E,0x11C0 ],
    [ 0xF54C, 0x1112,0x119E,0x11C2 ],
    [ 0xF54D, 0x1112,0x119E,0x11F9 ],
    [ 0xF54F, 0x1112,0x119F,0x11AB ],
    [ 0xF551, 0x1112,0x11A1,0x11A8 ],
    [ 0xF552, 0x1112,0x11A1,0x11AB ],
    [ 0xF553, 0x1112,0x11A1,0x11AE ],
    [ 0xF554, 0x1112,0x11A1,0x11AF ],
    [ 0xF555, 0x1112,0x11A1,0x11B0 ],
    [ 0xF556, 0x1112,0x11A1,0x11D9 ],
    [ 0xF557, 0x1112,0x11A1,0x11B7 ],
    [ 0xF558, 0x1112,0x11A1,0x11B8 ],
    [ 0xF559, 0x1112,0x11A1,0xD7E8 ],
    [ 0xF55A, 0x1112,0x11A1,0x11BA ],
    [ 0xF55B, 0x1112,0x11A1,0x11BC ],
    [ 0xF55C, 0x1112,0x11A1,0x11F0 ],
    [ 0xF55E, 0xA97B,0x1163,0x11BC ],
    [ 0xF561, 0xA97B,0x116A,0x11AB ],
    [ 0xF567, 0xA97B,0x1172,0x11BC ],
    [ 0xF569, 0xA97B,0x118E,0x11AB ],
    [ 0xF56D, 0xA97B,0x1194,0x11AB ],
    [ 0xF56F, 0xA97B,0x1175,0x11AB ],
    [ 0xF570, 0xA97B,0x1175,0x11BC ],
    [ 0xF572, 0x1158,0x1161,0x11A8 ],
    [ 0xF573, 0x1158,0x1161,0x11AB ],
    [ 0xF574, 0x1158,0x1161,0x11AF ],
    [ 0xF575, 0x1158,0x1161,0x11D9 ],
    [ 0xF576, 0x1158,0x1161,0x11B7 ],
    [ 0xF577, 0x1158,0x1161,0x11E2 ],
    [ 0xF578, 0x1158,0x1161,0x11B8 ],
    [ 0xF579, 0x1158,0x1161,0x11E6 ],
    [ 0xF57A, 0x1158,0x1161,0x11BC ],
    [ 0xF57B, 0x1158,0x1161,0x11F0 ],
    [ 0xF57D, 0x1158,0x1162,0x11BC ],
    [ 0xF57E, 0x1158,0x1162,0x11F0 ],
    [ 0xF580, 0x1158,0x1163,0x11AB ],
    [ 0xF581, 0x1158,0x1163,0x11B7 ],
    [ 0xF582, 0x1158,0x1163,0x11E2 ],
    [ 0xF583, 0x1158,0x1163,0x11B8 ],
    [ 0xF584, 0x1158,0x1163,0x11E6 ],
    [ 0xF585, 0x1158,0x1163,0x11F0 ],
    [ 0xF586, 0x1158,0x1163,0x11F9 ],
    [ 0xF589, 0x1158,0x1165,0x11AB ],
    [ 0xF58A, 0x1158,0x1165,0x11AF ],
    [ 0xF58B, 0x1158,0x1165,0x11B7 ],
    [ 0xF58C, 0x1158,0x1165,0x11E2 ],
    [ 0xF58D, 0x1158,0x1165,0x11F9 ],
    [ 0xF58F, 0x1158,0x1166,0x11BC ],
    [ 0xF591, 0x1158,0x1167,0x11A8 ],
    [ 0xF592, 0x1158,0x1167,0x11AB ],
    [ 0xF593, 0x1158,0x1167,0x11AF ],
    [ 0xF594, 0x1158,0x1167,0x11D9 ],
    [ 0xF595, 0x1158,0x1167,0x11B7 ],
    [ 0xF596, 0x1158,0x1167,0x11B8 ],
    [ 0xF597, 0x1158,0x1167,0x11F0 ],
    [ 0xF598, 0x1158,0x1167,0x11F9 ],
    [ 0xF59A, 0x1158,0x1168,0x11BC ],
    [ 0xF59C, 0x1158,0x1169,0x11A8 ],
    [ 0xF59D, 0x1158,0x1169,0x11AB ],
    [ 0xF59E, 0x1158,0x1169,0x11AF ],
    [ 0xF59F, 0x1158,0x1169,0x11D9 ],
    [ 0xF5A0, 0x1158,0x1169,0x11B7 ],
    [ 0xF5A1, 0x1158,0x1169,0x11E2 ],
    [ 0xF5A2, 0x1158,0x1169,0x11BC ],
    [ 0xF5A3, 0x1158,0x1169,0x11F0 ],
    [ 0xF5A5, 0x1158,0x116A,0x11A8 ],
    [ 0xF5A6, 0x1158,0x116A,0x11AB ],
    [ 0xF5A7, 0x1158,0x116A,0x11AF ],
    [ 0xF5A8, 0x1158,0x116A,0x11D9 ],
    [ 0xF5A9, 0x1158,0x116A,0x11BC ],
    [ 0xF5AA, 0x1158,0x116A,0x11F0 ],
    [ 0xF5AB, 0x1158,0x116A,0x11F9 ],
    [ 0xF5AD, 0x1158,0x116B,0x11BC ],
    [ 0xF5AF, 0x1158,0x116C,0x11A8 ],
    [ 0xF5B0, 0x1158,0x116C,0x11BC ],
    [ 0xF5B1, 0x1158,0x116C,0x11F0 ],
    [ 0xF5B3, 0x1158,0x116D,0x11B7 ],
    [ 0xF5B4, 0x1158,0x116D,0x11E2 ],
    [ 0xF5B6, 0x1158,0x116E,0x11AB ],
    [ 0xF5B7, 0x1158,0x116E,0x11B7 ],
    [ 0xF5B8, 0x1158,0x116E,0x11E2 ],
    [ 0xF5B9, 0x1158,0x116E,0x11B8 ],
    [ 0xF5BA, 0x1158,0x116E,0x11E6 ],
    [ 0xF5BB, 0x1158,0x116E,0x11BC ],
    [ 0xF5BC, 0x1158,0x116E,0x11F0 ],
    [ 0xF5BD, 0x1158,0x116E,0x11F9 ],
    [ 0xF5BF, 0x1158,0x116F,0x11AB ],
    [ 0xF5C0, 0x1158,0x116F,0x11F9 ],
    [ 0xF5C2, 0x1158,0x1171,0x11F9 ],
    [ 0xF5C4, 0x1158,0x1172,0x11B7 ],
    [ 0xF5C6, 0x1158,0x118F,0x11AB ],
    [ 0xF5C8, 0x1158,0x1191,0x11AB ],
    [ 0xF5C9, 0x1158,0x1191,0x11AF ],
    [ 0xF5CA, 0x1158,0x1191,0x11D9 ],
    [ 0xF5CB, 0x1158,0x1191,0x11BC ],
    [ 0xF5CC, 0x1158,0x1191,0x11F0 ],
    [ 0xF5CE, 0x1158,0x1192,0x11BC ],
    [ 0xF5D0, 0x1158,0x1173,0x11A8 ],
    [ 0xF5D1, 0x1158,0x1173,0x11AB ],
    [ 0xF5D2, 0x1158,0x1173,0x11B7 ],
    [ 0xF5D3, 0x1158,0x1173,0x11E2 ],
    [ 0xF5D4, 0x1158,0x1173,0x11BC ],
    [ 0xF5D5, 0x1158,0x1173,0x11F0 ],
    [ 0xF5D8, 0x1158,0x1174,0x11A8 ],
    [ 0xF5D9, 0x1158,0x1174,0x11F0 ],
    [ 0xF5DA, 0x1158,0x1174,0x11C2 ],
    [ 0xF5DB, 0x1158,0x1174,0x11F9 ],
    [ 0xF5DD, 0x1158,0x1175,0x11AB ],
    [ 0xF5DE, 0x1158,0x1175,0x11B7 ],
    [ 0xF5DF, 0x1158,0x1175,0x11E2 ],
    [ 0xF5E0, 0x1158,0x1175,0x11F0 ],
    [ 0xF5E1, 0x1158,0x1175,0x11F9 ],
    [ 0xF5E3, 0x1158,0x119E,0x11AB ],
    [ 0xF5E5, 0x1158,0x11A1,0x11A8 ],
    [ 0xF5E6, 0x1158,0x11A1,0x11BC ],
    [ 0xF5E7, 0x1158,0x11A1,0x11F0 ],
    [ 0xF5E9, 0x1159,0x1161,0x11A8 ],
    [ 0xF5EA, 0x1159,0x1161,0x11AB ],
    [ 0xF5EB, 0x1159,0x1161,0x11AF ],
    [ 0xF5EC, 0x1159,0x1161,0x11D9 ],
    [ 0xF5ED, 0x1159,0x1161,0x11B7 ],
    [ 0xF5EE, 0x1159,0x1161,0x11E2 ],
    [ 0xF5EF, 0x1159,0x1161,0x11B8 ],
    [ 0xF5F0, 0x1159,0x1161,0x11E6 ],
    [ 0xF5F1, 0x1159,0x1161,0x11BC ],
    [ 0xF5F2, 0x1159,0x1161,0x11F0 ],
    [ 0xF5F4, 0x1159,0x1162,0x11BC ],
    [ 0xF5F5, 0x1159,0x1162,0x11F0 ],
    [ 0xF5F7, 0x1159,0x1163,0x11A8 ],
    [ 0xF5F8, 0x1159,0x1163,0x11AB ],
    [ 0xF5F9, 0x1159,0x1163,0x11B7 ],
    [ 0xF5FA, 0x1159,0x1163,0x11E2 ],
    [ 0xF5FB, 0x1159,0x1163,0x11B8 ],
    [ 0xF5FC, 0x1159,0x1163,0x11E6 ],
    [ 0xF5FD, 0x1159,0x1163,0x11F0 ],
    [ 0xF5FE, 0x1159,0x1163,0x11F9 ],
    [ 0xF601, 0x1159,0x1165,0x11AB ],
    [ 0xF602, 0x1159,0x1165,0x11AF ],
    [ 0xF603, 0x1159,0x1165,0x11D9 ],
    [ 0xF604, 0x1159,0x1165,0x11B7 ],
    [ 0xF605, 0x1159,0x1165,0x11B8 ],
    [ 0xF606, 0x1159,0x1165,0x11BC ],
    [ 0xF608, 0x1159,0x1166,0x11BC ],
    [ 0xF60A, 0x1159,0x1167,0x11A8 ],
    [ 0xF60B, 0x1159,0x1167,0x11AB ],
    [ 0xF60C, 0x1159,0x1167,0x11AF ],
    [ 0xF60D, 0x1159,0x1167,0x11D9 ],
    [ 0xF60E, 0x1159,0x1167,0x11B7 ],
    [ 0xF60F, 0x1159,0x1167,0x11B8 ],
    [ 0xF610, 0x1159,0x1167,0x11F0 ],
    [ 0xF611, 0x1159,0x1167,0x11C2 ],
    [ 0xF613, 0x1159,0x1168,0x11BC ],
    [ 0xF615, 0x1159,0x1169,0x11A8 ],
    [ 0xF616, 0x1159,0x1169,0x11AB ],
    [ 0xF617, 0x1159,0x1169,0x11B7 ],
    [ 0xF618, 0x1159,0x1169,0x11E2 ],
    [ 0xF619, 0x1159,0x1169,0x11BC ],
    [ 0xF61A, 0x1159,0x1169,0x11F0 ],
    [ 0xF61C, 0x1159,0x116A,0x11A8 ],
    [ 0xF61D, 0x1159,0x116A,0x11AB ],
    [ 0xF61E, 0x1159,0x116A,0x11AF ],
    [ 0xF61F, 0x1159,0x116A,0x11D9 ],
    [ 0xF620, 0x1159,0x116A,0x11B8 ],
    [ 0xF621, 0x1159,0x116A,0x11E6 ],
    [ 0xF622, 0x1159,0x116A,0x11BC ],
    [ 0xF623, 0x1159,0x116A,0x11F0 ],
    [ 0xF624, 0x1159,0x116A,0x11F9 ],
    [ 0xF626, 0x1159,0x116B,0x11BC ],
    [ 0xF628, 0x1159,0x116C,0x11A8 ],
    [ 0xF629, 0x1159,0x116C,0x11BC ],
    [ 0xF62A, 0x1159,0x116C,0x11F0 ],
    [ 0xF62C, 0x1159,0x116D,0x11A8 ],
    [ 0xF62D, 0x1159,0x116D,0x11B7 ],
    [ 0xF62E, 0x1159,0x116D,0x11E2 ],
    [ 0xF62F, 0x1159,0x116D,0x11BC ],
    [ 0xF631, 0x1159,0x116E,0x11A8 ],
    [ 0xF632, 0x1159,0x116E,0x11AB ],
    [ 0xF633, 0x1159,0x116E,0x11AF ],
    [ 0xF634, 0x1159,0x116E,0x11D9 ],
    [ 0xF635, 0x1159,0x116E,0x11B7 ],
    [ 0xF636, 0x1159,0x116E,0x11E2 ],
    [ 0xF637, 0x1159,0x116E,0x11BC ],
    [ 0xF638, 0x1159,0x116E,0x11F0 ],
    [ 0xF639, 0x1159,0x116E,0x11F9 ],
    [ 0xF63B, 0x1159,0x116F,0x11AB ],
    [ 0xF63C, 0x1159,0x116F,0x11AF ],
    [ 0xF63D, 0x1159,0x116F,0x11D9 ],
    [ 0xF63E, 0x1159,0x116F,0x11BC ],
    [ 0xF640, 0x1159,0x1171,0x11BC ],
    [ 0xF642, 0x1159,0x1172,0x11AB ],
    [ 0xF643, 0x1159,0x1172,0x11B7 ],
    [ 0xF644, 0x1159,0x1172,0x11E2 ],
    [ 0xF645, 0x1159,0x1172,0x11F9 ],
    [ 0xF647, 0x1159,0x1191,0x11A8 ],
    [ 0xF648, 0x1159,0x1191,0x11AB ],
    [ 0xF649, 0x1159,0x1191,0x11AF ],
    [ 0xF64A, 0x1159,0x1191,0x11D9 ],
    [ 0xF64B, 0x1159,0x1191,0x11BC ],
    [ 0xF64C, 0x1159,0x1191,0x11F0 ],
    [ 0xF64E, 0x1159,0x1192,0x11BC ],
    [ 0xF650, 0x1159,0x1194,0x11BC ],
    [ 0xF652, 0x1159,0x1173,0x11A8 ],
    [ 0xF653, 0x1159,0x1173,0x11AB ],
    [ 0xF654, 0x1159,0x1173,0x11B7 ],
    [ 0xF655, 0x1159,0x1173,0x11E2 ],
    [ 0xF656, 0x1159,0x1173,0x11B8 ],
    [ 0xF657, 0x1159,0x1173,0x11F0 ],
    [ 0xF658, 0x1159,0x1173,0x11F9 ],
    [ 0xF65A, 0x1159,0x1174,0x11AB ],
    [ 0xF65B, 0x1159,0x1174,0x11BC ],
    [ 0xF65D, 0x1159,0x1175,0x11AB ],
    [ 0xF65E, 0x1159,0x1175,0x11AF ],
    [ 0xF65F, 0x1159,0x1175,0x11D9 ],
    [ 0xF660, 0x1159,0x1175,0x11B7 ],
    [ 0xF661, 0x1159,0x1175,0x11E2 ],
    [ 0xF662, 0x1159,0x1175,0x11BC ],
    [ 0xF663, 0x1159,0x1175,0x11F0 ],
    [ 0xF664, 0x1159,0x1175,0x11F9 ],
    [ 0xF666, 0x1159,0x119E,0x11AB ],
    [ 0xF667, 0x1159,0x119E,0x11B7 ],
    [ 0xF669, 0x1159,0x11A1,0x11A8 ],
    [ 0xF66A, 0x1159,0x11A1,0x11AF ],
    [ 0xF66B, 0x1159,0x11A1,0x11CD ],
    [ 0xF66C, 0x1159,0x11A1,0x11B1 ],
    [ 0xF66D, 0x1159,0x11A1,0x11BC ],
    [ 0xF66E, 0x1159,0x11A1,0x11F0 ],
    [ 0xE0C6, 0x115F,0x11A3 ],
    [ 0xE0E9, 0x1100,0x1176 ],
    [ 0xE0EA, 0x1100,0x1177 ],
    [ 0xE0F2, 0x1100,0x1178 ],
    [ 0xE100, 0x1100,0x117C ],
    [ 0xE124, 0x1100,0x11A6 ],
    [ 0xE126, 0x1100,0x117F ],
    [ 0xE129, 0x1100,0x1182 ],
    [ 0xE12F, 0x1100,0xD7B2 ],
    [ 0xE130, 0x1100,0xD7B3 ],
    [ 0xE132, 0x1100,0x1184 ],
    [ 0xE133, 0x1100,0x1185 ],
    [ 0xE136, 0x1100,0x1188 ],
    [ 0xE146, 0x1100,0x1189 ],
    [ 0xE14A, 0x1100,0x118A ],
    [ 0xE14E, 0x1100,0x118B ],
    [ 0xE150, 0x1100,0xD7B5 ],
    [ 0xE151, 0x1100,0x118C ],
    [ 0xE152, 0x1100,0x118D ],
    [ 0xE15A, 0x1100,0x118E ],
    [ 0xE15C, 0x1100,0x118F ],
    [ 0xE15E, 0x1100,0x1191 ],
    [ 0xE166, 0x1100,0x1192 ],
    [ 0xE16A, 0x1100,0x1194 ],
    [ 0xE181, 0x1100,0xD7BA ],
    [ 0xE182, 0x1100,0xD7BB ],
    [ 0xE183, 0x1100,0x1195 ],
    [ 0xE184, 0x1100,0x1196 ],
    [ 0xE19A, 0x1100,0x1198 ],
    [ 0xE19B, 0x1100,0x1199 ],
    [ 0xE19E, 0x1100,0xD7BF ],
    [ 0xE19F, 0x1100,0xD7C0 ],
    [ 0xE1A0, 0x1100,0xD7C2 ],
    [ 0xE1A1, 0x1100,0x119B ],
    [ 0xE1A2, 0x1100,0xD7C3 ],
    [ 0xE1A4, 0x1100,0x119C ],
    [ 0xE1A5, 0x1100,0xD7C4 ],
    [ 0xE1A6, 0x1100,0x119D ],
    [ 0xE1A7, 0x1100,0x119E ],
    [ 0xE1CA, 0x1100,0x119F ],
    [ 0xE1CD, 0x1100,0x11A1 ],
    [ 0xE1D8, 0x1101,0x1176 ],
    [ 0xE1DF, 0x1101,0x117B ],
    [ 0xE1E6, 0x1101,0x1188 ],
    [ 0xE1EB, 0x1101,0x1189 ],
    [ 0xE1EE, 0x1101,0xD7B5 ],
    [ 0xE1EF, 0x1101,0x118C ],
    [ 0xE1F3, 0x1101,0x1191 ],
    [ 0xE1F8, 0x1101,0x1194 ],
    [ 0xE201, 0x1101,0x1198 ],
    [ 0xE202, 0x1101,0x119E ],
    [ 0xE203, 0x1101,0x11A1 ],
    [ 0xE204, 0x115A,0x1165 ],
    [ 0xE205, 0x115A,0x1173 ],
    [ 0xE21C, 0x1102,0x1176 ],
    [ 0xE21D, 0x1102,0x1177 ],
    [ 0xE224, 0x1102,0x1178 ],
    [ 0xE22E, 0x1102,0x117B ],
    [ 0xE230, 0x1102,0x117A ],
    [ 0xE230, 0x1102,0x117C ],
    [ 0xE24A, 0x1102,0x117F ],
    [ 0xE24F, 0x1102,0xD7B2 ],
    [ 0xE251, 0x1102,0x1188 ],
    [ 0xE25A, 0x1102,0x1189 ],
    [ 0xE25C, 0x1102,0xD7B5 ],
    [ 0xE25D, 0x1102,0x118C ],
    [ 0xE25F, 0x1102,0xD7B6 ],
    [ 0xE263, 0x1102,0x118F ],
    [ 0xE264, 0x1102,0x1194 ],
    [ 0xE26C, 0x1102,0x1196 ],
    [ 0xE26E, 0x1102,0x1197 ],
    [ 0xE27C, 0x1102,0x1198 ],
    [ 0xE27D, 0x1102,0x1199 ],
    [ 0xE280, 0x1102,0xD7BF ],
    [ 0xE281, 0x1102,0xD7C0 ],
    [ 0xE282, 0x1102,0x119B ],
    [ 0xE283, 0x1102,0x119E ],
    [ 0xE299, 0x1102,0x11A1 ],
    [ 0xE2A5, 0x1113,0x1161 ],
    [ 0xE2A6, 0x1113,0x1162 ],
    [ 0xE2A7, 0x1113,0x1163 ],
    [ 0xE2A8, 0x1113,0x1165 ],
    [ 0xE2A9, 0x1113,0x1166 ],
    [ 0xE2AA, 0x1113,0x1167 ],
    [ 0xE2AB, 0x1113,0x1168 ],
    [ 0xE2AD, 0x1113,0x1169 ],
    [ 0xE2AE, 0x1113,0x116A ],
    [ 0xE2AF, 0x1113,0x116B ],
    [ 0xE2B0, 0x1113,0x116C ],
    [ 0xE2B1, 0x1113,0x116D ],
    [ 0xE2B2, 0x1113,0x116E ],
    [ 0xE2B3, 0x1113,0x116F ],
    [ 0xE2B4, 0x1113,0x1170 ],
    [ 0xE2B5, 0x1113,0xD7B5 ],
    [ 0xE2B6, 0x1113,0x118C ],
    [ 0xE2B7, 0x1113,0x1171 ],
    [ 0xE2B8, 0x1113,0x1172 ],
    [ 0xE2B9, 0x1113,0x1194 ],
    [ 0xE2BA, 0x1113,0x1173 ],
    [ 0xE2BB, 0x1113,0x1174 ],
    [ 0xE2BC, 0x1113,0x1175 ],
    [ 0xE2BD, 0x1113,0x1198 ],
    [ 0xE2BE, 0x1113,0x119E ],
    [ 0xE2BF, 0x1113,0x11A1 ],
    [ 0xE2C0, 0x1114,0x1161 ],
    [ 0xE2C1, 0x1114,0x1163 ],
    [ 0xE2C3, 0x1114,0x1165 ],
    [ 0xE2C4, 0x1114,0x1166 ],
    [ 0xE2C5, 0x1114,0x1169 ],
    [ 0xE2C6, 0x1114,0x116E ],
    [ 0xE2C7, 0x1114,0x1173 ],
    [ 0xE2C9, 0x1114,0x119E ],
    [ 0xE2CB, 0x1115,0x1161 ],
    [ 0xE2CD, 0x1115,0x1162 ],
    [ 0xE2CE, 0x1115,0x1163 ],
    [ 0xE2CF, 0x1115,0x1165 ],
    [ 0xE2D0, 0x1115,0x1166 ],
    [ 0xE2D2, 0x1115,0x1167 ],
    [ 0xE2D3, 0x1115,0x1168 ],
    [ 0xE2D4, 0x1115,0x1169 ],
    [ 0xE2D6, 0x1115,0x116A ],
    [ 0xE2D7, 0x1115,0x116B ],
    [ 0xE2D8, 0x1115,0x116C ],
    [ 0xE2D9, 0x1115,0x116D ],
    [ 0xE2DA, 0x1115,0x116E ],
    [ 0xE2DB, 0x1115,0x116F ],
    [ 0xE2DC, 0x1115,0x1170 ],
    [ 0xE2DD, 0x1115,0xD7B5 ],
    [ 0xE2DE, 0x1115,0x118C ],
    [ 0xE2DF, 0x1115,0x1171 ],
    [ 0xE2E0, 0x1115,0x1172 ],
    [ 0xE2E1, 0x1115,0x1194 ],
    [ 0xE2E2, 0x1115,0x1173 ],
    [ 0xE2E3, 0x1115,0x1174 ],
    [ 0xE2E4, 0x1115,0x1175 ],
    [ 0xE2E5, 0x1115,0x1198 ],
    [ 0xE2E6, 0x1115,0x119E ],
    [ 0xE2E7, 0x1115,0x11A1 ],
    [ 0xE2E8, 0x1116,0x1161 ],
    [ 0xE2E9, 0x1116,0x1162 ],
    [ 0xE2EA, 0x1116,0x1163 ],
    [ 0xE2EB, 0x1116,0x1165 ],
    [ 0xE2EC, 0x1116,0x1166 ],
    [ 0xE2ED, 0x1116,0x1167 ],
    [ 0xE2EE, 0x1116,0x1168 ],
    [ 0xE2EF, 0x1116,0x1169 ],
    [ 0xE2F0, 0x1116,0x116A ],
    [ 0xE2F1, 0x1116,0x116B ],
    [ 0xE2F2, 0x1116,0x116C ],
    [ 0xE2F3, 0x1116,0x116D ],
    [ 0xE2F4, 0x1116,0x116E ],
    [ 0xE2F5, 0x1116,0x116F ],
    [ 0xE2F6, 0x1116,0x1170 ],
    [ 0xE2F7, 0x1116,0xD7B5 ],
    [ 0xE2F8, 0x1116,0x118C ],
    [ 0xE2F9, 0x1116,0x1171 ],
    [ 0xE2FA, 0x1116,0x1172 ],
    [ 0xE2FB, 0x1116,0x1194 ],
    [ 0xE2FC, 0x1116,0x1173 ],
    [ 0xE2FD, 0x1116,0x1174 ],
    [ 0xE2FE, 0x1116,0x1175 ],
    [ 0xE2FF, 0x1116,0x1198 ],
    [ 0xE300, 0x1116,0x119E ],
    [ 0xE301, 0x1116,0x11A1 ],
    [ 0xE302, 0x115B,0x1175 ],
    [ 0xE303, 0x115B,0x1198 ],
    [ 0xE304, 0x115C,0x1173 ],
    [ 0xE305, 0x115C,0x1175 ],
    [ 0xE306, 0x115D,0x1163 ],
    [ 0xE308, 0x115D,0x1169 ],
    [ 0xE309, 0x115D,0x1172 ],
    [ 0xE30B, 0x115D,0x1175 ],
    [ 0xE31F, 0x1103,0x1176 ],
    [ 0xE325, 0x1103,0x1178 ],
    [ 0xE32B, 0x1103,0x117A ],
    [ 0xE32C, 0x1103,0x117B ],
    [ 0xE32E, 0x1103,0x117C ],
    [ 0xE348, 0x1103,0x1182 ],
    [ 0xE34D, 0x1103,0x1188 ],
    [ 0xE356, 0x1103,0x1189 ],
    [ 0xE359, 0x1103,0xD7B5 ],
    [ 0xE35A, 0x1103,0x118C ],
    [ 0xE361, 0x1103,0x1191 ],
    [ 0xE366, 0x1103,0x1192 ],
    [ 0xE368, 0x1103,0x1194 ],
    [ 0xE376, 0x1103,0x1195 ],
    [ 0xE377, 0x1103,0x1196 ],
    [ 0xE383, 0x1103,0x1198 ],
    [ 0xE384, 0x1103,0x1199 ],
    [ 0xE387, 0x1103,0xD7BF ],
    [ 0xE388, 0x1103,0xD7C0 ],
    [ 0xE389, 0x1103,0x119B ],
    [ 0xE38A, 0x1103,0x119E ],
    [ 0xE3A8, 0x1103,0x11A1 ],
    [ 0xE3B4, 0x1117,0x1173 ],
    [ 0xE3B6, 0x1117,0x1175 ],
    [ 0xE3C1, 0x1104,0x117B ],
    [ 0xE3C2, 0x1104,0x117C ],
    [ 0xE3CD, 0x1104,0x117F ],
    [ 0xE3D2, 0x1104,0xD7B5 ],
    [ 0xE3D3, 0x1104,0x118C ],
    [ 0xE3D8, 0x1104,0x1191 ],
    [ 0xE3DA, 0x1104,0x1194 ],
    [ 0xE3E4, 0x1104,0x1198 ],
    [ 0xE3E5, 0x1104,0x119E ],
    [ 0xE3EB, 0x1104,0x11A1 ],
    [ 0xE3F0, 0x115E,0x1175 ],
    [ 0xE3F3, 0xA960,0x1161 ],
    [ 0xE3F5, 0xA961,0x1161 ],
    [ 0xE3F6, 0xA961,0x1162 ],
    [ 0xE3F7, 0xA961,0x116E ],
    [ 0xE3F9, 0xA962,0x1172 ],
    [ 0xE3FC, 0xA962,0x1173 ],
    [ 0xE3FD, 0xA962,0x1175 ],
    [ 0xE3FF, 0xA963,0x1163 ],
    [ 0xE401, 0xA963,0x119E ],
    [ 0xE40F, 0x1105,0x1176 ],
    [ 0xE411, 0x1105,0x1177 ],
    [ 0xE415, 0x1105,0x1178 ],
    [ 0xE41B, 0x1105,0x117A ],
    [ 0xE41C, 0x1105,0x117B ],
    [ 0xE430, 0x1105,0x1182 ],
    [ 0xE435, 0x1105,0x1186 ],
    [ 0xE436, 0x1105,0x1187 ],
    [ 0xE437, 0x1105,0x1188 ],
    [ 0xE43E, 0x1105,0x1189 ],
    [ 0xE440, 0x1105,0xD7B5 ],
    [ 0xE441, 0x1105,0x118C ],
    [ 0xE442, 0x1105,0x118D ],
    [ 0xE447, 0x1105,0x1191 ],
    [ 0xE44B, 0x1105,0x1194 ],
    [ 0xE455, 0x1105,0x1195 ],
    [ 0xE456, 0x1105,0x1196 ],
    [ 0xE45F, 0x1105,0x1198 ],
    [ 0xE460, 0x1105,0x1199 ],
    [ 0xE463, 0x1105,0xD7BD ],
    [ 0xE464, 0x1105,0xD7BF ],
    [ 0xE465, 0x1105,0xD7C0 ],
    [ 0xE466, 0x1105,0x119A ],
    [ 0xE468, 0x1105,0xD7C2 ],
    [ 0xE469, 0x1105,0x119B ],
    [ 0xE46B, 0x1105,0xD7C4 ],
    [ 0xE46C, 0x1105,0x119E ],
    [ 0xE486, 0x1105,0xD7C5 ],
    [ 0xE487, 0x1105,0x11A1 ],
    [ 0xE490, 0xA964,0x1161 ],
    [ 0xE491, 0xA964,0x1169 ],
    [ 0xE493, 0xA965,0x1161 ],
    [ 0xE494, 0x1118,0x1161 ],
    [ 0xE498, 0x1118,0x1165 ],
    [ 0xE499, 0x1118,0x1166 ],
    [ 0xE49C, 0x1118,0x1169 ],
    [ 0xE4A1, 0x1118,0x116D ],
    [ 0xE4A3, 0x1118,0x116E ],
    [ 0xE4A4, 0x1118,0x1172 ],
    [ 0xE4A6, 0x1118,0x1173 ],
    [ 0xE4A7, 0x1118,0x1174 ],
    [ 0xE4AB, 0x1118,0x119E ],
    [ 0xE4AE, 0x1118,0x11A1 ],
    [ 0xE4B1, 0xA967,0x1161 ],
    [ 0xE4B2, 0x1119,0x1161 ],
    [ 0xE4B6, 0x1119,0x1176 ],
    [ 0xE4B7, 0x1119,0x1162 ],
    [ 0xE4B9, 0x1119,0x1163 ],
    [ 0xE4BB, 0x1119,0x1178 ],
    [ 0xE4BC, 0x1119,0x1165 ],
    [ 0xE4BF, 0x1119,0x117B ],
    [ 0xE4C0, 0x1119,0x1166 ],
    [ 0xE4C2, 0x1119,0x1167 ],
    [ 0xE4C4, 0x1119,0x1168 ],
    [ 0xE4C6, 0x1119,0x1169 ],
    [ 0xE4C8, 0x1119,0x116D ],
    [ 0xE4C9, 0x1119,0x116E ],
    [ 0xE4CC, 0x1119,0x1189 ],
    [ 0xE4CE, 0x1119,0x116F ],
    [ 0xE4CF, 0x1119,0x1170 ],
    [ 0xE4D0, 0x1119,0x1171 ],
    [ 0xE4D1, 0x1119,0x1172 ],
    [ 0xE4D2, 0x1119,0x118F ],
    [ 0xE4D3, 0x1119,0x1190 ],
    [ 0xE4D4, 0x1119,0x1173 ],
    [ 0xE4D8, 0x1119,0x1175 ],
    [ 0xE4DB, 0x1119,0x119B ],
    [ 0xE4DC, 0xA968,0x1161 ],
    [ 0xE4DE, 0xA969,0x1161 ],
    [ 0xE4DF, 0xA96A,0x1161 ],
    [ 0xE4E0, 0xA96B,0x119E ],
    [ 0xE4E1, 0xA96B,0x11A1 ],
    [ 0xE4E2, 0xA96C,0x1169 ],
    [ 0xE4E4, 0xA96C,0x1173 ],
    [ 0xE4E5, 0xA96D,0x1165 ],
    [ 0xE4E6, 0xA96D,0x1166 ],
    [ 0xE4E7, 0xA96D,0x1172 ],
    [ 0xE4E9, 0xA96E,0x1161 ],
    [ 0xE4EA, 0x111A,0x1161 ],
    [ 0xE4EC, 0x111A,0x1163 ],
    [ 0xE4EE, 0x111A,0x1169 ],
    [ 0xE4EF, 0x111A,0x116A ],
    [ 0xE4F0, 0x111A,0x1175 ],
    [ 0xE502, 0x1106,0x1176 ],
    [ 0xE503, 0x1106,0x1177 ],
    [ 0xE506, 0x1106,0x1178 ],
    [ 0xE50D, 0x1106,0x117A ],
    [ 0xE50E, 0x1106,0x117C ],
    [ 0xE51F, 0x1106,0x117F ],
    [ 0xE521, 0x1106,0x1182 ],
    [ 0xE522, 0x1106,0xD7B1 ],
    [ 0xE523, 0x1106,0x1183 ],
    [ 0xE525, 0x1106,0x1187 ],
    [ 0xE526, 0x1106,0x1188 ],
    [ 0xE536, 0x1106,0xD7B5 ],
    [ 0xE537, 0x1106,0x118C ],
    [ 0xE538, 0x1106,0x118D ],
    [ 0xE53B, 0x1106,0x1194 ],
    [ 0xE544, 0x1106,0x1195 ],
    [ 0xE545, 0x1106,0x1196 ],
    [ 0xE559, 0x1106,0x1198 ],
    [ 0xE55A, 0x1106,0x1199 ],
    [ 0xE55C, 0x1106,0xD7BF ],
    [ 0xE55D, 0x1106,0xD7C0 ],
    [ 0xE55E, 0x1106,0x119C ],
    [ 0xE560, 0x1106,0x119E ],
    [ 0xE578, 0x1106,0x119F ],
    [ 0xE57B, 0x1106,0x11A1 ],
    [ 0xE58C, 0xA96F,0x1163 ],
    [ 0xE58D, 0xA96F,0x1178 ],
    [ 0xE58E, 0xA96F,0x1167 ],
    [ 0xE58F, 0xA96F,0x1168 ],
    [ 0xE590, 0xA970,0x1167 ],
    [ 0xE592, 0xA970,0x1169 ],
    [ 0xE593, 0x111C,0x1161 ],
    [ 0xE596, 0x111C,0x1162 ],
    [ 0xE597, 0x111C,0x1163 ],
    [ 0xE598, 0x111C,0x1165 ],
    [ 0xE599, 0x111C,0x1166 ],
    [ 0xE59A, 0x111C,0x1167 ],
    [ 0xE59B, 0x111C,0x1168 ],
    [ 0xE59E, 0x111C,0x1169 ],
    [ 0xE5A2, 0x111C,0x116A ],
    [ 0xE5A3, 0x111C,0x116B ],
    [ 0xE5A4, 0x111C,0x116C ],
    [ 0xE5A5, 0x111C,0x116D ],
    [ 0xE5A6, 0x111C,0x116E ],
    [ 0xE5A9, 0x111C,0x116F ],
    [ 0xE5AA, 0x111C,0x1170 ],
    [ 0xE5AB, 0x111C,0xD7B5 ],
    [ 0xE5AC, 0x111C,0x118C ],
    [ 0xE5AD, 0x111C,0x1171 ],
    [ 0xE5AE, 0x111C,0x1172 ],
    [ 0xE5AF, 0x111C,0x1194 ],
    [ 0xE5B0, 0x111C,0x1173 ],
    [ 0xE5B1, 0x111C,0x1174 ],
    [ 0xE5B2, 0x111C,0x1175 ],
    [ 0xE5B4, 0x111C,0x1198 ],
    [ 0xE5B5, 0x111C,0x119E ],
    [ 0xE5B6, 0x111C,0x11A1 ],
    [ 0xE5B7, 0xA971,0x1169 ],
    [ 0xE5B9, 0x111D,0x1161 ],
    [ 0xE5BD, 0x111D,0x1169 ],
    [ 0xE5BE, 0x111D,0x116A ],
    [ 0xE5C1, 0x111D,0x116E ],
    [ 0xE5C5, 0x111D,0x1171 ],
    [ 0xE5C6, 0x111D,0x1172 ],
    [ 0xE5C8, 0x111D,0x1173 ],
    [ 0xE5CC, 0x111D,0x1175 ],
    [ 0xE5E9, 0x1107,0x1176 ],
    [ 0xE5EB, 0x1107,0x1177 ],
    [ 0xE5EF, 0x1107,0x1178 ],
    [ 0xE5F5, 0x1107,0x117B ],
    [ 0xE5F7, 0x1107,0x117C ],
    [ 0xE60C, 0x1107,0x1182 ],
    [ 0xE60D, 0x1107,0xD7B1 ],
    [ 0xE613, 0x1107,0x1187 ],
    [ 0xE614, 0x1107,0x1188 ],
    [ 0xE623, 0x1107,0xD7B5 ],
    [ 0xE624, 0x1107,0x118C ],
    [ 0xE625, 0x1107,0x118D ],
    [ 0xE62A, 0x1107,0x1194 ],
    [ 0xE635, 0x1107,0x1196 ],
    [ 0xE643, 0x1107,0x1198 ],
    [ 0xE644, 0x1107,0x1199 ],
    [ 0xE646, 0x1107,0xD7BF ],
    [ 0xE647, 0x1107,0xD7C0 ],
    [ 0xE648, 0x1107,0x119A ],
    [ 0xE649, 0x1107,0x119B ],
    [ 0xE64A, 0x1107,0x119E ],
    [ 0xE663, 0x1107,0x119F ],
    [ 0xE667, 0x1107,0xD7C6 ],
    [ 0xE668, 0x1107,0x11A1 ],
    [ 0xE679, 0x111E,0x1161 ],
    [ 0xE67C, 0x111E,0x1162 ],
    [ 0xE67E, 0x111E,0x1165 ],
    [ 0xE680, 0x111E,0x1166 ],
    [ 0xE682, 0x111E,0x1167 ],
    [ 0xE685, 0x111E,0x1168 ],
    [ 0xE687, 0x111E,0x1169 ],
    [ 0xE689, 0x111E,0x116C ],
    [ 0xE68B, 0x111E,0x116E ],
    [ 0xE68F, 0x111E,0x116F ],
    [ 0xE690, 0x111E,0x1170 ],
    [ 0xE691, 0x111E,0x1171 ],
    [ 0xE692, 0x111E,0x1173 ],
    [ 0xE697, 0x111E,0x1174 ],
    [ 0xE699, 0x111E,0x1175 ],
    [ 0xE69D, 0x111E,0x119E ],
    [ 0xE69F, 0x111E,0x11A1 ],
    [ 0xE6A2, 0x111F,0x1167 ],
    [ 0xE6A3, 0x1120,0x1161 ],
    [ 0xE6AA, 0x1120,0x1162 ],
    [ 0xE6AB, 0x1120,0x1163 ],
    [ 0xE6AC, 0x1120,0x1165 ],
    [ 0xE6B5, 0x1120,0x1166 ],
    [ 0xE6B8, 0x1120,0x1167 ],
    [ 0xE6BA, 0x1120,0x1168 ],
    [ 0xE6BC, 0x1120,0x1169 ],
    [ 0xE6C1, 0x1120,0x116A ],
    [ 0xE6C2, 0x1120,0x116B ],
    [ 0xE6C3, 0x1120,0x116C ],
    [ 0xE6C5, 0x1120,0x116D ],
    [ 0xE6C6, 0x1120,0x116E ],
    [ 0xE6CE, 0x1120,0x116F ],
    [ 0xE6CF, 0x1120,0x1170 ],
    [ 0xE6D0, 0x1120,0xD7B5 ],
    [ 0xE6D1, 0x1120,0x118C ],
    [ 0xE6D2, 0x1120,0x1171 ],
    [ 0xE6D5, 0x1120,0x1172 ],
    [ 0xE6D6, 0x1120,0x1194 ],
    [ 0xE6D7, 0x1120,0x1173 ],
    [ 0xE6E4, 0x1120,0x1174 ],
    [ 0xE6E7, 0x1120,0x1175 ],
    [ 0xE6EE, 0x1120,0x1198 ],
    [ 0xE6EF, 0x1120,0x119E ],
    [ 0xE6F5, 0x1120,0x11A1 ],
    [ 0xE70F, 0x1108,0xD7B5 ],
    [ 0xE71D, 0x1108,0x119E ],
    [ 0xE721, 0x1108,0x11A1 ],
    [ 0xE725, 0x1121,0x1161 ],
    [ 0xE733, 0x1121,0x1162 ],
    [ 0xE735, 0x1121,0x1163 ],
    [ 0xE73A, 0x1121,0x1164 ],
    [ 0xE73B, 0x1121,0x1165 ],
    [ 0xE741, 0x1121,0x1166 ],
    [ 0xE744, 0x1121,0x1167 ],
    [ 0xE74A, 0x1121,0x1168 ],
    [ 0xE74C, 0x1121,0x1169 ],
    [ 0xE755, 0x1121,0x116A ],
    [ 0xE759, 0x1121,0x116B ],
    [ 0xE75B, 0x1121,0x116C ],
    [ 0xE75C, 0x1121,0x116D ],
    [ 0xE75D, 0x1121,0x116E ],
    [ 0xE766, 0x1121,0x116F ],
    [ 0xE767, 0x1121,0x1170 ],
    [ 0xE768, 0x1121,0xD7B5 ],
    [ 0xE769, 0x1121,0x118C ],
    [ 0xE76A, 0x1121,0x1171 ],
    [ 0xE76B, 0x1121,0x1172 ],
    [ 0xE76F, 0x1121,0x1194 ],
    [ 0xE770, 0x1121,0x1173 ],
    [ 0xE77A, 0x1121,0x1174 ],
    [ 0xE77C, 0x1121,0x1175 ],
    [ 0xE785, 0x1121,0x1198 ],
    [ 0xE786, 0x1121,0x119E ],
    [ 0xE792, 0x1121,0x11A1 ],
    [ 0xE794, 0x1122,0x1161 ],
    [ 0xE795, 0x1122,0x1162 ],
    [ 0xE798, 0x1122,0x1165 ],
    [ 0xE79C, 0x1122,0x1166 ],
    [ 0xE79E, 0x1122,0x1167 ],
    [ 0xE7A1, 0x1122,0x1168 ],
    [ 0xE7A3, 0x1122,0x1169 ],
    [ 0xE7A4, 0x1122,0x116C ],
    [ 0xE7A6, 0x1122,0x116E ],
    [ 0xE7AB, 0x1122,0x116F ],
    [ 0xE7AC, 0x1122,0x1170 ],
    [ 0xE7AD, 0x1122,0x1171 ],
    [ 0xE7AF, 0x1122,0x1173 ],
    [ 0xE7B4, 0x1122,0x1174 ],
    [ 0xE7B9, 0x1122,0x1175 ],
    [ 0xE7BC, 0x1122,0x119E ],
    [ 0xE7BE, 0x1123,0x1161 ],
    [ 0xE7C0, 0x1123,0x1162 ],
    [ 0xE7C1, 0x1123,0x1165 ],
    [ 0xE7C5, 0x1123,0x1167 ],
    [ 0xE7C7, 0x1123,0x1169 ],
    [ 0xE7C8, 0x1123,0x116C ],
    [ 0xE7C9, 0x1123,0x116E ],
    [ 0xE7CC, 0x1123,0x1172 ],
    [ 0xE7CE, 0x1123,0x1173 ],
    [ 0xE7D1, 0x1123,0x1174 ],
    [ 0xE7D2, 0x1123,0x1175 ],
    [ 0xE7D4, 0x1123,0x119E ],
    [ 0xE7D6, 0x1123,0x11A1 ],
    [ 0xE7D7, 0x1124,0x1167 ],
    [ 0xE7D8, 0x1124,0x1168 ],
    [ 0xE7D9, 0x1125,0x1165 ],
    [ 0xE7DA, 0x1125,0x1169 ],
    [ 0xE7DB, 0x1125,0x1173 ],
    [ 0xE7DC, 0x1126,0x119E ],
    [ 0xE7DE, 0xA972,0x1161 ],
    [ 0xE7DF, 0x1127,0x1161 ],
    [ 0xE7E6, 0x1127,0x1162 ],
    [ 0xE7E7, 0x1127,0x1163 ],
    [ 0xE7EB, 0x1127,0x1165 ],
    [ 0xE7ED, 0x1127,0x1166 ],
    [ 0xE7EE, 0x1127,0x1167 ],
    [ 0xE7EF, 0x1127,0x1169 ],
    [ 0xE7F8, 0x1127,0x116A ],
    [ 0xE7FB, 0x1127,0x116B ],
    [ 0xE7FC, 0x1127,0x116C ],
    [ 0xE7FF, 0x1127,0x116D ],
    [ 0xE800, 0x1127,0x116E ],
    [ 0xE804, 0x1127,0x1171 ],
    [ 0xE805, 0x1127,0x1173 ],
    [ 0xE80D, 0x1127,0x1174 ],
    [ 0xE810, 0x1127,0x1175 ],
    [ 0xE81A, 0x1127,0x119E ],
    [ 0xE81F, 0x1127,0x11A1 ],
    [ 0xE823, 0x1128,0x1169 ],
    [ 0xE825, 0x1128,0x119E ],
    [ 0xE827, 0xA973,0x1169 ],
    [ 0xE828, 0xA973,0x116A ],
    [ 0xE829, 0xA973,0x116B ],
    [ 0xE82A, 0xA973,0x1173 ],
    [ 0xE82B, 0x1129,0x1161 ],
    [ 0xE82C, 0x1129,0x1165 ],
    [ 0xE833, 0x1129,0x1167 ],
    [ 0xE834, 0x1129,0x1169 ],
    [ 0xE837, 0x1129,0x116E ],
    [ 0xE83A, 0x1129,0x1171 ],
    [ 0xE83D, 0x1129,0x1173 ],
    [ 0xE841, 0x1129,0x1174 ],
    [ 0xE843, 0x1129,0x1175 ],
    [ 0xE847, 0x1129,0x119E ],
    [ 0xE84A, 0x1129,0x11A1 ],
    [ 0xE84B, 0x112A,0x1161 ],
    [ 0xE84D, 0x112A,0x1169 ],
    [ 0xE84F, 0x112A,0x116D ],
    [ 0xE850, 0x112A,0x1173 ],
    [ 0xE852, 0x112A,0x1175 ],
    [ 0xE853, 0x112A,0x119E ],
    [ 0xE855, 0xA974,0x1161 ],
    [ 0xE857, 0x112B,0x1161 ],
    [ 0xE865, 0x112B,0x1162 ],
    [ 0xE868, 0x112B,0x1165 ],
    [ 0xE86D, 0x112B,0x1166 ],
    [ 0xE86F, 0x112B,0x1167 ],
    [ 0xE874, 0x112B,0x1168 ],
    [ 0xE876, 0x112B,0x1169 ],
    [ 0xE880, 0x112B,0x116A ],
    [ 0xE882, 0x112B,0x116D ],
    [ 0xE883, 0x112B,0x1187 ],
    [ 0xE884, 0x112B,0x1188 ],
    [ 0xE885, 0x112B,0x116E ],
    [ 0xE88F, 0x112B,0x116F ],
    [ 0xE890, 0x112B,0x1171 ],
    [ 0xE893, 0x112B,0x1172 ],
    [ 0xE895, 0x112B,0x1173 ],
    [ 0xE89F, 0x112B,0x1195 ],
    [ 0xE8A0, 0x112B,0x1174 ],
    [ 0xE8A1, 0x112B,0x1175 ],
    [ 0xE8A7, 0x112B,0x119E ],
    [ 0xE8B2, 0x112B,0x11A1 ],
    [ 0xE8B6, 0x112C,0x1161 ],
    [ 0xE8BB, 0x112C,0x1169 ],
    [ 0xE8BD, 0x112C,0x116A ],
    [ 0xE8BF, 0x112C,0x116E ],
    [ 0xE8C4, 0x112C,0x1172 ],
    [ 0xE8C6, 0x112C,0x1173 ],
    [ 0xE8CB, 0x112C,0x1174 ],
    [ 0xE8CC, 0x112C,0x1175 ],
    [ 0xE8DD, 0x1109,0x1176 ],
    [ 0xE8DE, 0x1109,0x1177 ],
    [ 0xE8DF, 0x1109,0x11A3 ],
    [ 0xE8E8, 0x1109,0x1178 ],
    [ 0xE8F2, 0x1109,0x117A ],
    [ 0xE8F3, 0x1109,0x117B ],
    [ 0xE8F4, 0x1109,0x117C ],
    [ 0xE907, 0x1109,0x117D ],
    [ 0xE908, 0x1109,0x117E ],
    [ 0xE916, 0x1109,0x11A6 ],
    [ 0xE917, 0x1109,0x11A7 ],
    [ 0xE918, 0x1109,0x1182 ],
    [ 0xE91E, 0x1109,0x1187 ],
    [ 0xE920, 0x1109,0x1188 ],
    [ 0xE92E, 0x1109,0x1189 ],
    [ 0xE931, 0x1109,0xD7B5 ],
    [ 0xE933, 0x1109,0x118C ],
    [ 0xE93B, 0x1109,0x118E ],
    [ 0xE93E, 0x1109,0xD7B7 ],
    [ 0xE93F, 0x1109,0x118F ],
    [ 0xE940, 0x1109,0x1190 ],
    [ 0xE941, 0x1109,0x1191 ],
    [ 0xE947, 0x1109,0x1192 ],
    [ 0xE949, 0x1109,0xD7B8 ],
    [ 0xE94B, 0x1109,0x1193 ],
    [ 0xE94D, 0x1109,0x1194 ],
    [ 0xE95E, 0x1109,0x1195 ],
    [ 0xE95F, 0x1109,0x1196 ],
    [ 0xE96F, 0x1109,0x1198 ],
    [ 0xE970, 0x1109,0x1199 ],
    [ 0xE973, 0x1109,0xD7BD ],
    [ 0xE974, 0x1109,0xD7BE ],
    [ 0xE975, 0x1109,0xD7BF ],
    [ 0xE976, 0x1109,0xD7C0 ],
    [ 0xE977, 0x1109,0x119A ],
    [ 0xE978, 0x1109,0xD7C1 ],
    [ 0xE979, 0x1109,0xD7C2 ],
    [ 0xE97A, 0x1109,0x119B ],
    [ 0xE97B, 0x1109,0xD7C3 ],
    [ 0xE97D, 0x1109,0x119E ],
    [ 0xE999, 0x1109,0x11A1 ],
    [ 0xE9A5, 0x112D,0x1161 ],
    [ 0xE9B4, 0x112D,0x1162 ],
    [ 0xE9BB, 0x112D,0x1163 ],
    [ 0xE9BD, 0x112D,0x1164 ],
    [ 0xE9BF, 0x112D,0x1165 ],
    [ 0xE9CD, 0x112D,0x1166 ],
    [ 0xE9D5, 0x112D,0x1167 ],
    [ 0xE9E0, 0x112D,0x1168 ],
    [ 0xE9E1, 0x112D,0x1169 ],
    [ 0xE9F0, 0x112D,0x116A ],
    [ 0xE9F6, 0x112D,0x116B ],
    [ 0xE9FA, 0x112D,0x116C ],
    [ 0xEA01, 0x112D,0x116D ],
    [ 0xEA05, 0x112D,0x116E ],
    [ 0xEA16, 0x112D,0x116F ],
    [ 0xEA1D, 0x112D,0x1170 ],
    [ 0xEA20, 0x112D,0xD7B5 ],
    [ 0xEA21, 0x112D,0x118C ],
    [ 0xEA22, 0x112D,0x1171 ],
    [ 0xEA27, 0x112D,0x1172 ],
    [ 0xEA2B, 0x112D,0x118F ],
    [ 0xEA2D, 0x112D,0x1194 ],
    [ 0xEA2E, 0x112D,0x1173 ],
    [ 0xEA45, 0x112D,0x1174 ],
    [ 0xEA4B, 0x112D,0x1175 ],
    [ 0xEA57, 0x112D,0x1198 ],
    [ 0xEA58, 0x112D,0x119E ],
    [ 0xEA61, 0x112D,0x11A1 ],
    [ 0xEA6A, 0x112E,0x1161 ],
    [ 0xEA6C, 0x112E,0x1167 ],
    [ 0xEA6E, 0x112E,0x1169 ],
    [ 0xEA6F, 0x112E,0x1175 ],
    [ 0xEA70, 0x112E,0x119E ],
    [ 0xEA71, 0x112F,0x1161 ],
    [ 0xEA7F, 0x112F,0x1162 ],
    [ 0xEA88, 0x112F,0x1163 ],
    [ 0xEA8C, 0x112F,0x1164 ],
    [ 0xEA8D, 0x112F,0x1165 ],
    [ 0xEA9A, 0x112F,0x117C ],
    [ 0xEA9E, 0x112F,0x1166 ],
    [ 0xEAA5, 0x112F,0x1167 ],
    [ 0xEAAA, 0x112F,0x1168 ],
    [ 0xEAAB, 0x112F,0x1169 ],
    [ 0xEAB6, 0x112F,0x116A ],
    [ 0xEAB7, 0x112F,0x116B ],
    [ 0xEAB8, 0x112F,0x116C ],
    [ 0xEABB, 0x112F,0x116D ],
    [ 0xEABF, 0x112F,0x116E ],
    [ 0xEAC9, 0x112F,0x116F ],
    [ 0xEACA, 0x112F,0x1170 ],
    [ 0xEACB, 0x112F,0xD7B5 ],
    [ 0xEACC, 0x112F,0x118C ],
    [ 0xEACD, 0x112F,0x1171 ],
    [ 0xEAD2, 0x112F,0x1172 ],
    [ 0xEAD8, 0x112F,0x1194 ],
    [ 0xEAD9, 0x112F,0x1173 ],
    [ 0xEAE7, 0x112F,0x1174 ],
    [ 0xEAED, 0x112F,0x1175 ],
    [ 0xEAF7, 0x112F,0x1198 ],
    [ 0xEAF8, 0x112F,0x119E ],
    [ 0xEB04, 0x112F,0x11A1 ],
    [ 0xEB0D, 0x1130,0x1161 ],
    [ 0xEB0E, 0x1130,0x1162 ],
    [ 0xEB10, 0x1130,0x1163 ],
    [ 0xEB11, 0x1130,0x1165 ],
    [ 0xEB13, 0x1130,0x1166 ],
    [ 0xEB14, 0x1130,0x1167 ],
    [ 0xEB16, 0x1130,0x116E ],
    [ 0xEB17, 0x1130,0x1172 ],
    [ 0xEB1A, 0x1130,0x1173 ],
    [ 0xEB1E, 0x1130,0x1175 ],
    [ 0xEB20, 0x1131,0x1161 ],
    [ 0xEB22, 0x1131,0x1169 ],
    [ 0xEB23, 0x1131,0x1173 ],
    [ 0xEB25, 0x1131,0x119E ],
    [ 0xEB27, 0x1132,0x1161 ],
    [ 0xEB34, 0x1132,0x1162 ],
    [ 0xEB3B, 0x1132,0x1163 ],
    [ 0xEB3D, 0x1132,0x1165 ],
    [ 0xEB47, 0x1132,0x1166 ],
    [ 0xEB4C, 0x1132,0x1167 ],
    [ 0xEB55, 0x1132,0x1168 ],
    [ 0xEB56, 0x1132,0x1169 ],
    [ 0xEB62, 0x1132,0x116A ],
    [ 0xEB63, 0x1132,0x116B ],
    [ 0xEB64, 0x1132,0x116C ],
    [ 0xEB65, 0x1132,0x116D ],
    [ 0xEB66, 0x1132,0x116E ],
    [ 0xEB71, 0x1132,0x116F ],
    [ 0xEB72, 0x1132,0x1170 ],
    [ 0xEB73, 0x1132,0xD7B5 ],
    [ 0xEB74, 0x1132,0x118C ],
    [ 0xEB75, 0x1132,0x1171 ],
    [ 0xEB76, 0x1132,0x1172 ],
    [ 0xEB77, 0x1132,0x1194 ],
    [ 0xEB78, 0x1132,0x1173 ],
    [ 0xEB80, 0x1132,0x1174 ],
    [ 0xEB82, 0x1132,0x1175 ],
    [ 0xEB8D, 0x1132,0x1198 ],
    [ 0xEB8E, 0x1132,0x119E ],
    [ 0xEB97, 0x1132,0x11A1 ],
    [ 0xEBA0, 0x1133,0x1161 ],
    [ 0xEBA1, 0x1133,0x1162 ],
    [ 0xEBA2, 0x1133,0x116E ],
    [ 0xEBA4, 0x1133,0x1173 ],
    [ 0xEBA6, 0x1133,0x1174 ],
    [ 0xEBA7, 0x1133,0x1175 ],
    [ 0xEBB5, 0x110A,0x1188 ],
    [ 0xEBB9, 0x110A,0x1189 ],
    [ 0xEBBC, 0x110A,0xD7B5 ],
    [ 0xEBBE, 0x110A,0x118C ],
    [ 0xEBC3, 0x110A,0x1191 ],
    [ 0xEBC6, 0x110A,0x1192 ],
    [ 0xEBC8, 0x110A,0x1194 ],
    [ 0xEBD3, 0x110A,0x1198 ],
    [ 0xEBD4, 0x110A,0x119E ],
    [ 0xEBE1, 0x110A,0x11A1 ],
    [ 0xEBE4, 0xA975,0x1169 ],
    [ 0xEBE5, 0x1134,0x1173 ],
    [ 0xEBE6, 0x1135,0x1173 ],
    [ 0xEBE9, 0x1135,0x1175 ],
    [ 0xEBEB, 0x1136,0x1161 ],
    [ 0xEBF6, 0x1136,0x1162 ],
    [ 0xEBFD, 0x1136,0x1163 ],
    [ 0xEC01, 0x1136,0x1165 ],
    [ 0xEC09, 0x1136,0x1166 ],
    [ 0xEC0E, 0x1136,0x1167 ],
    [ 0xEC14, 0x1136,0x1168 ],
    [ 0xEC16, 0x1136,0x1169 ],
    [ 0xEC20, 0x1136,0x116A ],
    [ 0xEC22, 0x1136,0x116C ],
    [ 0xEC27, 0x1136,0x116D ],
    [ 0xEC2D, 0x1136,0x116E ],
    [ 0xEC34, 0x1136,0x116F ],
    [ 0xEC35, 0x1136,0x1170 ],
    [ 0xEC36, 0x1136,0x1171 ],
    [ 0xEC38, 0x1136,0x1172 ],
    [ 0xEC3C, 0x1136,0x1173 ],
    [ 0xEC43, 0x1136,0x1174 ],
    [ 0xEC45, 0x1136,0x1175 ],
    [ 0xEC51, 0x1136,0x119E ],
    [ 0xEC59, 0x1136,0x11A1 ],
    [ 0xEC5E, 0x1137,0x1169 ],
    [ 0xEC5F, 0x1137,0x1175 ],
    [ 0xEC60, 0x1137,0x119E ],
    [ 0xEC63, 0x1138,0x1161 ],
    [ 0xEC65, 0x1138,0x1165 ],
    [ 0xEC68, 0x1138,0x1167 ],
    [ 0xEC69, 0x1138,0x1169 ],
    [ 0xEC6E, 0x1138,0x116A ],
    [ 0xEC6F, 0x1138,0x116B ],
    [ 0xEC70, 0x1138,0x116C ],
    [ 0xEC73, 0x1138,0x116E ],
    [ 0xEC74, 0x1138,0x116F ],
    [ 0xEC75, 0x1138,0x1173 ],
    [ 0xEC77, 0x1138,0x1174 ],
    [ 0xEC78, 0x1138,0x1175 ],
    [ 0xEC79, 0x1138,0x119E ],
    [ 0xEC7A, 0x1139,0x1161 ],
    [ 0xEC7C, 0x1139,0x1165 ],
    [ 0xEC7E, 0x1139,0x1166 ],
    [ 0xEC7F, 0x1139,0x1169 ],
    [ 0xEC81, 0x1139,0x116E ],
    [ 0xEC83, 0x1139,0x1171 ],
    [ 0xEC84, 0x1139,0x1173 ],
    [ 0xEC88, 0x1139,0x1175 ],
    [ 0xEC8A, 0x1139,0x119E ],
    [ 0xEC8D, 0x113A,0x1167 ],
    [ 0xEC8F, 0x113A,0x116E ],
    [ 0xEC91, 0x113A,0x1175 ],
    [ 0xEC92, 0x113A,0x119E ],
    [ 0xEC94, 0x113B,0x1165 ],
    [ 0xEC96, 0x113B,0x1167 ],
    [ 0xEC97, 0x113B,0x1173 ],
    [ 0xEC99, 0x113C,0x1161 ],
    [ 0xEC9C, 0x113C,0x1162 ],
    [ 0xEC9D, 0x113C,0x1163 ],
    [ 0xECA1, 0x113C,0x1165 ],
    [ 0xECA3, 0x113C,0x1167 ],
    [ 0xECA5, 0x113C,0x1168 ],
    [ 0xECA6, 0x113C,0x116E ],
    [ 0xECA9, 0x113C,0x1171 ],
    [ 0xECAA, 0x113C,0x1172 ],
    [ 0xECAC, 0x113C,0x1191 ],
    [ 0xECAE, 0x113C,0x1173 ],
    [ 0xECB1, 0x113C,0x1175 ],
    [ 0xECB3, 0x113D,0x1163 ],
    [ 0xECB6, 0x113D,0x1167 ],
    [ 0xECB8, 0x113D,0x116E ],
    [ 0xECB9, 0x113D,0x1171 ],
    [ 0xECBA, 0x113D,0x1172 ],
    [ 0xECBD, 0x113D,0x1191 ],
    [ 0xECBF, 0x113D,0x1194 ],
    [ 0xECC0, 0x113D,0x1173 ],
    [ 0xECC2, 0x113D,0x1175 ],
    [ 0xECC6, 0x113E,0x1161 ],
    [ 0xECC9, 0x113E,0x1162 ],
    [ 0xECCA, 0x113E,0x1163 ],
    [ 0xECCD, 0x113E,0x1167 ],
    [ 0xECCF, 0x113E,0x1169 ],
    [ 0xECD0, 0x113E,0x116A ],
    [ 0xECD1, 0x113E,0x116B ],
    [ 0xECD2, 0x113E,0x116E ],
    [ 0xECD5, 0x113E,0x1171 ],
    [ 0xECD6, 0x113E,0x1172 ],
    [ 0xECD8, 0x113E,0x1191 ],
    [ 0xECDA, 0x113E,0x1173 ],
    [ 0xECDD, 0x113E,0x1175 ],
    [ 0xECE1, 0x113F,0x1163 ],
    [ 0xECE3, 0x113F,0x1167 ],
    [ 0xECE5, 0x113F,0x116E ],
    [ 0xECE6, 0x113F,0x1171 ],
    [ 0xECE7, 0x113F,0x1172 ],
    [ 0xECEB, 0x113F,0x1173 ],
    [ 0xECED, 0x113F,0x1175 ],
    [ 0xECF0, 0x1140,0x1161 ],
    [ 0xECFB, 0x1140,0x1162 ],
    [ 0xED01, 0x1140,0x1163 ],
    [ 0xED0B, 0x1140,0x1178 ],
    [ 0xED0C, 0x1140,0x1165 ],
    [ 0xED18, 0x1140,0x117B ],
    [ 0xED19, 0x1140,0x1166 ],
    [ 0xED1C, 0x1140,0x1167 ],
    [ 0xED27, 0x1140,0x1168 ],
    [ 0xED2D, 0x1140,0x1169 ],
    [ 0xED36, 0x1140,0x116A ],
    [ 0xED37, 0x1140,0x116B ],
    [ 0xED38, 0x1140,0x116C ],
    [ 0xED39, 0x1140,0x116D ],
    [ 0xED41, 0x1140,0x116E ],
    [ 0xED4E, 0x1140,0x1189 ],
    [ 0xED50, 0x1140,0x116F ],
    [ 0xED51, 0x1140,0x1170 ],
    [ 0xED52, 0x1140,0xD7B5 ],
    [ 0xED53, 0x1140,0x118C ],
    [ 0xED54, 0x1140,0x1171 ],
    [ 0xED56, 0x1140,0x1172 ],
    [ 0xED61, 0x1140,0x118F ],
    [ 0xED63, 0x1140,0x1191 ],
    [ 0xED67, 0x1140,0x1192 ],
    [ 0xED69, 0x1140,0x1194 ],
    [ 0xED6B, 0x1140,0x1173 ],
    [ 0xED7C, 0x1140,0x1174 ],
    [ 0xED7F, 0x1140,0x1175 ],
    [ 0xED8E, 0x1140,0x1198 ],
    [ 0xED8F, 0x1140,0x119E ],
    [ 0xEDA0, 0x1140,0x11A1 ],
    [ 0xEDBA, 0x110B,0x1176 ],
    [ 0xEDC8, 0x110B,0x1178 ],
    [ 0xEDC9, 0x110B,0x11A4 ],
    [ 0xEDD8, 0x110B,0x117A ],
    [ 0xEDD9, 0x110B,0x117C ],
    [ 0xEDFA, 0x110B,0x117D ],
    [ 0xEDFB, 0x110B,0x117E ],
    [ 0xEE0F, 0x110B,0x117F ],
    [ 0xEE12, 0x110B,0x1182 ],
    [ 0xEE1A, 0x110B,0xD7B2 ],
    [ 0xEE1B, 0x110B,0xD7B3 ],
    [ 0xEE1D, 0x110B,0xD7B4 ],
    [ 0xEE1F, 0x110B,0x1187 ],
    [ 0xEE20, 0x110B,0x1188 ],
    [ 0xEE34, 0x110B,0x1189 ],
    [ 0xEE38, 0x110B,0x118A ],
    [ 0xEE3E, 0x110B,0x118B ],
    [ 0xEE40, 0x110B,0xD7B5 ],
    [ 0xEE41, 0x110B,0x118C ],
    [ 0xEE48, 0x110B,0x118E ],
    [ 0xEE4B, 0x110B,0x118F ],
    [ 0xEE4C, 0x110B,0x1190 ],
    [ 0xEE4E, 0x110B,0x1191 ],
    [ 0xEE55, 0x110B,0x1192 ],
    [ 0xEE58, 0x110B,0x1193 ],
    [ 0xEE59, 0x110B,0x1194 ],
    [ 0xEE6E, 0x110B,0x1195 ],
    [ 0xEE6F, 0x110B,0x1196 ],
    [ 0xEE82, 0x110B,0x1198 ],
    [ 0xEE83, 0x110B,0x119A ],
    [ 0xEE84, 0x110B,0xD7C1 ],
    [ 0xEE85, 0x110B,0x119B ],
    [ 0xEE86, 0x110B,0x119C ],
    [ 0xEE88, 0x110B,0x119E ],
    [ 0xEE9E, 0x110B,0x119F ],
    [ 0xEEA4, 0x110B,0x11A1 ],
    [ 0xEEAE, 0x110B,0x11A2 ],
    [ 0xEEB0, 0x1141,0x1161 ],
    [ 0xEEB2, 0x1141,0x1162 ],
    [ 0xEEB3, 0x1141,0x1163 ],
    [ 0xEEB5, 0x1141,0x1165 ],
    [ 0xEEB6, 0x1141,0x1166 ],
    [ 0xEEB7, 0x1141,0x1167 ],
    [ 0xEEB8, 0x1141,0x1168 ],
    [ 0xEEBA, 0x1141,0x1169 ],
    [ 0xEEBD, 0x1141,0x116A ],
    [ 0xEEBF, 0x1141,0x116B ],
    [ 0xEEC0, 0x1141,0x116C ],
    [ 0xEEC1, 0x1141,0x116D ],
    [ 0xEEC2, 0x1141,0x116E ],
    [ 0xEEC4, 0x1141,0x116F ],
    [ 0xEEC5, 0x1141,0x1170 ],
    [ 0xEEC6, 0x1141,0xD7B5 ],
    [ 0xEEC7, 0x1141,0x118C ],
    [ 0xEEC8, 0x1141,0x1171 ],
    [ 0xEEC9, 0x1141,0x1172 ],
    [ 0xEECA, 0x1141,0x1194 ],
    [ 0xEECB, 0x1141,0x1173 ],
    [ 0xEECC, 0x1141,0x1174 ],
    [ 0xEECD, 0x1141,0x1175 ],
    [ 0xEECF, 0x1141,0x1198 ],
    [ 0xEED0, 0x1141,0x119E ],
    [ 0xEED1, 0x1141,0x11A1 ],
    [ 0xEED2, 0x1142,0x1161 ],
    [ 0xEED4, 0x1142,0x1165 ],
    [ 0xEED6, 0x1142,0x1166 ],
    [ 0xEED8, 0x1142,0x1169 ],
    [ 0xEEDB, 0x1142,0x1173 ],
    [ 0xEEDD, 0x1142,0x1174 ],
    [ 0xEEDE, 0x1142,0x1175 ],
    [ 0xEEE0, 0x1142,0x119E ],
    [ 0xEEE2, 0xA976,0x1165 ],
    [ 0xEEE3, 0xA976,0x1169 ],
    [ 0xEEE4, 0xA976,0x1175 ],
    [ 0xEEE5, 0x1143,0x116E ],
    [ 0xEEE6, 0x1143,0x119E ],
    [ 0xEEE7, 0x1143,0x11A1 ],
    [ 0xEEE8, 0x1144,0x1161 ],
    [ 0xEEEC, 0x1144,0x1162 ],
    [ 0xEEEF, 0x1144,0x1165 ],
    [ 0xEEF2, 0x1144,0x1166 ],
    [ 0xEEF6, 0x1144,0x1167 ],
    [ 0xEEF7, 0x1144,0x1168 ],
    [ 0xEEF9, 0x1144,0x1169 ],
    [ 0xEEFD, 0x1144,0x116A ],
    [ 0xEEFF, 0x1144,0x116D ],
    [ 0xEF00, 0x1144,0x116E ],
    [ 0xEF03, 0x1144,0x116F ],
    [ 0xEF04, 0x1144,0x1170 ],
    [ 0xEF06, 0x1144,0x1171 ],
    [ 0xEF08, 0x1144,0x1172 ],
    [ 0xEF09, 0x1144,0x1173 ],
    [ 0xEF0C, 0x1144,0x1174 ],
    [ 0xEF0D, 0x1144,0x1175 ],
    [ 0xEF13, 0x1144,0x119E ],
    [ 0xEF14, 0x1144,0x11A1 ],
    [ 0xEF15, 0x1145,0x1161 ],
    [ 0xEF18, 0x1145,0x1176 ],
    [ 0xEF19, 0x1145,0x1162 ],
    [ 0xEF1A, 0x1145,0x1165 ],
    [ 0xEF1F, 0x1145,0x117B ],
    [ 0xEF20, 0x1145,0x1166 ],
    [ 0xEF21, 0x1145,0x1167 ],
    [ 0xEF22, 0x1145,0x1168 ],
    [ 0xEF24, 0x1145,0x1169 ],
    [ 0xEF26, 0x1145,0x116E ],
    [ 0xEF28, 0x1145,0x1189 ],
    [ 0xEF2B, 0x1145,0x118A ],
    [ 0xEF2C, 0x1145,0x116F ],
    [ 0xEF2D, 0x1145,0x1170 ],
    [ 0xEF2E, 0x1145,0x1172 ],
    [ 0xEF2F, 0x1145,0x118E ],
    [ 0xEF31, 0x1145,0x1191 ],
    [ 0xEF32, 0x1145,0x1192 ],
    [ 0xEF33, 0x1145,0x1194 ],
    [ 0xEF34, 0x1145,0x1173 ],
    [ 0xEF36, 0x1145,0x1174 ],
    [ 0xEF38, 0x1145,0x1175 ],
    [ 0xEF3A, 0x1145,0x119E ],
    [ 0xEF3C, 0x1146,0x1161 ],
    [ 0xEF3D, 0x1146,0x1162 ],
    [ 0xEF3E, 0x1146,0x1163 ],
    [ 0xEF3F, 0x1146,0x1165 ],
    [ 0xEF40, 0x1146,0x1166 ],
    [ 0xEF41, 0x1146,0x1167 ],
    [ 0xEF42, 0x1146,0x1168 ],
    [ 0xEF43, 0x1146,0x1169 ],
    [ 0xEF44, 0x1146,0x116A ],
    [ 0xEF45, 0x1146,0x116B ],
    [ 0xEF46, 0x1146,0x116C ],
    [ 0xEF47, 0x1146,0x116D ],
    [ 0xEF48, 0x1146,0x116E ],
    [ 0xEF49, 0x1146,0x116F ],
    [ 0xEF4A, 0x1146,0x1170 ],
    [ 0xEF4B, 0x1146,0xD7B5 ],
    [ 0xEF4C, 0x1146,0x118C ],
    [ 0xEF4D, 0x1146,0x1171 ],
    [ 0xEF4E, 0x1146,0x1172 ],
    [ 0xEF4F, 0x1146,0x1194 ],
    [ 0xEF50, 0x1146,0x1173 ],
    [ 0xEF51, 0x1146,0x1174 ],
    [ 0xEF52, 0x1146,0x1175 ],
    [ 0xEF53, 0x1146,0x1198 ],
    [ 0xEF54, 0x1146,0x119E ],
    [ 0xEF55, 0x1146,0x11A1 ],
    [ 0xEF56, 0x1147,0x1163 ],
    [ 0xEF58, 0x1147,0x1167 ],
    [ 0xEF5B, 0x1147,0x1168 ],
    [ 0xEF5D, 0x1147,0x116D ],
    [ 0xEF62, 0x1147,0x116E ],
    [ 0xEF63, 0x1147,0x116F ],
    [ 0xEF64, 0x1147,0x1172 ],
    [ 0xEF67, 0x1147,0x118E ],
    [ 0xEF69, 0x1147,0x1175 ],
    [ 0xEF6B, 0x1148,0x1161 ],
    [ 0xEF6E, 0x1148,0x1176 ],
    [ 0xEF6F, 0x1148,0x1162 ],
    [ 0xEF70, 0x1148,0x1165 ],
    [ 0xEF74, 0x1148,0x117B ],
    [ 0xEF75, 0x1148,0x1169 ],
    [ 0xEF76, 0x1148,0x116D ],
    [ 0xEF77, 0x1148,0x116E ],
    [ 0xEF7A, 0x1148,0x1189 ],
    [ 0xEF7C, 0x1148,0x116F ],
    [ 0xEF7D, 0x1148,0x1170 ],
    [ 0xEF7E, 0x1148,0x1171 ],
    [ 0xEF7F, 0x1148,0x1172 ],
    [ 0xEF80, 0x1148,0x1194 ],
    [ 0xEF81, 0x1148,0x1173 ],
    [ 0xEF82, 0x1148,0x1175 ],
    [ 0xEF83, 0x1149,0x1161 ],
    [ 0xEF86, 0x1149,0x1162 ],
    [ 0xEF87, 0x1149,0x1165 ],
    [ 0xEF8A, 0x1149,0x116E ],
    [ 0xEF8D, 0x1149,0x1189 ],
    [ 0xEF8F, 0x1149,0x116F ],
    [ 0xEF90, 0x1149,0x1170 ],
    [ 0xEF91, 0x1149,0x1172 ],
    [ 0xEF92, 0x1149,0x118E ],
    [ 0xEF94, 0x1149,0x1194 ],
    [ 0xEF95, 0x1149,0x1173 ],
    [ 0xEF96, 0x114A,0x1173 ],
    [ 0xEF97, 0x114B,0x1161 ],
    [ 0xEF9D, 0x114B,0x1162 ],
    [ 0xEFA0, 0x114B,0x1165 ],
    [ 0xEFA5, 0x114B,0x1166 ],
    [ 0xEFAA, 0x114B,0x1169 ],
    [ 0xEFB1, 0x114B,0x116A ],
    [ 0xEFB2, 0x114B,0x116B ],
    [ 0xEFB4, 0x114B,0x116E ],
    [ 0xEFB9, 0x114B,0x1189 ],
    [ 0xEFBB, 0x114B,0x116F ],
    [ 0xEFBD, 0x114B,0x1170 ],
    [ 0xEFBF, 0x114B,0x1171 ],
    [ 0xEFC3, 0x114B,0x1172 ],
    [ 0xEFC5, 0x114B,0x1173 ],
    [ 0xEFC7, 0x114B,0x1175 ],
    [ 0xEFCE, 0x114B,0x119E ],
    [ 0xEFCF, 0x114B,0x11A1 ],
    [ 0xEFD0, 0xA977,0x1169 ],
    [ 0xEFD1, 0xA977,0x116E ],
    [ 0xEFD2, 0x114C,0x1161 ],
    [ 0xEFDD, 0x114C,0x1162 ],
    [ 0xEFE0, 0x114C,0x1163 ],
    [ 0xEFE5, 0x114C,0x1164 ],
    [ 0xEFE7, 0x114C,0x1165 ],
    [ 0xEFF2, 0x114C,0x1166 ],
    [ 0xEFF8, 0x114C,0x1167 ],
    [ 0xF104, 0x114C,0x1168 ],
    [ 0xF107, 0x114C,0x1169 ],
    [ 0xF112, 0x114C,0x116A ],
    [ 0xF11B, 0x114C,0x116B ],
    [ 0xF11D, 0x114C,0x116C ],
    [ 0xF121, 0x114C,0x116D ],
    [ 0xF125, 0x114C,0x1184 ],
    [ 0xF126, 0x114C,0x1185 ],
    [ 0xF128, 0x114C,0x116E ],
    [ 0xF130, 0x114C,0x116F ],
    [ 0xF136, 0x114C,0x1170 ],
    [ 0xF137, 0x114C,0x1171 ],
    [ 0xF13C, 0x114C,0x1172 ],
    [ 0xF143, 0x114C,0x1191 ],
    [ 0xF148, 0x114C,0x1194 ],
    [ 0xF149, 0x114C,0x1173 ],
    [ 0xF152, 0x114C,0x1174 ],
    [ 0xF154, 0x114C,0x1175 ],
    [ 0xF15C, 0x114C,0x119E ],
    [ 0xF162, 0x114C,0x11A1 ],
    [ 0xF172, 0x110C,0x1176 ],
    [ 0xF17C, 0x110C,0x1178 ],
    [ 0xF17D, 0x110C,0x11A4 ],
    [ 0xF18A, 0x110C,0x117B ],
    [ 0xF18B, 0x110C,0x117C ],
    [ 0xF19B, 0x110C,0x11A5 ],
    [ 0xF19D, 0x110C,0x117D ],
    [ 0xF19E, 0x110C,0x117E ],
    [ 0xF1AE, 0x110C,0x1187 ],
    [ 0xF1AF, 0x110C,0x1188 ],
    [ 0xF1BA, 0x110C,0x1189 ],
    [ 0xF1BC, 0x110C,0x118A ],
    [ 0xF1BD, 0x110C,0xD7B5 ],
    [ 0xF1C1, 0x110C,0x118C ],
    [ 0xF1C7, 0x110C,0x118E ],
    [ 0xF1C9, 0x110C,0x118F ],
    [ 0xF1CC, 0x110C,0x1190 ],
    [ 0xF1CD, 0x110C,0x1191 ],
    [ 0xF1D2, 0x110C,0x1192 ],
    [ 0xF1D6, 0x110C,0x1194 ],
    [ 0xF1E3, 0x110C,0x1195 ],
    [ 0xF1E4, 0x110C,0x1196 ],
    [ 0xF1F4, 0x110C,0x1198 ],
    [ 0xF1F5, 0x110C,0x1199 ],
    [ 0xF1F7, 0x110C,0xD7BF ],
    [ 0xF1F8, 0x110C,0xD7C0 ],
    [ 0xF1F9, 0x110C,0x119A ],
    [ 0xF1FB, 0x110C,0x119B ],
    [ 0xF1FC, 0x110C,0x119E ],
    [ 0xF210, 0x110C,0xD7C5 ],
    [ 0xF212, 0x110C,0x11A1 ],
    [ 0xF21C, 0x114D,0x1173 ],
    [ 0xF21E, 0x114D,0x1175 ],
    [ 0xF233, 0x110D,0x117F ],
    [ 0xF23E, 0x110D,0x1191 ],
    [ 0xF243, 0x110D,0x1194 ],
    [ 0xF250, 0x110D,0x119E ],
    [ 0xF25A, 0x110D,0x11A1 ],
    [ 0xF263, 0xA978,0x1165 ],
    [ 0xF264, 0xA978,0x1166 ],
    [ 0xF268, 0x114E,0x1161 ],
    [ 0xF26C, 0x114E,0x1162 ],
    [ 0xF26D, 0x114E,0x1163 ],
    [ 0xF26F, 0x114E,0x1165 ],
    [ 0xF271, 0x114E,0x1167 ],
    [ 0xF274, 0x114E,0x1168 ],
    [ 0xF275, 0x114E,0x1169 ],
    [ 0xF276, 0x114E,0x116A ],
    [ 0xF277, 0x114E,0x116D ],
    [ 0xF278, 0x114E,0x1184 ],
    [ 0xF279, 0x114E,0x116E ],
    [ 0xF27C, 0x114E,0x1171 ],
    [ 0xF27D, 0x114E,0x1172 ],
    [ 0xF280, 0x114E,0x1191 ],
    [ 0xF282, 0x114E,0x1173 ],
    [ 0xF286, 0x114E,0x1175 ],
    [ 0xF28B, 0x114F,0x1161 ],
    [ 0xF290, 0x114F,0x1162 ],
    [ 0xF291, 0x114F,0x1163 ],
    [ 0xF294, 0x114F,0x1165 ],
    [ 0xF296, 0x114F,0x1167 ],
    [ 0xF298, 0x114F,0x1168 ],
    [ 0xF299, 0x114F,0x116E ],
    [ 0xF29A, 0x114F,0x1171 ],
    [ 0xF29B, 0x114F,0x1172 ],
    [ 0xF29D, 0x114F,0x1191 ],
    [ 0xF29F, 0x114F,0x1173 ],
    [ 0xF2A2, 0x114F,0x1175 ],
    [ 0xF2A4, 0x114F,0x119E ],
    [ 0xF2A5, 0x114F,0x11A1 ],
    [ 0xF2A6, 0x1150,0x1161 ],
    [ 0xF2AC, 0x1150,0x1162 ],
    [ 0xF2AD, 0x1150,0x1163 ],
    [ 0xF2B1, 0x1150,0x1167 ],
    [ 0xF2B3, 0x1150,0x1169 ],
    [ 0xF2B4, 0x1150,0x116A ],
    [ 0xF2B6, 0x1150,0x116E ],
    [ 0xF2B9, 0x1150,0x1172 ],
    [ 0xF2BD, 0x1150,0x1191 ],
    [ 0xF2BF, 0x1150,0x1173 ],
    [ 0xF2C2, 0x1150,0x1175 ],
    [ 0xF2C6, 0x1151,0x1161 ],
    [ 0xF2C9, 0x1151,0x1162 ],
    [ 0xF2CA, 0x1151,0x1163 ],
    [ 0xF2CF, 0x1151,0x1167 ],
    [ 0xF2D1, 0x1151,0x116E ],
    [ 0xF2D2, 0x1151,0x1171 ],
    [ 0xF2D3, 0x1151,0x1172 ],
    [ 0xF2D5, 0x1151,0x1191 ],
    [ 0xF2D7, 0x1151,0x1173 ],
    [ 0xF2DA, 0x1151,0x1175 ],
    [ 0xF2E4, 0x110E,0x1176 ],
    [ 0xF2E9, 0x110E,0x1178 ],
    [ 0xF2EB, 0x110E,0x11A4 ],
    [ 0xF2EE, 0x110E,0x117C ],
    [ 0xF2F8, 0x110E,0x117D ],
    [ 0xF2F9, 0x110E,0x117E ],
    [ 0xF301, 0x110E,0x1182 ],
    [ 0xF304, 0x110E,0x1187 ],
    [ 0xF305, 0x110E,0x1188 ],
    [ 0xF30C, 0x110E,0x1189 ],
    [ 0xF30F, 0x110E,0x118A ],
    [ 0xF311, 0x110E,0xD7B5 ],
    [ 0xF312, 0x110E,0x118C ],
    [ 0xF319, 0x110E,0x118E ],
    [ 0xF31B, 0x110E,0x118F ],
    [ 0xF31D, 0x110E,0x1190 ],
    [ 0xF31E, 0x110E,0x1191 ],
    [ 0xF323, 0x110E,0x1192 ],
    [ 0xF325, 0x110E,0x1194 ],
    [ 0xF32F, 0x110E,0x1195 ],
    [ 0xF330, 0x110E,0x1196 ],
    [ 0xF339, 0x110E,0x1198 ],
    [ 0xF33A, 0x110E,0x1199 ],
    [ 0xF33D, 0x110E,0xD7BF ],
    [ 0xF33E, 0x110E,0xD7C0 ],
    [ 0xF33F, 0x110E,0xD7C2 ],
    [ 0xF340, 0x110E,0x119B ],
    [ 0xF341, 0x110E,0x119E ],
    [ 0xF353, 0x110E,0x11A1 ],
    [ 0xF35B, 0x1152,0x116E ],
    [ 0xF35C, 0x1152,0x116F ],
    [ 0xF35D, 0x1153,0x1175 ],
    [ 0xF35F, 0x1154,0x1161 ],
    [ 0xF361, 0x1154,0x1176 ],
    [ 0xF362, 0x1154,0x1162 ],
    [ 0xF363, 0x1154,0x1165 ],
    [ 0xF365, 0x1154,0x1167 ],
    [ 0xF367, 0x1154,0x1168 ],
    [ 0xF368, 0x1154,0x1169 ],
    [ 0xF369, 0x1154,0x116D ],
    [ 0xF36A, 0x1154,0xD7B2 ],
    [ 0xF36B, 0x1154,0x1184 ],
    [ 0xF36C, 0x1154,0x116E ],
    [ 0xF36F, 0x1154,0x1171 ],
    [ 0xF370, 0x1154,0x1172 ],
    [ 0xF372, 0x1154,0x1191 ],
    [ 0xF374, 0x1154,0x1173 ],
    [ 0xF375, 0x1154,0x1175 ],
    [ 0xF37A, 0x1155,0x1161 ],
    [ 0xF37F, 0x1155,0x1162 ],
    [ 0xF380, 0x1155,0x1163 ],
    [ 0xF382, 0x1155,0x1167 ],
    [ 0xF384, 0x1155,0x1169 ],
    [ 0xF385, 0x1155,0x116A ],
    [ 0xF387, 0x1155,0x116B ],
    [ 0xF388, 0x1155,0x116E ],
    [ 0xF389, 0x1155,0x1171 ],
    [ 0xF38A, 0x1155,0x1172 ],
    [ 0xF38F, 0x1155,0x1191 ],
    [ 0xF391, 0x1155,0x1173 ],
    [ 0xF394, 0x1155,0x1175 ],
    [ 0xF39A, 0x1155,0x119B ],
    [ 0xF3A0, 0x110F,0x1176 ],
    [ 0xF3A4, 0x110F,0x1178 ],
    [ 0xF3A7, 0x110F,0x117A ],
    [ 0xF3A8, 0x110F,0x117B ],
    [ 0xF3AD, 0x110F,0x117E ],
    [ 0xF3B3, 0x110F,0x1182 ],
    [ 0xF3B7, 0x110F,0x1187 ],
    [ 0xF3BC, 0x110F,0x1189 ],
    [ 0xF3BF, 0x110F,0x118A ],
    [ 0xF3C2, 0x110F,0xD7B5 ],
    [ 0xF3C4, 0x110F,0x118C ],
    [ 0xF3C7, 0x110F,0x118F ],
    [ 0xF3C8, 0x110F,0x1190 ],
    [ 0xF3C9, 0x110F,0x1191 ],
    [ 0xF3D0, 0x110F,0x1192 ],
    [ 0xF3D2, 0x110F,0x1194 ],
    [ 0xF3D7, 0x110F,0x1195 ],
    [ 0xF3D8, 0x110F,0x1196 ],
    [ 0xF3DD, 0x110F,0x1198 ],
    [ 0xF3DE, 0x110F,0x1199 ],
    [ 0xF3E1, 0x110F,0xD7BD ],
    [ 0xF3E2, 0x110F,0xD7BF ],
    [ 0xF3E3, 0x110F,0xD7C0 ],
    [ 0xF3E4, 0x110F,0x119A ],
    [ 0xF3E5, 0x110F,0xD7C2 ],
    [ 0xF3E6, 0x110F,0x119B ],
    [ 0xF3E8, 0x110F,0xD7C3 ],
    [ 0xF3EA, 0x110F,0x119E ],
    [ 0xF3F3, 0x110F,0x11A1 ],
    [ 0xF3FF, 0x1110,0x1176 ],
    [ 0xF400, 0x1110,0x1177 ],
    [ 0xF403, 0x1110,0x1178 ],
    [ 0xF408, 0x1110,0x117B ],
    [ 0xF417, 0x1110,0x1182 ],
    [ 0xF41C, 0x1110,0x1188 ],
    [ 0xF422, 0x1110,0x1189 ],
    [ 0xF425, 0x1110,0xD7B5 ],
    [ 0xF426, 0x1110,0x118C ],
    [ 0xF427, 0x1110,0x118D ],
    [ 0xF42B, 0x1110,0x1194 ],
    [ 0xF431, 0x1110,0x1195 ],
    [ 0xF432, 0x1110,0x1196 ],
    [ 0xF437, 0x1110,0x1198 ],
    [ 0xF438, 0x1110,0x1199 ],
    [ 0xF43A, 0x1110,0xD7BF ],
    [ 0xF43B, 0x1110,0xD7C0 ],
    [ 0xF43C, 0x1110,0x119E ],
    [ 0xF448, 0x1110,0x11A1 ],
    [ 0xF451, 0xA979,0x116E ],
    [ 0xF452, 0xA979,0x119E ],
    [ 0xF45C, 0x1111,0x1176 ],
    [ 0xF45F, 0x1111,0x1178 ],
    [ 0xF462, 0x1111,0x117A ],
    [ 0xF463, 0x1111,0x117C ],
    [ 0xF46E, 0x1111,0x1182 ],
    [ 0xF471, 0x1111,0x1188 ],
    [ 0xF478, 0x1111,0xD7B5 ],
    [ 0xF479, 0x1111,0x118C ],
    [ 0xF47D, 0x1111,0x118E ],
    [ 0xF47F, 0x1111,0x1194 ],
    [ 0xF482, 0x1111,0x1196 ],
    [ 0xF488, 0x1111,0x1198 ],
    [ 0xF489, 0x1111,0x1199 ],
    [ 0xF48B, 0x1111,0xD7BD ],
    [ 0xF48C, 0x1111,0xD7BF ],
    [ 0xF48D, 0x1111,0xD7C0 ],
    [ 0xF48E, 0x1111,0x119E ],
    [ 0xF49E, 0x1111,0x119F ],
    [ 0xF4A0, 0x1111,0x11A1 ],
    [ 0xF4A9, 0x1156,0x1161 ],
    [ 0xF4AB, 0xA97A,0x1175 ],
    [ 0xF4AD, 0x1157,0x116E ],
    [ 0xF4B8, 0x1112,0x1176 ],
    [ 0xF4B9, 0x1112,0x1177 ],
    [ 0xF4C1, 0x1112,0x1178 ],
    [ 0xF4C8, 0x1112,0x117A ],
    [ 0xF4C9, 0x1112,0x117B ],
    [ 0xF4CA, 0x1112,0x117C ],
    [ 0xF4DF, 0x1112,0x11A6 ],
    [ 0xF4E0, 0x1112,0x117F ],
    [ 0xF4E3, 0x1112,0x1180 ],
    [ 0xF4E4, 0x1112,0x1182 ],
    [ 0xF4E9, 0x1112,0xD7B2 ],
    [ 0xF4EB, 0x1112,0x1184 ],
    [ 0xF4ED, 0x1112,0x1187 ],
    [ 0xF4EE, 0x1112,0x1188 ],
    [ 0xF4F6, 0x1112,0x1189 ],
    [ 0xF4F9, 0x1112,0x118A ],
    [ 0xF4FD, 0x1112,0x118B ],
    [ 0xF500, 0x1112,0xD7B5 ],
    [ 0xF501, 0x1112,0x118C ],
    [ 0xF507, 0x1112,0x118E ],
    [ 0xF509, 0x1112,0x1191 ],
    [ 0xF511, 0x1112,0x1192 ],
    [ 0xF514, 0x1112,0x1194 ],
    [ 0xF51C, 0x1112,0xD7B9 ],
    [ 0xF51D, 0x1112,0xD7BC ],
    [ 0xF51E, 0x1112,0x1195 ],
    [ 0xF51F, 0x1112,0x1196 ],
    [ 0xF52D, 0x1112,0x1198 ],
    [ 0xF52E, 0x1112,0x1199 ],
    [ 0xF531, 0x1112,0xD7BF ],
    [ 0xF532, 0x1112,0xD7C0 ],
    [ 0xF533, 0x1112,0xD7C2 ],
    [ 0xF534, 0x1112,0x119B ],
    [ 0xF535, 0x1112,0xD7C3 ],
    [ 0xF537, 0x1112,0x119E ],
    [ 0xF54E, 0x1112,0x119F ],
    [ 0xF550, 0x1112,0x11A1 ],
    [ 0xF55D, 0xA97B,0x1163 ],
    [ 0xF55F, 0xA97B,0x1169 ],
    [ 0xF560, 0xA97B,0x116A ],
    [ 0xF562, 0xA97B,0x116E ],
    [ 0xF563, 0xA97B,0x116F ],
    [ 0xF564, 0xA97B,0x1170 ],
    [ 0xF565, 0xA97B,0x1171 ],
    [ 0xF566, 0xA97B,0x1172 ],
    [ 0xF568, 0xA97B,0x118E ],
    [ 0xF56A, 0xA97B,0x118F ],
    [ 0xF56B, 0xA97B,0x1190 ],
    [ 0xF56C, 0xA97B,0x1194 ],
    [ 0xF56E, 0xA97B,0x1175 ],
    [ 0xF571, 0x1158,0x1161 ],
    [ 0xF57C, 0x1158,0x1162 ],
    [ 0xF57F, 0x1158,0x1163 ],
    [ 0xF587, 0x1158,0x1164 ],
    [ 0xF588, 0x1158,0x1165 ],
    [ 0xF58E, 0x1158,0x1166 ],
    [ 0xF590, 0x1158,0x1167 ],
    [ 0xF599, 0x1158,0x1168 ],
    [ 0xF59B, 0x1158,0x1169 ],
    [ 0xF5A4, 0x1158,0x116A ],
    [ 0xF5AC, 0x1158,0x116B ],
    [ 0xF5AE, 0x1158,0x116C ],
    [ 0xF5B2, 0x1158,0x116D ],
    [ 0xF5B5, 0x1158,0x116E ],
    [ 0xF5BE, 0x1158,0x116F ],
    [ 0xF5C1, 0x1158,0x1171 ],
    [ 0xF5C3, 0x1158,0x1172 ],
    [ 0xF5C5, 0x1158,0x118F ],
    [ 0xF5C7, 0x1158,0x1191 ],
    [ 0xF5CD, 0x1158,0x1192 ],
    [ 0xF5CF, 0x1158,0x1173 ],
    [ 0xF5D6, 0x1158,0x1195 ],
    [ 0xF5D7, 0x1158,0x1174 ],
    [ 0xF5DC, 0x1158,0x1175 ],
    [ 0xF5E2, 0x1158,0x119E ],
    [ 0xF5E4, 0x1158,0x11A1 ],
    [ 0xF5E8, 0x1159,0x1161 ],
    [ 0xF5F3, 0x1159,0x1162 ],
    [ 0xF5F6, 0x1159,0x1163 ],
    [ 0xF5FF, 0x1159,0x1164 ],
    [ 0xF600, 0x1159,0x1165 ],
    [ 0xF607, 0x1159,0x1166 ],
    [ 0xF609, 0x1159,0x1167 ],
    [ 0xF612, 0x1159,0x1168 ],
    [ 0xF614, 0x1159,0x1169 ],
    [ 0xF61B, 0x1159,0x116A ],
    [ 0xF625, 0x1159,0x116B ],
    [ 0xF627, 0x1159,0x116C ],
    [ 0xF62B, 0x1159,0x116D ],
    [ 0xF630, 0x1159,0x116E ],
    [ 0xF63A, 0x1159,0x116F ],
    [ 0xF63F, 0x1159,0x1171 ],
    [ 0xF641, 0x1159,0x1172 ],
    [ 0xF646, 0x1159,0x1191 ],
    [ 0xF64D, 0x1159,0x1192 ],
    [ 0xF64F, 0x1159,0x1194 ],
    [ 0xF651, 0x1159,0x1173 ],
    [ 0xF659, 0x1159,0x1174 ],
    [ 0xF65C, 0x1159,0x1175 ],
    [ 0xF665, 0x1159,0x119E ],
    [ 0xF668, 0x1159,0x11A1 ],
  );
}
