#!/data/data/com.termux/files/usr/bin/env python3
# Copyright 2016 The Emscripten Authors.  All rights reserved.
# Emscripten is available under two separate licenses, the MIT license and the
# University of Illinois/NCSA Open Source License.  Both these licenses can be
# found in the LICENSE file.

import os
import shlex
import shutil
import sys

from tools import config, shared, utils


#
# Main run() function
#
def run():
  if len(sys.argv) < 2 or sys.argv[1] in ('--version', '--help'):
    print('''\
emcmake is a helper for cmake, setting various environment
variables so that emcc etc. are used. Typical usage:

  emcmake cmake [FLAGS]
''', file=sys.stderr)
    return 1

  args = sys.argv[1:]

  def has_substr(args, substr):
    return any(substr in s for s in args)

  # Append the Emscripten toolchain file if the user didn't specify one.
  if not has_substr(args, '-DCMAKE_TOOLCHAIN_FILE') and 'CMAKE_TOOLCHAIN_FILE' not in os.environ:
    args.append('-DCMAKE_TOOLCHAIN_FILE=' + utils.path_from_root('cmake/Modules/Platform/Emscripten.cmake'))

  if not has_substr(args, '-DCMAKE_CROSSCOMPILING_EMULATOR'):
    node_js = [config.NODE_JS[0]]
    # In order to allow cmake to run code built with pthreads we need to pass
    # some extra flags to node.
    node_js += shared.node_pthread_flags(config.NODE_JS)
    node_js = ';'.join(node_js)
    # See https://github.com/emscripten-core/emscripten/issues/15522
    args.append(f'-DCMAKE_CROSSCOMPILING_EMULATOR={node_js}')

  # Print a better error if we have no CMake executable on the PATH
  if not os.path.dirname(args[0]) and not shutil.which(args[0]):
    print(f'emcmake: cmake executable not found on PATH: `{args[0]}`', file=sys.stderr)
    return 1

  # On Windows specify MinGW Makefiles or ninja if we have them and no other
  # toolchain was specified, to keep CMake from pulling in a native Visual
  # Studio, or Unix Makefiles.
  if utils.WINDOWS and not any(arg.startswith('-G') for arg in args):
    if shutil.which('mingw32-make'):
      args += ['-G', 'MinGW Makefiles']
    elif shutil.which('ninja'):
      args += ['-G', 'Ninja']
    else:
      print('emcmake: no compatible cmake generator found; Please install ninja or mingw32-make, or specify a generator explicitly using -G', file=sys.stderr)
      return 1

  print(f'emcmake: {shlex.join(args)} in directory {os.getcwd()}', file=sys.stderr)
  shared.exec_process(args)


if __name__ == '__main__':
  sys.exit(run())
