/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2009-2016 Bernhard Gschaider
    Copyright (C) 2016-2024 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::profilingTrigger

Description
    Triggers for starting/stopping code profiling.

SourceFiles
    profilingTrigger.C

\*---------------------------------------------------------------------------*/

#ifndef Foam_profilingTrigger_H
#define Foam_profilingTrigger_H

#include <string>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward Declarations
class profilingInformation;

/*---------------------------------------------------------------------------*\
                      Class profilingTrigger Declaration
\*---------------------------------------------------------------------------*/

class profilingTrigger
{
    // Private Data Members

        //- The profiling information
        profilingInformation* ptr_;


    // Private Member Functions

        //- True if any profiling hooks are possible
        static bool possible() noexcept;

        //- Helper: string concatenation (no-op)
        static inline void string_concat(std::string& output) {}

        //- Helper: string concatenation (loop)
        template<class StringType, class... StringArgs>
        static inline void string_concat
        (
            std::string& output,
            const StringType& str,
            StringArgs&&... args
        )
        {
            output += str;
            string_concat(output, std::forward<StringArgs>(args)...);
        }

        //- Enter profiling section
        void enter(const std::string& name);

        //- Trigger entering of profiling section
        template<class... StringArgs>
        void trigger(std::string name, StringArgs&&... args)
        {
            string_concat(name, std::forward<StringArgs>(args)...);
            // No checking for empty name (should not occur)
            enter(name);
        }


public:

    // Generated Methods

        //- No copy construct
        profilingTrigger(const profilingTrigger&) = delete;

        //- No copy assignment
        void operator=(const profilingTrigger&) = delete;


    // Constructors

        //- Default construct, no profiling trigger
        constexpr profilingTrigger() noexcept : ptr_(nullptr) {}

        //- Start profiling section (if profiling is active)
        //- with given description.
        //  The description is generated by string concatenation of the
        //  parameters.
        //  Descriptions beginning with 'application::' are reserved for
        //  internal use.
        template<class... StringArgs>
        explicit profilingTrigger(StringArgs&&... description)
        :
            ptr_(nullptr)
        {
            if (possible())
            {
                // Delay string concatenation until actually needed
                trigger(std::forward<StringArgs>(description)...);
            }
        }


    //- Destructor, calls stop()
    ~profilingTrigger();


    // Member Functions

        //- True if the triggered profiling section is active
        bool running() const noexcept;

        //- Stop triggered profiling section
        void stop();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

// Macros

//- Define profiling trigger with specified name and description string.
//- The description is generated by string concatenation.
//  \sa endProfiling
#define addProfiling(Name, ...) \
    ::Foam::profilingTrigger  profilingTriggerFor##Name(__VA_ARGS__)

//- Remove profiling with specified name
//  \sa addProfiling
#define endProfiling(Name)    profilingTriggerFor##Name.stop()

//- Define profiling trigger with specified name and description
//- corresponding to the compiler-defined function name string
//  \sa addProfiling
#ifdef __GNUC__
#define addProfilingInFunction(Name) \
    ::Foam::profilingTrigger  profilingTriggerFor##Name(__PRETTY_FUNCTION__)
#else
#define addProfilingInFunction(Name) \
    ::Foam::profilingTrigger  profilingTriggerFor##Name(__func__)
#endif


#endif

// ************************************************************************* //
