#ifndef __FRIDA_CORE_H__
#define __FRIDA_CORE_H__

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <json-glib/json-glib.h>

#define FRIDA_VERSION "17.2.14"

#define FRIDA_MAJOR_VERSION 17
#define FRIDA_MINOR_VERSION 2
#define FRIDA_MICRO_VERSION 14
#define FRIDA_NANO_VERSION 0

#define FRIDA_CHECK_VERSION(maj, min, mic) \
    (FRIDA_CURRENT_VERSION >= FRIDA_VERSION_ENCODE ((maj), (min), (mic)))

#define FRIDA_CURRENT_VERSION \
    FRIDA_VERSION_ENCODE (    \
        FRIDA_MAJOR_VERSION,  \
        FRIDA_MINOR_VERSION,  \
        FRIDA_MICRO_VERSION)

#define FRIDA_VERSION_ENCODE(maj, min, mic) \
    (((maj) * 1000000U) + ((min) * 1000U) + (mic))

G_BEGIN_DECLS

typedef struct _FridaDeviceManager FridaDeviceManager;
typedef struct _FridaDeviceList FridaDeviceList;
typedef struct _FridaDevice FridaDevice;
typedef struct _FridaRemoteDeviceOptions FridaRemoteDeviceOptions;
typedef struct _FridaApplicationList FridaApplicationList;
typedef struct _FridaApplication FridaApplication;
typedef struct _FridaProcessList FridaProcessList;
typedef struct _FridaProcess FridaProcess;
typedef struct _FridaProcessMatchOptions FridaProcessMatchOptions;
typedef struct _FridaSpawnOptions FridaSpawnOptions;
typedef struct _FridaFrontmostQueryOptions FridaFrontmostQueryOptions;
typedef struct _FridaApplicationQueryOptions FridaApplicationQueryOptions;
typedef struct _FridaProcessQueryOptions FridaProcessQueryOptions;
typedef struct _FridaSessionOptions FridaSessionOptions;
typedef struct _FridaSpawnList FridaSpawnList;
typedef struct _FridaSpawn FridaSpawn;
typedef struct _FridaChildList FridaChildList;
typedef struct _FridaChild FridaChild;
typedef struct _FridaCrash FridaCrash;
typedef struct _FridaBus FridaBus;
typedef struct _FridaService FridaService;
typedef struct _FridaSession FridaSession;
typedef struct _FridaScript FridaScript;
typedef struct _FridaSnapshotOptions FridaSnapshotOptions;
typedef struct _FridaScriptOptions FridaScriptOptions;
typedef struct _FridaPeerOptions FridaPeerOptions;
typedef struct _FridaRelay FridaRelay;
typedef struct _FridaPortalOptions FridaPortalOptions;
typedef struct _FridaPortalMembership FridaPortalMembership;
typedef struct _FridaRpcClient FridaRpcClient;
typedef struct _FridaRpcPeer FridaRpcPeer;
typedef struct _FridaRpcPeerIface FridaRpcPeerIface;
typedef struct _FridaInjector FridaInjector;
typedef struct _FridaInjectorIface FridaInjectorIface;
typedef struct _FridaPackageManager FridaPackageManager;
typedef struct _FridaPackage FridaPackage;
typedef struct _FridaPackageList FridaPackageList;
typedef struct _FridaPackageSearchOptions FridaPackageSearchOptions;
typedef struct _FridaPackageSearchResult FridaPackageSearchResult;
typedef struct _FridaPackageInstallOptions FridaPackageInstallOptions;
typedef struct _FridaPackageInstallResult FridaPackageInstallResult;
typedef struct _FridaControlService FridaControlService;
typedef struct _FridaControlServiceOptions FridaControlServiceOptions;
typedef struct _FridaPortalService FridaPortalService;
typedef struct _FridaEndpointParameters FridaEndpointParameters;
typedef struct _FridaAuthenticationService FridaAuthenticationService;
typedef struct _FridaAuthenticationServiceIface FridaAuthenticationServiceIface;
typedef struct _FridaStaticAuthenticationService FridaStaticAuthenticationService;
typedef struct _FridaFileMonitor FridaFileMonitor;
typedef struct _FridaCompiler FridaCompiler;
typedef struct _FridaCompilerOptions FridaCompilerOptions;
typedef struct _FridaBuildOptions FridaBuildOptions;
typedef struct _FridaWatchOptions FridaWatchOptions;

typedef enum {
  FRIDA_RUNTIME_GLIB,
  FRIDA_RUNTIME_OTHER
} FridaRuntime;

typedef enum {
  FRIDA_DEVICE_TYPE_LOCAL,
  FRIDA_DEVICE_TYPE_REMOTE,
  FRIDA_DEVICE_TYPE_USB
} FridaDeviceType;

typedef enum {
  FRIDA_PACKAGE_INSTALL_PHASE_INITIALIZING,
  FRIDA_PACKAGE_INSTALL_PHASE_PREPARING_DEPENDENCIES,
  FRIDA_PACKAGE_INSTALL_PHASE_RESOLVING_PACKAGE,
  FRIDA_PACKAGE_INSTALL_PHASE_FETCHING_RESOURCE,
  FRIDA_PACKAGE_INSTALL_PHASE_PACKAGE_ALREADY_INSTALLED,
  FRIDA_PACKAGE_INSTALL_PHASE_DOWNLOADING_PACKAGE,
  FRIDA_PACKAGE_INSTALL_PHASE_PACKAGE_INSTALLED,
  FRIDA_PACKAGE_INSTALL_PHASE_RESOLVING_AND_INSTALLING_ALL,
  FRIDA_PACKAGE_INSTALL_PHASE_COMPLETE
} FridaPackageInstallPhase;

typedef enum {
  FRIDA_PACKAGE_ROLE_RUNTIME,
  FRIDA_PACKAGE_ROLE_DEVELOPMENT,
  FRIDA_PACKAGE_ROLE_OPTIONAL,
  FRIDA_PACKAGE_ROLE_PEER
} FridaPackageRole;

typedef enum {
  FRIDA_OUTPUT_FORMAT_UNESCAPED,
  FRIDA_OUTPUT_FORMAT_HEX_BYTES,
  FRIDA_OUTPUT_FORMAT_C_STRING
} FridaOutputFormat;

typedef enum {
  FRIDA_BUNDLE_FORMAT_ESM,
  FRIDA_BUNDLE_FORMAT_IIFE
} FridaBundleFormat;

typedef enum {
  FRIDA_TYPE_CHECK_MODE_FULL,
  FRIDA_TYPE_CHECK_MODE_NONE
} FridaTypeCheckMode;

typedef enum {
  FRIDA_SOURCE_MAPS_INCLUDED,
  FRIDA_SOURCE_MAPS_OMITTED
} FridaSourceMaps;

typedef enum {
  FRIDA_JS_COMPRESSION_NONE,
  FRIDA_JS_COMPRESSION_TERSER
} FridaJsCompression;

typedef enum {
  FRIDA_CHILD_ORIGIN_FORK,
  FRIDA_CHILD_ORIGIN_EXEC,
  FRIDA_CHILD_ORIGIN_SPAWN
} FridaChildOrigin;

typedef enum {
  FRIDA_REALM_NATIVE,
  FRIDA_REALM_EMULATED
} FridaRealm;

typedef enum {
  FRIDA_RELAY_KIND_TURN_UDP,
  FRIDA_RELAY_KIND_TURN_TCP,
  FRIDA_RELAY_KIND_TURN_TLS
} FridaRelayKind;

typedef enum {
  FRIDA_SCOPE_MINIMAL,
  FRIDA_SCOPE_METADATA,
  FRIDA_SCOPE_FULL
} FridaScope;

typedef enum {
  FRIDA_SCRIPT_RUNTIME_DEFAULT,
  FRIDA_SCRIPT_RUNTIME_QJS,
  FRIDA_SCRIPT_RUNTIME_V8
} FridaScriptRuntime;

typedef enum {
  FRIDA_SESSION_DETACH_REASON_APPLICATION_REQUESTED = 1,
  FRIDA_SESSION_DETACH_REASON_PROCESS_REPLACED,
  FRIDA_SESSION_DETACH_REASON_PROCESS_TERMINATED,
  FRIDA_SESSION_DETACH_REASON_CONNECTION_TERMINATED,
  FRIDA_SESSION_DETACH_REASON_DEVICE_LOST
} FridaSessionDetachReason;

typedef enum {
  FRIDA_SNAPSHOT_TRANSPORT_INLINE,
  FRIDA_SNAPSHOT_TRANSPORT_SHARED_MEMORY
} FridaSnapshotTransport;

typedef enum {
  FRIDA_STDIO_INHERIT,
  FRIDA_STDIO_PIPE
} FridaStdio;

/* Library lifetime */
void frida_init (void);
void frida_shutdown (void);
void frida_deinit (void);
GMainContext * frida_get_main_context (void);

/* Object lifetime */
void frida_unref (gpointer obj);

/* Library versioning */
void frida_version (guint * major, guint * minor, guint * micro, guint * nano);
const gchar * frida_version_string (void);

/* DeviceManager */
typedef gboolean (* FridaDeviceManagerPredicate) (FridaDevice * device, gpointer user_data);

FridaDeviceManager * frida_device_manager_new (void);
FridaDeviceManager * frida_device_manager_new_with_nonlocal_backends_only (void);
FridaDeviceManager * frida_device_manager_new_with_socket_backend_only (void);

FridaDeviceManager * frida_device_manager_construct_with_nonlocal_backends_only (GType object_type);
FridaDeviceManager * frida_device_manager_construct_with_socket_backend_only (GType object_type);
void frida_device_manager_close (FridaDeviceManager * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_manager_close_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
void frida_device_manager_close_sync (FridaDeviceManager * self, GCancellable * cancellable, GError ** error);
void frida_device_manager_get_device_by_id (FridaDeviceManager * self, const gchar * id, gint timeout, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_get_device_by_id_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_get_device_by_id_sync (FridaDeviceManager * self, const gchar * id, gint timeout, GCancellable * cancellable, GError ** error);
void frida_device_manager_get_device_by_type (FridaDeviceManager * self, FridaDeviceType type, gint timeout, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_get_device_by_type_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_get_device_by_type_sync (FridaDeviceManager * self, FridaDeviceType type, gint timeout, GCancellable * cancellable, GError ** error);
void frida_device_manager_get_device (FridaDeviceManager * self, FridaDeviceManagerPredicate predicate, gpointer predicate_target, gint timeout, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_get_device_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_get_device_sync (FridaDeviceManager * self, FridaDeviceManagerPredicate predicate, gpointer predicate_target, gint timeout, GCancellable * cancellable, GError ** error);
void frida_device_manager_find_device_by_id (FridaDeviceManager * self, const gchar * id, gint timeout, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_find_device_by_id_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_find_device_by_id_sync (FridaDeviceManager * self, const gchar * id, gint timeout, GCancellable * cancellable, GError ** error);
void frida_device_manager_find_device_by_type (FridaDeviceManager * self, FridaDeviceType type, gint timeout, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_find_device_by_type_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_find_device_by_type_sync (FridaDeviceManager * self, FridaDeviceType type, gint timeout, GCancellable * cancellable, GError ** error);
void frida_device_manager_find_device (FridaDeviceManager * self, FridaDeviceManagerPredicate predicate, gpointer predicate_target, gint timeout, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_find_device_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_find_device_sync (FridaDeviceManager * self, FridaDeviceManagerPredicate predicate, gpointer predicate_target, gint timeout, GCancellable * cancellable, GError ** error);
void frida_device_manager_enumerate_devices (FridaDeviceManager * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDeviceList * frida_device_manager_enumerate_devices_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDeviceList * frida_device_manager_enumerate_devices_sync (FridaDeviceManager * self, GCancellable * cancellable, GError ** error);
void frida_device_manager_add_remote_device (FridaDeviceManager * self, const gchar * address, FridaRemoteDeviceOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaDevice * frida_device_manager_add_remote_device_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
FridaDevice * frida_device_manager_add_remote_device_sync (FridaDeviceManager * self, const gchar * address, FridaRemoteDeviceOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_manager_remove_remote_device (FridaDeviceManager * self, const gchar * address, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_manager_remove_remote_device_finish (FridaDeviceManager * self, GAsyncResult * result, GError ** error);
void frida_device_manager_remove_remote_device_sync (FridaDeviceManager * self, const gchar * address, GCancellable * cancellable, GError ** error);

/* DeviceList */
gint frida_device_list_size (FridaDeviceList * self);
FridaDevice * frida_device_list_get (FridaDeviceList * self, gint index);

/* Device */
typedef gboolean (* FridaDeviceProcessPredicate) (FridaProcess * process, gpointer user_data);

const gchar * frida_device_get_id (FridaDevice * self);
const gchar * frida_device_get_name (FridaDevice * self);
GVariant * frida_device_get_icon (FridaDevice * self);
FridaDeviceType frida_device_get_dtype (FridaDevice * self);
FridaBus * frida_device_get_bus (FridaDevice * self);

gboolean frida_device_is_lost (FridaDevice * self);
void frida_device_query_system_parameters (FridaDevice * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
GHashTable * frida_device_query_system_parameters_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
GHashTable * frida_device_query_system_parameters_sync (FridaDevice * self, GCancellable * cancellable, GError ** error);
void frida_device_get_frontmost_application (FridaDevice * self, FridaFrontmostQueryOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaApplication * frida_device_get_frontmost_application_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaApplication * frida_device_get_frontmost_application_sync (FridaDevice * self, FridaFrontmostQueryOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_enumerate_applications (FridaDevice * self, FridaApplicationQueryOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaApplicationList * frida_device_enumerate_applications_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaApplicationList * frida_device_enumerate_applications_sync (FridaDevice * self, FridaApplicationQueryOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_get_process_by_pid (FridaDevice * self, guint pid, FridaProcessMatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcess * frida_device_get_process_by_pid_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcess * frida_device_get_process_by_pid_sync (FridaDevice * self, guint pid, FridaProcessMatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_get_process_by_name (FridaDevice * self, const gchar * name, FridaProcessMatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcess * frida_device_get_process_by_name_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcess * frida_device_get_process_by_name_sync (FridaDevice * self, const gchar * name, FridaProcessMatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_get_process (FridaDevice * self, FridaDeviceProcessPredicate predicate, gpointer predicate_target, FridaProcessMatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcess * frida_device_get_process_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcess * frida_device_get_process_sync (FridaDevice * self, FridaDeviceProcessPredicate predicate, gpointer predicate_target, FridaProcessMatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_find_process_by_pid (FridaDevice * self, guint pid, FridaProcessMatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcess * frida_device_find_process_by_pid_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcess * frida_device_find_process_by_pid_sync (FridaDevice * self, guint pid, FridaProcessMatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_find_process_by_name (FridaDevice * self, const gchar * name, FridaProcessMatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcess * frida_device_find_process_by_name_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcess * frida_device_find_process_by_name_sync (FridaDevice * self, const gchar * name, FridaProcessMatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_find_process (FridaDevice * self, FridaDeviceProcessPredicate predicate, gpointer predicate_target, FridaProcessMatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcess * frida_device_find_process_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcess * frida_device_find_process_sync (FridaDevice * self, FridaDeviceProcessPredicate predicate, gpointer predicate_target, FridaProcessMatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_enumerate_processes (FridaDevice * self, FridaProcessQueryOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaProcessList * frida_device_enumerate_processes_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaProcessList * frida_device_enumerate_processes_sync (FridaDevice * self, FridaProcessQueryOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_enable_spawn_gating (FridaDevice * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_enable_spawn_gating_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
void frida_device_enable_spawn_gating_sync (FridaDevice * self, GCancellable * cancellable, GError ** error);
void frida_device_disable_spawn_gating (FridaDevice * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_disable_spawn_gating_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
void frida_device_disable_spawn_gating_sync (FridaDevice * self, GCancellable * cancellable, GError ** error);
void frida_device_enumerate_pending_spawn (FridaDevice * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaSpawnList * frida_device_enumerate_pending_spawn_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaSpawnList * frida_device_enumerate_pending_spawn_sync (FridaDevice * self, GCancellable * cancellable, GError ** error);
void frida_device_enumerate_pending_children (FridaDevice * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaChildList * frida_device_enumerate_pending_children_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaChildList * frida_device_enumerate_pending_children_sync (FridaDevice * self, GCancellable * cancellable, GError ** error);
void frida_device_spawn (FridaDevice * self, const gchar * program, FridaSpawnOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
guint frida_device_spawn_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
guint frida_device_spawn_sync (FridaDevice * self, const gchar * program, FridaSpawnOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_input (FridaDevice * self, guint pid, GBytes * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_input_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
void frida_device_input_sync (FridaDevice * self, guint pid, GBytes * data, GCancellable * cancellable, GError ** error);
void frida_device_resume (FridaDevice * self, guint pid, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_resume_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
void frida_device_resume_sync (FridaDevice * self, guint pid, GCancellable * cancellable, GError ** error);
void frida_device_kill (FridaDevice * self, guint pid, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_kill_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
void frida_device_kill_sync (FridaDevice * self, guint pid, GCancellable * cancellable, GError ** error);
void frida_device_attach (FridaDevice * self, guint pid, FridaSessionOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaSession * frida_device_attach_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaSession * frida_device_attach_sync (FridaDevice * self, guint pid, FridaSessionOptions * options, GCancellable * cancellable, GError ** error);
void frida_device_inject_library_file (FridaDevice * self, guint pid, const gchar * path, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
guint frida_device_inject_library_file_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
guint frida_device_inject_library_file_sync (FridaDevice * self, guint pid, const gchar * path, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GError ** error);
void frida_device_inject_library_blob (FridaDevice * self, guint pid, GBytes * blob, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
guint frida_device_inject_library_blob_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
guint frida_device_inject_library_blob_sync (FridaDevice * self, guint pid, GBytes * blob, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GError ** error);
void frida_device_open_channel (FridaDevice * self, const gchar * address, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
GIOStream * frida_device_open_channel_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
GIOStream * frida_device_open_channel_sync (FridaDevice * self, const gchar * address, GCancellable * cancellable, GError ** error);
void frida_device_open_service (FridaDevice * self, const gchar * address, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaService * frida_device_open_service_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
FridaService * frida_device_open_service_sync (FridaDevice * self, const gchar * address, GCancellable * cancellable, GError ** error);
void frida_device_unpair (FridaDevice * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_device_unpair_finish (FridaDevice * self, GAsyncResult * result, GError ** error);
void frida_device_unpair_sync (FridaDevice * self, GCancellable * cancellable, GError ** error);

/* RemoteDeviceOptions */
FridaRemoteDeviceOptions * frida_remote_device_options_new (void);

GTlsCertificate * frida_remote_device_options_get_certificate (FridaRemoteDeviceOptions * self);
const gchar * frida_remote_device_options_get_origin (FridaRemoteDeviceOptions * self);
const gchar * frida_remote_device_options_get_token (FridaRemoteDeviceOptions * self);
gint frida_remote_device_options_get_keepalive_interval (FridaRemoteDeviceOptions * self);

void frida_remote_device_options_set_certificate (FridaRemoteDeviceOptions * self, GTlsCertificate * value);
void frida_remote_device_options_set_origin (FridaRemoteDeviceOptions * self, const gchar * value);
void frida_remote_device_options_set_token (FridaRemoteDeviceOptions * self, const gchar * value);
void frida_remote_device_options_set_keepalive_interval (FridaRemoteDeviceOptions * self, gint value);

/* ApplicationList */
gint frida_application_list_size (FridaApplicationList * self);
FridaApplication * frida_application_list_get (FridaApplicationList * self, gint index);

/* Application */
const gchar * frida_application_get_identifier (FridaApplication * self);
const gchar * frida_application_get_name (FridaApplication * self);
guint frida_application_get_pid (FridaApplication * self);
GHashTable * frida_application_get_parameters (FridaApplication * self);

/* ProcessList */
gint frida_process_list_size (FridaProcessList * self);
FridaProcess * frida_process_list_get (FridaProcessList * self, gint index);

/* Process */
guint frida_process_get_pid (FridaProcess * self);
const gchar * frida_process_get_name (FridaProcess * self);
GHashTable * frida_process_get_parameters (FridaProcess * self);

/* ProcessMatchOptions */
FridaProcessMatchOptions * frida_process_match_options_new (void);

gint frida_process_match_options_get_timeout (FridaProcessMatchOptions * self);
FridaScope frida_process_match_options_get_scope (FridaProcessMatchOptions * self);

void frida_process_match_options_set_timeout (FridaProcessMatchOptions * self, gint value);
void frida_process_match_options_set_scope (FridaProcessMatchOptions * self, FridaScope value);

/* SpawnOptions */
FridaSpawnOptions * frida_spawn_options_new (void);

gchar ** frida_spawn_options_get_argv (FridaSpawnOptions * self, gint * result_length);
gchar ** frida_spawn_options_get_envp (FridaSpawnOptions * self, gint * result_length);
gchar ** frida_spawn_options_get_env (FridaSpawnOptions * self, gint * result_length);
const gchar * frida_spawn_options_get_cwd (FridaSpawnOptions * self);
FridaStdio frida_spawn_options_get_stdio (FridaSpawnOptions * self);
GHashTable * frida_spawn_options_get_aux (FridaSpawnOptions * self);

void frida_spawn_options_set_argv (FridaSpawnOptions * self, gchar ** value, gint value_length);
void frida_spawn_options_set_envp (FridaSpawnOptions * self, gchar ** value, gint value_length);
void frida_spawn_options_set_env (FridaSpawnOptions * self, gchar ** value, gint value_length);
void frida_spawn_options_set_cwd (FridaSpawnOptions * self, const gchar * value);
void frida_spawn_options_set_stdio (FridaSpawnOptions * self, FridaStdio value);
void frida_spawn_options_set_aux (FridaSpawnOptions * self, GHashTable * value);

/* FrontmostQueryOptions */
FridaFrontmostQueryOptions * frida_frontmost_query_options_new (void);

FridaScope frida_frontmost_query_options_get_scope (FridaFrontmostQueryOptions * self);

void frida_frontmost_query_options_set_scope (FridaFrontmostQueryOptions * self, FridaScope value);

/* ApplicationQueryOptions */
FridaApplicationQueryOptions * frida_application_query_options_new (void);

FridaScope frida_application_query_options_get_scope (FridaApplicationQueryOptions * self);

void frida_application_query_options_select_identifier (FridaApplicationQueryOptions * self, const gchar * identifier);
gboolean frida_application_query_options_has_selected_identifiers (FridaApplicationQueryOptions * self);
void frida_application_query_options_enumerate_selected_identifiers (FridaApplicationQueryOptions * self, GFunc func, gpointer user_data);
void frida_application_query_options_set_scope (FridaApplicationQueryOptions * self, FridaScope value);

/* ProcessQueryOptions */
FridaProcessQueryOptions * frida_process_query_options_new (void);

FridaScope frida_process_query_options_get_scope (FridaProcessQueryOptions * self);

void frida_process_query_options_select_pid (FridaProcessQueryOptions * self, guint pid);
gboolean frida_process_query_options_has_selected_pids (FridaProcessQueryOptions * self);
void frida_process_query_options_enumerate_selected_pids (FridaProcessQueryOptions * self, GFunc func, gpointer user_data);
void frida_process_query_options_set_scope (FridaProcessQueryOptions * self, FridaScope value);

/* SessionOptions */
FridaSessionOptions * frida_session_options_new (void);

FridaRealm frida_session_options_get_realm (FridaSessionOptions * self);
guint frida_session_options_get_persist_timeout (FridaSessionOptions * self);
const gchar * frida_session_options_get_emulated_agent_path (FridaSessionOptions * self);

void frida_session_options_set_realm (FridaSessionOptions * self, FridaRealm value);
void frida_session_options_set_persist_timeout (FridaSessionOptions * self, guint value);
void frida_session_options_set_emulated_agent_path (FridaSessionOptions * self, const gchar * value);

/* SpawnList */
gint frida_spawn_list_size (FridaSpawnList * self);
FridaSpawn * frida_spawn_list_get (FridaSpawnList * self, gint index);

/* Spawn */
guint frida_spawn_get_pid (FridaSpawn * self);
const gchar * frida_spawn_get_identifier (FridaSpawn * self);

/* ChildList */
gint frida_child_list_size (FridaChildList * self);
FridaChild * frida_child_list_get (FridaChildList * self, gint index);

/* Child */
guint frida_child_get_pid (FridaChild * self);
guint frida_child_get_parent_pid (FridaChild * self);
FridaChildOrigin frida_child_get_origin (FridaChild * self);
const gchar * frida_child_get_identifier (FridaChild * self);
const gchar * frida_child_get_path (FridaChild * self);
gchar ** frida_child_get_argv (FridaChild * self, gint * result_length);
gchar ** frida_child_get_envp (FridaChild * self, gint * result_length);

/* Crash */
guint frida_crash_get_pid (FridaCrash * self);
const gchar * frida_crash_get_process_name (FridaCrash * self);
const gchar * frida_crash_get_summary (FridaCrash * self);
const gchar * frida_crash_get_report (FridaCrash * self);
GHashTable * frida_crash_get_parameters (FridaCrash * self);

/* Bus */
gboolean frida_bus_is_detached (FridaBus * self);
void frida_bus_attach (FridaBus * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_bus_attach_finish (FridaBus * self, GAsyncResult * result, GError ** error);
void frida_bus_attach_sync (FridaBus * self, GCancellable * cancellable, GError ** error);
void frida_bus_post (FridaBus * self, const gchar * json, GBytes * data);

/* Service */
gboolean frida_service_is_closed (FridaService * self);
void frida_service_activate (FridaService * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_service_activate_finish (FridaService * self, GAsyncResult * result, GError ** error);
void frida_service_activate_sync (FridaService * self, GCancellable * cancellable, GError ** error);
void frida_service_cancel (FridaService * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_service_cancel_finish (FridaService * self, GAsyncResult * result, GError ** error);
void frida_service_cancel_sync (FridaService * self, GCancellable * cancellable, GError ** error);
void frida_service_request (FridaService * self, GVariant * parameters, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
GVariant * frida_service_request_finish (FridaService * self, GAsyncResult * result, GError ** error);
GVariant * frida_service_request_sync (FridaService * self, GVariant * parameters, GCancellable * cancellable, GError ** error);

/* Session */
guint frida_session_get_pid (FridaSession * self);
guint frida_session_get_persist_timeout (FridaSession * self);

gboolean frida_session_is_detached (FridaSession * self);
void frida_session_detach (FridaSession * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_session_detach_finish (FridaSession * self, GAsyncResult * result, GError ** error);
void frida_session_detach_sync (FridaSession * self, GCancellable * cancellable, GError ** error);
void frida_session_resume (FridaSession * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_session_resume_finish (FridaSession * self, GAsyncResult * result, GError ** error);
void frida_session_resume_sync (FridaSession * self, GCancellable * cancellable, GError ** error);
void frida_session_enable_child_gating (FridaSession * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_session_enable_child_gating_finish (FridaSession * self, GAsyncResult * result, GError ** error);
void frida_session_enable_child_gating_sync (FridaSession * self, GCancellable * cancellable, GError ** error);
void frida_session_disable_child_gating (FridaSession * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_session_disable_child_gating_finish (FridaSession * self, GAsyncResult * result, GError ** error);
void frida_session_disable_child_gating_sync (FridaSession * self, GCancellable * cancellable, GError ** error);
void frida_session_create_script (FridaSession * self, const gchar * source, FridaScriptOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaScript * frida_session_create_script_finish (FridaSession * self, GAsyncResult * result, GError ** error);
FridaScript * frida_session_create_script_sync (FridaSession * self, const gchar * source, FridaScriptOptions * options, GCancellable * cancellable, GError ** error);
void frida_session_create_script_from_bytes (FridaSession * self, GBytes * bytes, FridaScriptOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaScript * frida_session_create_script_from_bytes_finish (FridaSession * self, GAsyncResult * result, GError ** error);
FridaScript * frida_session_create_script_from_bytes_sync (FridaSession * self, GBytes * bytes, FridaScriptOptions * options, GCancellable * cancellable, GError ** error);
void frida_session_compile_script (FridaSession * self, const gchar * source, FridaScriptOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
GBytes * frida_session_compile_script_finish (FridaSession * self, GAsyncResult * result, GError ** error);
GBytes * frida_session_compile_script_sync (FridaSession * self, const gchar * source, FridaScriptOptions * options, GCancellable * cancellable, GError ** error);
void frida_session_snapshot_script (FridaSession * self, const gchar * embed_script, FridaSnapshotOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
GBytes * frida_session_snapshot_script_finish (FridaSession * self, GAsyncResult * result, GError ** error);
GBytes * frida_session_snapshot_script_sync (FridaSession * self, const gchar * embed_script, FridaSnapshotOptions * options, GCancellable * cancellable, GError ** error);
void frida_session_setup_peer_connection (FridaSession * self, FridaPeerOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_session_setup_peer_connection_finish (FridaSession * self, GAsyncResult * result, GError ** error);
void frida_session_setup_peer_connection_sync (FridaSession * self, FridaPeerOptions * options, GCancellable * cancellable, GError ** error);
void frida_session_join_portal (FridaSession * self, const gchar * address, FridaPortalOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaPortalMembership * frida_session_join_portal_finish (FridaSession * self, GAsyncResult * result, GError ** error);
FridaPortalMembership * frida_session_join_portal_sync (FridaSession * self, const gchar * address, FridaPortalOptions * options, GCancellable * cancellable, GError ** error);

/* Script */
gboolean frida_script_is_destroyed (FridaScript * self);
void frida_script_load (FridaScript * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_script_load_finish (FridaScript * self, GAsyncResult * result, GError ** error);
void frida_script_load_sync (FridaScript * self, GCancellable * cancellable, GError ** error);
void frida_script_unload (FridaScript * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_script_unload_finish (FridaScript * self, GAsyncResult * result, GError ** error);
void frida_script_unload_sync (FridaScript * self, GCancellable * cancellable, GError ** error);
void frida_script_eternalize (FridaScript * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_script_eternalize_finish (FridaScript * self, GAsyncResult * result, GError ** error);
void frida_script_eternalize_sync (FridaScript * self, GCancellable * cancellable, GError ** error);
void frida_script_post (FridaScript * self, const gchar * json, GBytes * data);
void frida_script_enable_debugger (FridaScript * self, guint16 port, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_script_enable_debugger_finish (FridaScript * self, GAsyncResult * result, GError ** error);
void frida_script_enable_debugger_sync (FridaScript * self, guint16 port, GCancellable * cancellable, GError ** error);
void frida_script_disable_debugger (FridaScript * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_script_disable_debugger_finish (FridaScript * self, GAsyncResult * result, GError ** error);
void frida_script_disable_debugger_sync (FridaScript * self, GCancellable * cancellable, GError ** error);

/* SnapshotOptions */
FridaSnapshotOptions * frida_snapshot_options_new (void);

const gchar * frida_snapshot_options_get_warmup_script (FridaSnapshotOptions * self);
FridaScriptRuntime frida_snapshot_options_get_runtime (FridaSnapshotOptions * self);

void frida_snapshot_options_set_warmup_script (FridaSnapshotOptions * self, const gchar * value);
void frida_snapshot_options_set_runtime (FridaSnapshotOptions * self, FridaScriptRuntime value);

/* ScriptOptions */
FridaScriptOptions * frida_script_options_new (void);

const gchar * frida_script_options_get_name (FridaScriptOptions * self);
GBytes * frida_script_options_get_snapshot (FridaScriptOptions * self);
FridaSnapshotTransport frida_script_options_get_snapshot_transport (FridaScriptOptions * self);
FridaScriptRuntime frida_script_options_get_runtime (FridaScriptOptions * self);

void frida_script_options_set_name (FridaScriptOptions * self, const gchar * value);
void frida_script_options_set_snapshot (FridaScriptOptions * self, GBytes * value);
void frida_script_options_set_snapshot_transport (FridaScriptOptions * self, FridaSnapshotTransport value);
void frida_script_options_set_runtime (FridaScriptOptions * self, FridaScriptRuntime value);

/* PeerOptions */
FridaPeerOptions * frida_peer_options_new (void);

const gchar * frida_peer_options_get_stun_server (FridaPeerOptions * self);

void frida_peer_options_clear_relays (FridaPeerOptions * self);
void frida_peer_options_add_relay (FridaPeerOptions * self, FridaRelay * relay);
void frida_peer_options_enumerate_relays (FridaPeerOptions * self, GFunc func, gpointer user_data);
void frida_peer_options_set_stun_server (FridaPeerOptions * self, const gchar * value);

/* Relay */
FridaRelay * frida_relay_new (const gchar * address, const gchar * username, const gchar * password, FridaRelayKind kind);

const gchar * frida_relay_get_address (FridaRelay * self);
const gchar * frida_relay_get_username (FridaRelay * self);
const gchar * frida_relay_get_password (FridaRelay * self);
FridaRelayKind frida_relay_get_kind (FridaRelay * self);

/* PortalOptions */
FridaPortalOptions * frida_portal_options_new (void);

GTlsCertificate * frida_portal_options_get_certificate (FridaPortalOptions * self);
const gchar * frida_portal_options_get_token (FridaPortalOptions * self);
gchar ** frida_portal_options_get_acl (FridaPortalOptions * self, gint * result_length);

void frida_portal_options_set_certificate (FridaPortalOptions * self, GTlsCertificate * value);
void frida_portal_options_set_token (FridaPortalOptions * self, const gchar * value);
void frida_portal_options_set_acl (FridaPortalOptions * self, gchar ** value, gint value_length);

/* PortalMembership */
void frida_portal_membership_terminate (FridaPortalMembership * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_portal_membership_terminate_finish (FridaPortalMembership * self, GAsyncResult * result, GError ** error);
void frida_portal_membership_terminate_sync (FridaPortalMembership * self, GCancellable * cancellable, GError ** error);

/* RpcClient */
FridaRpcClient * frida_rpc_client_new (FridaRpcPeer * peer);

FridaRpcPeer * frida_rpc_client_get_peer (FridaRpcClient * self);

void frida_rpc_client_call (FridaRpcClient * self, const gchar * method, JsonNode ** args, gint args_length, GBytes * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
JsonNode * frida_rpc_client_call_finish (FridaRpcClient * self, GAsyncResult * result, GError ** error);
gboolean frida_rpc_client_try_handle_message (FridaRpcClient * self, const gchar * json);

/* RpcPeer */
struct _FridaRpcPeerIface {
  GTypeInterface parent_iface;
  void (* post_rpc_message) (FridaRpcPeer * self, const gchar * json, GBytes * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  void (* post_rpc_message_finish) (FridaRpcPeer * self, GAsyncResult * result, GError ** error);
};

void frida_rpc_peer_post_rpc_message (FridaRpcPeer * self, const gchar * json, GBytes * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_rpc_peer_post_rpc_message_finish (FridaRpcPeer * self, GAsyncResult * result, GError ** error);

/* Injector */
struct _FridaInjectorIface {
  GTypeInterface parent_iface;
  void (* close) (FridaInjector * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  void (* close_finish) (FridaInjector * self, GAsyncResult * result, GError ** error);
  void (* inject_library_file) (FridaInjector * self, guint pid, const gchar * path, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  guint (* inject_library_file_finish) (FridaInjector * self, GAsyncResult * result, GError ** error);
  void (* inject_library_blob) (FridaInjector * self, guint pid, GBytes * blob, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  guint (* inject_library_blob_finish) (FridaInjector * self, GAsyncResult * result, GError ** error);
  void (* demonitor) (FridaInjector * self, guint id, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  void (* demonitor_finish) (FridaInjector * self, GAsyncResult * result, GError ** error);
  void (* demonitor_and_clone_state) (FridaInjector * self, guint id, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  guint (* demonitor_and_clone_state_finish) (FridaInjector * self, GAsyncResult * result, GError ** error);
  void (* recreate_thread) (FridaInjector * self, guint pid, guint id, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  void (* recreate_thread_finish) (FridaInjector * self, GAsyncResult * result, GError ** error);
};

FridaInjector * frida_injector_new (void);
FridaInjector * frida_injector_new_inprocess (void);

void frida_injector_close (FridaInjector * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_injector_close_finish (FridaInjector * self, GAsyncResult * result, GError ** error);
void frida_injector_close_sync (FridaInjector * self, GCancellable * cancellable, GError ** error);
void frida_injector_inject_library_file (FridaInjector * self, guint pid, const gchar * path, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
guint frida_injector_inject_library_file_finish (FridaInjector * self, GAsyncResult * result, GError ** error);
guint frida_injector_inject_library_file_sync (FridaInjector * self, guint pid, const gchar * path, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GError ** error);
void frida_injector_inject_library_blob (FridaInjector * self, guint pid, GBytes * blob, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
guint frida_injector_inject_library_blob_finish (FridaInjector * self, GAsyncResult * result, GError ** error);
guint frida_injector_inject_library_blob_sync (FridaInjector * self, guint pid, GBytes * blob, const gchar * entrypoint, const gchar * data, GCancellable * cancellable, GError ** error);
void frida_injector_demonitor (FridaInjector * self, guint id, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_injector_demonitor_finish (FridaInjector * self, GAsyncResult * result, GError ** error);
void frida_injector_demonitor_sync (FridaInjector * self, guint id, GCancellable * cancellable, GError ** error);
void frida_injector_demonitor_and_clone_state (FridaInjector * self, guint id, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
guint frida_injector_demonitor_and_clone_state_finish (FridaInjector * self, GAsyncResult * result, GError ** error);
guint frida_injector_demonitor_and_clone_state_sync (FridaInjector * self, guint id, GCancellable * cancellable, GError ** error);
void frida_injector_recreate_thread (FridaInjector * self, guint pid, guint id, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_injector_recreate_thread_finish (FridaInjector * self, GAsyncResult * result, GError ** error);
void frida_injector_recreate_thread_sync (FridaInjector * self, guint pid, guint id, GCancellable * cancellable, GError ** error);

/* PackageManager */
FridaPackageManager * frida_package_manager_new (void);

const gchar * frida_package_manager_get_registry (FridaPackageManager * self);

void frida_package_manager_search (FridaPackageManager * self, const gchar * query, FridaPackageSearchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaPackageSearchResult * frida_package_manager_search_finish (FridaPackageManager * self, GAsyncResult * result, GError ** error);
FridaPackageSearchResult * frida_package_manager_search_sync (FridaPackageManager * self, const gchar * query, FridaPackageSearchOptions * options, GCancellable * cancellable, GError ** error);
void frida_package_manager_install (FridaPackageManager * self, FridaPackageInstallOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
FridaPackageInstallResult * frida_package_manager_install_finish (FridaPackageManager * self, GAsyncResult * result, GError ** error);
FridaPackageInstallResult * frida_package_manager_install_sync (FridaPackageManager * self, FridaPackageInstallOptions * options, GCancellable * cancellable, GError ** error);
void frida_package_manager_set_registry (FridaPackageManager * self, const gchar * value);

/* Package */
const gchar * frida_package_get_name (FridaPackage * self);
const gchar * frida_package_get_version (FridaPackage * self);
const gchar * frida_package_get_description (FridaPackage * self);
const gchar * frida_package_get_url (FridaPackage * self);

/* PackageList */
gint frida_package_list_size (FridaPackageList * self);
FridaPackage * frida_package_list_get (FridaPackageList * self, gint index);

/* PackageSearchOptions */
FridaPackageSearchOptions * frida_package_search_options_new (void);

guint frida_package_search_options_get_offset (FridaPackageSearchOptions * self);
guint frida_package_search_options_get_limit (FridaPackageSearchOptions * self);

void frida_package_search_options_set_offset (FridaPackageSearchOptions * self, guint value);
void frida_package_search_options_set_limit (FridaPackageSearchOptions * self, guint value);

/* PackageSearchResult */
FridaPackageList * frida_package_search_result_get_packages (FridaPackageSearchResult * self);
guint frida_package_search_result_get_total (FridaPackageSearchResult * self);

/* PackageInstallOptions */
FridaPackageInstallOptions * frida_package_install_options_new (void);

const gchar * frida_package_install_options_get_project_root (FridaPackageInstallOptions * self);
FridaPackageRole frida_package_install_options_get_role (FridaPackageInstallOptions * self);

void frida_package_install_options_clear_specs (FridaPackageInstallOptions * self);
void frida_package_install_options_add_spec (FridaPackageInstallOptions * self, const gchar * spec);
void frida_package_install_options_clear_omits (FridaPackageInstallOptions * self);
void frida_package_install_options_add_omit (FridaPackageInstallOptions * self, FridaPackageRole role);
void frida_package_install_options_set_project_root (FridaPackageInstallOptions * self, const gchar * value);
void frida_package_install_options_set_role (FridaPackageInstallOptions * self, FridaPackageRole value);

/* PackageInstallResult */
FridaPackageList * frida_package_install_result_get_packages (FridaPackageInstallResult * self);

/* ControlService */
FridaControlService * frida_control_service_new (FridaEndpointParameters * endpoint_params, FridaControlServiceOptions * options, GError ** error);
FridaControlService * frida_control_service_new_with_device_finish (GAsyncResult * result, GError ** error);

FridaEndpointParameters * frida_control_service_get_endpoint_params (FridaControlService * self);
FridaControlServiceOptions * frida_control_service_get_options (FridaControlService * self);

FridaControlService * frida_control_service_construct_with_device_finish (GAsyncResult * result, GError ** error);
void frida_control_service_start (FridaControlService * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_control_service_start_finish (FridaControlService * self, GAsyncResult * result, GError ** error);
void frida_control_service_start_sync (FridaControlService * self, GCancellable * cancellable, GError ** error);
void frida_control_service_stop (FridaControlService * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_control_service_stop_finish (FridaControlService * self, GAsyncResult * result, GError ** error);
void frida_control_service_stop_sync (FridaControlService * self, GCancellable * cancellable, GError ** error);

/* ControlServiceOptions */
FridaControlServiceOptions * frida_control_service_options_new (void);

const gchar * frida_control_service_options_get_sysroot (FridaControlServiceOptions * self);
gboolean frida_control_service_options_get_enable_preload (FridaControlServiceOptions * self);
gboolean frida_control_service_options_get_report_crashes (FridaControlServiceOptions * self);

void frida_control_service_options_set_sysroot (FridaControlServiceOptions * self, const gchar * value);
void frida_control_service_options_set_enable_preload (FridaControlServiceOptions * self, gboolean value);
void frida_control_service_options_set_report_crashes (FridaControlServiceOptions * self, gboolean value);

/* PortalService */
FridaPortalService * frida_portal_service_new (FridaEndpointParameters * cluster_params, FridaEndpointParameters * control_params);

FridaDevice * frida_portal_service_get_device (FridaPortalService * self);
FridaEndpointParameters * frida_portal_service_get_cluster_params (FridaPortalService * self);
FridaEndpointParameters * frida_portal_service_get_control_params (FridaPortalService * self);

void frida_portal_service_start (FridaPortalService * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_portal_service_start_finish (FridaPortalService * self, GAsyncResult * result, GError ** error);
void frida_portal_service_start_sync (FridaPortalService * self, GCancellable * cancellable, GError ** error);
void frida_portal_service_stop (FridaPortalService * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_portal_service_stop_finish (FridaPortalService * self, GAsyncResult * result, GError ** error);
void frida_portal_service_stop_sync (FridaPortalService * self, GCancellable * cancellable, GError ** error);
void frida_portal_service_kick (FridaPortalService * self, guint connection_id);
void frida_portal_service_post (FridaPortalService * self, guint connection_id, const gchar * json, GBytes * data);
void frida_portal_service_narrowcast (FridaPortalService * self, const gchar * tag, const gchar * json, GBytes * data);
void frida_portal_service_broadcast (FridaPortalService * self, const gchar * json, GBytes * data);
gchar ** frida_portal_service_enumerate_tags (FridaPortalService * self, guint connection_id, gint * result_length);
void frida_portal_service_tag (FridaPortalService * self, guint connection_id, const gchar * tag);
void frida_portal_service_untag (FridaPortalService * self, guint connection_id, const gchar * tag);

/* EndpointParameters */
FridaEndpointParameters * frida_endpoint_parameters_new (const gchar * address, guint16 port, GTlsCertificate * certificate, const gchar * origin, FridaAuthenticationService * auth_service, GFile * asset_root);

const gchar * frida_endpoint_parameters_get_address (FridaEndpointParameters * self);
guint16 frida_endpoint_parameters_get_port (FridaEndpointParameters * self);
GTlsCertificate * frida_endpoint_parameters_get_certificate (FridaEndpointParameters * self);
const gchar * frida_endpoint_parameters_get_origin (FridaEndpointParameters * self);
FridaAuthenticationService * frida_endpoint_parameters_get_auth_service (FridaEndpointParameters * self);
GFile * frida_endpoint_parameters_get_asset_root (FridaEndpointParameters * self);

void frida_endpoint_parameters_set_asset_root (FridaEndpointParameters * self, GFile * value);

/* AuthenticationService */
struct _FridaAuthenticationServiceIface {
  GTypeInterface parent_iface;
  void (* authenticate) (FridaAuthenticationService * self, const gchar * token, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
  gchar * (* authenticate_finish) (FridaAuthenticationService * self, GAsyncResult * result, GError ** error);
};

void frida_authentication_service_authenticate (FridaAuthenticationService * self, const gchar * token, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
gchar * frida_authentication_service_authenticate_finish (FridaAuthenticationService * self, GAsyncResult * result, GError ** error);

/* StaticAuthenticationService */
FridaStaticAuthenticationService * frida_static_authentication_service_new (const gchar * token);

const gchar * frida_static_authentication_service_get_token_hash (FridaStaticAuthenticationService * self);

/* FileMonitor */
FridaFileMonitor * frida_file_monitor_new (const gchar * path);

const gchar * frida_file_monitor_get_path (FridaFileMonitor * self);

void frida_file_monitor_enable (FridaFileMonitor * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_file_monitor_enable_finish (FridaFileMonitor * self, GAsyncResult * result, GError ** error);
void frida_file_monitor_enable_sync (FridaFileMonitor * self, GCancellable * cancellable, GError ** error);
void frida_file_monitor_disable (FridaFileMonitor * self, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_file_monitor_disable_finish (FridaFileMonitor * self, GAsyncResult * result, GError ** error);
void frida_file_monitor_disable_sync (FridaFileMonitor * self, GCancellable * cancellable, GError ** error);

/* Compiler */
FridaCompiler * frida_compiler_new (FridaDeviceManager * manager);

void frida_compiler_build (FridaCompiler * self, const gchar * entrypoint, FridaBuildOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
gchar * frida_compiler_build_finish (FridaCompiler * self, GAsyncResult * result, GError ** error);
gchar * frida_compiler_build_sync (FridaCompiler * self, const gchar * entrypoint, FridaBuildOptions * options, GCancellable * cancellable, GError ** error);
void frida_compiler_watch (FridaCompiler * self, const gchar * entrypoint, FridaWatchOptions * options, GCancellable * cancellable, GAsyncReadyCallback callback, gpointer user_data);
void frida_compiler_watch_finish (FridaCompiler * self, GAsyncResult * result, GError ** error);
void frida_compiler_watch_sync (FridaCompiler * self, const gchar * entrypoint, FridaWatchOptions * options, GCancellable * cancellable, GError ** error);
void frida_compiler_schedule_on_frida_thread (FridaCompiler * self, GSourceFunc function, gpointer function_target, GDestroyNotify function_target_destroy_notify);

/* CompilerOptions */
FridaCompilerOptions * frida_compiler_options_new (void);

const gchar * frida_compiler_options_get_project_root (FridaCompilerOptions * self);
FridaOutputFormat frida_compiler_options_get_output_format (FridaCompilerOptions * self);
FridaBundleFormat frida_compiler_options_get_bundle_format (FridaCompilerOptions * self);
FridaTypeCheckMode frida_compiler_options_get_type_check (FridaCompilerOptions * self);
FridaSourceMaps frida_compiler_options_get_source_maps (FridaCompilerOptions * self);
FridaJsCompression frida_compiler_options_get_compression (FridaCompilerOptions * self);

void frida_compiler_options_set_project_root (FridaCompilerOptions * self, const gchar * value);
void frida_compiler_options_set_output_format (FridaCompilerOptions * self, FridaOutputFormat value);
void frida_compiler_options_set_bundle_format (FridaCompilerOptions * self, FridaBundleFormat value);
void frida_compiler_options_set_type_check (FridaCompilerOptions * self, FridaTypeCheckMode value);
void frida_compiler_options_set_source_maps (FridaCompilerOptions * self, FridaSourceMaps value);
void frida_compiler_options_set_compression (FridaCompilerOptions * self, FridaJsCompression value);

/* BuildOptions */
FridaBuildOptions * frida_build_options_new (void);

/* WatchOptions */
FridaWatchOptions * frida_watch_options_new (void);

/* Toplevel functions */

/* Errors */
GQuark frida_error_quark (void);

typedef enum {
  FRIDA_ERROR_SERVER_NOT_RUNNING,
  FRIDA_ERROR_EXECUTABLE_NOT_FOUND,
  FRIDA_ERROR_EXECUTABLE_NOT_SUPPORTED,
  FRIDA_ERROR_PROCESS_NOT_FOUND,
  FRIDA_ERROR_PROCESS_NOT_RESPONDING,
  FRIDA_ERROR_INVALID_ARGUMENT,
  FRIDA_ERROR_INVALID_OPERATION,
  FRIDA_ERROR_PERMISSION_DENIED,
  FRIDA_ERROR_ADDRESS_IN_USE,
  FRIDA_ERROR_TIMED_OUT,
  FRIDA_ERROR_NOT_SUPPORTED,
  FRIDA_ERROR_PROTOCOL,
  FRIDA_ERROR_TRANSPORT
} FridaError;

/* GTypes */
GType frida_runtime_get_type (void) G_GNUC_CONST;
GType frida_device_type_get_type (void) G_GNUC_CONST;
GType frida_package_install_phase_get_type (void) G_GNUC_CONST;
GType frida_package_role_get_type (void) G_GNUC_CONST;
GType frida_output_format_get_type (void) G_GNUC_CONST;
GType frida_bundle_format_get_type (void) G_GNUC_CONST;
GType frida_type_check_mode_get_type (void) G_GNUC_CONST;
GType frida_source_maps_get_type (void) G_GNUC_CONST;
GType frida_js_compression_get_type (void) G_GNUC_CONST;
GType frida_child_origin_get_type (void) G_GNUC_CONST;
GType frida_realm_get_type (void) G_GNUC_CONST;
GType frida_relay_kind_get_type (void) G_GNUC_CONST;
GType frida_scope_get_type (void) G_GNUC_CONST;
GType frida_script_runtime_get_type (void) G_GNUC_CONST;
GType frida_session_detach_reason_get_type (void) G_GNUC_CONST;
GType frida_snapshot_transport_get_type (void) G_GNUC_CONST;
GType frida_stdio_get_type (void) G_GNUC_CONST;
GType frida_device_manager_get_type (void) G_GNUC_CONST;
GType frida_device_list_get_type (void) G_GNUC_CONST;
GType frida_device_get_type (void) G_GNUC_CONST;
GType frida_remote_device_options_get_type (void) G_GNUC_CONST;
GType frida_application_list_get_type (void) G_GNUC_CONST;
GType frida_application_get_type (void) G_GNUC_CONST;
GType frida_process_list_get_type (void) G_GNUC_CONST;
GType frida_process_get_type (void) G_GNUC_CONST;
GType frida_process_match_options_get_type (void) G_GNUC_CONST;
GType frida_spawn_options_get_type (void) G_GNUC_CONST;
GType frida_frontmost_query_options_get_type (void) G_GNUC_CONST;
GType frida_application_query_options_get_type (void) G_GNUC_CONST;
GType frida_process_query_options_get_type (void) G_GNUC_CONST;
GType frida_session_options_get_type (void) G_GNUC_CONST;
GType frida_spawn_list_get_type (void) G_GNUC_CONST;
GType frida_spawn_get_type (void) G_GNUC_CONST;
GType frida_child_list_get_type (void) G_GNUC_CONST;
GType frida_child_get_type (void) G_GNUC_CONST;
GType frida_crash_get_type (void) G_GNUC_CONST;
GType frida_bus_get_type (void) G_GNUC_CONST;
GType frida_service_get_type (void) G_GNUC_CONST;
GType frida_session_get_type (void) G_GNUC_CONST;
GType frida_script_get_type (void) G_GNUC_CONST;
GType frida_snapshot_options_get_type (void) G_GNUC_CONST;
GType frida_script_options_get_type (void) G_GNUC_CONST;
GType frida_peer_options_get_type (void) G_GNUC_CONST;
GType frida_relay_get_type (void) G_GNUC_CONST;
GType frida_portal_options_get_type (void) G_GNUC_CONST;
GType frida_portal_membership_get_type (void) G_GNUC_CONST;
GType frida_rpc_client_get_type (void) G_GNUC_CONST;
GType frida_rpc_peer_get_type (void) G_GNUC_CONST;
GType frida_injector_get_type (void) G_GNUC_CONST;
GType frida_package_manager_get_type (void) G_GNUC_CONST;
GType frida_package_get_type (void) G_GNUC_CONST;
GType frida_package_list_get_type (void) G_GNUC_CONST;
GType frida_package_search_options_get_type (void) G_GNUC_CONST;
GType frida_package_search_result_get_type (void) G_GNUC_CONST;
GType frida_package_install_options_get_type (void) G_GNUC_CONST;
GType frida_package_install_result_get_type (void) G_GNUC_CONST;
GType frida_control_service_get_type (void) G_GNUC_CONST;
GType frida_control_service_options_get_type (void) G_GNUC_CONST;
GType frida_portal_service_get_type (void) G_GNUC_CONST;
GType frida_endpoint_parameters_get_type (void) G_GNUC_CONST;
GType frida_authentication_service_get_type (void) G_GNUC_CONST;
GType frida_static_authentication_service_get_type (void) G_GNUC_CONST;
GType frida_file_monitor_get_type (void) G_GNUC_CONST;
GType frida_compiler_get_type (void) G_GNUC_CONST;
GType frida_compiler_options_get_type (void) G_GNUC_CONST;
GType frida_build_options_get_type (void) G_GNUC_CONST;
GType frida_watch_options_get_type (void) G_GNUC_CONST;

/* Macros */
#define FRIDA_TYPE_RUNTIME (frida_runtime_get_type ())

#define FRIDA_TYPE_DEVICE_TYPE (frida_device_type_get_type ())

#define FRIDA_TYPE_PACKAGE_INSTALL_PHASE (frida_package_install_phase_get_type ())

#define FRIDA_TYPE_PACKAGE_ROLE (frida_package_role_get_type ())

#define FRIDA_TYPE_OUTPUT_FORMAT (frida_output_format_get_type ())

#define FRIDA_TYPE_BUNDLE_FORMAT (frida_bundle_format_get_type ())

#define FRIDA_TYPE_TYPE_CHECK_MODE (frida_type_check_mode_get_type ())

#define FRIDA_TYPE_SOURCE_MAPS (frida_source_maps_get_type ())

#define FRIDA_TYPE_JS_COMPRESSION (frida_js_compression_get_type ())

#define FRIDA_TYPE_CHILD_ORIGIN (frida_child_origin_get_type ())

#define FRIDA_TYPE_REALM (frida_realm_get_type ())

#define FRIDA_TYPE_RELAY_KIND (frida_relay_kind_get_type ())

#define FRIDA_TYPE_SCOPE (frida_scope_get_type ())

#define FRIDA_TYPE_SCRIPT_RUNTIME (frida_script_runtime_get_type ())

#define FRIDA_TYPE_SESSION_DETACH_REASON (frida_session_detach_reason_get_type ())

#define FRIDA_TYPE_SNAPSHOT_TRANSPORT (frida_snapshot_transport_get_type ())

#define FRIDA_TYPE_STDIO (frida_stdio_get_type ())

#define FRIDA_TYPE_DEVICE_MANAGER (frida_device_manager_get_type ())
#define FRIDA_DEVICE_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_DEVICE_MANAGER, FridaDeviceManager))
#define FRIDA_IS_DEVICE_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_DEVICE_MANAGER))

#define FRIDA_TYPE_DEVICE_LIST (frida_device_list_get_type ())
#define FRIDA_DEVICE_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_DEVICE_LIST, FridaDeviceList))
#define FRIDA_IS_DEVICE_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_DEVICE_LIST))

#define FRIDA_TYPE_DEVICE (frida_device_get_type ())
#define FRIDA_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_DEVICE, FridaDevice))
#define FRIDA_IS_DEVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_DEVICE))

#define FRIDA_TYPE_REMOTE_DEVICE_OPTIONS (frida_remote_device_options_get_type ())
#define FRIDA_REMOTE_DEVICE_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_REMOTE_DEVICE_OPTIONS, FridaRemoteDeviceOptions))
#define FRIDA_IS_REMOTE_DEVICE_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_REMOTE_DEVICE_OPTIONS))

#define FRIDA_TYPE_APPLICATION_LIST (frida_application_list_get_type ())
#define FRIDA_APPLICATION_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_APPLICATION_LIST, FridaApplicationList))
#define FRIDA_IS_APPLICATION_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_APPLICATION_LIST))

#define FRIDA_TYPE_APPLICATION (frida_application_get_type ())
#define FRIDA_APPLICATION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_APPLICATION, FridaApplication))
#define FRIDA_IS_APPLICATION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_APPLICATION))

#define FRIDA_TYPE_PROCESS_LIST (frida_process_list_get_type ())
#define FRIDA_PROCESS_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PROCESS_LIST, FridaProcessList))
#define FRIDA_IS_PROCESS_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PROCESS_LIST))

#define FRIDA_TYPE_PROCESS (frida_process_get_type ())
#define FRIDA_PROCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PROCESS, FridaProcess))
#define FRIDA_IS_PROCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PROCESS))

#define FRIDA_TYPE_PROCESS_MATCH_OPTIONS (frida_process_match_options_get_type ())
#define FRIDA_PROCESS_MATCH_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PROCESS_MATCH_OPTIONS, FridaProcessMatchOptions))
#define FRIDA_IS_PROCESS_MATCH_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PROCESS_MATCH_OPTIONS))

#define FRIDA_TYPE_SPAWN_OPTIONS (frida_spawn_options_get_type ())
#define FRIDA_SPAWN_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SPAWN_OPTIONS, FridaSpawnOptions))
#define FRIDA_IS_SPAWN_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SPAWN_OPTIONS))

#define FRIDA_TYPE_FRONTMOST_QUERY_OPTIONS (frida_frontmost_query_options_get_type ())
#define FRIDA_FRONTMOST_QUERY_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_FRONTMOST_QUERY_OPTIONS, FridaFrontmostQueryOptions))
#define FRIDA_IS_FRONTMOST_QUERY_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_FRONTMOST_QUERY_OPTIONS))

#define FRIDA_TYPE_APPLICATION_QUERY_OPTIONS (frida_application_query_options_get_type ())
#define FRIDA_APPLICATION_QUERY_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_APPLICATION_QUERY_OPTIONS, FridaApplicationQueryOptions))
#define FRIDA_IS_APPLICATION_QUERY_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_APPLICATION_QUERY_OPTIONS))

#define FRIDA_TYPE_PROCESS_QUERY_OPTIONS (frida_process_query_options_get_type ())
#define FRIDA_PROCESS_QUERY_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PROCESS_QUERY_OPTIONS, FridaProcessQueryOptions))
#define FRIDA_IS_PROCESS_QUERY_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PROCESS_QUERY_OPTIONS))

#define FRIDA_TYPE_SESSION_OPTIONS (frida_session_options_get_type ())
#define FRIDA_SESSION_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SESSION_OPTIONS, FridaSessionOptions))
#define FRIDA_IS_SESSION_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SESSION_OPTIONS))

#define FRIDA_TYPE_SPAWN_LIST (frida_spawn_list_get_type ())
#define FRIDA_SPAWN_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SPAWN_LIST, FridaSpawnList))
#define FRIDA_IS_SPAWN_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SPAWN_LIST))

#define FRIDA_TYPE_SPAWN (frida_spawn_get_type ())
#define FRIDA_SPAWN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SPAWN, FridaSpawn))
#define FRIDA_IS_SPAWN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SPAWN))

#define FRIDA_TYPE_CHILD_LIST (frida_child_list_get_type ())
#define FRIDA_CHILD_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_CHILD_LIST, FridaChildList))
#define FRIDA_IS_CHILD_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_CHILD_LIST))

#define FRIDA_TYPE_CHILD (frida_child_get_type ())
#define FRIDA_CHILD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_CHILD, FridaChild))
#define FRIDA_IS_CHILD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_CHILD))

#define FRIDA_TYPE_CRASH (frida_crash_get_type ())
#define FRIDA_CRASH(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_CRASH, FridaCrash))
#define FRIDA_IS_CRASH(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_CRASH))

#define FRIDA_TYPE_BUS (frida_bus_get_type ())
#define FRIDA_BUS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_BUS, FridaBus))
#define FRIDA_IS_BUS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_BUS))

#define FRIDA_TYPE_SERVICE (frida_service_get_type ())
#define FRIDA_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SERVICE, FridaService))
#define FRIDA_IS_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SERVICE))

#define FRIDA_TYPE_SESSION (frida_session_get_type ())
#define FRIDA_SESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SESSION, FridaSession))
#define FRIDA_IS_SESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SESSION))

#define FRIDA_TYPE_SCRIPT (frida_script_get_type ())
#define FRIDA_SCRIPT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SCRIPT, FridaScript))
#define FRIDA_IS_SCRIPT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SCRIPT))

#define FRIDA_TYPE_SNAPSHOT_OPTIONS (frida_snapshot_options_get_type ())
#define FRIDA_SNAPSHOT_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SNAPSHOT_OPTIONS, FridaSnapshotOptions))
#define FRIDA_IS_SNAPSHOT_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SNAPSHOT_OPTIONS))

#define FRIDA_TYPE_SCRIPT_OPTIONS (frida_script_options_get_type ())
#define FRIDA_SCRIPT_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_SCRIPT_OPTIONS, FridaScriptOptions))
#define FRIDA_IS_SCRIPT_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_SCRIPT_OPTIONS))

#define FRIDA_TYPE_PEER_OPTIONS (frida_peer_options_get_type ())
#define FRIDA_PEER_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PEER_OPTIONS, FridaPeerOptions))
#define FRIDA_IS_PEER_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PEER_OPTIONS))

#define FRIDA_TYPE_RELAY (frida_relay_get_type ())
#define FRIDA_RELAY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_RELAY, FridaRelay))
#define FRIDA_IS_RELAY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_RELAY))

#define FRIDA_TYPE_PORTAL_OPTIONS (frida_portal_options_get_type ())
#define FRIDA_PORTAL_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PORTAL_OPTIONS, FridaPortalOptions))
#define FRIDA_IS_PORTAL_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PORTAL_OPTIONS))

#define FRIDA_TYPE_PORTAL_MEMBERSHIP (frida_portal_membership_get_type ())
#define FRIDA_PORTAL_MEMBERSHIP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PORTAL_MEMBERSHIP, FridaPortalMembership))
#define FRIDA_IS_PORTAL_MEMBERSHIP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PORTAL_MEMBERSHIP))

#define FRIDA_TYPE_RPC_CLIENT (frida_rpc_client_get_type ())
#define FRIDA_RPC_CLIENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_RPC_CLIENT, FridaRpcClient))
#define FRIDA_IS_RPC_CLIENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_RPC_CLIENT))

#define FRIDA_TYPE_RPC_PEER (frida_rpc_peer_get_type ())
#define FRIDA_RPC_PEER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_RPC_PEER, FridaRpcPeer))
#define FRIDA_IS_RPC_PEER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_RPC_PEER))

#define FRIDA_TYPE_INJECTOR (frida_injector_get_type ())
#define FRIDA_INJECTOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_INJECTOR, FridaInjector))
#define FRIDA_IS_INJECTOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_INJECTOR))

#define FRIDA_TYPE_PACKAGE_MANAGER (frida_package_manager_get_type ())
#define FRIDA_PACKAGE_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE_MANAGER, FridaPackageManager))
#define FRIDA_IS_PACKAGE_MANAGER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE_MANAGER))

#define FRIDA_TYPE_PACKAGE (frida_package_get_type ())
#define FRIDA_PACKAGE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE, FridaPackage))
#define FRIDA_IS_PACKAGE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE))

#define FRIDA_TYPE_PACKAGE_LIST (frida_package_list_get_type ())
#define FRIDA_PACKAGE_LIST(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE_LIST, FridaPackageList))
#define FRIDA_IS_PACKAGE_LIST(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE_LIST))

#define FRIDA_TYPE_PACKAGE_SEARCH_OPTIONS (frida_package_search_options_get_type ())
#define FRIDA_PACKAGE_SEARCH_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE_SEARCH_OPTIONS, FridaPackageSearchOptions))
#define FRIDA_IS_PACKAGE_SEARCH_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE_SEARCH_OPTIONS))

#define FRIDA_TYPE_PACKAGE_SEARCH_RESULT (frida_package_search_result_get_type ())
#define FRIDA_PACKAGE_SEARCH_RESULT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE_SEARCH_RESULT, FridaPackageSearchResult))
#define FRIDA_IS_PACKAGE_SEARCH_RESULT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE_SEARCH_RESULT))

#define FRIDA_TYPE_PACKAGE_INSTALL_OPTIONS (frida_package_install_options_get_type ())
#define FRIDA_PACKAGE_INSTALL_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE_INSTALL_OPTIONS, FridaPackageInstallOptions))
#define FRIDA_IS_PACKAGE_INSTALL_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE_INSTALL_OPTIONS))

#define FRIDA_TYPE_PACKAGE_INSTALL_RESULT (frida_package_install_result_get_type ())
#define FRIDA_PACKAGE_INSTALL_RESULT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PACKAGE_INSTALL_RESULT, FridaPackageInstallResult))
#define FRIDA_IS_PACKAGE_INSTALL_RESULT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PACKAGE_INSTALL_RESULT))

#define FRIDA_TYPE_CONTROL_SERVICE (frida_control_service_get_type ())
#define FRIDA_CONTROL_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_CONTROL_SERVICE, FridaControlService))
#define FRIDA_IS_CONTROL_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_CONTROL_SERVICE))

#define FRIDA_TYPE_CONTROL_SERVICE_OPTIONS (frida_control_service_options_get_type ())
#define FRIDA_CONTROL_SERVICE_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_CONTROL_SERVICE_OPTIONS, FridaControlServiceOptions))
#define FRIDA_IS_CONTROL_SERVICE_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_CONTROL_SERVICE_OPTIONS))

#define FRIDA_TYPE_PORTAL_SERVICE (frida_portal_service_get_type ())
#define FRIDA_PORTAL_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_PORTAL_SERVICE, FridaPortalService))
#define FRIDA_IS_PORTAL_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_PORTAL_SERVICE))

#define FRIDA_TYPE_ENDPOINT_PARAMETERS (frida_endpoint_parameters_get_type ())
#define FRIDA_ENDPOINT_PARAMETERS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_ENDPOINT_PARAMETERS, FridaEndpointParameters))
#define FRIDA_IS_ENDPOINT_PARAMETERS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_ENDPOINT_PARAMETERS))

#define FRIDA_TYPE_AUTHENTICATION_SERVICE (frida_authentication_service_get_type ())
#define FRIDA_AUTHENTICATION_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_AUTHENTICATION_SERVICE, FridaAuthenticationService))
#define FRIDA_IS_AUTHENTICATION_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_AUTHENTICATION_SERVICE))

#define FRIDA_TYPE_STATIC_AUTHENTICATION_SERVICE (frida_static_authentication_service_get_type ())
#define FRIDA_STATIC_AUTHENTICATION_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_STATIC_AUTHENTICATION_SERVICE, FridaStaticAuthenticationService))
#define FRIDA_IS_STATIC_AUTHENTICATION_SERVICE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_STATIC_AUTHENTICATION_SERVICE))

#define FRIDA_TYPE_FILE_MONITOR (frida_file_monitor_get_type ())
#define FRIDA_FILE_MONITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_FILE_MONITOR, FridaFileMonitor))
#define FRIDA_IS_FILE_MONITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_FILE_MONITOR))

#define FRIDA_TYPE_COMPILER (frida_compiler_get_type ())
#define FRIDA_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_COMPILER, FridaCompiler))
#define FRIDA_IS_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_COMPILER))

#define FRIDA_TYPE_COMPILER_OPTIONS (frida_compiler_options_get_type ())
#define FRIDA_COMPILER_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_COMPILER_OPTIONS, FridaCompilerOptions))
#define FRIDA_IS_COMPILER_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_COMPILER_OPTIONS))

#define FRIDA_TYPE_BUILD_OPTIONS (frida_build_options_get_type ())
#define FRIDA_BUILD_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_BUILD_OPTIONS, FridaBuildOptions))
#define FRIDA_IS_BUILD_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_BUILD_OPTIONS))

#define FRIDA_TYPE_WATCH_OPTIONS (frida_watch_options_get_type ())
#define FRIDA_WATCH_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), FRIDA_TYPE_WATCH_OPTIONS, FridaWatchOptions))
#define FRIDA_IS_WATCH_OPTIONS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), FRIDA_TYPE_WATCH_OPTIONS))

#define FRIDA_ERROR (frida_error_quark ())

G_END_DECLS

#endif
